/************************************************************************/
/*									*/
/*  A find/replace tool.						*/
/*									*/
/************************************************************************/

#   include	"appFrameConfig.h"

#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<stddef.h>
#   include	<appDebugon.h>

#   include	<reg.h>

#   include	"appFrame.h"

#   ifdef USE_MOTIF
#   include	<X11/keysym.h>
#   endif

/************************************************************************/
/*									*/
/*  Represents a find tool.						*/
/*									*/
/************************************************************************/

#   define	FILEL	400

typedef struct AppFindToolResources
    {
    char *	aftrFindTitle;
    char *	aftrFindNext;

    char *	aftrUseRegex;

    char *	aftrFindPrevious;

    char *	aftrReplaceTitle;
    char *	aftrReplaceFound;
    char *	aftrReplaceNext;
    } AppFindToolResources;

typedef struct AppFindTool
    {
    APP_WIDGET			aftTopWidget;
    APP_WIDGET			aftMainWidget;

    APP_WIDGET			aftPatternText;
    APP_WIDGET			aftReplaceText;
    APP_WIDGET			aftReplaceFrame;

    APP_WIDGET			aftReplaceButton;
    APP_WIDGET			aftReplaceNextButton;
    APP_WIDGET			aftFindNextButton;
    APP_WIDGET			aftFindPrevButton;

    APP_WIDGET			aftRegexToggle;

    AppToolDestroy		aftDestroy;
    FindToolFind		aftFindNext;
    FindToolFind		aftFindPrev;
    FindToolReplace		aftReplace;

    void *			aftTarget;

    int				aftVisible;
    int				aftLastFindResult;
    int				aftUseRegex;
    } AppFindTool;

/************************************************************************/
/*									*/
/*  Turn buttons on and off depending on the situation.			*/
/*									*/
/************************************************************************/

static void appFindToolReflectPatternText(	AppFindTool *	aft )
    {
    int		sensitive;
    char *	pattern;

    pattern= appGetStringFromTextWidget( aft->aftPatternText );

    sensitive= pattern[0] != '\0';

    appFreeStringFromTextWidget( pattern );

    appGuiEnableWidget( aft->aftFindNextButton, sensitive );
    appGuiEnableWidget( aft->aftFindPrevButton, sensitive );

    return;
    }

static void appFindToolReflectReplaceText(	AppFindTool *	aft )
    {
    int		sensitive;
    char *	replacement;

    replacement= appGetStringFromTextWidget( aft->aftReplaceText );

    sensitive= replacement[0] != '\0' && aft->aftLastFindResult == 0;

    appFreeStringFromTextWidget( replacement );

    appGuiEnableWidget( aft->aftReplaceButton, sensitive );
    appGuiEnableWidget( aft->aftReplaceNextButton, sensitive );

    return;
    }

static void appFindToolSetFindResult(	AppFindTool *	aft,
					int		result )
    {
    int		sensitive;

    aft->aftLastFindResult= result;

    if  ( result )
	{ sensitive= 0;	}
    else{
	char *	replacement;

	appGuiEnableWidget( aft->aftReplaceFrame, 1 );

	replacement= appGetStringFromTextWidget( aft->aftReplaceText );

	sensitive= replacement[0] != '\0';

	appFreeStringFromTextWidget( replacement );
	}

    appGuiEnableWidget( aft->aftReplaceButton, sensitive );
    appGuiEnableWidget( aft->aftReplaceNextButton, sensitive );

    return;
    }

static void appFindToolPatternChanged(	APP_WIDGET	w,
					void *		voidaft,
					void *		call_data )
    {
    AppFindTool *	aft= (AppFindTool *)voidaft;

    appFindToolReflectPatternText( aft );

    return;
    }

static void appFindToolReplacementChanged(	APP_WIDGET	w,
						void *		voidaft,
						void *		call_data )
    {
    AppFindTool *	aft= (AppFindTool *)voidaft;

    appFindToolReflectReplaceText( aft );

    return;
    }

/************************************************************************/
/*									*/
/*  'Find' button has been pushed.					*/
/*									*/
/************************************************************************/

static int appFindToolGetProgram(	regProg **	pProg,
					AppFindTool *	aft )
    {
    char *		pattern;
    regProg *		prog;

    pattern= appGetStringFromTextWidget( aft->aftPatternText );

    if  ( aft->aftUseRegex )
	{
	prog= regCompile( (const unsigned char *)pattern );
	if  ( ! prog )
	    {
	    SXDEB(pattern,prog);
	    appFreeStringFromTextWidget( pattern ); return -1;
	    }
	}
    else{
	prog= regCompileEscaped( (const unsigned char *)pattern );
	if  ( ! prog )
	    {
	    SXDEB(pattern,prog);
	    appFreeStringFromTextWidget( pattern ); return -1;
	    }
	}

    appFreeStringFromTextWidget( pattern );

    *pProg= prog; return 0;
    }

static void appFindToolFindNext(	APP_WIDGET	w,
					void *		voidaft,
					void *		voidpbcs )
    {
    AppFindTool *	aft= (AppFindTool *)voidaft;
    regProg *		prog;
    int			result;

    if  ( ! aft->aftFindNext )
	{
	XDEB(aft->aftFindNext);
	appFindToolSetFindResult( aft, -1 ); return;
	}

    if  ( appFindToolGetProgram( &prog, aft ) )
	{ appFindToolSetFindResult( aft, -1 ); return; }

    result= (*aft->aftFindNext)( aft->aftTarget, prog );
    appFindToolSetFindResult( aft, result );

    free( prog );

    return;
    }

static void appFindToolFindPrev(	APP_WIDGET	w,
					void *		voidaft,
					void *		voidpbcs )
    {
    AppFindTool *	aft= (AppFindTool *)voidaft;
    regProg *		prog;
    int			result;

    if  ( ! aft->aftFindPrev )
	{
	XDEB(aft->aftFindPrev);
	appFindToolSetFindResult( aft, -1 ); return;
	}

    if  ( appFindToolGetProgram( &prog, aft ) )
	{ appFindToolSetFindResult( aft, -1 ); return; }

    result= (*aft->aftFindPrev)( aft->aftTarget, prog );
    appFindToolSetFindResult( aft, result );

    free( prog );

    return;
    }

/************************************************************************/
/*									*/
/*  'Replace' button has been pushed.					*/
/*									*/
/************************************************************************/

static void appFindToolReplaceSelection(	APP_WIDGET	w,
						void *		voidaft,
						void *		voidpbcs )
    {
    AppFindTool *	aft= (AppFindTool *)voidaft;
    char *		replacement;

    if  ( ! aft->aftReplace )
	{ XDEB(aft->aftReplace); return;	}

    replacement= appGetStringFromTextWidget( aft->aftReplaceText );

    (*aft->aftReplace)( aft->aftTarget, (const unsigned char *)replacement );

    appFreeStringFromTextWidget( replacement );

    appGuiEnableWidget( aft->aftReplaceButton, 0 );

    return;
    }

static void appFindToolReplaceNext(	APP_WIDGET	w,
					void *		voidaft,
					void *		voidpbcs )
    {
    appFindToolReplaceSelection( w, voidaft, voidpbcs );

    appFindToolFindNext( w, voidaft, voidpbcs );

    return;
    }

static void appFindRegexToggled(	APP_WIDGET	w,
					void *		voidaft,
					void *		voidtbcs )
    {
    AppFindTool *	aft= (AppFindTool *)voidaft;
    int			set;

    set= appGuiGetToggleStateFromCallback( w, voidtbcs );

    aft->aftUseRegex= ( set != 0 );

    return;
    }

/************************************************************************/
/*									*/
/*  Make a row of buttons for the Find Tool.				*/
/*									*/
/************************************************************************/

static void appFindToolMakeButtonRow(	APP_WIDGET *		pLeftButton,
					APP_WIDGET *		pRightButton,
					char *			leftLabel,
					char *			rightLabel,
					APP_BUTTON_CALLBACK	leftCallback,
					APP_BUTTON_CALLBACK	rightCallback,
					AppFindTool *		aft,
					APP_WIDGET		parent )

    {
    APP_WIDGET		row;

    APP_WIDGET		leftButton;
    APP_WIDGET		rightButton;

    const int		heightResizable= 0;
    const int		showAsDefault= 0;

    row= appMakeRowInColumn( parent, 2, heightResizable );

    appMakeButtonInRow( &(leftButton), row, leftLabel,
			    leftCallback, (void *)aft, 0, showAsDefault );
    appMakeButtonInRow( &(rightButton), row, rightLabel,
			    rightCallback, (void *)aft, 1, showAsDefault );

#   ifdef USE_MOTIF
    XtVaSetValues( XtParent( parent ),
			XmNdefaultButton,	leftButton,
			NULL );
#   endif


    *pLeftButton= leftButton, *pRightButton= rightButton; return;
    }

/************************************************************************/
/*									*/
/*  Make the button part of the find Tool.				*/
/*									*/
/************************************************************************/

static APP_WIDGET appFindMakeFindFrame(	APP_WIDGET		parent,
					AppFindToolResources *	aftr,
					AppFindTool *		aft 	)
    {
    APP_WIDGET		frame;
    APP_WIDGET		paned;
    APP_WIDGET		row;

    const int		textEnabled= 1;
    const int		heightResizable= 0;

    appMakeColumnFrameInColumn( &frame, &paned, parent, aftr->aftrFindTitle );

    appMakeTextInColumn( &(aft->aftPatternText), paned, 0, textEnabled );

    appGuiSetTypingCallbackForText( aft->aftPatternText,
				    appFindToolPatternChanged, (void *)aft );

#   ifdef USE_MOTIF
    XtVaSetValues( XtParent( paned ),
			XmNinitialFocus,	aft->aftPatternText,
			NULL );
#   endif

    /***************/

    row= appMakeRowInColumn( paned, 1, heightResizable );
    aft->aftRegexToggle=
		    appMakeToggleInRow( row, aftr->aftrUseRegex,
					appFindRegexToggled, (void *)aft, 0 );

    /***************/

    appFindToolMakeButtonRow(
		&(aft->aftFindNextButton), &(aft->aftFindPrevButton),
		aftr->aftrFindNext, aftr->aftrFindPrevious,
		appFindToolFindNext, appFindToolFindPrev,
		aft, paned );

    /***************/

    return frame;
    }

/************************************************************************/
/*									*/
/*  Make the replace part of the find tool.				*/
/*									*/
/************************************************************************/

static APP_WIDGET appFindMakeReplaceFrame( APP_WIDGET		parent,
					AppFindToolResources *	aftr,
					AppFindTool *		aft )
    {
    APP_WIDGET		frame;
    APP_WIDGET		paned;

    const int		textEnabled= 1;

    appMakeColumnFrameInColumn( &frame, &paned,
					    parent, aftr->aftrReplaceTitle );
    aft->aftReplaceFrame= frame;

    appMakeTextInColumn( &(aft->aftReplaceText), paned, 0, textEnabled );

    appGuiSetTypingCallbackForText( aft->aftReplaceText,
				appFindToolReplacementChanged, (void *)aft );

#   ifdef USE_MOTIF
    XtVaSetValues( XtParent( paned ),
			XmNinitialFocus,	aft->aftReplaceText,
			NULL );
#   endif

    /***************/

    appFindToolMakeButtonRow(
		&(aft->aftReplaceButton), &(aft->aftReplaceNextButton),
		aftr->aftrReplaceFound, aftr->aftrReplaceNext,
		appFindToolReplaceSelection, appFindToolReplaceNext,
		aft, paned );

    /***************/

    return frame;
    }

/************************************************************************/
/*  A find tool must be destroyed.					*/
/************************************************************************/

static void appDestroyFindTool(	AppFindTool *	aft )
    {
    if  ( aft->aftDestroy )
	{ (*aft->aftDestroy)( aft->aftTarget );	}

    appDestroyShellWidget( aft->aftTopWidget );

    free( aft );
    }

static APP_CLOSE_CALLBACK( appCloseFindTool, w, voidaft )
    {
    AppFindTool *	aft= (AppFindTool *)voidaft;

    appDestroyFindTool( aft );

    return;
    }

#   ifdef USE_MOTIF
static void appFindToolCancel(		APP_WIDGET		w,
					void *			voidaft,
					XEvent *		event,
					Boolean *		pRefused )
    {
    if  ( event->type == KeyRelease )
	{
	XKeyEvent *		kevent= &(event->xkey);
	AppFindTool *		aft= (AppFindTool *)voidaft;

	if  ( XKeysymToKeycode( XtDisplay(w), XK_Escape ) == kevent->keycode )
	    { appDestroyFindTool( aft ); *pRefused= 0; return; }
	}

    *pRefused= 1; return;
    }
#   endif

/************************************************************************/
/*									*/
/*  make a find tool.							*/
/*									*/
/************************************************************************/

#   ifdef USE_MOTIF
static void appFindToolConfigure(	APP_WIDGET		w,
					void *			voidaft,
					XEvent *		event,
					Boolean *		pRefused )
    {
    XConfigureEvent *		cevent= &(event->xconfigure);

    if  ( cevent->type != ConfigureNotify )
	{ return;	}

    XtVaSetValues( w,	XmNminHeight,	cevent->height,
			XmNmaxHeight,	cevent->height,
			XmNminWidth,	cevent->width,
			NULL );

    XtRemoveEventHandler( w, StructureNotifyMask, False,
					    appFindToolConfigure, voidaft );

    *pRefused= 1;
    return;
    }
#   endif

#   ifdef USE_MOTIF
static void appFindToolVisible(		APP_WIDGET		w,
					void *			voidaft,
					XEvent *		event,
					Boolean *		pRefused )
    {
    AppFindTool *	aft= (AppFindTool *)voidaft;

    if  ( event->type == MapNotify )
	{
	XSetInputFocus( XtDisplay( w ), XtWindow( w ),
						RevertToNone, CurrentTime );
	aft->aftVisible= 1;
	}

    if  ( event->type == UnmapNotify )
	{ aft->aftVisible= 0; }

    *pRefused= 1;
    return;
    }
#   endif

static AppConfigurableResource APP_FindToolResourceTable[]=
    {
	APP_RESOURCE( "findToolFindTitle",
		    offsetof(AppFindToolResources,aftrFindTitle),
		    "Find" ),

	APP_RESOURCE( "findToolUseRegex",
		    offsetof(AppFindToolResources,aftrUseRegex),
		    "Find Regular Expression" ),

	APP_RESOURCE( "findToolFindNext",
		    offsetof(AppFindToolResources,aftrFindNext),
		    "Find" ),
	APP_RESOURCE( "findToolFindPrevious",
		    offsetof(AppFindToolResources,aftrFindPrevious),
		    "Previous" ),

	APP_RESOURCE( "findToolReplaceTitle",
		    offsetof(AppFindToolResources,aftrReplaceTitle),
		    "Replace" ),

	APP_RESOURCE( "findToolReplaceFound",
		    offsetof(AppFindToolResources,aftrReplaceFound),
		    "Replace" ),
	APP_RESOURCE( "findToolReplaceNext",
		    offsetof(AppFindToolResources,aftrReplaceNext),
		    "Replace, Next" ),
    };

void * appMakeFindTool(		APP_WIDGET		findOption,
				EditApplication *	ea,
				const char *		widgetName,
				APP_BITMAP_IMAGE	iconPixmap,
				APP_BITMAP_MASK		iconMask,
				AppToolDestroy		destroy,
				FindToolFind		findNext,
				FindToolFind		findPrev,
				FindToolReplace		replace,
				void *			target )
    {
    AppFindTool *	aft;
    
    APP_WIDGET		findFrame;
    APP_WIDGET		replaceFrame;

    const int		userResizable= 1;
    
    static AppFindToolResources	aftr;
    static int			gotResources;

    if  ( ! gotResources )
	{
	appGuiGetResourceValues( ea, (void *)&aftr, APP_FindToolResourceTable,
					sizeof(APP_FindToolResourceTable)/
					sizeof(AppConfigurableResource) );

	gotResources= 1;
	}

    aft= (AppFindTool *)malloc( sizeof(AppFindTool) );
    if  ( ! aft )
	{ XDEB(aft); return (void *)0;	}

    aft->aftDestroy= destroy;
    aft->aftFindNext= findNext;
    aft->aftFindPrev= findPrev;
    aft->aftReplace= replace;
    aft->aftTarget= target;
    aft->aftVisible= 0;
    aft->aftUseRegex= 0;
    
    appMakeVerticalTool( &(aft->aftTopWidget), &(aft->aftMainWidget), ea,
			    iconPixmap, iconMask,
			    widgetName, userResizable, findOption,
			    appCloseFindTool, (void *)aft );

#   ifdef USE_MOTIF
    XtVaSetValues(	aft->aftTopWidget,
				XmNkeyboardFocusPolicy,	XmEXPLICIT,
				NULL );

    XtAddEventHandler( aft->aftTopWidget, StructureNotifyMask, False,
					appFindToolConfigure, (void *)aft );
    XtAddEventHandler( aft->aftTopWidget, StructureNotifyMask, False,
					appFindToolVisible, (void *)aft );
#   endif

    findFrame= appFindMakeFindFrame( aft->aftMainWidget, &aftr, aft );

    replaceFrame= appFindMakeReplaceFrame( aft->aftMainWidget,
							&aftr, aft );

#   ifdef USE_MOTIF
    XtInsertEventHandler( aft->aftPatternText, KeyReleaseMask, False,
				appFindToolCancel, (void *)aft, XtListHead );
    XtInsertEventHandler( aft->aftReplaceText, KeyReleaseMask, False,
				appFindToolCancel, (void *)aft, XtListHead );

    XtInsertEventHandler( aft->aftReplaceButton, KeyReleaseMask, False,
				appFindToolCancel, (void *)aft, XtListHead );
    XtInsertEventHandler( aft->aftReplaceNextButton, KeyReleaseMask, False,
				appFindToolCancel, (void *)aft, XtListHead );

    XtInsertEventHandler( aft->aftFindNextButton, KeyReleaseMask, False,
				appFindToolCancel, (void *)aft, XtListHead );
    XtInsertEventHandler( aft->aftFindPrevButton, KeyReleaseMask, False,
				appFindToolCancel, (void *)aft, XtListHead );
#   endif

    return (void *)aft;
    }

/************************************************************************/
/*									*/
/*  Draw a find tool to front.						*/
/*									*/
/************************************************************************/

void appShowFindTool(		APP_WIDGET	relative,
				void *		voidaft	)
    {
    AppFindTool *		aft= (AppFindTool *)voidaft;

    if  ( ! aft->aftVisible )
	{
#	ifdef USE_MOTIF
	Dimension			x;
	Dimension			y;
	Dimension			width;
	Dimension			height;

	XtVaGetValues( relative,
			    XmNx,		&x,
			    XmNy,		&y,
			    XmNwidth,		&width,
			    XmNheight,		&height,
			    NULL );

	XtVaSetValues( aft->aftTopWidget,
			    XmNx,		x+ width/10,
			    XmNy,		y+ height/10,
			    NULL );
#	endif

	appShowShellWidget( aft->aftTopWidget );
	}
    else{
	appShowShellWidget( aft->aftTopWidget );

#	ifdef USE_MOTIF
	{
	Window		win=  XtWindow( aft->aftTopWidget );
	Display *	display= XtDisplay( aft->aftTopWidget );

	XSetInputFocus( display, win, RevertToNone, CurrentTime );
	}
#	endif

	}

    appFindToolSetFindResult( aft, -1 );
    appFindToolReflectPatternText( aft );
    appFindToolReflectReplaceText( aft );

#   ifdef USE_MOTIF
    XmProcessTraversal( aft->aftPatternText, XmTRAVERSE_CURRENT );
#   endif
    }

void appEnableFindTool(		void *	voidaft,
				int	enabled )
    {
    AppFindTool *	aft= (AppFindTool *)voidaft;

    appGuiEnableWidget( aft->aftMainWidget, enabled != 0 );

    return;
    }

void appFindToolDisableReplace(		void *	voidaft )
    {
    AppFindTool *	aft= (AppFindTool *)voidaft;

    appGuiEnableWidget( aft->aftReplaceFrame, 0 );

    return;
    }

