/*
 * key.c - a module for keyboard command
 * by Hirotsugu Kakugawa
 */
/*
 * Copyright (C) 1996-1997 Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#ifdef STDC_HEADERS
#  include <string.h>
#else
#  include <strings.h>
#endif
#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/Xresource.h>
#include <Xm/Xm.h>

#include "cf-xmdvi.h"
#include "libdvi29.h"
#include "defs.h"
#include "dev.h"
#include "resource.h"
#include "ui.h"
#include "window.h"
#include "key.h"


#define CTL(ch)         ((ch)-0x40)  /* ASCII CODE ONLY */
#define DIGIT2NUM(ch)   ((ch)-'0')   /* ASCII CODE ONLY */

Private void  key_emacs(XEvent*);
Private void  key_vi(XEvent*);
Private void  key_common_mouse(XEvent*,int,int*);
Private void  key_common_complexkey(KeySym,int,int*,int);
Private void  key_common_key(char,int,int*);

#define PREFIX_NONE             0
#define PREFIX_EMACS_ESC      101
#define PREFIX_EMACS_CTLX     102
#define PREFIX_VI_NONE        200
#define PREFIX_VI_Z           201
#define PREFIX_VI_Col         202
#define PREFIX_VI_ColM        203
#define PREFIX_VI_ColMA       204
#define PREFIX_VI_ColMAR      205
#define PREFIX_VI_ColMARK     206
#define PREFIX_VI_ColMARKspc  207
#define PREFIX_VI_ColMAspc    208
#define PREFIX_VI_ColW        209
#define PREFIX_VI_Quote       210

Private int  argType = 0;
Private int  numArg  = 1;
Private int  numSign = 1;
Private int  prefix  = PREFIX_NONE;
Private int  cmdDone = 0;
Private int  marked_page = -1;

Private int VI_MarkTable[256];

#define COMMAND_DONE() { \
    argType = 0;  \
    numArg  = 1;  \
    numSign = 1;  \
    prefix  = PREFIX_NONE;  \
    cmdDone = 0;  \
    marked_page = -1;  }


Public int 
key_parse_keymap_name(char *name)
{
  if (strcasecmp(name, "emacs") == 0)
    return KEYMAP_EMACS;
  if (strcasecmp(name, "vi") == 0)
    return KEYMAP_VI;
  return KEYMAP_UNKNOWN;
}

Public void
key_command(XEvent *xev)
{
  switch (RES(DviDev,keymap)){
  default:
  case KEYMAP_UNKNOWN:
  case KEYMAP_EMACS:    /* My favorite (^o^)/~ */
    key_emacs(xev);
    break;
  case KEYMAP_VI:
    key_vi(xev);
    break;
  }
}

Public void
key_reset(void)
{
  int  i;

  switch (RES(DviDev,keymap)){
  default:
  case KEYMAP_UNKNOWN:
  case KEYMAP_EMACS:    /* My favorite (^o^)/~ */
    COMMAND_DONE();
    break;
  case KEYMAP_VI:
    for (i = 0; i < sizeof(VI_MarkTable); i++)
      VI_MarkTable[i] = -1; 
    break;
  }
}



/*
 * Keymap: Emacs 
 */
Private void  key_emacs_none(int,char,int*);
Private void  key_emacs_esc(int,char,int*);
Private void  key_emacs_ctlx(int,char,int*);

Private void
key_emacs(XEvent *xev)
{
  char   keyin[16], ch;
  KeySym ks;

  switch (xev->type){
  case ButtonPress:
    key_common_mouse(xev, numArg, &cmdDone);
    break;
  case KeyPress:
    if (XLookupString(&(xev->xkey), keyin, sizeof(keyin), &ks, NULL) != 1){
      key_common_complexkey(ks, numArg, &cmdDone, KEYMAP_EMACS);
      break;
    }
  default:
    ch = keyin[0];
    switch (prefix){
    case PREFIX_NONE:
      switch (ch){
      case '-':       /* Signs for num arg */
	numSign = -1;
	argType = 0;
	numArg  = -1;
	break;
      case '+':
	numSign = +1;
	argType = 0;
	numArg  = +1;
	break;
      case CTL('U'):  /* Universal Arg */
	if (argType != 1)
	  numArg = numSign;
	numArg  = 4*numArg;
	argType = 1;
	break;
      case '0': case '1': case '2': case '3': case '4': 
      case '5': case '6': case '7': case '8': case '9': 
	/* Number Arg */	
	if (argType != 2)
	  numArg  = 0;
	if (numSign > 0)
	  numArg  = numArg*10 + DIGIT2NUM(keyin[0]);
	else 
	  numArg  = numArg*10 - DIGIT2NUM(keyin[0]);
	argType = 2;
	break;
      case '\033':    /* ESC */
	prefix = PREFIX_EMACS_ESC;
	break;
      case CTL('X'): /* Ctl-x */
	prefix = PREFIX_EMACS_CTLX;
	break;
      default:
	key_emacs_none(numArg, ch, &cmdDone);
	break;
      }
      break;
    case PREFIX_EMACS_ESC:
      key_emacs_esc(numArg, ch, &cmdDone);
      break;
    case PREFIX_EMACS_CTLX:
      key_emacs_ctlx(numArg, ch, &cmdDone);
      break;
    default:
      COMMAND_DONE();
      x_bell();
    }
  }
}

Private void
key_emacs_none(int narg, char ch, int *cmdDonep)
{
  switch (ch){
  /* ^G */
  case CTL('G'): 
    COMMAND_DONE();
    x_bell();
    break;
  /* save-buffers-kill-emacs */
  case 'q': 
    COMMAND_DONE();
    cmd_quit();
    break;
  /* save-buffers-kill-emacs (without confirmation) */  
  case 'Q':
    COMMAND_DONE();
    cmd_quit_anyway();
    break;
  /* recenter */
  case CTL('L'):
    cmd_redraw();
    COMMAND_DONE();
    break;
  /* scroll down */
  case 'p':
  case '\b': case '\177':
    cmd_page_prev(narg);
    break;
  /* scroll-up */
  case CTL('V'): 
  case 'n': 
  case ' ': case '\t':
    cmd_page_next(narg);
    COMMAND_DONE();
    break;
  /* previous-line */
  case CTL('P'):
    cmd_preview_move_v(-SCROLL_DELTA_V*narg);
    COMMAND_DONE();
    break;
  /* next-line */
  case CTL('N'):
    cmd_preview_move_v(+SCROLL_DELTA_V*narg);
    COMMAND_DONE();
    break;
  /* backward-char */
  case CTL('B'):
    cmd_preview_move_h(-SCROLL_DELTA_H*narg);
    COMMAND_DONE();
    break;
  /* forward-char */
  case CTL('F'):
    cmd_preview_move_h(+SCROLL_DELTA_H*narg);
    COMMAND_DONE();
    break;
  /* beginning-of-line */
  case CTL('A'):
    cmd_preview_move_h(SCROLL_LEFTMOST);
    COMMAND_DONE();
    break;
  /* end-of-line */
  case CTL('E'):
    cmd_preview_move_h(SCROLL_RIGHTMOST);
    COMMAND_DONE();
    break;
  /* set-mark-command */
  case CTL(' '): case CTL('@'):
    if (DviFile == NULL){
      x_bell();
    } else {
      marked_page = DviCurrentPage;
      x_message_advice("Mark set");
    }
    COMMAND_DONE();
    break;
  /* Other key... */
  default:
    key_common_key(ch, narg, cmdDonep);
    COMMAND_DONE();
    break;
  }
}

Private void
key_emacs_esc(int narg, char ch, int *cmdDonep)
{
  switch (ch){
  /* ^G */
  case CTL('G'): 
    COMMAND_DONE();
    x_bell();
    break;
  /* scroll down */
  case 'v': 
    cmd_page_prev(narg);
    COMMAND_DONE();
    break;
  /* beginning-of-buffer */
  case '<':
    if (DviFile == NULL){
      x_bell();
    } else {
      marked_page = DviCurrentPage;
      x_message_advice("Mark set");
      cmd_goto_page(1);
    }
    COMMAND_DONE();
    break;
  /* end-of-buffer */
  case '>':
    if (DviFile == NULL){
      x_bell();
    } else {
      marked_page = DviCurrentPage;
      x_message_advice("Mark set");
      cmd_goto_page(DviFile->pages);
    }
    COMMAND_DONE();
    break;
  /* forward-word */
  case 'B':
    cmd_preview_move_h(-SCROLL_DELTA_H2*narg); 
    COMMAND_DONE();
    break;
  /* backward-word */
  case 'F':
    cmd_preview_move_h(+SCROLL_DELTA_H2*narg);
    COMMAND_DONE();
    break;
  /* Other key... */
  default:
    COMMAND_DONE();
    x_bell();
  }
}

Private void
key_emacs_ctlx(int narg, char ch, int *cmdDonep)
{
  int  page; 

  switch (ch){
  /* ^G */
  case CTL('G'): 
    COMMAND_DONE();
    x_bell();
    break;
  /* find-file / find-file-read-only */
  case CTL('F'):
  case CTL('R'):
    cmd_find_file();
    COMMAND_DONE();
    break;
  /* exit-emacs */
  case CTL('C'): 
    cmd_quit();
    COMMAND_DONE();
    break;
  /* exhange-point-and-mark */
  case CTL('X'):
    if (DviFile == NULL){
      x_bell();
    } else {
      if ((marked_page == -1) || (marked_page > DviFile->pages)){
	x_error_message("No mark set");
	marked_page = -1;
      } else {
	page = marked_page;
	marked_page = DviCurrentPage;
	cmd_goto_page(page);
      }
    }
    COMMAND_DONE();
    break;
  /* Other key... */
  default:
    COMMAND_DONE();
    x_bell();
    break;
  }
}



/* 
 * Keymap: Vi
 */
Private void  key_vi_none(int,int,char,int*);
Private void  key_vi_z(int,char,int*);
Private void  key_vi_col(int,char,int*,int*);
Private void  key_vi_colm(int,char,int*,int*);
Private void  key_vi_colma(int,char,int*,int*);
Private void  key_vi_colmar(int,char,int*,int*);
Private void  key_vi_colmark(int,char,int*,int*);
Private void  key_vi_colmarkspc(int,char,int*,int*);
Private void  key_vi_colmaspc(int,char,int*,int*);
Private void  key_vi_colw(int,char,int*,int*);
Private void  key_vi_quote(int,char,int*);

Private void
key_vi(XEvent *xev)
{
  char   keyin[16], ch;
  KeySym ks;

  switch (xev->type){
  case ButtonPress:
    key_common_mouse(xev, numArg, &cmdDone);
    break;
  case KeyPress:
    if (XLookupString(&(xev->xkey), keyin, sizeof(keyin), &ks, NULL) != 1){
      key_common_complexkey(ks, numArg, &cmdDone, KEYMAP_VI);
      break;
    }
    ch = keyin[0];
    switch (prefix){
    case PREFIX_NONE:
      switch (ch){
      /* Signs for num arg */
      case '-':
	numSign = -1;
	argType = 0;
	numArg  = -1;
	break;
      case '+':
	numSign = +1;
	argType = 0;
	numArg  = +1;
	break;
      /* Number Arg */
      case '0': case '1': case '2': case '3': case '4': 
      case '5': case '6': case '7': case '8': case '9': 
	if (argType != 2)
	  numArg  = 0;
	if (numSign > 0)
	  numArg  = numArg*10 + DIGIT2NUM(ch);
	else 
	  numArg  = numArg*10 - DIGIT2NUM(ch);
	argType = 2;
	break;
      case '\033':    /* ESC */
	x_bell();
	break;
      case 'Z':
	prefix = PREFIX_VI_Z;
	break;
      case ':':
	prefix = PREFIX_VI_Col;
	break;
      case '\'':
	prefix = PREFIX_VI_Quote;
	break;
      default:
	key_vi_none(numArg, argType, ch, &cmdDone);
	break;
      }
      break;
    case PREFIX_VI_Z:
      key_vi_z(numArg, ch, &cmdDone);
      break;
    case PREFIX_VI_Col:
      key_vi_col(numArg, ch, &cmdDone, &prefix);
      break;
    case PREFIX_VI_ColM:
      key_vi_colm(numArg, ch, &cmdDone, &prefix);
      break;
    case PREFIX_VI_ColMA:
      key_vi_colma(numArg, ch, &cmdDone, &prefix);
      break;
    case PREFIX_VI_ColMAR:
      key_vi_colmar(numArg, ch, &cmdDone, &prefix);
      break;
    case PREFIX_VI_ColMARK:
      key_vi_colmark(numArg, ch, &cmdDone, &prefix);
      break;
    case PREFIX_VI_ColMARKspc:
      key_vi_colmarkspc(numArg, ch, &cmdDone, &prefix);
      break;
    case PREFIX_VI_ColMAspc:
      key_vi_colmaspc(numArg, ch, &cmdDone, &prefix);
      break;
    case PREFIX_VI_ColW:
      key_vi_colw(numArg, ch, &cmdDone, &prefix);
      break;
    case PREFIX_VI_Quote:
      key_vi_quote(numArg, ch, &cmdDone);
      break;
    default:
      x_bell();
    }
  }
}

Private void
key_vi_none(int narg, int atype, char ch, int *cmdDonep)
{
  int  page;

  *cmdDonep = 1;
  switch (ch){
  /* RET */
  case '\n':
  case '\r':
    COMMAND_DONE();
    break;
  /* ESC */
  case '\033':
    COMMAND_DONE();
    break;
  /* Quit */
  case 'q': 
  case 'Z': 
    cmd_quit();
    COMMAND_DONE();
    break;
  case 'Q':
    cmd_quit_anyway();
    COMMAND_DONE();
    break;
  /* Next Page */
  case CTL('F'): 
  case 'n': 
  case ' ': case '\t':
    cmd_page_next(narg);
    COMMAND_DONE();
    break;
  /* Prev Page */
  case CTL('B'): 
  case 'p': 
  case '\b': case '\177':
    cmd_page_prev(narg); 
    COMMAND_DONE();
    break;
  /* Goto Page */
  case 'G':
    if (DviFile == NULL){
      x_bell();
    } else {
      if (atype == 0){
	cmd_goto_page(DviFile->pages);
      } else {
	page = narg;
	if (DviFile->pages < page)
	  cmd_goto_page(DviFile->pages);
	if (page < 1)
	  cmd_goto_page(1);
      }
    }
    COMMAND_DONE();
    break;
  /* Up */
  case CTL('U'):
  case 'k':
    cmd_preview_move_v(-SCROLL_DELTA_V*narg);
    COMMAND_DONE();
    break;
  /* Down */
  case CTL('D'): 
  case 'j':
    cmd_preview_move_v(+SCROLL_DELTA_V*narg);
    COMMAND_DONE();
    break;
  /* Left */
  case 'h':
    cmd_preview_move_h(-SCROLL_DELTA_H*narg);
    COMMAND_DONE();
    break;
  /* Right */
  case 'l':
    cmd_preview_move_h(+SCROLL_DELTA_H*narg);
    COMMAND_DONE();
    break;
  /* beginning of line */
  case 'O':
  case '^':
    cmd_preview_move_h(SCROLL_LEFTMOST);
    COMMAND_DONE();
    break;
  /* end of line */
  case '$':
    cmd_preview_move_h(SCROLL_RIGHTMOST);
    COMMAND_DONE();
    break;
  /* redraw */
  case CTL('L'):
    cmd_redraw();
    COMMAND_DONE();
    break;
  /* Other Commands... */
  default:
    key_common_key(ch, narg, cmdDonep);
    COMMAND_DONE();
    break;
  }
}

Private void
key_vi_z(int narg, char ch, int *cmdDonep)
{
  *cmdDonep = 1;
  switch (ch){
  /* ZZ */
  case 'Z':
    cmd_quit_anyway();
    COMMAND_DONE();
    break;
  /* Other Commands... */
  default:
    x_bell();
    COMMAND_DONE();
    break;
  }
}

Private void
key_vi_col(int narg, char ch, int *cmdDonep, int *prefixp)
{
  switch (ch){
  /* :q */
  case 'q':
    cmd_quit();
    COMMAND_DONE();
    break;
  /* :w */
  case 'w':
    *prefixp = PREFIX_VI_ColW;
    break;
  /* :r */
  case 'r':
    cmd_find_file();
    COMMAND_DONE();
    break;
  /* :m */
  case 'm':
    *prefixp = PREFIX_VI_ColM;
    break;
  /* Other Commands... */
  default:
    COMMAND_DONE();
    x_bell();
    break;
  }
}

Private void
key_vi_colw(int narg, char ch, int *cmdDonep, int *prefixp)
{
  *cmdDonep = 1;
  switch (ch){
  /* :wq */
  case 'q':
    cmd_quit();
    COMMAND_DONE();
    break;
  /* Other Commands... */
  default:
    COMMAND_DONE();
    x_bell();
    break;
  }
}

Private void
key_vi_colm(int narg, char ch, int *cmdDonep, int *prefixp)
{
  switch (ch){
  /* :ma */
  case 'a':
    *prefixp = PREFIX_VI_ColMA;
    break;
  /* Other Commands... */
  default:
    COMMAND_DONE();
    x_bell();
    break;
  }
}
Private void
key_vi_colma(int narg, char ch, int *cmdDonep, int *prefixp)
{
  switch (ch){
  /* :mar */
  case 'r':
    *prefixp = PREFIX_VI_ColMAR;
    break;
  /* :maSPC */
  case ' ':
    *prefixp = PREFIX_VI_ColMAspc;
    break;
  /* Other Commands... */
  default:
    COMMAND_DONE();
    x_bell();
    break;
  }
}
Private void
key_vi_colmar(int narg, char ch, int *cmdDonep, int *prefixp)
{
  switch (ch){
  /* :mark */
  case 'k':
    *prefixp = PREFIX_VI_ColMARK;
    break;
  /* Other Commands... */
  default:
    COMMAND_DONE();
    x_bell();
    break;
  }
}
Private void
key_vi_colmark(int narg, char ch, int *cmdDonep, int *prefixp)
{
  switch (ch){
  /* :mark */
  case ' ':
    *prefixp = PREFIX_VI_ColMARKspc;
    break;
  /* Other Commands... */
  default:
    COMMAND_DONE();
    x_bell();
    break;
  }
}

Private void
key_vi_colmarkspc(int narg, char ch, int *cmdDonep, int *prefixp)
{
  if (DviFile == NULL){
    x_bell();
  } else {
    VI_MarkTable[(unsigned int)ch] = DviCurrentPage;
  }
  COMMAND_DONE();
}
Private void
key_vi_colmaspc(int narg, char ch, int *cmdDonep, int *prefixp)
{
  if (DviFile == NULL){
    x_bell();
  } else {
    VI_MarkTable[(unsigned int)ch] = DviCurrentPage;
  }
  COMMAND_DONE(); 
}

Private void
key_vi_quote(int narg, char ch, int *cmdDonep)
{
  int page;

  if (DviFile == NULL){
    x_bell();
  } else {
    page = VI_MarkTable[(unsigned int)ch];
    if ((marked_page < 0) || (page < 0))
      x_bell();
    else 
      cmd_goto_page(page);
  }
  COMMAND_DONE(); 
}


/*
 * Common Actions 
 */
Private void
key_common_mouse(XEvent *xev, int narg, int *cmdDonep)
{
  switch (xev->xbutton.button){
  case Button1:  /* Left Button */
    cmd_page_prev(1);
    COMMAND_DONE(); 
    break;
  case Button2:  /* Center Button */
    break;
    COMMAND_DONE(); 
  case Button3:  /* Right Button */
    cmd_page_next(1);
    COMMAND_DONE(); 
    break;
  }
}

Private void
key_common_complexkey(KeySym ks, int narg, int *cmdDonep, int mode)
{
  switch (ks){
  case XK_Right:
    cmd_preview_move_h(+SCROLL_DELTA_H*narg);
    COMMAND_DONE(); 
    break;
  case XK_Left:
    cmd_preview_move_h(-SCROLL_DELTA_H*narg);
    COMMAND_DONE(); 
    break;
  case XK_Up:
    cmd_preview_move_v(-SCROLL_DELTA_V*narg);
    COMMAND_DONE(); 
    break;
  case XK_Down:
    cmd_preview_move_v(+SCROLL_DELTA_V*narg);
    COMMAND_DONE(); 
    break;
  case XK_Shift_L:
  case XK_Shift_R:
  case XK_Control_L:
  case XK_Control_R:
    break;
  default:
    COMMAND_DONE(); 
    x_bell();
  }
}

Private void
key_common_key(char ch, int narg, int *cmdDonep)
{
  switch (ch){
  /* Magnify */
  case 'M': case ']':
    cmd_magnify_enlarge();  
    COMMAND_DONE(); 
    break;
  /* Shrink */
  case 'm': case '[':
    cmd_magnify_shrink();
    COMMAND_DONE(); 
    break;
  /* Enlarge preview window */
  case '}':
    cmd_preview_enlarge();  
    COMMAND_DONE(); 
    break;
  /* Shrink preview window */
  case '{':
    cmd_preview_shrink();
    COMMAND_DONE(); 
    break;
  /* Reload */
  case 'R':
    cmd_reload();
    COMMAND_DONE();
    break;
  /* Other key ... */  
  default:
    COMMAND_DONE();
    x_bell();
  }
}

/*EOF*/
