/*
 * resource.c - a module for getting resource values
 * by Hirotsugu Kakugawa
 */
/*
 * Copyright (C) 1996-1997 Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#if defined(HAVE_STRING_H) || defined(STDC_HEADERS)
# include <string.h>
#else
# include <strings.h>
#endif
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#if HAVE_MALLOC_H
# include <malloc.h>
#endif
#include <ctype.h>
#include <sys/param.h>
#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>

#include "libdvi29.h"
#include "defs.h"
#include "dev.h"
#include "cf-xmdvi.h"
#include "resource.h"
#include "paper.h"
#include "key.h"
#include "window.h"

#define PR(s) fprintf(stderr, s);

Private int      resource_parse_args(DVI_DEVICE,RESOURCE,int,char**,int);
Private int      resource_conv(RESOURCE);
Private int      visual_type(char *name);

Private char*    find_dvi_file(RESOURCE,char*);
Private void         bogus_resource_int(char*,int);
Private void         usage(void);
Private void         license(void);
Private int          atob(char*);
extern double        atof();



Public int
resource_create(DVI_DEVICE dev, int argc, char **argv, int is_cmdline)
{
  RESOURCE     res;

  if ((res = calloc(1, sizeof(struct s_resource))) == NULL)
    return -1;

  SET_RES(dev, res);
#if 0
  if (x_init_application_resources() == NULL)
    return -1;
#endif
  x_resource_set_defaults(res);
  if (resource_parse_args(dev, res, argc, argv, is_cmdline) < 0)
    return -1;
  if (resource_conv(res) < 0)
    return -1;
  return 0;
}

Public void
resource_dispose(RESOURCE res)
{
  if (res != NULL)
    free(res);
}


Private int
resource_parse_args(DVI_DEVICE dev, RESOURCE res,
		    int argc, char **argv, int is_cmdline)
{
  char          *dvi_name;
  double        m, s;
  static char   shrink_str[64];

  dvi_name = NULL;
  --argc; argv++;
  while (argc > 0){
    if ((strcmp(*argv, "-ver") == 0) || (strcmp(*argv, "-version") == 0)){
      if (is_cmdline == 1){
	fprintf(stderr, "%s %s\n", PROG_NAME, XMDVI_VERSION);
	exit(0);
      } else {
	return 0;
      }
    } else if (paper_name2id(&argv[0][1], 0) >= 0){
      res->s_paper = &argv[0][1];
    } else if (**argv != '-') {
      dvi_name = *argv;
    } else if (strcmp(*argv, "-l") == 0){
      res->s_orient = "Landscape";
    } else if (strcmp(*argv, "-epsf.sty") == 0){
      res->s_eps_style = EPS_STYLE_LATEX209_EPSF_STY_STR;
    } else if (strcmp(*argv, "-eps-mono") == 0){
      res->s_visual_eps = VISUAL_MONO_STR;
    } else if (strcmp(*argv, "-eps-gray") == 0){
      res->s_visual_eps = VISUAL_GRAY_STR;
    } else if (strcmp(*argv, "-eps-rgb") == 0){
      res->s_visual_eps = VISUAL_RGB_STR;
    } else if (strcmp(*argv, "-text-mono") == 0){
      res->s_visual_text = VISUAL_MONO_STR;
    } else if (strcmp(*argv, "-text-gray") == 0){
      res->s_visual_text = VISUAL_GRAY_STR;
    } else if (strcmp(*argv, "-text-rgb") == 0){
      res->s_visual_text = VISUAL_RGB_STR;
    } else if (strcmp(*argv, "-help") == 0){
      usage();
      exit(0);
    } else if (strcmp(*argv, "-license") == 0){
      license();
      exit(0);
    } else if (argc >= 2){
      if (strcmp(*argv, "-v") == 0){
	argc--;  argv++;
	res->vflibcap = *argv;
      } else if (strcmp(*argv, "-m") == 0){
	argc--; argv++;
	m = atof(*argv);
	s = atof(res->s_shrink);
	if ((atof(res->s_mag_min) <= m) || (m <= atof(res->s_mag_max))){
	  sprintf(shrink_str, "%.5f", s/m);
	  res->s_shrink = shrink_str;
	} else {
	  fprintf(stderr, "Maginification is out of range.\n");
	  if (is_cmdline == 1){
	    exit(1);
	  } else
	    return -1;
	}
      } else if (strcmp(*argv, "-dpi") == 0){
	argc--;  argv++;
	res->s_dpi = *argv;
      } else if (strcmp(*argv, "-mode") == 0){
	argc--;  argv++;
	res->kpathsea_mode = *argv;
      } else if (strcmp(*argv, "-k") == 0){
	argc--;  argv++;
	res->s_keymap = *argv;
      } else if (strcmp(*argv, "-fg") == 0){
	argc--;  argv++;
	res->color_char = *argv;
      } else if (strcmp(*argv, "-bg") == 0){
	argc--;  argv++;
	res->color_paper = *argv;
      } else if (strcmp(*argv, "-frame") == 0){
	argc--;  argv++;
	res->color_frame = *argv;
      } else if (strcmp(*argv, "-ox") == 0){
	argc--;  argv++;
	res->s_offset_x = *argv;
      } else if (strcmp(*argv, "-oy") == 0){
	argc--;  argv++;
	res->s_offset_y = *argv;
      } else if (strcmp(*argv, "-cache") == 0){
	argc--;  argv++;
	res->s_page_cache = *argv;
      } else if (strcmp(*argv, "-f") == 0){
	argc--;  argv++;
	dvi_name = *argv;
      } else 
	goto ArgError;
    } else {
ArgError:
      fprintf(stderr, "Unknow option: %s\n", *argv);
      usage();
      return -1;
    }
    argc--;  argv++;
  }
  DviFileNameNew = find_dvi_file(res, dvi_name);

  return 0;
}

Private int
resource_conv(RESOURCE res)
{
  int  p; 
  int  err = 0;

  res->dpi = atoi(res->s_dpi);
  if (res->dpi <= 0){
    --err;
    bogus_resource_int("DPI", res->dpi);
  }

  p = paper_name2id(res->s_paper, res->s_orient);
  res->paper      = PAPER_BASIC_ID(p);
  res->orient     = PAPER_ORIENTATION(p);
#if 0
  printf("** paper  %d\n", res->paper);
  printf("** orient %d\n", res->orient);
#endif
  res->shrink     = atof(res->s_shrink);
  res->mag_min    = atof(res->s_mag_min);
  res->mag_max    = atof(res->s_mag_max);
  if (res->mag_min <= 0)
    res->mag_min = 0.01;
  if (res->mag_min > 1)
    res->mag_min = 1.0;
  if (res->mag_max < 1)
    res->mag_max = 1.0;
  res->prevwin_w  = atoi(res->s_prevwin_w);
  res->prevwin_h  = atoi(res->s_prevwin_h);

  res->offset_x   = atof(res->s_offset_x);
  res->offset_y   = atof(res->s_offset_y);

  res->msg_window_size  = atoi(res->s_msg_window_size);
  res->msg_history_size = atoi(res->s_msg_history_size);

  res->keymap     = key_parse_keymap_name(res->s_keymap);

  res->novice     = atob(res->s_novice);

  res->visual_eps  = visual_type(res->s_visual_eps);
  res->visual_text  = visual_type(res->s_visual_text);
  res->eps_style = EPS_STYLE_LATEX2E_GRAPHICS_STY;
  if (res->s_eps_style != NULL){
    if (strcmp(res->s_eps_style, EPS_STYLE_LATEX209_EPSF_STY_STR) == 0)
      res->eps_style = EPS_STYLE_LATEX209_EPSF_STY;
  }
  res->draw_eps   = atob(res->s_draw_eps);
  res->gs_timeout = atoi(res->s_gs_timeout);

  res->print_msg_window_columns  = atoi(res->s_print_msg_window_columns);
  res->print_msg_window_rows     = atoi(res->s_print_msg_window_rows);
  res->print_msg_history_size    = atoi(res->s_print_msg_history_size);
  res->debug_lpspec     = atob(res->s_debug_lpspec);

  res->poll       = atoi(res->s_poll);

  res->page_cache = atoi(res->s_page_cache);

  return err;
}

Private void
usage(void)
{
  fprintf(stderr, "%s version %s\n", PROG_NAME, XMDVI_VERSION);
  PR("Copyright (C) 1997-1999  Hirotsugu Kakugawa.\n");
  PR("This program is distributed in the hope that it will be useful,\n");
  PR("but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
  PR("MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n");
  PR("\n");
  fprintf(stderr, "Usage: %s [Paper Size] [Options] dvi-file\n", PROG_NAME);
  PR("Paper Size:\n");
  PR("   -A3, -A4, -A5, -B4, -B5, -B6, -LETTER, -US, etc\n");
  PR("Options:\n");
  PR("  -l             Landscape mode\n");
  PR("  -m M           Magnification\n");
  PR("  -v VFLIBCAP    Vflibcap file\n");
  PR("  -k KEYMAP      Select keymap ('emacs' or 'vi')\n");
  PR("  -fg COLOR      Character color\n");
  PR("  -bg COLOR      Paper color\n");
  PR("  -frame COLOR   Paper frame color\n");
  PR("  -eps-{mono,gray,rgb}     Monochrome/Grayscale/RGB EPS figures\n");
  PR("  -text-{mono,gray,rgb}    Monochrome/Grayscale/RGB text color\n");
  PR("  -epsf.sty      EPS figure mode for LaTeX 2.09 epsf.sty\n");
  PR("  -ox OFFSET     Horizontal offset (in inch)\n");
  PR("  -oy OFFSET     Vertical offset (in inch)\n");
  PR("  -cache N       Set page cache size (N pages are cached.)\n");
  PR("  -version       Show version\n");
  PR("  -license       Show lisence\n");
  PR("Key and Mouse operation:\n");
  PR("  SPC (mouse right button)    Next page\n");
  PR("  BS  (mouse left button)     Previous page\n");
  PR("  q                           Quit\n");
}

Private void
license(void)
{
  fprintf(stderr, "%s version %s\n", PROG_NAME, XMDVI_VERSION);
  PR("Copyright (C) 1997 Hirotsugu Kakugawa.\n");
  PR("All rights reserved.\n");
  PR("This program is free software; you can redistribute it and/or modify\n");
  PR("it under the terms of the GNU General Public License as published by\n");
  PR("the Free Software Foundation; either version 2, or (at your option)\n");
  PR("any later version.\n");
  PR("\n");
  PR("This program is distributed in the hope that it will be useful,\n");
  PR("but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
  PR("MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
  PR("GNU General Public License for more details.\n");
  PR("\n");
  PR("This program uses DVIlib and VFlib which are distributed under the\n");
  PR("terms of GNU Lribrary General Public License\n");
}



Private void
bogus_resource_int(char *reason, int val)
{
  fprintf(stderr, "Illegal resource value for %s: %d\n", reason, val);
}

Private char*
find_dvi_file(RESOURCE res, char *name)
{
  int  name_len;
  char *path;
  
  if (name == NULL)
    return NULL;

  if (access(name, R_OK) >= 0){
    if ((path = (char*)malloc(strlen(name) + 1)) == NULL)
      return NULL;
    strcpy(path, name);
    return path;
  }

  name_len = strlen(name);
  if ((name_len > strlen(DVI_FILE_EXTENSION)) 
      && (strcmp(name+name_len-strlen(DVI_FILE_EXTENSION),
		 DVI_FILE_EXTENSION) == 0)){
    return NULL;
  }
  if ((path = (char*)malloc(name_len+strlen(DVI_FILE_EXTENSION)+1)) == NULL)
    return NULL;
  sprintf(path, "%s%s", name, DVI_FILE_EXTENSION);
  if (access(path, R_OK) < 0){
    free(path);
    return NULL;
  }
  return path;
}

Private int 
visual_type(char *name)
{
  int  t = VISUAL_MONO;
  
  if (tolower(*name) == tolower(VISUAL_GRAY_STR[0]))
    t = VISUAL_GRAY;
  else if (tolower(*name) == tolower(VISUAL_RGB_STR[0]))
    t = VISUAL_RGB;
  return t;
}

Private int
atob(char *s)
{
  if (   (strcmp(s, "true") == 0)
      || (strcmp(s, "True") == 0)
      || (strcmp(s, "TRUE") == 0)
      || (strcmp(s, "yes") == 0) 
      || (strcmp(s, "Yes") == 0)
      || (strcmp(s, "YES") == 0)
      || (strcmp(s, "ok") == 0)
      || (strcmp(s, "Ok") == 0)
      || (strcmp(s, "OK") == 0)
      || (strcmp(s, "1") == 0)   )
    return 1;
  return 0;
}


/*EOF*/
