/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "frequencyinfo.h"

#include "definitionbuilder.h"
#include "definitionparser.h"

FrequencyInfo::FrequencyInfo() :
    m_type( Invalid ),
    m_day( 0 )
{
}

FrequencyInfo::~FrequencyInfo()
{
}

QString FrequencyInfo::toString() const
{
    if ( !isValid() )
        return QString();

    DefinitionBuilder builder;

    switch ( m_type ) {
        case Immediately:
            builder.setType( "IMMEDIATELY" );
            break;
        case Daily:
            builder.setType( "DAILY" );
            break;
        case Weekly:
            builder.setType( "WEEKLY" );
            break;
        default:
            return QString();
    };

    if ( m_type == Weekly )
        builder.addMetadata( "day", m_day );

    if ( m_type == Daily || m_type == Weekly )
        builder.addMetadata( "time", m_time.toString( "hh:mm" ) );

    return builder.result();
}

FrequencyInfo FrequencyInfo::fromString( const QString& text )
{
    FrequencyInfo info;

    DefinitionParser parser;

    if ( !parser.parse( text ) )
        return info;

    if ( parser.type() == QLatin1String( "IMMEDIATELY" ) )
        info.m_type = Immediately;
    else if ( parser.type() == QLatin1String( "DAILY" ) )
        info.m_type = Daily;
    else if ( parser.type() == QLatin1String( "WEEKLY" ) )
        info.m_type = Weekly;
    else
        return info;

    QMap<QString, QVariant> metadata = parser.metadata();

    QVariant value = metadata.value( "day" );
    if ( value.isValid() )
        info.m_day = value.toInt();

    value = metadata.value( "time" );
    if ( value.isValid() )
        info.m_time = QTime::fromString( value.toString(), "hh:mm" );

    return info;
}
