/* -*-c-*- */
/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

%{

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/* ugly hack ALL@@1 */
#ifndef __USE_GNU
#define __USE_GNU /* necessary */
#include <string.h>
#undef __USE_GNU
#else
#include <string.h>
#endif

#define __USE_XOPEN /* strptime needs this */
#include <time.h> /* strptime */

#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
#include <sys/types.h> /* <netinet/in.h> needs this */
#endif
#include <netinet/in.h>       /* for IPPROTO_* */
#include <netdb.h> /* for getservbyname */

#include "wfipaddr.h"
#include "wfnetwork.h"
#include "wfprotocol.h" /* for wf_protocol_icmp_totype */

#include "filter.h"
#include "filter_y.h"
#include "defs.h"


#define YY_NO_UNPUT  /* to avoid warning */

#define LEXRET(a, b) { yylval.intval = b; return a; }

extern void yline(const char* str, ...);

%}


/* We have to do this in order to avoid 'undefined symbol: yywrap' when
   dynamically opening wallfire module.
   Note: only a static fl library is available.
         That means that libfl.a cannot be linked with a dynamic library.
         For example, we would need this when we build dynamic wallfire module.
*/
%option noyywrap

DIGIT		[0-9]
XDIGIT		[A-Fa-f0-9]

BYTE		{DIGIT}{1,3}
IPADDRESS	{BYTE}\.{BYTE}\.{BYTE}\.{BYTE}

NETBYTE		{BYTE}|\*
NETBYTES	{BYTE}\.{NETBYTE}\.{NETBYTE}\.\*
NETMASK		{IPADDRESS}|{DIGIT}{1,2}
IPNETWORK	{NETBYTES}|({IPADDRESS}\/{NETMASK})

MACBYTE		{XDIGIT}{1,2}
MACADDRESS	({MACBYTE}:){5}{MACBYTE}

%%

"&&"	return LAND_OP;
"||"	return LOR_OP;
"<="	return LE_OP;
">="	return GE_OP;
"=="	return EQ_OP;
"!="	return NE_OP;
"=~"	return REG_OP; /* regexp */

all|any	return ALL;
"0/0"	return NULLNETWORK;

 /* TCP flags */
FIN	LEXRET(TCP_FLAGS, TCP_FIN);
SYN	LEXRET(TCP_FLAGS, TCP_SYN);
RST	LEXRET(TCP_FLAGS, TCP_RST);
PSH	LEXRET(TCP_FLAGS, TCP_PSH);
ACK	LEXRET(TCP_FLAGS, TCP_ACK);
URG	LEXRET(TCP_FLAGS, TCP_URG);
ECE	LEXRET(TCP_FLAGS, TCP_ECE);
CWR	LEXRET(TCP_FLAGS, TCP_CWR);

{IPNETWORK}	{
  yylval.network = new wf_network();
  if (yylval.network == NULL)
    yyterminate();
  if (yylval.network->network_set(yytext) == false) {
    yline(_("wrong network `%s'"), yytext);
    yyterminate();
  }
  return IPNET;
}

{IPADDRESS}	{
  yylval.ipaddr = new wf_ipaddr();
  if (yylval.ipaddr == NULL)
    yyterminate();
  if (yylval.ipaddr->set(yytext) == false) {
    yline(_("wrong IP address `%s'"), yytext);
    yyterminate();
  }
  return IPADDR;
}

{MACADDRESS}	{
  yylval.macaddr = new wf_macaddr();
  if (yylval.macaddr == NULL)
    yyterminate();
  if (yylval.macaddr->set(yytext) == false) {
    yline(_("wrong MAC address `%s'"), yytext);
    yyterminate();
  }
  return MACADDR;
}

\"[^\"]*\"	{
  yytext[yyleng - 1] = '\0';
  yylval.str = strdup(yytext + 1);
  return STRING_LITERAL;
}

\/[^/]*\/(i)?	{
  yylval.str = strdup(yytext);
  return STRING_REGEXP;
}

\$[a-zA-Z][a-zA-Z0-9_\-]*	{
  yylval.str = strdup(yytext + 1);
  return IDENTIFIER;
}

\[[^\]]+\]	{
  yytext[yyleng - 1] =  '\0';
  yylval.str = strdup(yytext + 1);
  return DATE_STRING;
}

[a-zA-Z][a-zA-Z0-9_\-]*		{
  /* maybe it is an icmp type */
  int code, type;
  if (wf_protocol_icmp_totype(yytext, &type, &code)) {
    yylval.intval = type;
    /* RV@@8
    yylval.protocol_icmp = new wf_protocol_icmp(type, code);
    TEST_ALLOC(yylval.protocol_icmp != NULL, _("ICMP protocol"));
    */
    return INTNUM;
  }

  /* maybe it is a service name */
  struct servent* serv = getservbyname(yytext, "tcp"); // udp? RV@@6
  if (serv != NULL) {
    yylval.intval = ntohs(serv->s_port);
    return INTNUM;
  }
  struct protoent* protoent = getprotobyname(yytext);
  if (protoent != NULL) {
    yylval.intval = protoent->p_proto;
    return INTNUM;
  }
  ostream_printf(cerr, _("Error: unknown word `%s'.\n"), yytext);
  yyterminate();
}

{DIGIT}+	{ yylval.intval = atoi(yytext); return INTNUM; }

[ \t]	; /* ignore whitespace */

.	return yytext[0];

%%
