/* fieldlistview.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2002 Ralf Hoffmann.
 * You can contact me at: ralf.hoffmann@epost.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: fieldlistview.cc,v 1.5 2002/04/06 16:02:18 ralf Exp $ */

#include "fieldlistview.h"
#include <stdlib.h>
#include "awindow.h"
#include "guielement.h"

const char *FieldListView::type="FieldListView";

FieldListView::FieldListView( AGUIX *aguix, int x, int y, int width, int height, int data ):GUIElement( aguix )
{
  int i;

  fields = 1;
  fieldwidth = new int[fields];
  for ( i = 0; i < fields; i++ )
    fieldwidth[i] = -1;
  used_width = new int[fields];
  for ( i = 0; i < fields; i++ )
    used_width[i] = 0;
  elements = 0;
  arraysize = 100;
  elementarray = new Content*[arraysize];
  yoffset = 0;
  xoffset = 0;

  fieldalign = new align_t[fields];
  for ( i = 0; i < fields; i++ )
    fieldalign[i] = ALIGN_LEFT;
  activerow = -1;

  this->data = data;
  this->x = x;
  this->y = y;
  this->w = width;
  this->h = height;
  bgset = false;
  hbar = 1;
  vbar = 1;
  vbar_width = 10;
  hbar_height = 10;
  hbar_dir = 0;
  vbar_dir = 0;
  vbar_pressed = false;
  hbar_pressed = false;
  subwins = NULL;
  winbgs = NULL;
  hbarwin = 0;
  vbarwin = 0;
  selecthandler=NULL;
  font = NULL;
  gettimeofday( &lastwheel, NULL );
  mbg = 0;
}

FieldListView::~FieldListView()
{
  int i;

  if ( parent != NULL ) {
    for ( i = 0; i < maxdisplayv; i++ ) {
      parent->removeSubWin( subwins[ i ] );
    }
    parent->removeSubWin( hbarwin );
    parent->removeSubWin( vbarwin );    
    parent->removeSubWin( win );
  }
  if ( subwins != NULL ) delete subwins;

  delete fieldwidth;
  delete used_width;
  delete fieldalign;
  for ( i = 0; i < elements; i++ ) {
    delete elementarray[i];
  }
  delete elementarray;
}

#if 0
void FieldListView::setFields( int nv )
{
  int *nf, i;
  
  if ( nv < 1 ) return;
  
  nf = new int[nv];

  if ( nv < fields ) {
    for ( i = 0; i < nv; i++ ) {
      nf[i] = fieldwidth[i];
    }
  } else {
    for ( i = 0; i < fields; i++ ) {
      nf[i] = fieldwidth[i];
    }
    for ( i = fields; i < nv; i++ ) {
      nf[i] = -1;
    }
  }
  delete fieldwidth;
  fieldwidth = nf;
  fields = nv;
}
#endif

FieldListView::Content::Content()
{
  int i;

  cur_fields = 1;
  texts = new std::string*[cur_fields];
  for ( i = 0; i < cur_fields; i++)
    texts[i] = new std::string("");

  // default colors
  fg[ 0 ] = 1;
  fg[ 1 ] = 2;
  fg[ 2 ] = 1;
  fg[ 3 ] = 2;
  bg[ 0 ] = 0;
  bg[ 1 ] = 1;
  bg[ 2 ] = 7;
  bg[ 3 ] = 7;

  select = false;
  data = 0;
}

FieldListView::Content::~Content()
{
  int i;

  for ( i = 0; i < cur_fields; i++)
    delete texts[i];
  delete texts;
}

void FieldListView::Content::setText( int field, std::string text )
{
  if ( field < 0 ) return;
  if ( field >= cur_fields ) trim( field + 1 );
  *texts[ field ] = text;
}

void FieldListView::Content::trim( int fields )
{
  std::string **nt;
  int i;

  if ( fields < 1 ) return;
  if ( fields == cur_fields ) return;

  nt = new std::string*[ fields ];
  if ( fields < cur_fields ) {
    for ( i = 0; i < fields; i++ ) {
      nt[i] = texts[i];
    }
    for ( i = fields; i < cur_fields; i++ ) {
      delete texts[i];
    }
  } else {
    for ( i = 0; i < cur_fields; i++ ) {
      nt[i] = texts[i];
    }
    for ( i = cur_fields; i < fields; i++ ) {
      nt[i] = new std::string( "" );
    }
  }
  cur_fields = fields;
  delete texts;
  texts = nt;
}

std::string FieldListView::Content::getText( int field )
{
  if ( ( field < 0 ) || ( field >= cur_fields ) ) return "";
  return *texts[field];

}

void FieldListView::setFieldWidth( int field, int width )
{
  if ( ( field < 0 ) || ( field >= fields ) ) return;
  if ( width < -1 ) {
    width = -1;
    used_width[ field ] = -1;
  }
  fieldwidth[ field ] = width;

  redraw();
}

int FieldListView::Content::getLength( int field )
{
  if ( ( field < 0 ) || ( field >= cur_fields ) ) return 0;
  return texts[field]->length();

}

/*
 * setText
 *
 * sets the text for the specified row and field
 *
 * this will not redraw the lv because of the performance
 */
void FieldListView::setText( int row, int field, std::string text )
{
  Content *te;

  if ( ( row < 0 ) || ( row >= elements ) ) return;
  if ( field < 0 ) return;

  te = elementarray[ row ];
  // make changes to te
  
  // test if field already exists
  if ( field >= fields ) {
    // increase fieldwidth
    setNrOfFields( field + 1 );
  }

  // test if new text will change dynamic width
  if ( fieldwidth[ field ] < 0 ) {
    // dynamic field width
    if ( used_width[ field ] >= 0 ) {
      // only change if not unknown
      if ( ( (int)text.length() ) > used_width[ field ] ) {
        // text is longer so only change used_width
        used_width[ field ] = text.length();
      } else if ( ( (int)text.length() ) < used_width[ field ] ) {
        // text is shorter
        // we don't know new width so mark it as unknown
        used_width[ field ] = -1;
      }
    }
  }
  elementarray[ row ]->setText( field, text );
}

void FieldListView::increaseArray( int newsize )
{
  Content **newarray;
  int i;

  if ( newsize < arraysize ) return;

  newarray = new Content*[newsize];
  for ( i = 0; i < elements; i++ ) {
    newarray[i] = elementarray[i];
  }
  delete elementarray;
  elementarray = newarray;
  arraysize = newsize;
}

void FieldListView::checkShrink()
{
  double loadfactor;
  int newsize, i;
  Content **newarray;

  if ( arraysize <= 100 ) return;  // never less then 100 (possible) entires
  
  loadfactor = (double)elements;
  loadfactor /= arraysize;
  if ( loadfactor < .1 ) {
    // only if less the 10 percent used
    newsize = arraysize / 5;  // decrease to 20%
    if ( newsize < 100 ) newsize = 100;
    newarray = new Content*[newsize];
    for ( i = 0; i < elements; i++ ) {
      newarray[i] = elementarray[i];
    }
    delete elementarray;
    elementarray = newarray;
    arraysize = newsize;
  }
}

void FieldListView::setNrOfFields( int nv )
{
  int *nw, *nu, i;
  align_t *na;

  if ( nv < 1 ) return;

  nw = new int[nv];
  nu = new int[nv];
  na = new align_t[nv];
  if ( nv < fields ) {
    for ( i = 0; i < nv; i++ ) {
      nw[i] = fieldwidth[i];
      nu[i] = used_width[i];
      na[i] = fieldalign[i];
    }
  } else {
    for ( i = 0; i < fields; i++ ) {
      nw[i] = fieldwidth[i];
      nu[i] = used_width[i];
      na[i] = fieldalign[i];
    }
    for ( i = fields; i < nv; i++ ) {
      nw[i] = -1;
      nu[i] = -1;
      na[i] = ALIGN_LEFT;
    }
  }
  delete fieldwidth;
  delete used_width;
  delete fieldalign;
  fieldwidth = nw;
  used_width = nu;
  fieldalign = na;
  fields = nv;

  redraw();
}

int FieldListView::getUsedWidth( int field )
{
  int i, w, tw;

  if ( ( field < 0 ) || ( field >= fields ) ) return -1;
  
  if ( used_width[ field ] < 0 ) {
    // unknown => calculate it
    w = 0;
    for ( i = 0; i < elements; i++ ) {
      tw = elementarray[ i ]->getLength( field );
      if ( tw > w ) w = tw;
    }
    used_width[ field ] = w;
  }
  return used_width[ field ];
}

int FieldListView::addRow()
{
  int newrow;

  if ( elements == arraysize ) increaseArray( arraysize * 2 );
  newrow = elements++;
  elementarray[ newrow ] = new Content();
  return newrow;
}

int FieldListView::deleteRow( int row )
{
  int i;

  if ( ( row < 0 ) || ( row >= elements ) ) return -1;
  delete elementarray[ row ];
  for ( i = row + 1; i < elements; i++ ) {
    elementarray[ i - 1 ] = elementarray[ i ];
  }
  elements--;
  if ( activerow >= elements ) activerow = -1;
  checkShrink();

  for ( i = 0; i < fields; i++ ) {
    // invalid dynamic fields
    if ( fieldwidth[i] == -1 )
      used_width[i] = -1;
  }

  // check yoffset
  if ( ( yoffset + maxdisplayv ) >= elements ) yoffset = elements - maxdisplayv;
  if ( yoffset < 0 ) yoffset = 0;

  if ( elements < maxdisplayv ) {
    // not all lines used so reset the unused subwindows
    setupsubwins();
  }

  return 0;
}

void FieldListView::setXOffset( int nv )
{
  int t;

  t = getMaxTextWidth();
  if ( ( nv + maxdisplayh ) >= t ) nv = t - maxdisplayh;
  if ( nv < 0 ) nv = 0;
  if ( nv != xoffset ) {
    xoffset = nv;
    
    redrawContent();
    hbarredraw();
  }
}

int FieldListView::getMaxTextWidth()
{
  //TODO: Vielleicht zwischenspeichern und nur bei deleteRow und setText aktualisieren
  int i, wi;

  wi = 0;
  for ( i = 0; i < fields; i++ ) {
    if ( fieldwidth[i] < 0 ) {
      wi += getUsedWidth( i );
    } else {
      wi += fieldwidth[i];
    }
  }
  return wi;
}

void FieldListView::redrawContent()
{
  char *line;
  int i;
  
  line = (char*)malloc( sizeof(char) * ( maxdisplayh + 1 ) );
  
  for ( i = 0; i < maxdisplayv; i++ ) {
    if ( ( i + yoffset ) >= elements ) break;
    redraw( yoffset + i, line );
  }
  
  free( line );
}

void FieldListView::setYOffset( int nv )
{
  if ( ( nv + maxdisplayv ) >= elements ) nv = elements - maxdisplayv;
  if ( nv < 0 ) nv = 0;
  if ( nv != yoffset ) {
    yoffset = nv;

    redrawContent();
    setupsubwins();
    vbarredraw();
  }
}

void FieldListView::setFieldAlign( int field, align_t nv )
{
  if ( ( field < 0 ) || ( field >= fields ) ) return;
  fieldalign[ field ] = nv;

  redraw();
}

void FieldListView::setSize( int rows )
{
  int i;
  if ( rows < elements ) {
    for ( i = rows; i < elements; i++ )
      delete elementarray[ i ];
    elements = rows;
    checkShrink();
  } else if ( rows > elements ) {
    if ( rows > arraysize ) increaseArray( rows );
    for ( i = elements; i < rows; i++ ) {
      elementarray[ i ] = new Content();
    }
    elements = rows;
  }
  if ( activerow >= elements ) activerow = -1;

  for ( i = 0; i < fields; i++ ) {
    // invalid dynamic fields
    if ( fieldwidth[i] == -1 )
      used_width[i] = -1;
  }

  // check yoffset
  if ( ( yoffset + maxdisplayv ) >= elements ) yoffset = elements - maxdisplayv;
  if ( yoffset < 0 ) yoffset = 0;

  if ( elements < maxdisplayv ) {
    // not all lines used so reset the unused subwindows
    setupsubwins();
  }
}

void FieldListView::redraw( int element )
{
  char *buf;

  if ( ( element < 0 ) || ( element >= elements ) ) return;
  if ( ( element >= yoffset ) && ( element < ( yoffset + maxdisplayv ) ) ) {
    buf = (char*)malloc( sizeof(char) * ( maxdisplayh + 1 ) );
    redraw( element, buf );
    free( buf );
  }
}

void FieldListView::redraw( int element, char *line )
{
  std::string s;
  const char *cstr;
  int sx, cpos, ex, vi_offset_s, vi_offset_c, f, len, tlen, ssx, toff;
  int vis;
  int tfg, tbg;

  if ( ( element < 0 ) || ( element >= elements ) ) return;
  if ( ( element >= yoffset ) && ( element < ( yoffset + maxdisplayv ) ) ) {
    memset( line, ' ', maxdisplayh );
    
    sx = 0;
    cpos = 0;
    for ( f = 0; f < fields; f++ ) {
      if ( fieldwidth[ f ] < 0)
	ex = sx + getUsedWidth( f );
      else
	ex = sx + fieldwidth[ f ];
      
      if ( ( sx < ( xoffset + maxdisplayh ) ) && ( ex >= xoffset ) ) {
	// field visible
	vi_offset_c = a_min( xoffset + maxdisplayh, ex ) - a_max( sx, xoffset );
	if ( fieldalign[ f ] == ALIGN_RIGHT ) {
	  if ( ex < ( xoffset + maxdisplayh ) )
	    vi_offset_s = 0;
	  else
	    vi_offset_s = ex - ( xoffset + maxdisplayh ); 
	  // remember vi_offset_s is now from right
	  
	  s = elementarray[ element ]->getText( f );
	  len = s.length();
	  if ( len > 0 ) {
	    cstr = s.c_str();
	    if ( len > vi_offset_s ) {
	      // something visible from this entry
	      tlen = len - vi_offset_s;
	      // only up to tlen chars
	      if ( tlen <= vi_offset_c ) {
		// everything from beginning visible
		ssx = 0;
		toff = vi_offset_c - tlen;
	      } else {
		ssx = tlen - vi_offset_c;
		toff = 0;
	      }
	      // toff + bytes copyied are always <= vi_offset-c
	      // if toff is 0 then because of the min we only copy up to vi_offset_c chars
	      // otherwise toff is vi_offset_c - tlen and because tlen<= vi_offset_c we copy
	      // only tlen bytes => toff + bytes copyied = vi_offset_c - tlen + tlen
	      // 
	      // done to show we do not copy more bytes then available
	      memcpy( line + cpos + toff, cstr + ssx, a_min( tlen, vi_offset_c ) );
	    }
	  }	  
	} else {
	  if ( sx >= xoffset )
	    vi_offset_s = 0;
	  else
	    vi_offset_s = xoffset - sx;
	  
	  // append at cpos from field beginning with vi_offset_s vi_offset_c chars
	  s = elementarray[ element ]->getText( f );
	  len = s.length();
	  if ( len > 0 ) {
	    cstr = s.c_str();
	    if ( len > vi_offset_s ) {
	      // something visible from this entry
	      memcpy( line + cpos, cstr + vi_offset_s, a_min( vi_offset_c, len - vi_offset_s ) );
	    }
	  }
	}
	cpos += vi_offset_c;
      }
      
      sx = ex;
    }
    line[ cpos ] = '\0';

    vis = element - yoffset;
    // element IS visible so vis correct for the winbgs/subwins array!

    if ( element == activerow ) {
      if ( elementarray[ element ]->getSelect() == true ) {
	tfg = elementarray[ element ]->getFG( CC_SELACT );
	tbg = elementarray[ element ]->getBG( CC_SELACT );
      } else {
	tfg = elementarray[ element ]->getFG( CC_ACTIVE );
	tbg = elementarray[ element ]->getBG( CC_ACTIVE );
      }
    } else {
      if ( elementarray[ element ]->getSelect() == true ) {
	tfg = elementarray[ element ]->getFG( CC_SELECT );
	tbg = elementarray[ element ]->getBG( CC_SELECT );
      } else {
	tfg = elementarray[ element ]->getFG( CC_NORMAL );
	tbg = elementarray[ element ]->getBG( CC_NORMAL );
      }
    }
    if ( winbgs[ vis ] != tbg ) {
      aguix->SetWindowBG( subwins[ vis ], tbg );
      winbgs[ vis ] = tbg;
    }
    aguix->ClearWin( subwins[ vis ] );
    if ( font == NULL )
      aguix->setFG( tfg );
    else
      aguix->setFG( font->getGC(), tfg );
    if ( font == NULL )
      aguix->DrawText( subwins[ vis ], line, 1, 0 );
    else
      aguix->DrawText( subwins[ vis ], font, line, 1, 0 );
  }
}

void FieldListView::Content::setFG( colorclass_t cc, int col )
{
  fg[ cc ] = col;
}

int FieldListView::Content::getFG( colorclass_t cc )
{
  return fg[ cc ];
}

void FieldListView::Content::setBG( colorclass_t cc, int col )
{
  bg[ cc ] = col;
}

int FieldListView::Content::getBG( colorclass_t cc )
{
  return bg[ cc ];
}

void FieldListView::setFG( int row, colorclass_t cc, int col )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return;
  elementarray[ row ]->setFG( cc, col );
}

int FieldListView::getFG( int row, colorclass_t cc )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return 0;
  return elementarray[ row ]->getFG( cc );
}

void FieldListView::setBG( int row, colorclass_t cc, int col )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return;
  elementarray[ row ]->setBG( cc, col );
}

int FieldListView::getBG( int row, colorclass_t cc )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return 0;
  return elementarray[ row ]->getBG( cc );
}

void FieldListView::Content::setSelect( bool nv )
{
  select = nv;
}

bool FieldListView::Content::getSelect()
{
  return select;
}

/*
 * setSelectQ: set select status
 * only disadvantage to setSelect is that the row isn't redrawed
 */
void FieldListView::setSelectQ( int row, bool nv )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return;
  elementarray[ row ]->setSelect( nv );
}

void FieldListView::setSelect( int row, bool nv )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return;
  setSelectQ( row, nv );
  redraw( row );
}

bool FieldListView::getSelect( int row )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return false;
  return elementarray[ row ]->getSelect();
}

/*
 * setActiveRowQ
 * sets the active row without redrawing of affected rows
 */
void FieldListView::setActiveRowQ( int nv )
{
  if ( ( nv < -1 ) || ( nv >= elements ) ) return;
  activerow = nv;
}

void FieldListView::setActiveRow( int nv )
{
  int oldact;

  if ( ( nv < -1 ) || ( nv >= elements ) ) return;

  // remember old activerow for redraw
  oldact = activerow;

  setActiveRowQ( nv );

  redraw( activerow );
  if ( oldact >= 0 ) redraw( oldact );  // redraw old active element
}

int FieldListView::getActiveRow()
{
  return activerow;
}

void FieldListView::resize( int tw, int th )
{
  int oldmaxdv;

  if ( ( tw < 1 ) || ( th < 1 ) ) return;
  this->w = tw;
  this->h = th;
  if ( parent != NULL ) {
    parent->resizeSubWin( win, tw, th );
    oldmaxdv = maxdisplayv;
    recalcMaxValues();
    rebuildSubWins( oldmaxdv );
    reconfSubWins();
    setupBars();
    mredraw();
  }
}

int FieldListView::getData()
{
  return data;
}

void FieldListView::setData( int tdata )
{
  this->data = tdata;
}

void FieldListView::redraw()
{
  mredraw();
  hbarredraw();
  vbarredraw();
  redrawContent();
}

void FieldListView::flush()
{
}

void FieldListView::handleExpose( Window msgwin )
{
  if ( msgwin == win ) mredraw();
  else if ( msgwin == hbarwin ) hbarredraw();
  else if ( msgwin == vbarwin ) vbarredraw();
  else {
    int elements = getElements();
    for ( int i = 0; i < maxdisplayv ; i++ ) {
      if ( ( i + yoffset ) >= elements ) break;
      if ( subwins[ i ] == msgwin ) {
	redraw( i + yoffset );
	break;
      }
    }
  }
}

bool FieldListView::handleMessage( XEvent *E, Message *msg )
{
  bool returnvalue;
  struct timeval t2;
  int dt, s, us, scrollspeed;

  returnvalue = false;
  if ( msg->type == Expose ) {
    handleExpose( msg->window );
  } else if ( msg->type == ButtonPress ) {
    // jetzt men wir was machen
    bool isc = false;
    if ( msg->window == hbarwin ) {
      handleHBar( msg );
      isc = true;
    } else if ( msg->window == vbarwin ) {
      handleVBar( msg );
      isc = true;
    } else if ( msg->window == win ) {
      isc = true;
    } else {
      bool fulls;
      if ( msg->button == Button3 ) fulls = true;
      else fulls = false;
      int elements = getElements();
      bool handle = true, found = false;
      for ( int i = 0; i < maxdisplayv; i++ ) {
        if ( ( ( i + yoffset ) >= elements ) && ( fulls == false ) ) handle = false;
        if ( subwins[ i ] == msg->window ) {
	  found = true;
	  break;
	}
      }
      if ( ( handle == true ) && ( found == true ) ) {
        // handleSelect, handleThird
        if ( msg->button == Button3 ) handleThird( msg );
	else if ( ( msg->button == Button4 ) || ( msg->button == Button5 ) ) {
	  gettimeofday( &t2, NULL );
	  s = abs( t2.tv_sec - lastwheel.tv_sec );
	  us = t2.tv_usec - lastwheel.tv_usec;
	  if ( us < 0 ) {
	    us += 1000000;
	    s--;
	  }
	  dt = us + s * 1000000;
	  dt /= 100000;
	  if ( dt < 2 ) scrollspeed = 5;
	  else if ( dt < 4 ) scrollspeed = 2;
	  else scrollspeed = 1;
	  if ( msg->button == Button4 ) scrollV1( -scrollspeed );
	  else if ( msg->button == Button5 ) scrollV1( scrollspeed );
	  lastwheel = t2;
        } else if ( msg->button == Button2 )
	  handleMiddle( msg );
	else
	  handleSelect( msg );
      }
      if ( found == true ) isc = true;
    }
    if ( isc == true ) {
      AGMessage *agmsg = (AGMessage*)_allocsafe( sizeof( AGMessage ) );
      agmsg->fieldlv.lv = this;
      agmsg->fieldlv.row = -1;
      agmsg->fieldlv.time = msg->time;
      agmsg->type = AG_FIELDLV_PRESSED;
      aguix->putAGMsg( agmsg );
    }
  } else if ( ( msg->type == KeyPress ) || ( msg->type == KeyRelease ) ) {
    if ( hasFocus() == true ) {
      // we have the focus so let's react to some keys
      // lets call an extra handler so it can be overwritten
      returnvalue = handleKeys( msg );
    }
  }
  return returnvalue;
}

void FieldListView::setParent(AWindow *tparent)
{
  int tbg = mbg;
  int tx, ty, tw, th;

  this->parent = tparent;
  win = tparent->getSubWindow( 0, x, y, w, h );
  recalcMaxValues();
  subwins = new Window[ maxdisplayv ];
  winbgs = new int[ maxdisplayv ];
  tw = w - 2;
  if ( vbar != 0 ) tw -= vbar_width;
  if ( font == NULL ) th = aguix->getCharHeight();
  else th = font->getCharHeight();
  if ( hbar == 1 ) ty = 1 + hbar_height;
  else ty = 1;
  if ( vbar == 1 ) tx = 1 + vbar_width;
  else tx = 1;
  for ( int i = 0; i < maxdisplayv; i++ ) {
    subwins[ i ] = tparent->getSubWindow( win, tx, ty + th * i, tw, th );
    aguix->SetWindowBG( subwins[ i ], tbg );
    aguix->ClearWin( subwins[ i ] );
    winbgs[ i ] = tbg;
  }
  if ( hbar != 0 ) hbarwin = tparent->getSubWindow( win, 0, 0, 5, 5 );
  if ( vbar != 0 ) vbarwin = tparent->getSubWindow( win, 0, 0, 5, 5 );  
  setupBars();
}

void FieldListView::hbarredraw()
{
  if ( hbar == 0 ) return;
  int tx, ty, tw, th;
  int maxlen = getMaxLen();

  tw = w - 2;
  if ( vbar != 0 ) tw -= vbar_width;
  th = hbar_height;
  ty = 0;
  tx = 0;

  aguix->SetWindowBG( hbarwin, parent->getBG() );
  aguix->ClearWin( hbarwin );
  if ( hbar_dir == 1 ) {
    aguix->setFG( 1 );
  } else {
    aguix->setFG( 2 );
  }
  aguix->DrawLine( hbarwin, tx, ty + th - 1, tx, ty );
  aguix->DrawLine( hbarwin, tx, ty, tx + th - 1, ty );
  if ( hbar_dir == 1 ) {
    aguix->setFG( 2 );
  } else {
    aguix->setFG( 1 );
  }
  aguix->DrawLine( hbarwin, tx + 1, ty + th - 1, tx + th - 1, ty + th - 1 );
  aguix->DrawLine( hbarwin, tx + th - 1, ty + th - 1, tx + th - 1, ty + 1 );
  // Left-Arrow
  aguix->setFG( 1 );
  aguix->DrawLine( hbarwin, tx + th * 4 / 5, ty + th / 5, tx + th / 5, ty + th / 2 );
  aguix->DrawLine( hbarwin, tx + th / 5, ty + th / 2, tx + th * 4 / 5, ty + th * 4 / 5 );

  aguix->DrawLine( hbarwin, tx + th, ty + th - 1, tx + tw - th - 1, ty + th - 1 );
  aguix->setFG( 2 );
  aguix->DrawLine( hbarwin, tx + th, ty, tx + tw - th - 1, ty );

  if ( hbar_dir == 2 ) {
    aguix->setFG( 1 );
  } else {
    aguix->setFG( 2 );
  }
  aguix->DrawLine( hbarwin,
		   tx + tw - th,
		   ty + th - 1,
		   tx + tw - th,
		   ty );
  aguix->DrawLine( hbarwin,
		   tx + tw - th,
		   ty,
		   tx + tw - 1 ,
		   ty );
  if ( hbar_dir == 2 ) {
    aguix->setFG( 2 );
  } else {
    aguix->setFG( 1 );
  }
  aguix->DrawLine( hbarwin,
		   tx + tw - th + 1,
		   ty + th - 1,
		   tx + tw - 1,
		   ty + th - 1 );
  aguix->DrawLine( hbarwin,
		   tx + tw - 1,
		   ty + th - 1,
		   tx + tw - 1,
		   ty + 1 );
  // Right-Arrow
  aguix->setFG( 1 );
  aguix->DrawLine( hbarwin,
		   tx + tw - th * 4 / 5,
		   ty + th / 5,
		   tx + tw - th / 5,
		   ty + th / 2 );
  aguix->DrawLine( hbarwin,
		   tx + tw - th / 5,
		   ty + th / 2,
		   tx + tw - th * 4 / 5,
		   ty + th * 4 / 5 );
  // jetzt noch die eigentliche Leiste
  tw -= 2 * th;  // Breite wenn alles anzeigbar ist
  th -= 2;
  ty++;
  if ( maxlen > maxdisplayh ) {
    int dw = tw * maxdisplayh;
    dw /= maxlen;
    int a = maxlen - maxdisplayh;
    double b = tw - dw;  // das was brig bleibt
    b /= a;            // verteilt sich auch die nichtsichtbaren Zeichen
    tx = tx + hbar_height + (int)( xoffset * b );
    tw = dw;
  } else {
    tx = tx + hbar_height;
  }
  if ( hbar_pressed == false ) {
    aguix->setFG( 2 );
  } else {
    aguix->setFG( 1 );
  }
  aguix->DrawLine( hbarwin,
		   tx,
		   ty + th - 1,
		   tx,
		   ty );
  aguix->DrawLine( hbarwin,
		   tx,
		   ty,
		   tx + tw - 1,
		   ty );
  if ( hbar_pressed == false ) {
    aguix->setFG( 1 );
  } else {
    aguix->setFG( 2 );
  }
  aguix->DrawLine( hbarwin,
		   tx + 1,
		   ty + th - 1,
		   tx + tw - 1,
		   ty + th - 1 );
  aguix->DrawLine( hbarwin,
		   tx + tw - 1,
		   ty + th - 1,
		   tx + tw - 1,
		   ty + 1 );  
}

void FieldListView::vbarredraw()
{
  if(vbar==0) return;
  int tx,ty,tw,th;

  th=h-2;
  if(hbar!=0) th-=hbar_height;
  tw=vbar_width;
  tx=0;
  ty=0;

  aguix->SetWindowBG(vbarwin,parent->getBG());
  aguix->ClearWin(vbarwin);
  if(vbar_dir==1) {
    aguix->setFG(1);
  } else {
    aguix->setFG(2);
  }
  aguix->DrawLine(vbarwin,tx,ty+tw-1,tx,ty);
  aguix->DrawLine(vbarwin,tx,ty,tx+tw-1,ty);
  if(vbar_dir==1) {
    aguix->setFG(2);
  } else {
    aguix->setFG(1);
  }
  aguix->DrawLine(vbarwin,tx+1,ty+tw-1,tx+tw-1,ty+tw-1);
  aguix->DrawLine(vbarwin,tx+tw-1,ty+tw-1,tx+tw-1,ty+1);
  // Up-Arrow
  aguix->setFG(1);
  aguix->DrawLine(vbarwin,tx+tw/5,ty+tw*4/5,tx+tw/2,ty+tw/5);
  aguix->DrawLine(vbarwin,tx+tw/2,ty+tw/5,tx+tw*4/5,ty+tw*4/5);

  aguix->DrawLine(vbarwin,tx+tw-1,ty+tw,tx+tw-1,ty+th-tw-1);
  aguix->setFG(2);
  aguix->DrawLine(vbarwin,tx,ty+tw,tx,ty+th-tw-1);

  if(vbar_dir==2) {
    aguix->setFG(1);
  } else {
    aguix->setFG(2);
  }
  aguix->DrawLine(vbarwin,tx,ty+th-1,tx,ty+th-tw);
  aguix->DrawLine(vbarwin,tx,ty+th-tw,tx+tw-1,ty+th-tw);
  if(vbar_dir==2) {
    aguix->setFG(2);
  } else {
    aguix->setFG(1);
  }
  aguix->DrawLine(vbarwin,tx+1,ty+th-1,tx+tw-1,ty+th-1);
  aguix->DrawLine(vbarwin,tx+tw-1,ty+th-1,tx+tw-1,ty+th-tw);
  // Down-Arrow
  aguix->setFG(1);
  aguix->DrawLine(vbarwin,tx+tw/5,ty+th-tw*4/5,tx+tw/2,ty+th-tw/5);
  aguix->DrawLine(vbarwin,tx+tw/2,ty+th-tw/5,tx+tw*4/5,ty+th-tw*4/5);
  // jetzt noch die eigentliche Leiste
  th-=2*tw;  // Hhe wenn alles anzeigbar ist
  tw-=2;
  tx++;
  int elements=getElements();
  if(elements>maxdisplayv) {
    int dh=th*maxdisplayv;
    dh/=elements;
    int a=elements-maxdisplayv;
    double b=th-dh;  // das was brig bleibt
    b/=a;            // verteilt sich auch die nichtsichtbaren Zeichen
    ty=ty+vbar_width+(int)(yoffset*b);
    th=dh;
  } else {
    ty=ty+vbar_width;
  }
  if(vbar_pressed==false) {
    aguix->setFG(2);
  } else {
    aguix->setFG(1);
  }
  aguix->DrawLine(vbarwin,tx,ty+th-1,tx,ty);
  aguix->DrawLine(vbarwin,tx,ty,tx+tw-1,ty);
  if(vbar_pressed==false) {
    aguix->setFG(1);
  } else {
    aguix->setFG(2);
  }
  aguix->DrawLine(vbarwin,tx+1,ty+th-1,tx+tw-1,ty+th-1);
  aguix->DrawLine(vbarwin,tx+tw-1,ty+th-1,tx+tw-1,ty+1);  
}

void FieldListView::mredraw()
{
  if ( bgset == false ) {
    aguix->SetWindowBG( win, mbg );
    bgset = true;
  }
  aguix->ClearWin( win );
  aguix->setFG( 2 );
  aguix->DrawLine( win, 0, h - 1, 0, 0 );
  aguix->DrawLine( win, 0, 0, w - 1, 0 );
  aguix->setFG( 1 );
  aguix->DrawLine( win, 1, h - 1, w - 1, h - 1 );
  aguix->DrawLine( win, w - 1, h - 1, w - 1, 1 );
  if ( ( hbar != 0 ) && ( vbar != 0 ) ) {
    int tx, ty;
    tx = ( vbar == 1 ) ? 1 : ( w - 1 - vbar_width );
    ty = ( hbar == 1 ) ? 1 : ( h - 1 - hbar_height );
    aguix->setFG( 0 );
    aguix->FillRectangle( win, tx, ty, vbar_width, hbar_height );
  }
  aguix->Flush();
}

void FieldListView::recalcMaxValues()
{
  int th, tw;

  if ( hbar != 0 ) {
    th = h - 2 - hbar_height;
  } else {
    th = h - 2;
  }
  if ( font == NULL ) maxdisplayv = th / ( aguix->getCharHeight() );
  else maxdisplayv = th / ( font->getCharHeight() );
  if ( vbar != 0 ) {
    tw = w - 2 - vbar_width;
  } else {
    tw = w - 2;
  }
  if ( font == NULL ) maxdisplayh = tw / ( aguix->getCharWidth() );
  else maxdisplayh = tw / ( font->getCharWidth() );
}

int FieldListView::getElements()
{
  return elements;
}

int FieldListView::getXOffset()
{
  return xoffset;
}

int FieldListView::getYOffset()
{
  return yoffset;
}

int FieldListView::getMaxLen()
{
  return getMaxTextWidth();
}

int FieldListView::getMaxDisplayV()
{
  return maxdisplayv;
}

int FieldListView::getMaxDisplayH()
{
  return maxdisplayh;
}

void FieldListView::setHBarState( int state )
{
  int oldmaxdv;

  oldmaxdv = maxdisplayv;
  if ( hbar == state ) return;
  if ( ( state == 0 ) && ( parent != NULL ) ) parent->removeSubWin( hbarwin );
  if ( ( hbar == 0 ) && ( state > 0 ) ) hbarwin = parent->getSubWindow( win, 0, 0, 5, 5 );
  hbar = state;
  recalcMaxValues();
  rebuildSubWins( oldmaxdv );
  reconfSubWins();
  setupBars();
}

void FieldListView::setVBarState(int state)
{
  if ( state == vbar ) return;
  if ( ( state == 0 ) && ( parent != NULL ) ) parent->removeSubWin( vbarwin );
  if ( ( vbar == 0 ) && ( state > 0 ) ) vbarwin = parent->getSubWindow( win, 0, 0, 5, 5 );
  vbar = state;
  recalcMaxValues();
  reconfSubWins();
  setupBars();
}

int FieldListView::getHBarState()
{
  return hbar;
}

int FieldListView::getVBarState()
{
  return vbar;
}

void FieldListView::setVBarWidth( int new_width )
{
  if ( ( new_width < 0 ) || ( ( new_width + 4 ) >= w ) ) return;
  vbar_width = new_width;
  recalcMaxValues();
  reconfSubWins();
  setupBars();
  vbarredraw();
}

int FieldListView::getVBarWidth()
{
  return vbar_width;
}

void FieldListView::setHBarHeight( int new_height )
{
  if ( ( new_height < 0 ) || ( ( new_height + 4 ) >= h ) ) return;
  int oldmaxdv;
  oldmaxdv = maxdisplayv;
  hbar_height = new_height;
  recalcMaxValues();
  rebuildSubWins( oldmaxdv );
  reconfSubWins();
  setupBars();
  hbarredraw();
}

int FieldListView::getHBarHeight()
{
  return hbar_height;
}

/*
 * setupsubwins
 *
 * will only reset subwindow-background for unused entries
 *
 */
void FieldListView::setupsubwins()
{
  int elements = getElements();
  if ( ( yoffset + maxdisplayv ) > elements ) {
    // Alle SubWindows, die kein LVC enthalten, auf parent->getBG() setzen
    int tbg = mbg;
    for ( int i = elements - yoffset; i < maxdisplayv; i++ ) {
      aguix->SetWindowBG( subwins[ i ], tbg );
      aguix->ClearWin( subwins[ i ] );
      winbgs[ i ] = tbg;
    }
  }
}

/*
 * rebuildSubWins
 *
 * will rebuild subwindows when maxdisplayv changed
 * old value needed
 */
void FieldListView::rebuildSubWins( int oldsize )
{
  if ( oldsize == maxdisplayv ) return;
  Window *nsubwins;
  int *nwinbgs;
  int i;

  nsubwins = new Window[ maxdisplayv ];
  nwinbgs = new int[ maxdisplayv ];
  if ( maxdisplayv < oldsize ) {
    for ( i = maxdisplayv; i < oldsize; i++ ) {
      parent->removeSubWin( subwins[ i ] );
    }
    for ( i = 0; i < maxdisplayv; i++ ) {
      nsubwins[ i ] = subwins[ i ];
      nwinbgs[ i ] = winbgs[ i ];
    }
    delete subwins;
    delete winbgs;
    subwins = nsubwins;
    winbgs = nwinbgs;
  } else {
    for ( i = 0; i < oldsize; i++ ) {
      nsubwins[ i ] = subwins[ i ];
      nwinbgs[ i ] = winbgs[ i ];
    }
    int tx, ty, tw, th;
    tw = w - 2;
    if ( vbar != 0 ) tw -= vbar_width;
    if ( font == NULL ) th = aguix->getCharHeight();
    else th = font->getCharHeight();
    if ( hbar == 1 ) ty = 1 + hbar_height;
    else ty = 1;
    if ( vbar == 1 ) tx = 1 + vbar_width;
    else tx = 1;
    int tbg = mbg;
    for ( i = oldsize; i < maxdisplayv; i++ ) {
      nsubwins[ i ] = parent->getSubWindow( win, tx, ty + th * i, tw, th );
      aguix->SetWindowBG( nsubwins[ i ], tbg );
      aguix->ClearWin( nsubwins[ i ] );
      nwinbgs[ i ] = tbg;
    }
    delete subwins;
    delete winbgs;
    subwins = nsubwins;
    winbgs = nwinbgs;
    int elements = getElements();
    if ( ( yoffset + maxdisplayv ) > elements ) {
      int ny;
      ny = elements - maxdisplayv;
      if ( ny < 0 ) ny = 0;
      setYOffset( ny );
    }
  }
}

/*
 * reconfSubWins
 *
 * will move and resize the subwindows
 * needed when vbar/hbar/widht... changed
 */
void FieldListView::reconfSubWins()
{
  int tx, ty, tw, th;
  tw = w - 2;
  if ( vbar != 0 ) tw -= vbar_width;
  if ( font == NULL ) th = aguix->getCharHeight();
  else th = font->getCharHeight();
  if ( hbar == 1 ) ty = 1 + hbar_height;
  else ty = 1;
  if ( vbar == 1 ) tx = 1 + vbar_width;
  else tx = 1;
  for ( int i = 0; i < maxdisplayv; i++ ) {
    parent->resizeSubWin( subwins[ i ], tw, th );
    parent->moveSubWin( subwins[ i ], tx, ty + i * th );
  }
}

void FieldListView::handleSelect( Message *msg )
{
  int j = -1;
  int dely, i;
  int elems = getElements();
  bool ende;

  for ( i = 0; i < maxdisplayv; i++ ) {
    if ( msg->window == subwins[ i ] ) j = i;
  }
  if ( j == -1 ) return;
  if ( ( j + yoffset ) > elems ) return;

  int ch;
  if ( font == NULL ) ch = aguix->getCharHeight();
  else ch = font->getCharHeight();

  int tpos, firstselect;
  tpos = j + yoffset;

  bool state = getSelect( tpos );
  state = ( state == false ) ? true : false;
  setSelect( tpos, state );
  setActiveRow( tpos );
  runSelectHandler( tpos );
  aguix->Flush();

  dely = j * ch;

  Message *tmsg;
  int newelement, element = j, my, elementh = ch;
  int dest, source;
  int selected = 1;
  bool ignore4end = false;

  firstselect = tpos;

  newelement = element;

  Time lasttime = 0;
  for ( ende = false; ende == false; ) {
    tmsg =aguix->wait4messReact( MES_GET, false );
    if ( tmsg != NULL ) {
      if ( tmsg->type == ButtonRelease ) {
        ende = true;
	lasttime = tmsg->time;

	my = tmsg->mousey;
	my += dely;
	newelement = my / elementh;
	if ( my < 0 ) newelement--;
	ignore4end = false;  // to allow apply new elementselection
      } else if ( ( tmsg->type == MotionNotify ) || ( tmsg->type == EnterNotify ) || ( tmsg->type == LeaveNotify ) ) {
        Window root, child;
        int root_x, root_y, win_x, win_y;
        unsigned int keys_buttons;

        XQueryPointer( aguix->getDisplay(),
		       msg->window,
		       &root,
		       &child,
		       &root_x,
		       &root_y,
		       &win_x,
		       &win_y,
		       &keys_buttons );
	if ( ( keys_buttons & Button1Mask ) == 0 ) {
	  // Release before Release-Msg from X
	  // this really happens
	  ignore4end = true;
	} else {
	  my = win_y;
	  my += dely;
	  newelement = my / elementh;
	  if ( my < 0 ) newelement--;
	}
      }
      _freesafe( tmsg );
    } else {
      waittime( 50 );
      Window root, child;
      int root_x, root_y, win_x, win_y;
      unsigned int keys_buttons;

      XQueryPointer( aguix->getDisplay(),
		     msg->window,
		     &root,
		     &child,
		     &root_x,
		     &root_y,
		     &win_x,
		     &win_y,
		     &keys_buttons );
      if ( ( keys_buttons & Button1Mask ) == 0 ) {
	// Release before Release-Msg from X
	// this really happens
	ignore4end = true;
      } else {
	my = win_y;
	my += dely;
	newelement = my / elementh;
	if ( my < 0 ) newelement--;
      }
    }
    if ( ignore4end == false ) {
      if ( ( element != newelement ) || ( newelement < 0 ) || ( newelement >= maxdisplayv ) ) {
	dest = newelement;
	if ( newelement < 0 ) {
	  scrollV1( -1 );
	  dest = 0;
	  element++;
	} else if ( newelement >= maxdisplayv ) {
	  scrollV1( 1 );
	  dest = maxdisplayv - 1;
	  element--;
	}
	source = element;
	if ( source < 0 ) source = 0;
	if ( source >= maxdisplayv ) source = maxdisplayv - 1;
	int starte, tende, dire;
	starte = source;
	tende = dest;
	if ( starte < tende ) {
	  starte++;
	  dire = 1;
	} else if ( starte > tende ) {
	  starte--;
	  dire = -1;
	} else dire = 0;

	// for i = starte to tende step dire
	for ( i = starte; ; i += dire ) {
	  if ( ( i + yoffset ) >= elems ) break;
	  tpos = i + yoffset;
	  setSelect( tpos, state );
	  runSelectHandler( tpos );
	  selected++;
	  if ( i == tende ) break;
	}
	element = dest;
	if ( ( element + yoffset ) >= elems ) {
	  element = elems - yoffset - 1;
	}
	tpos = element + yoffset;
	if ( tpos < elems ) {
	  setActiveRow( tpos );
	  runSelectHandler( tpos );
	}
      }
    }
  }
  AGMessage *agmsg = (AGMessage*)_allocsafe( sizeof( AGMessage ) );
  agmsg->fieldlv.lv = this;
  agmsg->fieldlv.row = firstselect;
  agmsg->fieldlv.time = lasttime;
  if ( selected == 1 ) {
    agmsg->type = AG_FIELDLV_ONESELECT;
  } else {
    agmsg->type = AG_FIELDLV_MULTISELECT;
  }
  aguix->putAGMsg( agmsg );
}

void FieldListView::handleThird(Message *msg)
{
  int midx, midy, dsx, dsy, xdir, ydir;
  int mx, my;
  int dely;
  bool ende;
  Window relwin;
  
  midx = w - 2;
  if ( vbar != 0 ) midx -= vbar_width;
  dsx = midx;
  midx /= 2;
  midy = h - 2;
  if ( hbar != 0 ) midy -= hbar_height;
  dsy = midy;
  midy /= 2;
  dsx /= 6;
  dsy /= 6;
  mx = msg->mousex;
  my = msg->mousey;

  Message *tmsg;

  int j = -1;
  for ( int i = 0; i < maxdisplayv; i++ ) {
    if ( msg->window == subwins[ i ] ) j = i;
  }
  if ( j == -1 ) return;
  relwin = subwins[ j ];
  if ( font == NULL ) dely = j * aguix->getCharHeight();
  else dely = j * font->getCharHeight();
  my = my + dely;

  int waitt;

  for ( ende = false; ende == false; ) {
    if ( mx < ( midx - dsx ) ) xdir = -1;
    else if ( mx > ( midx + dsx ) ) xdir = 1;
    else xdir = 0;

    if ( my < ( midy - dsy ) ) ydir = -1;
    else if ( my > ( midy + dsy ) ) ydir = 1;
    else ydir = 0;

    if ( xdir != 0 ) setXOffset( xoffset + xdir );
    if ( ydir != 0 ) scrollV( ydir );

    aguix->Flush();

    if ( ( my < dsy ) || ( my > ( midy + 2 * dsy ) ) ) waitt = 20;
    else if ( ( mx < dsx ) || ( mx > ( midx + 2 * dsx ) ) ) waitt = 20;
    else waitt = 50;
    if ( ende == false ) waittime( waitt );
    tmsg = aguix->wait4messReact( MES_GET, false );
    if ( tmsg != NULL ) {
      if ( tmsg->type == ButtonRelease ) ende = true;
      else if ( tmsg->type == MotionNotify ) {
	aguix->queryPointer( relwin, &mx, &my );
        my = my + dely;
      }
      _freesafe( tmsg );
    }
  }
}

bool FieldListView::handleHBar( Message *msg )
{
  if ( hbar == 0 ) return false;
  int mx, my;
  int dir;
  mx = msg->mousex;
  my = msg->mousey;
  int hx, hy, hw, hh;
  bool ende;
  int maxlen = getMaxLen();

  hw = w - 2;
  if ( vbar != 0 ) hw -= vbar_width;
  hh = hbar_height;
  hy = 0;
  hx = 0;
  dir = 0;
  if ( ( mx > hx ) && ( mx <= ( hx + hh ) ) ) {
    // left
    dir = -1;
    hbar_dir = 1;
  }
  if ( ( mx > ( hx + hw - hh ) ) && ( mx <= ( hx + hw ) ) ) {
    // right
    dir = 1;
    hbar_dir = 2;
  }
  if ( dir != 0 ) {
    hbarredraw();
    aguix->Flush();
    Message *tmsg,*tmsg2;
    for(ende=false;ende==false;) {
      setXOffset(xoffset+dir);
      aguix->Flush();
      if(ende==false) waittime(70);
      tmsg=aguix->wait4messReact(MES_GET,false);
      if(tmsg!=NULL) {
        if(tmsg->type==ButtonRelease) ende=true;
        else {
          while((tmsg2=aguix->wait4messReact(MES_GET,false))!=NULL) {
            if(tmsg2->type==ButtonRelease) ende=true;
            _freesafe(tmsg2);
          }
        }
        _freesafe(tmsg);
      }
    }
    hbar_dir=0;
    hbarredraw();
    aguix->Flush();
  } else {
    int tx,ty,tw,th;
    double b=0;
    tw=w-2;
    if(vbar!=0) tw-=vbar_width;
    th=hbar_height;
    ty=0;
    tx=0;
    tw-=2*th;  // Breite wenn alles anzeigbar ist
    th-=2;
    ty++;
    if(maxlen>maxdisplayh) {
      int dw=tw*maxdisplayh;
      dw/=maxlen;
      int a=maxlen-maxdisplayh;
      b=tw-dw;  // das was brig bleibt
      b/=a;            // verteilt sich auch die nichtsichtbaren Zeichen
      tx=tx+hbar_height+(int)(xoffset*b);
      tw=dw;
    } else {
      tx=tx+hbar_height;
    }
    if((mx>=tx)&&(mx<=(tx+tw))&&(my>=ty)&&(my<=(ty+th))) {
      // the hbar-scroller is pressed
      if(maxlen>maxdisplayh) {
        Display *dsp=aguix->getDisplay();
        int grabs=XGrabPointer(dsp,hbarwin,False,Button1MotionMask|ButtonReleaseMask,GrabModeAsync,GrabModeAsync,/*hbarwin*/None,None,CurrentTime);
	aguix->setCursor(hbarwin,AGUIX::SCROLLH_CURSOR);
        int dx=mx-tx;
	int newx=0;
	bool gotnewx=false;
        hbar_pressed=true;
        hbarredraw();
        aguix->Flush();
        Message *tmsg,*tmsg2;
        for(ende=false;ende==false;gotnewx=false) {
          tmsg=aguix->wait4messReact(MES_GET,false);
          if(tmsg!=NULL) {
            if(tmsg->type==ButtonRelease) ende=true;
            else if(tmsg->type==MotionNotify) {
              newx=tmsg->mousex;
	      gotnewx=true;
            } else {
              while((tmsg2=aguix->wait4messReact(MES_GET,false))!=NULL) {
                if(tmsg2->type==ButtonRelease) ende=true;
                _freesafe(tmsg2);
              }
            }
            _freesafe(tmsg);
          } else {
	    waittime(20);
	  }
	  if(gotnewx==true) {
            int tmx,pos;
            double f1;
            tmx=newx;
            tmx-=hbar_height;
            tmx-=dx;
            pos=tmx;
            f1=pos/b;
            setXOffset((int)f1);
	    aguix->Flush();
	  }
        }
	aguix->unsetCursor(hbarwin);
	if(grabs==GrabSuccess) XUngrabPointer(dsp,CurrentTime);
        hbar_pressed=false;
        hbarredraw();
        aguix->Flush();
      }
    } else {
      if(mx<tx) {
        setXOffset(xoffset-maxdisplayh+1);
      } else {
        setXOffset(xoffset+maxdisplayh-1);
      }
      aguix->Flush();
    }
  }
  return true;
}

bool FieldListView::handleVBar(Message *msg)
{
  if(vbar==0) return false;
  int mx,my;
  int dir;
  mx=msg->mousex;
  my=msg->mousey;
  int vx,vy,vw,vh;
  bool ende;

  vh=h-2;
  if(hbar!=0) vh-=hbar_height;
  vw=vbar_width;
  vx=0;
  vy=0;
  // vbar
  dir=0;
  if((my>vy)&&(my<=(vy+vw))) {
    // up
    dir=-1;
    vbar_dir=1;
  }
  if((my>(vy+vh-vw))&&(my<=(vy+vh))) {
    // down
    dir=1;
    vbar_dir=2;
  }
  if(dir!=0) {
    vbarredraw();
    aguix->Flush();
    Message *tmsg,*tmsg2;
    for(ende=false;ende==false;) {
      scrollV1(dir);
      aguix->Flush();
      if(ende==false) waittime(70);
      tmsg=aguix->wait4messReact(MES_GET,false);
      if(tmsg!=NULL) {
        if(tmsg->type==ButtonRelease) ende=true;
        else {
          while((tmsg2=aguix->wait4messReact(MES_GET,false))!=NULL) {
            if(tmsg2->type==ButtonRelease) ende=true;
            _freesafe(tmsg2);
          }
        }
        _freesafe(tmsg);
      }
    }
    vbar_dir=0;
    vbarredraw();
    aguix->Flush();
  } else {
    int tx,ty,tw,th;
    double b=0;
    int elements=getElements();
    th=h-2;
    if(hbar!=0) th-=hbar_height;
    tw=vbar_width;
    tx=0;
    ty=0;
    th-=2*tw;  // Hhe wenn alles anzeigbar ist
    tw-=2;
    tx++;
    if(elements>maxdisplayv) {
      int dh=th*maxdisplayv;
      dh/=elements;
      int a=elements-maxdisplayv;
      b=th-dh;  // das was brig bleibt
      b/=a;            // verteilt sich auch die nichtsichtbaren Zeichen
      ty=ty+vbar_width+(int)(yoffset*b);
      th=dh;
    } else {
      ty=ty+vbar_width;
    }
    ty++;
    if((mx>=tx)&&(mx<=(tx+tw))&&(my>=ty)&&(my<=(ty+th))) {
      // vbar-scroller pressed
      if(elements>maxdisplayv) {
        Display *dsp=aguix->getDisplay();
        int grabs=XGrabPointer(dsp,vbarwin,False,Button1MotionMask|ButtonReleaseMask|EnterWindowMask|LeaveWindowMask,GrabModeAsync,GrabModeAsync,/*vbarwin*/None,None,CurrentTime);
	aguix->setCursor(vbarwin,AGUIX::SCROLLV_CURSOR);
        int dy=my-ty;
	int newy=0;
	bool gotnewy=false;
        vbar_pressed=true;
        vbarredraw();
        aguix->Flush();
        Message *tmsg,*tmsg2;
        for(ende=false;ende==false;gotnewy=false) {
          tmsg=aguix->wait4messReact(MES_GET,false);
          if(tmsg!=NULL) {
            if(tmsg->type==ButtonRelease) ende=true;
            else if(tmsg->type==MotionNotify) {
              newy=tmsg->mousey;
	      gotnewy=true;
            } else {
              while((tmsg2=aguix->wait4messReact(MES_GET,false))!=NULL) {
                if(tmsg2->type==ButtonRelease) ende=true;
                _freesafe(tmsg2);
              }
            }
            _freesafe(tmsg);
          } else {
	    waittime(20);
	  }
	  if(gotnewy==true) {
     	    int tmy,pos;
            double f1;
            tmy=newy;
            tmy-=vbar_width;
            tmy-=dy;
            pos=tmy;
            f1=pos/b;
            setYOffset((int)f1);
	    aguix->Flush();
	  }
        }
	aguix->unsetCursor(vbarwin);
        if(grabs==GrabSuccess) XUngrabPointer(dsp,CurrentTime);
        vbar_pressed=false;
        vbarredraw();
        aguix->Flush();
      }
    } else {
      if(my<ty) {
        scrollV(-maxdisplayv+1);
      } else {
        scrollV(maxdisplayv-1);
      }
      aguix->Flush();
    }
  }
  return true;
}

void FieldListView::scrollV(int delta)
{
  //TODO
  setYOffset(yoffset+delta);
}

void FieldListView::scrollV1(int dir)
{
  //TODO
  setYOffset(yoffset+dir);
}

void FieldListView::setSelectHandler( void (*nsh)( FieldListView*, int p ) )
{
  selecthandler = nsh;
}

/*
 * showActive zeigt den aktiven Eintrag
 * sie tut nichts, wenn Eintrag sichtbar
 * ist der oberhalb, wird er oben als erstes Element sichtbar
 * ist er unterhalb, wird er als letztes sichtbar
 */
void FieldListView::showActive()
{
  if ( activerow >= 0 ) {
    if ( activerow < yoffset ) setYOffset( activerow );
    else if ( ( activerow - yoffset ) >= maxdisplayv ) setYOffset( activerow - maxdisplayv + 1 );
  }
}

/*
 * centerActive zentriert Ansicht auf aktiven Eintrag
 * egal, ob sichtbar oder nicht
 */
void FieldListView::centerActive()
{
  if ( activerow >= 0 ) {
    setYOffset( activerow - maxdisplayv / 2 );
  }
}

/*
 * setupBars
 *
 * will move/resize hbar/vbar
 */
void FieldListView::setupBars()
{
  if ( hbar != 0 ) {
    int tx, ty, tw, th;
    tw = w - 2;
    if ( vbar != 0 ) tw -= vbar_width;
    th = hbar_height;
    if ( hbar == 1 ) ty = 1;
    else ty = h - 1 - th;
    if ( vbar == 1 ) tx = 1 + vbar_width;
    else tx = 1;
    parent->resizeSubWin( hbarwin, tw, th );
    parent->moveSubWin( hbarwin, tx, ty );
  }
  if ( vbar != 0 ) {
    int tx, ty, tw, th;
    th = h - 2;
    if ( hbar != 0 ) th -= hbar_height;
    tw = vbar_width;
    if ( vbar == 1 ) tx = 1;
    else tx = w - 1 - tw;
    if ( hbar == 1 ) ty = 1 + hbar_height;
    else ty = 1;
    parent->resizeSubWin( vbarwin, tw, th );
    parent->moveSubWin( vbarwin, tx, ty );
  }
}

int FieldListView::setFont(char *fontname)
{
  font=aguix->getFont(fontname);
  if(font==NULL) return -1;
  int oldmaxdv;
  oldmaxdv=maxdisplayv;
  recalcMaxValues();
  rebuildSubWins(oldmaxdv);
  reconfSubWins();
  setupBars();
  return 0;
}

const char *FieldListView::getType()
{
  return type;
}

bool FieldListView::isType(const char *qtype)
{
  if(strcmp(type,qtype)==0) return true;
  return false;
}

bool FieldListView::isParent(Window child)
{
  if(child==win) return true;
  if(child==hbarwin) return true;
  if(child==vbarwin) return true;
  for(int i=0;i<maxdisplayv;i++) {
    if(child==subwins[i]) return true;
  }
  return false;
}

void FieldListView::runSelectHandler( int p )
{
  if ( selecthandler != NULL ) ( selecthandler )( this, p );
}

void FieldListView::showRow( int row )
{
  if ( ( row >=0 ) && ( row < elements ) ){
    if ( row < yoffset ) {
      if ( abs( row - yoffset ) == 1 ) scrollV1( -1 );
      else scrollV( row - yoffset );
    } else if ( ( row - yoffset ) >= maxdisplayv ) {
      if ( abs( row - yoffset - maxdisplayv + 1 ) == 1 ) scrollV1( 1 );
      else scrollV( row - yoffset - maxdisplayv + 1 );
    }
  }
}

int FieldListView::getRowHeight()
{
  if ( font == NULL ) return aguix->getCharHeight();
  else return font->getCharHeight();
}

int FieldListView::getMBG()
{
  return mbg;
}

void FieldListView::setMBG(int color)
{
  mbg = color;
  if ( parent != NULL ) {
    aguix->SetWindowBG( win, mbg );
    setupsubwins();
  }
}

int FieldListView::maximizeX()
{
  int elem_width, s;

  if ( font == NULL ) elem_width = aguix->getCharWidth();
  else elem_width = font->getCharWidth();
  s = elem_width * getMaxLen() + ( ( vbar != 0 ) ? vbar_width : 0 ) + 2;
  resize( s, getHeight() );
  return s;
}

int FieldListView::maximizeY()
{
  int elem_height, s;

  if ( font == NULL ) elem_height = aguix->getCharHeight();
  else elem_height = font->getCharHeight();
  s = elem_height * getElements() + ( ( hbar != 0 ) ? hbar_height : 0 ) + 2;
  resize( getWidth(), s );
  return s;
}

void FieldListView::handleMiddle(Message *msg)
{
  int j = -1;
  int dely, i;
  bool ende;

  for ( i = 0; i < maxdisplayv; i++ ) {
    if ( msg->window == subwins[ i ] ) j = i;
  }
  if ( j == -1 ) return;
  if ( ( j + yoffset ) > elements ) return;

  int ch;

  if ( font == NULL ) ch = aguix->getCharHeight();
  else ch = font->getCharHeight();

  int tpos, firstselect;

  tpos = j + yoffset;
  setActiveRow( tpos );
  runSelectHandler( tpos );
  aguix->Flush();
  
  Time lasttime = msg->time;

  dely = j * ch;

  Message *tmsg;
  int newelement, element = j, my, elementh = ch;
  int dest;

  firstselect = tpos;
  newelement = element;
  for ( ende = false; ende == false; ) {
    tmsg = aguix->wait4messReact( MES_GET, false );
    if ( tmsg != NULL ) {
      if ( tmsg->type == ButtonRelease ) {
        ende = true;
	lasttime = tmsg->time;
      } else if ( ( tmsg->type == MotionNotify ) || ( tmsg->type == EnterNotify ) || ( tmsg->type == LeaveNotify ) ) {
        Window root, child;
        int root_x, root_y, win_x, win_y;
        unsigned int keys_buttons;

        XQueryPointer( aguix->getDisplay(),
		       msg->window,
		       &root,
		       &child,
		       &root_x,
		       &root_y,
		       &win_x,
		       &win_y,
		       &keys_buttons );
        my = win_y;
        my += dely;
        newelement = my / elementh;
        if ( my < 0 ) newelement--;
      }
      _freesafe( tmsg );
    } else {
      waittime( 50 );
      Window root, child;
      int root_x, root_y, win_x, win_y;
      unsigned int keys_buttons;

      XQueryPointer( aguix->getDisplay(),
		     msg->window,
		     &root,
		     &child,
		     &root_x,
		     &root_y,
		     &win_x,
		     &win_y,
		     &keys_buttons );
      my = win_y;
      my += dely;
      newelement = my / elementh;
      if ( my < 0 ) newelement--;
    }
    if ( ( element != newelement ) || ( newelement < 0 ) || ( newelement >= maxdisplayv ) ) {
      dest = newelement;
      if ( newelement < 0 ) {
        scrollV1( -1 );
        dest = 0;
        element++;
      } else if ( newelement >= maxdisplayv ) {
        scrollV1( 1 );
        dest = maxdisplayv - 1;
        element--;
      }
      element = dest;
      if ( ( element + yoffset ) >= elements ) {
        element = elements - yoffset - 1;
      }
      tpos = element + yoffset;
      if ( tpos < elements ) {
	setActiveRow( tpos );
        runSelectHandler( tpos );
      }
    }
  }
  //TODO: Muss hier eine Message verschickt werden
  // Erstmal mache ich das, dann der User bei Worker auch per mittleren Doppelklick Dateien starten kann
  AGMessage *agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
  agmsg->fieldlv.lv=this;
  agmsg->fieldlv.row = firstselect;
  agmsg->fieldlv.time=lasttime;
  agmsg->type=AG_FIELDLV_ONESELECT;
  aguix->putAGMsg(agmsg);
}

bool FieldListView::handleKeys(Message *msg)
{
  bool returnvalue=false;

  if ( msg != NULL ) {
    //TODO2:
    // most important features:
    // 1.up/down
    //   should react like handleMiddle
    // 2.return for simulate doubleclick
    //   problem: there is no doubleclick-feature at all at the moment
    // 3.space for selecting/unselecting
    //   just like handleSelect
    if ( msg->type == KeyPress ) {
      if ( ( msg->key == XK_Up ) ||
	   ( msg->key == XK_Down ) ||
	   ( msg->key == XK_Home ) ||
	   ( msg->key == XK_End ) ||
	   ( msg->key == XK_Prior ) ||
	   ( msg->key == XK_Next ) ) {
	int pos;

	switch ( msg->key ) {
	case XK_Up:
	  if ( activerow > 0 )
	    pos = activerow - 1;
	  else
	    pos = 0;
	  break;
	case XK_Down:
	  if ( activerow < 0 )
	    pos = 0;
	  else if ( activerow >= ( elements - 1 ) )
	    pos = elements - 1;
	  else
	    pos = activerow + 1;
	  break;
	case XK_Home:
	  pos = 0;
	  break;
	case XK_End:
	  pos = elements - 1;
	  break;
	case XK_Prior:
	  if ( activerow < 0 )
	    pos = 0;
	  else {
	    pos = activerow - maxdisplayv + 1;
	    if ( pos < 0 )
	      pos = 0;
	  }
	  break;
	case XK_Next:
	  if ( activerow < 0 )
	    pos = 0;
	  else {
	    pos = activerow + maxdisplayv - 1;
	    if ( pos >= ( elements - 1 ) )
	      pos = elements - 1;
	  }
	  break;
	default:
	  pos = -1;
	  break;
	}
	if ( ( pos >= 0 ) && ( pos < elements ) ){
	  // now activate entry pos
	  setActiveRow( pos );
	  if ( ( ( pos - yoffset ) < 0 ) || ( ( pos - yoffset ) >= maxdisplayv ) )
	    showActive();
	  runSelectHandler( pos );
	  aguix->Flush();
	}
      } else if ( msg->key == XK_space ) {
	if ( activerow >= 0 ) {
	  bool state = getSelect( activerow );
	  state = ( state == false ) ? true : false;
	  setSelect( activerow, state );
	  runSelectHandler( activerow );
	  aguix->Flush();
	}
      } else if ( msg->key == XK_Return ) {
      }
    }
  }
  return returnvalue;
}

void FieldListView::setData( int row, int ndata )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return;
  elementarray[ row ]->setData( ndata );
}

int FieldListView::getData( int row )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return 0;
  return elementarray[ row ]->getData();
}

void FieldListView::Content::setData( int ndata )
{
  data = ndata;
}

int FieldListView::Content::getData()
{
  return data;
}

bool FieldListView::isValidRow( int row )
{
  if ( ( row >= 0 ) && ( row < elements ) ) return true;
  return false;
}

int FieldListView::insertRow( int row )
{
  int i;

  if ( row < 0 ) return -1;
  if ( row >= elements ) return addRow();

  if ( elements == arraysize ) increaseArray( arraysize * 2 );

  for ( i = elements - 1; i >= row; i-- ) {
    elementarray[ i + 1 ] = elementarray[ i ];
  }

  elements++;
  elementarray[ row ] = new Content();
  return row;
}

void FieldListView::swapRows( int row1, int row2 )
{
  Content *te;

  if ( ( isValidRow( row1 ) != true ) || ( isValidRow( row2 ) != true ) ) return;
  
  te = elementarray[ row1 ];
  elementarray[ row1 ] = elementarray[ row2 ];
  elementarray[ row2 ] = te;
  return;
}

void FieldListView::setPreColors( int row, precolor_t mode )
{
  if ( isValidRow( row ) == false ) return;

  switch( mode ) {
    case PRECOLOR_ONLYSELECT:
      setFG( row, CC_NORMAL, 1 );
      setFG( row, CC_SELECT, 2 );
      setFG( row, CC_ACTIVE, 1 );
      setFG( row, CC_SELACT, 2 );
      setBG( row, CC_NORMAL, 0 );
      setBG( row, CC_SELECT, 1 );
      setBG( row, CC_ACTIVE, 0 );
      setBG( row, CC_SELACT, 1 );
      break;
    case PRECOLOR_ONLYACTIVE:
      setFG( row, CC_NORMAL, 1 );
      setFG( row, CC_SELECT, 1 );
      setFG( row, CC_ACTIVE, 2 );
      setFG( row, CC_SELACT, 2 );
      setBG( row, CC_NORMAL, 0 );
      setBG( row, CC_SELECT, 0 );
      setBG( row, CC_ACTIVE, 1 );
      setBG( row, CC_SELACT, 1 );
      break;
    default:
      setFG( row, CC_NORMAL, 1 );
      setFG( row, CC_SELECT, 2 );
      setFG( row, CC_ACTIVE, 1 );
      setFG( row, CC_SELACT, 2 );
      setBG( row, CC_NORMAL, 0 );
      setBG( row, CC_SELECT, 1 );
      setBG( row, CC_ACTIVE, 7 );
      setBG( row, CC_SELACT, 7 );
      break;
  }
}
