/* font.c
 * 
 * This program is distributed under the GNU General Public License.
 * Copyright (C) 2000 Eugene Osintsev <gene@linuxave.net>
 */

#include <config.h>
#include <stdlib.h>
#include <curses.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include "font.h"
#include "scr.h"
#include "defines.h"

extern struct CFE_GLYPH glyph[2];

/*------------------------------------------------------------------*/

int create_font_file(struct CFE_FONT *font)
{
    if (font->type == F_PSF) {
        font->offset = 4;
        font->size = font->height * font->num_of_chars + 4;
        font->file = (unsigned char *) calloc(font->size, 1);
        font->file[0] = 0x36;
        font->file[1] = 0x04;
        font->file[2] = (font->num_of_chars == 256 ? 0 : 1);
        font->file[3] = font->height;
        font->data = font->file + 4;
    } else {
        font->offset = 0;
        font->num_of_chars = NUM_OF_CHARS;
        font->size = font->height * NUM_OF_CHARS;
        font->file = font->data = (unsigned char *) calloc(font->size, 1);
    }
    if (!font->file) {
        perror(PACKAGE);
        return 0;
    } else {
        return 1;
    }
}

/*------------------------------------------------------------------*/

int load_font_file(struct CFE_FONT *font,
                   int is_offset, int is_height)
{
    int fd;

    fd = open(font->name, O_RDONLY);
    if (fd == -1) {
        return -1;
    }
    font->size = lseek(fd, 0, SEEK_END);
    lseek(fd, 0, SEEK_SET);
    font->file = (unsigned char *) malloc(font->size);
    if (!font->file) {
        perror(PACKAGE);
        return 0;
    }
    read(fd, font->file, font->size);
    close(fd);

    if (is_offset || is_height) {
        font->num_of_chars = NUM_OF_CHARS;
        if (font->offset < 0 ||
            font->offset > (font->size - font->height * NUM_OF_CHARS)) {
            fprintf(stderr,
                    "%s: wrong font offset (or size)\n",
                    font->name);
            return 0;
        }
    } else {
        font->type = psf_or_raw(font);
        if (font->type == F_ERR) {
            fprintf(stderr, "%s: possibly corrupted font data\n",
                    font->name);
            return 0;
        }
    }
    font->data = font->file + font->offset;
    return 1;
}

/*------------------------------------------------------------------*/

int save_font_file(struct CFE_FONT *font)
{
    int fd;

    if ((fd = open(font->name, O_CREAT | O_TRUNC | O_WRONLY, 0644)) == -1) {
        return 0;
    }
    message("Saving...");
    refresh();
    write(fd, font->file, font->size);
    close(fd);
    return 1;
}

/*------------------------------------------------------------------*/

int psf_or_raw(struct CFE_FONT *font)
{
    int ret_type;
    
    font->num_of_chars = NUM_OF_CHARS;

    if (font->size > 4 &&            /* Linux/i386 PC Screen Font data */
        (font->file[0] == 0x36 && font->file[1] == 0x04)) {
        font->num_of_chars =
            ((font->file[2] == 1 || font->file[2] == 3) ? 512 : 256);
        font->height = (int) font->file[3];
        font->offset = 4;
        if (font->file[2] == 2 || font->file[2] == 3) {
            font->unicode_data = (unsigned short *)
                (font->file + font->offset + font->num_of_chars * font->height);
        }
        ret_type = F_PSF;
    } else {                        /* RAW font data */
        font->height = font->size / NUM_OF_CHARS;
        font->offset = 0;
        ret_type = F_RAW; 
    }
    
    if (((font->offset + font->num_of_chars * font->height) > font->size) ||
        (font->height < MIN_FONT_HEIGHT || font->height > MAX_FONT_HEIGHT))
        return F_ERR;
    else
        return ret_type;
}

/*------------------------------------------------------------------*/

int get_unicode_data(struct CFE_FONT *font, int glyph_num, char *str)
{
    unsigned short *val, *end;
    char buf[7];
    int num = 0;
    
    if (!font->unicode_data)
        return 0;

    end = (unsigned short *) (font->file + font->size - 2);
    val = font->unicode_data;
    strcpy(str, "Unicode:");

    while (val <= end) {
        if (*val == 0xFFFF) {
            num++;
            val++;
            continue;
        }
        if (glyph_num == num) {
            sprintf(buf, " 0x%04X", *val);
            if (strlen(str) < 74)
                strcat(str, buf);
            else
                break;
        }
        val++;
    }
    
    if (str[8] == '\0')
        strcat(str, " none");
    
    return 1;    
}

/*------------------------------------------------------------------*/

void set_glyph_row(int row, unsigned char data)
{
    glyph[0].data[row] = data;
}

/*------------------------------------------------------------------*/

void set_glyph_cur_row(unsigned char data)
{
    glyph[0].data[glyph[0].y] = data;
}

/*------------------------------------------------------------------*/

unsigned char get_glyph_row(int row)
{
    return glyph[0].data[row];
}

/*------------------------------------------------------------------*/

unsigned char get_glyph_cur_row()
{
    return glyph[0].data[glyph[0].y];
}


/* EOF */
