%  Copyright (C) 2004 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

\documentclass{article}

\usepackage{verbatim}
\newenvironment{code}{\verbatim}{\endverbatim}

\begin{document}

\title{Antimemoization: a nice monadic trick to save memory}
\author{David Roundy}

\maketitle

\begin{abstract}
This paper introduces the concept of ``antimemoization'', a technique which
lets a program save memory by allowing large data items to be
garbage-collected and recomputed as needed.  The method is implemented in
haskell as a monad.
\end{abstract}

[Add an introduction/motivation here]

Antimemoization is a relevant optimization when there is a relatively cheap
computation which takes a \emph{small} input and produces a \emph{large}
output.  In such a case, it may save memory to discard the output, and
recompute it later.  In simple situations, this can be done by hand.
However, as code complexity grows, performing this optimization by hand
could lead to situations where one has more than one copy of the same large
data structure, which obviously would be a Bad Thing.

\begin{comment}
\begin{code}
module AntiMemo ( AntiMemo, antimemoize, readAntiMemo,
                  (|++|),
                ) where

import Control.Monad ( liftM2 )
#ifdef ANTIMEMOIZE
import System.IO.Unsafe ( unsafePerformIO )
import Data.IORef ( newIORef, writeIORef, readIORef, IORef )
import System.Mem.Weak ( mkWeakPtr, deRefWeak, Weak )
#endif

antimemoize :: (a -> b) -> a -> AntiMemo b
readAntiMemo :: AntiMemo b -> b

(|++|) :: AntiMemo [a] -> AntiMemo [a] -> AntiMemo [a]
(|++|) = liftM2 (++)

instance Ord a => Ord (AntiMemo a) where
    x <= y = (readAntiMemo x) <= (readAntiMemo y)
\end{code}
\end{comment}

Antimemoization is implemented as a monadic type, \verb!AntiMemo!.  Any
computations performed ``within'' this monad may be garbage collected, in
which case they will be recomputed as needed.  While \emph{implementing}
antimemoization is a mind-twisting experience, actually \emph{using} the
AntiMemo module is pretty straightforward for anyone with experience in
monadic programming.

An \verb!AntiMemo! is created using \verb!antimemoize!:
\begin{verbatim}
antimemoize :: (a -> b) -> a -> AntiMemo b
\end{verbatim}

Once we have an \verb!AntiMemo!, we can read its value using the
\verb!readAntiMemo!:
\begin{verbatim}
readAntiMemo :: AntiMemo b -> b
\end{verbatim}
However, it's best to use readAntiMemo only sparingly, since it has the
unfortunate effect of creating a copy of a potentially large data structure
which cannot be garbage-collected (until it is no longer accessible).  If
you want to modify the result of an antimemoized calculation, you should do
either with the \verb!liftM! family of functions, or by writing monadic
code directly, perhaps using a \verb!do! statement.

\begin{comment}
\begin{code}
#ifdef ANTIMEMOIZE
data AntiMemo a = AntiMemo (IO a) (IORef (Weak a))

instance Monad AntiMemo where
    thea@(AntiMemo ioa _) >>= f =
        case f (readAntiMemo thea) of
        (AntiMemo _ iowb) -> AntiMemo (unantimemoize f `fmap` ioa) iowb
    return x = unsafePerformIO $ do wb <- mkWeakPtr x Nothing
                                    iowb <- newIORef wb
                                    return (AntiMemo (return x) iowb)

instance Functor AntiMemo where
    fmap f thea@(AntiMemo ioa _) =
        case readAntiMemo thea of
        a -> AntiMemo (f `fmap` ioa) $
             unsafePerformIO $ mkWeakPtr (f a) Nothing >>= newIORef

instance Eq a => Eq (AntiMemo a) where
    x@(AntiMemo _ ix) == y@(AntiMemo _ iy)
        = ix == iy || (readAntiMemo x) == (readAntiMemo y)

unantimemoize :: (a -> AntiMemo b) -> (a -> b)
unantimemoize f a = readAntiMemo $ f a

antimemoize f a = unsafePerformIO $ do wb <- mkWeakPtr (f a) Nothing
                                       iowb <- newIORef wb
                                       return (AntiMemo (lazy_apply f a) iowb)

lazy_apply :: (a -> b) -> a -> IO b
lazy_apply f a = do f' <- return f
                    a' <- return a
                    return $ f' a'

readAntiMemo (AntiMemo ioa iowa) = unsafePerformIO $
    do ma <- readIORef iowa >>= deRefWeak
       case ma of Nothing -> do a <- ioa
                                wa <- mkWeakPtr a Nothing
                                writeIORef iowa wa
                                return a
                  Just a -> return a

#else

newtype AntiMemo a = AM a

instance Functor AntiMemo where
    fmap f (AM a) = AM (f a)

instance Monad AntiMemo where
    (AM a) >>= f = f a
    return x = AM x

instance Eq a => Eq (AntiMemo a) where
    x == y = readAntiMemo x == readAntiMemo y

readAntiMemo (AM x) = x
antimemoize f a = AM (f a)

#endif
\end{code}
\end{comment}

[Add a usage example here?]

[Add a conclusion here?]

\end{document}
