/* dc_gui2 - a GTK+2 GUI for DCTC
 * Copyright (C) 2002 Eric Prevoteau
 *
 * dc_gui2_com_channel_client.c: Copyright (C) Eric Prevoteau <www@a2pb.gotdns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: dc_gui2_com_channel_client.c,v 1.2 2004/01/14 15:47:10 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <ctype.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <glib.h>

#include "mem_access.h"
#include "dc_gui2_com_channel_client.h"
#include "../../src/proc_serv_cmd.h"

static int com_fd=-1;

/********************************************/
/* initialize the comm channel with dc_gui2 */
/********************************************/
/* input: env var "EMB_IOC"             */
/* output: FALSE= EMB_IOC not available */
/*         TRUE= ok                     */
/****************************************/
gboolean init_comm_channel(void)
{
	char *t;

	t=getenv("EMB_IOC");
	if(t==NULL)
		return 0;

	com_fd=strtoul(t,NULL,10);

	return 1;
}


/* ---------------------------------------------------------------------- */
static inline void dump_packet(guint8 *data, int len)
{
#if 0
	printf("REMOTE packet dump\n");
	while(len-->0)
	{
		printf("%02x ",(unsigned int)*data++);
	}
	printf("\n");
#endif
}

/*************************************/
/* send a command on the com channel */
/*****************************************************************************/
/* optionnal paramete: nb_param * (1 unsigned int(=B), a pointer on B bytes) */
/*****************************************************************************/
static void send_cmd(int sock_fd, guint16 cmd, guint8 nb_param, ...)
{
   guint packet_size;
   GByteArray *gba;
   va_list ap;
   guint8 i;

   packet_size=2;	/* size of the 2 first bytes */
   gba=g_byte_array_new();
   g_byte_array_append(gba,(guint8*)&cmd,2);
   g_byte_array_append(gba,&nb_param,1);

   va_start(ap,nb_param);
   for(i=0;i<nb_param;i++)
   {
      guint16 psize;
      guint8 *pptr;

      psize=va_arg(ap,unsigned int);
      pptr=va_arg(ap,void *);

      g_byte_array_append(gba,(guint8*)&psize,2);
      if(psize)
         g_byte_array_append(gba,pptr,psize);
   }
   va_end(ap);

	packet_size+=gba->len;
   /* add the newly created packet to the array */
   g_byte_array_prepend(gba,(guint8*)&packet_size,2);

	dump_packet(gba->data, gba->len);
	if(write(sock_fd, gba->data,gba->len)!=gba->len)
		perror("send_cmd");
   g_byte_array_free(gba,TRUE);
}

/******************************/
/* get the reply of a command */
/******************************/
static GByteArray *get_reply(int sock_fd)
{
	char buf[5];
	guint16 packet_size;
	GByteArray *gba;

	if(read(sock_fd,buf,5)!=5)
	{
		perror("get_reply");
		return NULL;
	}

	dump_packet(buf,5);
	packet_size=GET_UAA_GUINT16(buf);

	gba=g_byte_array_new();
	g_byte_array_set_size(gba,packet_size);
	memcpy(gba->data,buf,5);
	
	if(packet_size>5)
	{
		if(read(sock_fd,gba->data+5,packet_size-5)!=(packet_size-5))
		{
			perror("get_reply (2)");
			g_byte_array_free(gba,TRUE);
			return NULL;
		}
		dump_packet(gba->data,gba->len);
	}

	return gba;
}

/*****************************************************/
/* send something to the client connected to the GUI */
/*****************************************************/
void remote_send_data_to_dctc(const char *cmd)
{
	send_cmd(com_fd,PS_SEND_CMD_TO_DCTC,1,strlen(cmd),cmd);
}

/*****************************************************/
/* send something to the master connected to the GUI */
/*****************************************************/
void remote_send_data_to_gdl_dctc(const char *cmd)
{
	send_cmd(com_fd,PS_SEND_CMD_TO_GDL_DCTC,1,strlen(cmd),cmd);
}

/*******************************/
/* retrieve the var of the GUI */
/***************************************/
/* output: var value (to free) or NULL */
/***************************************/
char *remote_get_var(const char *var_name)
{
	GByteArray *gba;
	guint16 proc_command;

	send_cmd(com_fd,PS_GET_VAR_VALUE,1,strlen(var_name),var_name);
	gba=get_reply(com_fd);

	if(gba==NULL)
		return NULL;

	proc_command=GET_UAA_GUINT16(gba->data+2);
	switch(proc_command)
	{
		case PS_VAR_VALUE:		if(gba->data[4]==1)		/* 1 parameter ? */
										{
											int param_size=GET_UAA_GUINT16(gba->data+5);
											char *output;

											output=malloc(param_size+1);
											memcpy(output,gba->data+7,param_size);
											output[param_size]='\0';
											g_byte_array_free(gba,TRUE);
											return output;
										}
										else
										{
											g_byte_array_free(gba,TRUE);
											return NULL;
										}
										break;

		case PS_INVALID_QUERY:	fprintf(stderr,"Invalid query for remote_get_var(%s)\n",var_name);
										g_byte_array_free(gba,TRUE);
										return NULL;

		default:						fprintf(stderr,"Unknown reply for remote_get_var(%x)\n",proc_command);
										g_byte_array_free(gba,TRUE);
										return NULL;
	}
}

/************************************************/
/* retrieve the value of a gtk_entry of the GUI */
/************************************************/
/* output: var value (to free) or NULL */
/***************************************/
char *remote_get_gtk_entry(const char *var_name)
{
	GByteArray *gba;
	guint16 proc_command;

	send_cmd(com_fd,PS_GET_TEXT_WIDGET_VALUE,1,strlen(var_name),var_name);
	gba=get_reply(com_fd);

	if(gba==NULL)
		return NULL;

	proc_command=GET_UAA_GUINT16(gba->data+2);
	switch(proc_command)
	{
		case PS_TEXT_WIDGET_VALUE:
										if(gba->data[4]==1)		/* 1 parameter ? */
										{
											int param_size=GET_UAA_GUINT16(gba->data+5);
											char *output;

											output=malloc(param_size+1);
											memcpy(output,gba->data+7,param_size);
											output[param_size]='\0';
											g_byte_array_free(gba,TRUE);
											return output;
										}
										else
										{
											g_byte_array_free(gba,TRUE);
											return NULL;
										}
										break;

		case PS_INVALID_QUERY:	fprintf(stderr,"Invalid query for remote_get_var(%s)\n",var_name);
										g_byte_array_free(gba,TRUE);
										return NULL;

		default:						fprintf(stderr,"Unknown reply for remote_get_var(%x)\n",proc_command);
										g_byte_array_free(gba,TRUE);
										return NULL;
	}
}

/************************************************/
/* retrieve the value of a gtk_entry of the GUI */
/**********************************************************************/
/* output: var value (as an integer) or 0 if gtk_entry does not exist */
/**********************************************************************/
int remote_get_gtk_entry_as_int(const char *var_name)
{
	char *v_txt;
	int v;

	v_txt=remote_get_gtk_entry(var_name);
	if(v_txt==NULL)
		return 0;

	v=atoi(v_txt);
	free(v_txt);
	return v;
}

/******************************************************/
/* retrieve the value of a gtk_checkbutton of the GUI */
/*************************************************************************/
/* output: var value (as a boolean) or FALSE if gtk_entry does not exist */
/*************************************************************************/
gboolean remote_get_gtk_checkbutton_as_boolean(const char *var_name)
{
	GByteArray *gba;
	guint16 proc_command;

	send_cmd(com_fd,PS_GET_CHECKBUTTON_WIDGET_VALUE,1,strlen(var_name),var_name);
	gba=get_reply(com_fd);

	if(gba==NULL)
		return FALSE;

	proc_command=GET_UAA_GUINT16(gba->data+2);
	switch(proc_command)
	{
		case PS_CHECKBUTTON_WIDGET_VALUE:
										if(gba->data[4]==1)		/* 1 parameter ? */
										{
											guint8 val;

											val=gba->data[7];		/* skip the param_size, it is always a boolean */
											g_byte_array_free(gba,TRUE);

											if(val)
												return TRUE;
											return FALSE;
										}
										else
										{
											g_byte_array_free(gba,TRUE);
											return FALSE;
										}
										break;

		case PS_INVALID_QUERY:	fprintf(stderr,"Invalid query for remote_get_var(%s)\n",var_name);
										g_byte_array_free(gba,TRUE);
										return FALSE;

		default:						fprintf(stderr,"Unknown reply for remote_get_var(%x)\n",proc_command);
										g_byte_array_free(gba,TRUE);
										return FALSE;
	}
}


