/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Copyright (C) 2000-2002 Florian Lohoff  <flo@rfc822.org>
 * Copyright (C) 2002,2003 Thiemo Seufer  <seufer@csv.ica.uni-stuttgart.de>
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdint.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <netinet/in.h>

#include <elf.h>

#include "isoio.h"
#include "extents.h"
#include "delo.h"

#define	FILE_TO_BOOT	"/boot/delo.2nd"
#define	TMP_DEV_DIR	"/tmp"
#define MAX_MAPS	51
#define DEC_BOOT_MAGIC	0x02757a
#define CONFIGFILE	"/etc/delo.conf"

/* Those were stolen from linux kernel headers. */

struct hd_geometry {
      unsigned char heads;
      unsigned char sectors;
      unsigned short cylinders;
      unsigned long start;
};

#define HDIO_GETGEO	0x0301	/* get device geometry */

#define FIBMAP		_IO(0x00,1)	/* bmap access */
#define FIGETBSZ	_IO(0x00,2)	/* get the block size used for bmap */

#define MAJOR(dev)	((dev)>>8)
#define MAJOR_SD	8

#define O_NOACCESS	3

int debug;

struct extent {
	uint32_t count;
	uint32_t start;
} __attribute__((packed));

struct dec_bootblock {
	int8_t pad[8];
	int32_t magic;		/* We are a DEC BootBlock */
	int32_t mode;		/* 0: Single extent, 1: Multi extent boot */
	int32_t loadAddr;	/* Load below kernel */
	int32_t execAddr;	/* And exec there */
	struct extent bootmap[MAX_MAPS];
} __attribute__((packed));

static int setup_delo_root(char *root);
static int d_open(const char *path, int flags);
static ssize_t d_read(int fd, void *buf, size_t count);
static int d_close(int fd);
static int opendev(dev_t device);
static int getpartoffset(struct stat *sb);
static int getextents(char *fname, struct extent *extents, int32_t *loadaddr,
		      int32_t *execaddr);
static int getextents_cdimage(char *fname, struct extent *extents,
			      int32_t *loadaddr, int32_t *execaddr);
static int writebootsec(char *fname, char *disk);
static char *device_from_config(char *file);
static void usage(void);

static char *delo_root = NULL;
static int image_p = 0;

static uint32_t u32_to_le(uint32_t value)
{
	union conv {
		uint32_t i;
		uint8_t b[4];
	} c, d;

	c.i = htonl(value);
	d.b[0] = c.b[3];
	d.b[1] = c.b[2];
	d.b[2] = c.b[1];
	d.b[3] = c.b[0];

	return d.i;
}

static uint16_t le_to_u16(uint16_t value)
{
	union conv {
		uint16_t i;
		uint8_t b[2];
	} c, d;

	c.i = value;
	d.b[0] = c.b[1];
	d.b[1] = c.b[0];

	return ntohs(d.i);
}

static uint32_t le_to_u32(uint32_t value)
{
	union conv {
		uint32_t i;
		uint8_t b[4];
	} c, d;

	c.i = value;
	d.b[0] = c.b[3];
	d.b[1] = c.b[2];
	d.b[2] = c.b[1];
	d.b[3] = c.b[0];

	return ntohl(d.i);
}

static void swap_in_elf32_ehdr(Elf32_Ehdr *ehdr)
{
	ehdr->e_type = le_to_u16(ehdr->e_type);
	ehdr->e_machine = le_to_u16(ehdr->e_machine);
	ehdr->e_version = le_to_u32(ehdr->e_version);
	ehdr->e_entry = le_to_u32(ehdr->e_entry);
	ehdr->e_phoff = le_to_u32(ehdr->e_phoff);
	ehdr->e_shoff = le_to_u32(ehdr->e_shoff);
	ehdr->e_flags = le_to_u32(ehdr->e_flags);
	ehdr->e_ehsize = le_to_u16(ehdr->e_ehsize);
	ehdr->e_phentsize = le_to_u16(ehdr->e_phentsize);
	ehdr->e_phnum = le_to_u16(ehdr->e_phnum);
	ehdr->e_shentsize = le_to_u16(ehdr->e_shentsize);
	ehdr->e_shnum = le_to_u16(ehdr->e_shnum);
	ehdr->e_shstrndx = le_to_u16(ehdr->e_shstrndx);
}

static void swap_in_elf32_phdr(Elf32_Phdr *phdr)
{
	phdr->p_type = le_to_u32(phdr->p_type);
	phdr->p_offset = le_to_u32(phdr->p_offset);
	phdr->p_vaddr = le_to_u32(phdr->p_vaddr);
	phdr->p_paddr = le_to_u32(phdr->p_paddr);
	phdr->p_filesz = le_to_u32(phdr->p_filesz);
	phdr->p_memsz = le_to_u32(phdr->p_memsz);
	phdr->p_flags = le_to_u32(phdr->p_flags);
	phdr->p_align = le_to_u32(phdr->p_align);
}

static int setup_delo_root(char *root)
{
	struct stat sb;

	if (!root)
		return 0;

	delo_root = root;

	if (stat(root, &sb)) {
		error("Unable to stat %s", root);
		return 1;
	}
	if (S_ISREG(sb.st_mode))
		image_p = 1;

	return iso_init(root);
}

static int d_open(const char *path, int flags)
{
	if (!delo_root)
		return open(path, flags);

	if (!image_p) {
		char buf[PATH_MAX];

		if (snprintf(buf, PATH_MAX, "%s%s", delo_root, path) >= PATH_MAX) {
			error("Path too long");
			return -1;
		}
		return open(buf, flags);
	}

	return iso_open(path);
}

static ssize_t d_read(int fd, void *buf, size_t count)
{
	if (image_p)
		return iso_read(fd, buf, count);

	return read(fd, buf, count);
}

static off_t d_lseek(int fd, off_t off, int whence)
{
	if (image_p)
		return iso_lseek(fd, off, whence);

	return lseek(fd, off, whence);
}

/* This works because we never need ioctl on an cd image. */
#define d_ioctl ioctl

static int d_fstat(int fd, struct stat *buf)
{
/* FIXME!
	if (image_p)
		return iso_fstat(fd, buf);
*/
	return fstat(fd, buf);
}

static int d_close(int fd)
{
	if (image_p)
		return iso_close(fd);

	return close(fd);
}

static int opendev(dev_t device)
{
	char devname[sizeof(TMP_DEV_DIR) + 20];
	int pfd;

	/* Create temp device name */
	sprintf(devname, "%s/dev.%d", TMP_DEV_DIR, getpid());

	/* Create device file */
	if(mknod(devname, 0600 | S_IFBLK, device)) {
		error("Unable to make device node for %s", devname);
		return 0;
	}

	/* And open it */
	if(!(pfd = open(devname, O_NOACCESS))) {
		error("Unable to open device %s", devname);
		unlink(devname);
		return 0;
	}

	unlink(devname);
	return(pfd);
}

static int getpartoffset(struct stat *sb)
{
	struct hd_geometry geo;
	int pfd;
	unsigned long start;

	if (!(pfd = opendev(sb->st_dev)))
		return -1;

	switch(MAJOR(sb->st_dev)) {
		case MAJOR_SD:
			if (ioctl(pfd, HDIO_GETGEO, &geo) != 0) {
				error("Unable to get disk geometry");
				close(pfd);
				return -1;
			}
			start = geo.start;
			break;
		default:
			error("Oops - unsupported device to install bootsector on");
			close(pfd);
			return -1;
	}

	close(pfd);
	debug("Start of partition %ld", start);
	return start;
}

static int getextents_cdimage(char *fname, struct extent *extents,
			      int32_t *loadaddr, int32_t *execaddr)
{
	int fd;
	Elf32_Ehdr ehdr;
	Elf32_Phdr phdr;
	struct extentlist *head;
	off_t offset;
	int i;
	int ext;
	unsigned int count;

	if ((fd = d_open(fname, O_RDONLY)) < 0) {
 		error("Unable to open %s", fname);
		return -1;
	}
	if (d_read(fd, &ehdr, sizeof(Elf32_Ehdr)) != sizeof(Elf32_Ehdr)) {
		error("Unable to read %s", fname);
		return -1;
	}
	swap_in_elf32_ehdr(&ehdr);
	if (!(ehdr.e_ident[EI_MAG0] == ELFMAG0
	      && ehdr.e_ident[EI_MAG1] == ELFMAG1
	      && ehdr.e_ident[EI_MAG2] == ELFMAG2
	      && ehdr.e_ident[EI_MAG3] == ELFMAG3
	      && ehdr.e_ident[EI_CLASS] == ELFCLASS32
	      && ehdr.e_ident[EI_DATA] == ELFDATA2LSB
	      && ehdr.e_ident[EI_VERSION] == EV_CURRENT
	      && ehdr.e_type == ET_EXEC
	      && ehdr.e_machine == EM_MIPS
	      && ehdr.e_version == EV_CURRENT)) {
		error("%s is not a MIPS ELF32 little endian file", fname);
		return -1;
	}
	if (ehdr.e_phnum != 1) {
		error("Sorry, %s has more than one ELF segment", fname);
		return -1;
	}
	if ((d_lseek(fd, ehdr.e_phoff, SEEK_SET) == -1)) {
		error("Unable to lseek %s", fname);
		return -1;
	}
	if (d_read(fd, &phdr, sizeof(Elf32_Phdr)) != sizeof(Elf32_Phdr)) {
		error("Unable to read %s", fname);
		return -1;
	}
	swap_in_elf32_phdr(&phdr);
	if (phdr.p_offset % SECTOR_SIZE) {
		error("Sorry, ELF segment in %s is not aligned to a sector boundary", fname);
		return -1;
	}
	*loadaddr = u32_to_le(phdr.p_vaddr);
	*execaddr = u32_to_le(ehdr.e_entry);

	if (!(head = iso_get_extentlist(fd))) {
		error("Unable to get extent list for %s", fname);
		return -1;
	}
	ext = -1;
	offset = (phdr.p_offset + SECTOR_SIZE - 1) / SECTOR_SIZE;
	count = (phdr.p_filesz + SECTOR_SIZE - 1) / SECTOR_SIZE;
	while (head) {
		if (++ext >= MAX_MAPS) {
			error("Oops - to many extents for the DECstation bootblock");
			return -1;
		}
		extents[ext].start = u32_to_le(head->extent + offset);
		extents[ext].count = u32_to_le(MIN(head->count - offset, count));
		count -= head->count;
		offset = 0;
		head = head->next;
	}

	debug("We have %d extents", ext + 1);
	for(i = 0; i <= ext; i++)
		debug("Extent %2d Start %8d Count %6d", i + 1,
		      le_to_u32(extents[i].start), le_to_u32(extents[i].count));

	d_close(fd);
	return 0;
}

static int getextents(char *fname, struct extent *extents, int32_t *loadaddr,
		      int32_t *execaddr)
{
	int fd;
	int i;
	int ext;
	int block;
	int no_blocks;
	int blocks_per_sector;
	int blocksize;
	long poffset;
	struct stat sb;
	Elf32_Ehdr ehdr;
	long textoffset;

	if ((fd = d_open(fname, O_RDONLY)) < 0) {
 		error("Unable to open %s", fname);
		return -1;
	}

	if (d_ioctl(fd, FIGETBSZ, &blocksize)) {
		error("Unable to get blocksize for %s", fname);
		return -1;
	}

	/* get stat (we want device and filesize) */
	if (d_fstat(fd, &sb)) {
		error("Unable to stat %s", fname);
		return -1;
	}

	/* Get partition offset */
	if ((poffset = getpartoffset(&sb)) < 0)
		return -1;

	if (d_read(fd, &ehdr, sizeof(Elf32_Ehdr)) != sizeof(Elf32_Ehdr)) {
		error("Unable to read %s", fname);
		return -1;
	}

	swap_in_elf32_ehdr(&ehdr);
	if (ehdr.e_ident[EI_MAG0] == ELFMAG0
	    && ehdr.e_ident[EI_MAG1] == ELFMAG1
	    && ehdr.e_ident[EI_MAG2] == ELFMAG2
	    && ehdr.e_ident[EI_MAG3] == ELFMAG3
	    && ehdr.e_ident[EI_CLASS] == ELFCLASS32
	    && ehdr.e_ident[EI_DATA] == ELFDATA2LSB
	    && ehdr.e_ident[EI_VERSION] == EV_CURRENT
	    && ehdr.e_type == ET_EXEC
	    && ehdr.e_machine == EM_MIPS
	    && ehdr.e_version == EV_CURRENT) {
		Elf32_Phdr phdr;

		/* It is an ELF bootloader */
		if (ehdr.e_phnum != 1) {
			error("Sorry, %s has more than one ELF segment", fname);
			return -1;
		}
		if ((d_lseek(fd, ehdr.e_phoff, SEEK_SET) == -1)) {
			error("Unable to lseek %s", fname);
			return -1;
		}
		if (d_read(fd, &phdr, sizeof(Elf32_Phdr)) != sizeof(Elf32_Phdr)) {
			error("Unable to read %s", fname);
			return -1;
		}
		swap_in_elf32_phdr(&phdr);
		if (phdr.p_offset % SECTOR_SIZE) {
			error("Sorry, ELF segment in %s is not aligned to a sector boundary", fname);
			return -1;
		}
		*loadaddr = u32_to_le(phdr.p_vaddr);
		*execaddr = u32_to_le(ehdr.e_entry);
		no_blocks = (phdr.p_offset + phdr.p_filesz + blocksize - 1) / blocksize;
		textoffset = phdr.p_offset;
	} else {
		/* We guess it is a old-style raw binary bootloader */

		/* This value must match the one in the loader's ld script */
#define LOAD_ADDRESS	0x80440000
		*loadaddr = u32_to_le(LOAD_ADDRESS);
		*execaddr = u32_to_le(LOAD_ADDRESS);
		no_blocks = (sb.st_size + blocksize - 1) / blocksize;
		textoffset = 0;
	}


	blocks_per_sector = blocksize / SECTOR_SIZE;

	debug("filesize: %ld textoffset: %ld blocks: %d blocksize %d",
	      (long)sb.st_size, textoffset, no_blocks, blocksize);

	/* Loop on blocks and get position via FIBMAP */
	for(i = 0, ext = block = -1; i < no_blocks; i++) {
		int prevblock = block;

		/* Ignore ELF header */
		if (textoffset >= blocksize) {
			textoffset -= blocksize;
			continue;
		}

		block = i;

		/* Get fibmap block */
		if (d_ioctl(fd, FIBMAP, &block) != 0) {
			error("Unable to get FIBMAP");
			return -1;
		}

		/* Is this block directly behind the last extent? */
		if (ext == -1) {
			ext = 0;
			extents[ext].start = u32_to_le(block * blocks_per_sector
						       + poffset
						       + textoffset / SECTOR_SIZE);
		} else if (block != prevblock + 1) {
			/* New extent */
			if (++ext >= MAX_MAPS) {
				error("Oops - too many extents for the DECstation bootblock");
				return -1;
			}
			extents[ext].start = u32_to_le(block * blocks_per_sector
						       + poffset
						       + textoffset / SECTOR_SIZE);
		}
		extents[ext].count += u32_to_le(blocks_per_sector
						- textoffset / SECTOR_SIZE);
		textoffset = 0;
	}

	debug("We have %d extents", ext + 1);
	for(i = 0; i <= ext; i++)
		debug("Extent %2d Start %8d Count %6d", i + 1,
		      le_to_u32(extents[i].start),
		      le_to_u32(extents[i].count));

	d_close(fd);
	return 0;
}

static int writebootsec(char *fname, char *disk)
{
	char sector[SECTOR_SIZE];
	struct dec_bootblock *bb = (struct dec_bootblock *)sector;
	int fd;

	debug("Writing bootsector to %s", disk);

	/* Open disk */
	if ((fd = open(disk, O_RDWR)) < 0) {
		error("Unable to open %s", disk);
		return -1;
	}

	/* Read the disk label's sector */
	if (read(fd, bb, SECTOR_SIZE) != SECTOR_SIZE) {
		error("Read from %s failed", disk);
		close(fd);
		return -1;
	}

	/* Fill in our values we care on */
	bb->magic = u32_to_le(DEC_BOOT_MAGIC);
	bb->mode = u32_to_le(1);

	memset(bb->bootmap, 0, sizeof(struct extent) * MAX_MAPS);
	if (image_p) {
		if (getextents_cdimage(fname, bb->bootmap, &bb->loadAddr,
				       &bb->execAddr))
		{
			close(fd);
			return -1;
		}
	} else {
		if (getextents(fname, bb->bootmap, &bb->loadAddr,
			       &bb->execAddr))
		{
			close(fd);
			return -1;
		}
	}

	/* And write back sector */
	if (lseek(fd, 0, SEEK_SET) == -1) {
		error("Seek on %s failed", disk);
		close(fd);
		return -1;
	}

	if (write(fd, sector, SECTOR_SIZE) != SECTOR_SIZE) {
		error("Write of bootsector to %s failed", disk);
		close(fd);
		return -1;
	}

	close(fd);
	return 0;
}

static char *device_from_config(char *file)
{
	int fd;
	char *bootdev = NULL;
	static char buf[BUFSIZ];
	int count;

	if ((fd = d_open(file, O_RDONLY)) < 0) {
 		error("Unable to open %s", file);
		return NULL;
	}

	memset(buf, 0, BUFSIZ);
	count = 0;
	while (d_read(fd, &buf[count], 1) > 0) {
		if (count > BUFSIZ - 1) {
			debug("Ridiculously long line in %s", file);
			memset(buf, 0, BUFSIZ);
			count = 0;
			continue;
		}

		if (buf[count] == '\n') {
			buf[count] = '\0';

			/* Look for boot device name */
			if (!strncmp(buf, "boot=/dev/", 10)) {
				bootdev = buf + 5;
				break;
			}
			memset(buf, 0, BUFSIZ);
			count = 0;
			continue;
		}

		count++;
	}

	d_close(fd);
	return bootdev;
}

static void usage(void)
{
	error("usage: delo [-r root] [-f config] [-d] [device]");
	exit(EXIT_FAILURE);
}

int main(int argc, char *argv[])
{
	int i;
	char *device = NULL;
	char *config = CONFIGFILE;
	char *root = NULL;

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				case 'd':
					debug=1;
					break;
				case 'f':
					if (i + 1 < argc)
						config = argv[++i];
					else
						usage();
					break;
				case 'r':
					if (i + 1 < argc)
						root = argv[++i];
					else
						usage();
					break;
				default:
					usage();
					break;
			}
		} else
			device=argv[i];
	}	

	if (setup_delo_root(root))
		return EXIT_FAILURE;

	if (image_p)
		device = root;

	if (!device)
		device = device_from_config(config);

	/* If we didn't get any device */
	if (!device)
		usage();

	if (writebootsec(FILE_TO_BOOT, device))
		return EXIT_FAILURE;

	return EXIT_SUCCESS;
}
