/* outboxjobs.cpp

    begin       : Sat Jun 08 2002
    copyright   : (C) 2001 by Martin Preuss
    email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/



#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "assert.h"

#include "outboxjobs.h"
#include <accountimpl.h>
#include <accountjobs.h>
#include <mediumrdhbase.h>


namespace HBCI {

// Only visible within this file
static Pointer<Customer> custPointer(Customer *c)
{
    Pointer<Customer> cp = c;
    cp.setAutoDelete(false);
    return cp;
}



/*___________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                           OutboxJobGetAccounts
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


OutboxJobGetAccounts::OutboxJobGetAccounts(Pointer<Customer> c)
    :OutboxJob(c)
{
}


OutboxJobGetAccounts::~OutboxJobGetAccounts(){
}


bool OutboxJobGetAccounts::createHBCIJobs(Pointer<MessageQueue> mbox, int n){
    _mbox = mbox;
    return true;
}


bool OutboxJobGetAccounts::evaluate(){
    _result=HBCI_JOB_RESULT_SUCCESS;
    _resultcode = _mbox.ref().getResult();
    _mbox=0;
    return true;
}


bool OutboxJobGetAccounts::commit(int msgNumber){
    return (_resultcode<9000);
}


string OutboxJobGetAccounts::description() const {
    return "Get account list";
}

list<int> OutboxJobGetAccounts::resultCodes() const
{
    list<int> res;
    res.push_back(_resultcode);
    return res;
}
} // namespace HBCI

HBCI_OutboxJobGetAccounts *HBCI_OutboxJobGetAccounts_new(HBCI_Customer *c)
{
    assert(c);
    return new HBCI_OutboxJobGetAccounts(custPointer(c));
}

    /** Upcast */
HBCI_OutboxJob *HBCI_OutboxJobGetAccounts_OutboxJob(HBCI_OutboxJobGetAccounts *j)
{
    return j;
}






namespace HBCI {

OutboxJobSynchronize::OutboxJobSynchronize(const API *api,
                                           Pointer<Customer> c,
                                           int syncwhat)
:OutboxJob(c)
,_hbciapi(api)
,_syncwhat(syncwhat)
{
}


OutboxJobSynchronize::~OutboxJobSynchronize(){
}


bool
OutboxJobSynchronize::createHBCIJobs(Pointer<MessageQueue> mbox,
                                     int n){
  if (n==0) {
    // create appropriate init job
    mbox.ref().setSyncMode(true);
    _initjob=new JOBDialogInit(_cust,
			       false, // anon
			       true,  // crypt
			       true,  // sign
			       false, // getkeys
                               true); // sync
    mbox.ref().addJob(_initjob.cast<Job>());
    _syncjob=new JOBSynchronize(_cust,_syncwhat);
    mbox.ref().addJob(_syncjob);
    addSignersToQueue(mbox);
    return true;
  }
  else if (n==1) {
    // create exit job (sign and crypt it, leave sync mode)
    mbox.ref().setSyncMode(false);
    _exitjob=new JOBDialogEnd(_cust,mbox.ref().dialogId(),true,true);
    mbox.ref().addJob(_exitjob);
    addSignersToQueue(mbox);
    return true;
  }
  else
    return false;
}


bool OutboxJobSynchronize::evaluate(){
  if (!_initjob.isValid() || !_exitjob.isValid() || !_syncjob.isValid())
    return false;
  if (_initjob.ref().hasErrors() ||
      _exitjob.ref().hasErrors() ||
      _syncjob.ref().hasErrors())
    _result=HBCI_JOB_RESULT_FAILED;
  else
    _result=HBCI_JOB_RESULT_SUCCESS;

  return _result==HBCI_JOB_RESULT_SUCCESS;
}


bool OutboxJobSynchronize::commit(int msgNumber){
    string sysid;
    Pointer<Medium> medium;

    // for the exitjob, there is nothing to commit
    // only for syncjob
    if (1 == msgNumber)
        return true;

    // check if needed job pointers are valid
    if (!_initjob.isValid() || !_syncjob.isValid())
        return false;

    // And Cheat around the const-correctness. FIXME: Maybe there's a
    // better way around that.
    API *a = const_cast<API*>(_hbciapi);
    // process the initJob (maybe update UPD, BPD). 
    if (!a->postProcessInitJob(_initjob.ref()).isOk())
        return false;

    // update according to syncmode (only SYSTEMID for now)
    sysid=dynamic_cast<JOBSynchronize&>(_syncjob.ref()).sysid();
    medium=_cust.ref().user().ref().medium();
    if (!medium.isValid())
        return false;
    if (_syncwhat==HBCI_SYNC_SYSTEMID &&
        medium.ref().securityMode()==HBCI_SECURITY_RDH &&
	!sysid.empty()) {
      medium.cast<MediumRDHBase>().ref().setSystemId(sysid);
    }
    else
        return false;
    return true;
}

list<int> OutboxJobSynchronize::resultCodes() const
{
    list<int> res, initres, exitres;
    if (_syncjob.isValid()) 
	res = resultCodesFromJob(_syncjob.ref());
    if (_initjob.isValid()) 
	initres = resultCodesFromJob(_initjob.ref());
    if (_exitjob.isValid()) 
	exitres = resultCodesFromJob(_exitjob.ref());
    // Note: this is not the most efficient implementation. However,
    // since this is only called when something went wrong, it isn't
    // used often.
    res.insert(res.end(), initres.begin(), initres.end());
    res.insert(res.end(), exitres.begin(), exitres.end());
    return res;
}



OutboxJobGetSystemId::OutboxJobGetSystemId(const API *api,
                                           Pointer<Customer> c)
:OutboxJobSynchronize(api,c,HBCI_SYNC_SYSTEMID)
{
}


OutboxJobGetSystemId::~OutboxJobGetSystemId(){
}


} // namespace HBCI

HBCI_OutboxJobGetSystemId *HBCI_OutboxJobGetSystemId_new(const HBCI_API *api, HBCI_Customer *c)
{
    assert(api);
    assert(c);
    return new HBCI_OutboxJobGetSystemId(api, custPointer(c));
}
HBCI_OutboxJob *HBCI_OutboxJobGetSystemId_OutboxJob(HBCI_OutboxJobGetSystemId *j) 
{
    return j;
}



namespace HBCI {

OutboxJobGetStatusReports::OutboxJobGetStatusReports(Pointer<Customer> c,
						     const Date &fromDate,
						     const Date &toDate,
						     int maxentries)
    :OutboxJob(c)
    ,_fromdate(fromDate)
    ,_todate(toDate)
    ,_maxentries(maxentries)
{
}


OutboxJobGetStatusReports::~OutboxJobGetStatusReports(){
}


bool OutboxJobGetStatusReports::createHBCIJobs(Pointer<MessageQueue> mbox,
					       int n){
  if (0 == n)
    _job=new JOBGetStatusReport(_cust, _fromdate, _todate, _maxentries);
  else {
    fprintf(stderr,"Will use jump point.\n");
    _job=new JOBGetStatusReport(_cust,
				_fromdate,
				_todate,
				_maxentries,
				_job.cast<Job>());
  }

  mbox.ref().addJob(_job.cast<Job>());
  addSignersToQueue(mbox);
  return true;
}


bool OutboxJobGetStatusReports::stillMessagesToSend(int nextMsg) const{
  // zero? of course we have a message to send!
  if (0 == nextMsg)
    return true;

  // any more?
  return _job.ref().attachMore();
}


bool OutboxJobGetStatusReports::evaluate(){
  if (_job.ref().hasErrors())
    _result=HBCI_JOB_RESULT_FAILED;
  else
    _result=HBCI_JOB_RESULT_SUCCESS;

  return _result==HBCI_JOB_RESULT_SUCCESS;
}


bool OutboxJobGetStatusReports::commit(int msgNumber){
  //bool doubled;
  list<StatusReport>::const_iterator it;

  // don't do anything when commit is called for the whole
  // job. otherwise, we would add the result of the last subjob twice
  if (HBCI_COMMIT_WHOLE_JOB == msgNumber)
    return true;

  // only commit if there was no error
  evaluate();
  if (_result!=HBCI_JOB_RESULT_SUCCESS)
    return false;

  // copy eventually received status reports
  for (it=_job.ref().statusReports().begin();
       it!=_job.ref().statusReports().end();
       it++) {
    _reports.push_back(*it);
  }

  return true;
}


string OutboxJobGetStatusReports::description() const{
  return "Get status reports";
}


list<int> OutboxJobGetStatusReports::resultCodes() const{
  list<int> res;
  if (_job.isValid())
    res = resultCodesFromJob(_job.ref());
  return res;
}

} // namespace HBCI


HBCI_OutboxJobGetStatusReports *
HBCI_OutboxJobGetStatusReports_new(HBCI_Customer *c,
				   const HBCI_Date *fromdate,
				   const HBCI_Date *todate,
				   int maxEntries)
{
    assert(c);
    assert(fromdate);
    assert(todate);
    return new HBCI_OutboxJobGetStatusReports(custPointer(c),
					      *fromdate,
					      *todate,
					      maxEntries);
}
HBCI_OutboxJob *
HBCI_OutboxJobGetStatusReports_OutboxJob(HBCI_OutboxJobGetStatusReports *j)
{
    return j;
}
const list_HBCI_StatusReport *
HBCI_OutboxJobGetStatusReports_statusReports(const
					     HBCI_OutboxJobGetStatusReports *j)
{
    assert(j);
    return &(j->statusReports());
}





