/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of the GNU C++ Library.  This library is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This library is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef COMPLEX_SUPPORTED

#ifdef __GNUG__
#pragma implementation
#endif
#include <MyComplex.h>
#include <stdlib.h>
#include <builtin.h>
#include <unistd.h>
#include <stdio.h>

// error handling

void default_MyComplex_error_handler(const char* msg)
{
  fprintf(stderr, "Fatal MyComplex arithmetic error. 'msg'\n");
  exit(1);
}

one_arg_error_handler_t MyComplex_error_handler = default_MyComplex_error_handler;

one_arg_error_handler_t set_MyComplex_error_handler(one_arg_error_handler_t f)
{
  one_arg_error_handler_t old = MyComplex_error_handler;
  MyComplex_error_handler = f;
  return old;
}

void  MyComplex::error(const char* msg) const
{
  (*MyComplex_error_handler)(msg);
}

/* from romine@xagsun.epm.ornl.gov */
MyComplex /* const */ operator / (const MyComplex& x, const MyComplex& y)
{
  double den = fabs(y.real()) + fabs(y.imag());
  if (den == 0.0) x.error ("Attempted division by zero.");
  double xrden = x.real() / den;
  double xiden = x.imag() / den;
  double yrden = y.real() / den;
  double yiden = y.imag() / den;
  double nrm   = yrden * yrden + yiden * yiden;
  return MyComplex((xrden * yrden + xiden * yiden) / nrm,
                 (xiden * yrden - xrden * yiden) / nrm);
}

MyComplex& MyComplex::operator /= (const MyComplex& y)
{
  double den = fabs(y.real()) + fabs(y.imag());
  if (den == 0.0) error ("Attempted division by zero.");
  double xrden = re / den;
  double xiden = im / den;
  double yrden = y.real() / den;
  double yiden = y.imag() / den;
  double nrm   = yrden * yrden + yiden * yiden;
  re = (xrden * yrden + xiden * yiden) / nrm;
  im = (xiden * yrden - xrden * yiden) / nrm;
  return *this;
}

MyComplex /* const */ operator / (double x, const MyComplex& y)
{
  double den = norm(y);
  if (den == 0.0) y.error ("Attempted division by zero.");
  return MyComplex((x * y.real()) / den, -(x * y.imag()) / den);
}

MyComplex /* const */ operator / (const MyComplex& x, double y)
{
  if (y == 0.0) x.error ("Attempted division by zero.");
  return MyComplex(x.real() / y, x.imag() / y);
}


MyComplex& MyComplex::operator /= (double y)
{
  if (y == 0.0) error ("Attempted division by zero.");
  re /= y;  im /= y;
  return *this;
}


MyComplex /* const */ exp(const MyComplex& x)
{
  double r = exp(x.real());
  return MyComplex(r * cos(x.imag()), 
                 r * sin(x.imag()));
}

MyComplex /* const */ cosh(const MyComplex& x)
{
  return MyComplex(cos(x.imag()) * cosh(x.real()), 
                 sin(x.imag()) * sinh(x.real()));
}

MyComplex /* const */ sinh(const MyComplex& x)
{
  return MyComplex(cos(x.imag()) * sinh(x.real()), 
                 sin(x.imag()) * cosh(x.real()));
}

MyComplex /* const */ cos(const MyComplex& x)
{
  return MyComplex(cos(x.real()) * cosh(x.imag()), 
                 -sin(x.real()) * sinh(x.imag()));
}

MyComplex /* const */ sin(const MyComplex& x)
{
  return MyComplex(sin(x.real()) * cosh(x.imag()), 
                 cos(x.real()) * sinh(x.imag()));
}

MyComplex /* const */ log(const MyComplex& x)
{
  double h = hypot(x.real(), x.imag());
  if (h <= 0.0) x.error("attempted log of zero magnitude number.");
  return MyComplex(log(h), atan2(x.imag(), x.real()));
}

// Corrections based on reports from: thc@cs.brown.edu & saito@sdr.slb.com
MyComplex /* const */ pow(const MyComplex& x, const MyComplex& p)
{
  double h = hypot(x.real(), x.imag());
  if (h <= 0.0) x.error("attempted power of zero magnitude number.");

  double a = atan2(x.imag(), x.real());
  double lr = pow(h, p.real());
  double li = p.real() * a;
  if (p.imag() != 0.0)
  {
    lr /= exp(p.imag() * a);
    li += p.imag() * log(h);
  }
  return MyComplex(lr * cos(li), lr * sin(li));
}

MyComplex /* const */ pow(const MyComplex& x, double p)
{
  double h = hypot(x.real(), x.imag());
  if (h <= 0.0) x.error("attempted power of zero magnitude number.");
  double lr = pow(h, p);
  double a = atan2(x.imag(), x.real());
  double li = p * a;
  return MyComplex(lr * cos(li), lr * sin(li));
}


MyComplex /* const */ sqrt(const MyComplex& x)
{
  if (x.real() == 0.0 && x.imag() == 0.0)
    return MyComplex(0.0, 0.0);
  else
  {
    double s = sqrt((fabs(x.real()) + hypot(x.real(), x.imag())) * 0.5);
    double d = (x.imag() / s) * 0.5;
    if (x.real() > 0.0)
      return MyComplex(s, d);
    else if (x.imag() >= 0.0)
      return MyComplex(d, s);
    else
      return MyComplex(-d, -s);
  }
}


MyComplex /* const */ pow(const MyComplex& x, int p)
{
  if (p == 0)
    return MyComplex(1.0, 0.0);
  else if (x == 0.0)
    return MyComplex(0.0, 0.0);
  else
  {
    MyComplex res(1.0, 0.0);
    MyComplex b = x;
    if (p < 0)
    {
      p = -p;
      b = 1.0 / b;
    }
    for(;;)
    {
      if (p & 1)
        res *= b;
      if ((p >>= 1) == 0)
        return res;
      else
        b *= b;
    }
  }
}

#endif	/* COMPLEX_SUPPORTED */
