// optionsetter.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#ifdef __GNUG__
#pragma implementation
#endif

#include "MyString.h"
#include "mxv_types.h"
#include "application.h"
#include "dialogbox.h"
#include "envelopeheader.h"
#include "fileselector.h"
#include "formatrequester.h"
#include "query_templates.h"
#include "request.h"
#include "requester.h"
#include "lpcdata.h"
#include "lpcheader.h"
#include "smartmem.h"
#include "sound.h"
#include "soundheader.h"
#include "optionsetter.h"

class GlobalOptionRequester : public TitledRequester {
	friend class GlobalOptionSetter;
protected:
	GlobalOptionRequester(GlobalOptionSetter *);
	redefined void configureRequest(Request *);
private:
	GlobalOptionSetter* client;
};

GlobalOptionRequester::GlobalOptionRequester(GlobalOptionSetter* gos)
	: TitledRequester("Set Global Program Options:"), client(gos) {}

void
GlobalOptionRequester::configureRequest(Request* request) {
	request->appendValue(
#ifndef __GNUG__
		createQuery(
			"Alert beep volume:",
			&DialogBox::setBeepLevel,
			&DialogBox::beepLevel,
			Range(0, 100),
			true
		)
#else
		new Query<int>(
			"Alert beep volume:",
			new FunctionSetterGetter<boolean, int>(
				&DialogBox::setBeepLevel,
				&DialogBox::beepLevel
			),
			Range(0, 100),
			true
		)
#endif
	);
	request->appendChoice("Dialog Panels Ignore Window Manager:",
	                      "|No|Yes|", &client->panelOverride, true);
	request->appendChoice("Auto-Place Windows on Screen:",
	                     "|No|Yes|", &client->autoPlace, true);
}

//********

GlobalOptionSetter::GlobalOptionSetter()
	  : panelOverride(DialogBox::overrideWindowManager() ?
		Requester::True : Requester::False),
	    autoPlace(
		Application::globalResourceIsTrue("AutoPlaceWindows") ?
			Requester::True : Requester::False) {}

Requester *
GlobalOptionSetter::createRequester() {
	return new GlobalOptionRequester(this);
}

int
GlobalOptionSetter::apply() {
	DialogBox::overrideWindowManager(panelOverride == Requester::True);
	Application::setGlobalResource(
		"AutoPlaceWindows",
		(autoPlace == Requester::True) ? "true" : "false"
	);
	return true;
}

//********

class MemoryOptionRequester : public TitledRequester {
	friend class MemoryOptionSetter;
protected:
	MemoryOptionRequester(MemoryOptionSetter *);
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	MemoryOptionSetter* client;
};

MemoryOptionRequester::MemoryOptionRequester(MemoryOptionSetter* mos)
	: TitledRequester("Set Data Memory Options:"), client(mos) {}

void
MemoryOptionRequester::configureRequest(Request* request) {
	request->appendLabel("(Use to adjust memory limits for waveforms only.)");
	request->appendLabel("----------------");
	static char label[120];
	sprintf(label, "Total memory currently in use for waveforms:  %.2f mB",
		SmartMemory::totalBytesAllocated()/1.0e+06);
	request->appendLabel(label);
	client->maxSingle = SmartMemory::maxSingleAllocation()/1.0e+06;
	client->maxTotal = SmartMemory::maxTotalAllocation()/1.0e+06;
	request->appendValue("Maximum Single Memory Allocation (mB.):",
	                    &client->maxSingle,  Range(1.0, 1000.0), true);
	request->appendValue("Maximum Total Memory Allocation (mB.):",
	                    &client->maxTotal, 
						Range(20.0, size_t(0xffffffff)/1.0e+06),
						true);
}

boolean
MemoryOptionRequester::confirmValues() {
	if(client->maxTotal <= client->maxSingle) {
		Application::alert("Total Allocation limit must be greater than",
			"the Single Allocation limit.  Please recheck settings.");
		return false;
	}
	return true;
}

//********

int
MemoryOptionSetter::apply() {
	SmartMemory::setMaxSingleAllocation(size_t(maxSingle * 1.0e+06));
	SmartMemory::setMaxTotalAllocation(size_t(maxTotal * 1.0e+06));
	char value[64];
	sprintf(value, "%g", maxSingle);
	Application::setGlobalResource("MaximumSingleAllocation", value);
	sprintf(value, "%g", maxTotal);
	Application::setGlobalResource("MaximumTotalAllocation", value);
	return true;
}

Requester *
MemoryOptionSetter::createRequester() {
	return new MemoryOptionRequester(this);
}

//********

class FileOptionRequester : public TitledRequester {
	friend class FileOptionSetter;
protected:
	FileOptionRequester(FileOptionSetter* fos)
		: TitledRequester("Set File Options:"), client(fos) {}
	redefined void configureRequest(Request *);
private:
	FileOptionSetter* client;
};

void
FileOptionRequester::configureRequest(Request* request) {
	request->appendChoice("Read Raw (Headerless) Files:",
	                     "|No|Yes|", &client->readRaw, true);
	request->appendChoice("Store/Recall Browser Path:",
	                     "|No|Yes|", &client->storePath, true);
	request->appendChoice("Browser Shows Invisible Files:",
	                     "|No|Yes|", &client->showInvisible, true);
}

//********

FileOptionSetter::FileOptionSetter()
	: readRaw(Application::globalResourceIsTrue("ReadRawFiles") ?
		Requester::True : Requester::False),
	  storePath(
		Application::globalResourceIsTrue("BrowserUseLastPath") ?
			Requester::True : Requester::False),
	  showInvisible(FileSelector::showDotFiles() ?
		Requester::True : Requester::False) {}

Requester *
FileOptionSetter::createRequester() {
	return new FileOptionRequester(this);
}

int
FileOptionSetter::apply() {
	Application::setGlobalResource(
		"ReadRawFiles", (readRaw == Requester::True) ? "true" : "false"
	);
	Application::setGlobalResource(
		"BrowserUseLastPath",
		(storePath == Requester::True) ? "true" : "false"
	);
	FileSelector::showDotFiles(showInvisible == Requester::True);
	return true;
}

//********

class EditorOptionRequester : public TitledRequester {
	friend class EditorOptionSetter;
protected:
	EditorOptionRequester(EditorOptionSetter* e)
		  : TitledRequester("Set Edit Command Options:"),
		    client(e) {}
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	EditorOptionSetter* client;
};

void
EditorOptionRequester::configureRequest(Request* request) {
	request->appendValue("Command Undo Depth: ",
						 &client->_undoDepth,
						 NonNegativeShorts, True);
	request->appendChoice("Clear Undo History on Save:",
	                     "|No|Yes|", &client->_clearUndoOnSave, true);
}

boolean
EditorOptionRequester::confirmValues() {
	if (client->_undoDepth == 0) {
		return Application::confirm("With a zero history depth you will not be",
						   "able to undo any editing operations.",
						   "Is this what you want?");
	}
	return true;
}

//

EditorOptionSetter::EditorOptionSetter()
{
	if (Application::globalResourceIsTrue("ClearUndoOnSave"))
		_clearUndoOnSave = Requester::True;
	else 
		_clearUndoOnSave = Requester::False;
	const char *res = Application::getGlobalResource("UndoDepth");
	_undoDepth = res ? max(0, atol(res)) : 65535;
}

Requester *
EditorOptionSetter::createRequester() {
	return new EditorOptionRequester(this);
}

int
EditorOptionSetter::apply() {
	Application::setGlobalResource(
		"ClearUndoOnSave", (_clearUndoOnSave == Requester::True) ? "true" : "false"
	);
	char string[6];
	sprintf(string, "%d", _undoDepth);
	Application::setGlobalResource("UndoDepth", string);
	return true;
}

//********

class SoundOptionRequester : public FormatRequester {
	friend class SoundOptionSetter;
protected:
	SoundOptionRequester(SoundOptionSetter* s)
		: FormatRequester(nil, "Default Sound Sample Format:",
		                  s->sampleFormat),
		  TitledRequester("Set Sound Options:"),
		  client(s) {}
	redefined void configureRequest(Request *);
private:
	SoundOptionSetter* client;
};

void
SoundOptionRequester::configureRequest(Request* request) {
	request->appendValue(
#ifndef __GNUG__
		createQuery(
			"Default Sampling Rate:",
			&Sound::setDefaultSampleRate,
			&Sound::defaultSampleRate,
			NonNegativeNumbers,
			false
		)
#else
		new Query<int>(
			"Default Sampling Rate:",
			new FunctionSetterGetter<boolean, int>(
				&Sound::setDefaultSampleRate,
				&Sound::defaultSampleRate
			),
			NonNegativeNumbers,
			false
		)
#endif
	);
	FormatRequester::configureRequest(request);
	request->appendChoice("Default Sound Header Format:",
		"|Raw (No Header)|Snd/au|Hybrid|BSD/IRCAM|AIF-C|WAVE|",
		&client->headerFormat
	);
}

//********

SoundOptionSetter::SoundOptionSetter()
	: sampleFormat(Sound::defaultDataType()),
	  headerFormat(SoundHeader::defaultHeaderType()) {}

Requester *
SoundOptionSetter::createRequester() {
	return new SoundOptionRequester(this);
}

int
SoundOptionSetter::apply() {
	Sound::setDefaultDataType(DataType(sampleFormat));
	SoundHeader::setDefaultHeaderType(SoundHeader::Type(headerFormat));
	return true;
}

//********

class LPCOptionRequester : public TitledRequester {
	friend class LPCOptionSetter;
protected:
	LPCOptionRequester(LPCOptionSetter* l)
		: TitledRequester("Set LPC Data Options:"), client(l) {}
	redefined void configureRequest(Request *);
private:
	LPCOptionSetter* client;
};

void
LPCOptionRequester::configureRequest(Request* request) {
	request->appendLabel("(Default samp rate = default sound samp rate.)");
	request->appendValue(
#ifndef __GNUG__
		createQuery(
			"Default Frame Rate:",
			&LPCData::setDefaultFrameRate,
			&LPCData::defaultFrameRate,
			NonNegativeNumbers,
			false
		)
#else
		new Query<double>(
			(const char *) "Default Frame Rate:",
			(SetterGetter<double> *)
			new FunctionSetterGetter<boolean, double>(
				&LPCData::setDefaultFrameRate,
				&LPCData::defaultFrameRate
			),
			NonNegativeNumbers,
			false
		)
#endif
	);
	request->appendValue(
#ifndef __GNUG__
		createQuery(
			"Default Number of Filter Poles:",
			&LPCData::setDefaultNumberOfPoles,
			&LPCData::defaultNumberOfPoles,
			Range(12, 64),
			true
		)
#else
		new Query<int>(
			(const char *) "Default Number of Filter Poles:",
			(SetterGetter<int> *)
			new FunctionSetterGetter<boolean, int>(
				&LPCData::setDefaultNumberOfPoles,
				&LPCData::defaultNumberOfPoles
			),
			Range(12, 64),
			true
		)
#endif
	);
	request->appendChoice("Default Byte Order: ",
		"|Little Endian|Big Endian|",
		&client->byteOrder
	);
	request->appendChoice("Default LPC Header Format: ",
		"|Raw (No Header)|Csound-format Header|",
		&client->headerFormat
	);
}

//********

LPCOptionSetter::LPCOptionSetter()
	: headerFormat(LPCHeader::defaultHeaderType()),
	  byteOrder(LPCHeader::defaultByteOrder()) {}

Requester *
LPCOptionSetter::createRequester() {
	return new LPCOptionRequester(this);
}

int
LPCOptionSetter::apply() {
	LPCHeader::setDefaultHeaderType(LPCHeader::Type(headerFormat));
	LPCHeader::setDefaultByteOrder(Header::ByteOrder(byteOrder));
	Application::setGlobalResource("DefaultLPCByteOrder",
								   byteOrder == Header::Big ? "Big" : "Little");
	return true;
}

//********

class EnvelopeOptionRequester : public TitledRequester {
	friend class EnvelopeOptionSetter;
protected:
	EnvelopeOptionRequester(EnvelopeOptionSetter* l)
		: TitledRequester("Set Envelope Data Options:"), client(l) {}
	redefined void configureRequest(Request *);
private:
	EnvelopeOptionSetter* client;
};

void
EnvelopeOptionRequester::configureRequest(Request* request) {
	request->appendChoice("Default Byte Order: ",
		"|Little Endian|Big Endian|",
		&client->byteOrder
	);
	request->appendChoice("Default Envelope Header Format: ",
		"|Raw (No Header)|With Header|",
		&client->headerFormat
	);
}

//********

EnvelopeOptionSetter::EnvelopeOptionSetter()
	: headerFormat(EnvelopeHeader::defaultHeaderType()),
	  byteOrder(EnvelopeHeader::defaultByteOrder()) {}

Requester *
EnvelopeOptionSetter::createRequester() {
	return new EnvelopeOptionRequester(this);
}

int
EnvelopeOptionSetter::apply() {
	EnvelopeHeader::setDefaultHeaderType(EnvelopeHeader::Type(headerFormat));
	EnvelopeHeader::setDefaultByteOrder(Header::ByteOrder(byteOrder));
	Application::setGlobalResource("DefaultEnvelopeByteOrder",
								   byteOrder == Header::Big ? "Big" : "Little");
	return true;
}

//********

class RawSoundfileOptionRequester : public FormatRequester {
	friend class RawSoundfileOptionSetter;
protected:
	RawSoundfileOptionRequester(RawSoundfileOptionSetter* s)
		: FormatRequester(nil, "Raw File Sample Format:",
		                  s->sampleFormat),
		  TitledRequester("Set Raw File Defaults:"),
		  client(s) {}
	redefined void configureRequest(Request *);
private:
	RawSoundfileOptionSetter* client;
};

void
RawSoundfileOptionRequester::configureRequest(Request* request) {
	request->appendValue(
#ifndef __GNUG__
		createQuery(
			"Raw File Channel Count:",
			&Sound::setDefaultRawChannelCount,
			&Sound::defaultRawChannelCount,
			NonNegativeNumbers,
			false
		)
#else
		new Query<int>(
			"Raw File Channel Count:",
			new FunctionSetterGetter<boolean, int>(
				&Sound::setDefaultRawChannelCount,
				&Sound::defaultRawChannelCount
			),
			NonNegativeNumbers,
			false
		)
#endif
	);
	request->appendValue(
#ifndef __GNUG__
		createQuery(
			"Raw File Sampling Rate:",
			&Sound::setDefaultRawSampleRate,
			&Sound::defaultRawSampleRate,
			NonNegativeNumbers,
			false
		)
#else
		new Query<int>(
			"Raw File Sampling Rate:",
			new FunctionSetterGetter<boolean, int>(
				&Sound::setDefaultRawSampleRate,
				&Sound::defaultRawSampleRate
			),
			NonNegativeNumbers,
			false
		)
#endif
	);
	FormatRequester::configureRequest(request);
}

//********

RawSoundfileOptionSetter::RawSoundfileOptionSetter()
	: sampleFormat(Sound::defaultRawDataType()) {}

Requester *
RawSoundfileOptionSetter::createRequester() {
	return new RawSoundfileOptionRequester(this);
}

int
RawSoundfileOptionSetter::apply() {
	Sound::setDefaultRawDataType(DataType(sampleFormat));
	return true;
}

//********

class SoundPlaybackOptionRequester : public TitledRequester {
	friend class SoundPlaybackOptionSetter;
protected:
	SoundPlaybackOptionRequester(SoundPlaybackOptionSetter* s)
		  : TitledRequester("Set Sound Playback Options:"),
		    client(s) {}
	redefined void configureRequest(Request *);
private:
	SoundPlaybackOptionSetter* client;
};

void
SoundPlaybackOptionRequester::configureRequest(Request* request) {
#ifdef OSS
	request->appendLabel("Make sure audio and mixer devices match");
	request->appendValue("Audio Device: ", &client->audioDeviceName);
	request->appendValue("Mixer Device: ", &client->mixerDeviceName);
#elif defined ALSA
	request->appendValue("Audio Device: ", &client->audioDeviceName);
#endif
	request->appendValue("Audio Buffer Length (seconds): ",
						 &client->audioBufferTime,
						 Range(0.01, 10.0), True);
}

#ifdef OSS
#define DEFAULT_DEVICE "/dev/dsp"
#define DEFAULT_MIXER "/dev/mixer"
#define DEFAULT_BUFTIME 0.1
#define BUFTIME_RESOURCE "OSSBufferTime"
#elif defined(MACOSX)
#define DEFAULT_DEVICE ""
#define DEFAULT_MIXER ""
#define DEFAULT_BUFTIME 0.1
#define BUFTIME_RESOURCE "OSSBufferTime"
#else
#define DEFAULT_DEVICE ""
#define DEFAULT_MIXER ""
#define DEFAULT_BUFTIME 0.1
#define BUFTIME_RESOURCE "BufferTime"
#endif

SoundPlaybackOptionSetter::SoundPlaybackOptionSetter()
{
#ifdef OSS
	audioDeviceName = 
		Application::getGlobalResource("OSSAudioDevice") ?
		Application::getGlobalResource("OSSAudioDevice") : DEFAULT_DEVICE;
	mixerDeviceName =
		Application::getGlobalResource("OSSMixerDevice") ?
		Application::getGlobalResource("OSSMixerDevice") : DEFAULT_MIXER;
#endif
	audioBufferTime =
		Application::getGlobalResource(BUFTIME_RESOURCE) ?
			atof(Application::getGlobalResource(BUFTIME_RESOURCE)) :
			DEFAULT_BUFTIME;
}

Requester *
SoundPlaybackOptionSetter::createRequester() {
	return new SoundPlaybackOptionRequester(this);
}

int
SoundPlaybackOptionSetter::apply() {
#ifdef OSS
	Application::setGlobalResource("OSSAudioDevice", audioDeviceName);
	Application::setGlobalResource("OSSMixerDevice", mixerDeviceName);
#endif
	char timeString[28];
	sprintf(timeString, "%g", audioBufferTime);
	Application::setGlobalResource(BUFTIME_RESOURCE, timeString);
	return true;
}

//********
//********

enum OptionType { StringType, IntType, DoubleType, BooleanType };

struct OptionEntry {
	const char* optionName;
	OptionType optionType;
};

static OptionEntry optionEntries[] = {
	// Global Options
	{ "AlertBeepVolume", IntType },
	{ "DialogIgnoreWindowManager", BooleanType },
	{ "AutoPlaceWindows", BooleanType },
	// Memory Options
	{ "MaximumSingleAllocation", DoubleType },
	{ "MaximumTotalAllocation", DoubleType },
	// File Options
	{ "ReadRawFiles", BooleanType },
	{ "BrowserUseLastPath", BooleanType },
	{ "BrowserShowDotFiles", BooleanType },

	{ "DefaultSoundFileDir", StringType },
	{ "DefaultLPCFileDir", StringType },
	{ "DefaultPitchFileDir", StringType },
	{ "DefaultFFTFileDir", StringType },
	{ "DefaultEnvelopeFileDir", StringType },
	{ "DefaultPvocFileDir", StringType },
	
	// raw file options
	
	{ "DefaultRawSoundfileSampleRate", IntType },
	{ "DefaultRawSoundfileSampleFormat", StringType },
	{ "DefaultRawSoundfileChannels", IntType },

	// edit options

	{ "UndoDepth", IntType },
	{ "ClearUndoOnSave", BooleanType },

	// Data Options
	{ "DefaultSoundSampleRate", IntType },
	{ "DefaultSoundSampleFormat", StringType },
	{ "DefaultSoundHeaderType", StringType },
	{ "DefaultLPCFrameRate", DoubleType },
	{ "DefaultLPCNumberOfPoles", IntType },
	{ "DefaultLPCHeaderType", StringType },
	{ "DefaultLPCByteOrder", StringType },	// "Big" or "Little"
	{ "DefaultEnvelopeByteOrder", StringType },	// "Big" or "Little"
#ifdef OSS
	// Sound Playback Options
	{ "OSSAudioDevice", StringType },
	{ "OSSMixerDevice", StringType },
	{ "OSSBufferTime", DoubleType },
#endif
#ifdef MACOSX
	{ "OSXBufferTime", DoubleType },
#endif
	{ nil }
};

#ifndef __GNUG__
extern "C" char* getenv(const char*);
#endif

void setStaticOptionValues();

void read_mxvrc() {
	char mxvrc_path[1024];
	const char* homeDir = getenv("HOME");
	if(!homeDir) homeDir = ".";
	sprintf(mxvrc_path, "%s/.mxvrc", homeDir);
	FILE* file = fopen(mxvrc_path, "r");
	if(file == nil) return;
	char oname[80];
	char ovalue[1024];
	// Some global entries default to specific values
	Application::setGlobalResource("ClearUndoOnSave", "True");
	// Scan entries set in file
	while(fscanf(file, "%s %s", oname, ovalue) != EOF) {
		String optionName(oname);
		for(OptionEntry* entry = optionEntries;
			entry->optionName != nil; 
			entry++)
		{
			if (fcompare(optionName, String(entry->optionName)) == 0) {
				Application::setGlobalResource(entry->optionName, ovalue);
				break;
			}
		}
	}
	fclose(file);
	setStaticOptionValues();
}

void write_mxvrc() {
	char mxvrc_path[1024];
	const char* homeDir = getenv("HOME");
	if(!homeDir) homeDir = ".";
	sprintf(mxvrc_path, "%s/.mxvrc", homeDir);
	FILE* file = fopen(mxvrc_path, "w+");
	if(file != nil) {
		for(OptionEntry* entry = optionEntries;
				entry->optionName != nil; entry++) {
			const char* resource =
				Application::getGlobalResource(entry->optionName);
			if(resource && strlen(resource))
				fprintf(file, "%s\t\t\t%s\n", entry->optionName, resource);
		}
		fclose(file);
	}
	else {
		Application::error("Unable to open .mxvrc file for writing.",
							mxvrc_path);
	}
}

void
setStaticOptionValues()
{
    for(OptionEntry* entry = optionEntries; entry->optionName != nil; entry++)
    {
	const char* res = Application::getGlobalResource(entry->optionName);
	if (!res || strlen(res) == 0)
	    continue;
	if (!strcmp(entry->optionName, "DefaultSoundSampleRate"))
	    Sound::setDefaultSampleRate(atoi(res));
	else if (!strcmp(entry->optionName, "DefaultSoundSampleFormat")) {
	    if (!strncasecmp(res, "char", 4)
		    || !strncasecmp(res, "signed char", 11))
		Sound::setDefaultDataType(SignedCharData);
	    else if (!strncasecmp(res, "unsigned char", 13))
		Sound::setDefaultDataType(UnsignedCharData);
	    else if (!strncasecmp(res, "short", 5))
		Sound::setDefaultDataType(ShortData);
	    else if (!strncasecmp(res, "integer", 7))
		Sound::setDefaultDataType(IntData);
	    else if (!strncasecmp(res, "float", 5))
		Sound::setDefaultDataType(FloatData);
	}
	else if (!strcmp(entry->optionName, "DefaultRawSoundfileSampleRate"))
	    Sound::setDefaultRawSampleRate(atoi(res));
	else if (!strcmp(entry->optionName, "DefaultRawSoundfileChannels"))
	    Sound::setDefaultRawChannelCount(atoi(res));
	else if (!strcmp(entry->optionName, "DefaultRawSoundfileSampleFormat")) {
	    if (!strncasecmp(res, "char", 4)
		    || !strncasecmp(res, "signed char", 11))
		Sound::setDefaultRawDataType(SignedCharData);
	    else if (!strncasecmp(res, "unsigned char", 13))
		Sound::setDefaultRawDataType(UnsignedCharData);
	    else if (!strncasecmp(res, "short", 5))
		Sound::setDefaultRawDataType(ShortData);
	    else if (!strncasecmp(res, "integer", 7))
		Sound::setDefaultRawDataType(IntData);
	    else if (!strncasecmp(res, "float", 5))
		Sound::setDefaultRawDataType(FloatData);
	}
	else if (!strcmp(entry->optionName, "DefaultLPCFrameRate"))
	    LPCData::setDefaultFrameRate(atof(res));
	else if (!strcmp(entry->optionName, "DefaultLPCNumberOfPoles"))
	    LPCData::setDefaultNumberOfPoles(atoi(res));
	else if (!strcmp(entry->optionName, "DefaultLPCByteOrder"))
	    LPCHeader::setDefaultByteOrder((!strncasecmp(res, "Big", 3)) ?
											Header::Big : Header::Little);
	else if (!strcmp(entry->optionName, "DefaultEnvelopeByteOrder"))
	    EnvelopeHeader::setDefaultByteOrder((!strncasecmp(res, "Big", 3)) ?
											Header::Big : Header::Little);
	else if (!strcmp(entry->optionName, "MaximumSingleAllocation"))
	    SmartMemory::setMaxSingleAllocation(size_t(atof(res) * 1.0e+06));
	else if (!strcmp(entry->optionName, "MaximumTotalAllocation"))
	    SmartMemory::setMaxTotalAllocation(size_t(atof(res) * 1.0e+06));
    }
}
