/*
 * mb-record.cc --
 *
 *      MediaBoard Archive Recorder
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @(#) $Header: /usr/mash/src/repository/mash/mash-1/archive/mb-record.cc,v 1.19 2002/02/03 03:09:26 lim Exp $
 */


#include "archive/archive-stream.h"
#include "mb/mb-obj.h"
#include "mb/mb-cmd.h"
#include "archive/mb-record.h"


DEFINE_OTCL_CLASS(MBRecordMgr, "MB_Manager/Record") {
	INSTPROC(cname_update);
}


DEFINE_OTCL_CLASS(MBRecordStream, "ArchiveStream/Record/Mediaboard") {
	INSTPROC(init_file_header);
	INSTPROC(write_headers);
}


MBRecordRcvr::MBRecordRcvr(MBBaseMgr *pMgr, const SrcId &srcId)
	: MBBaseRcvr(pMgr, srcId), stream_(NULL)
{
}


MBRecordRcvr::~MBRecordRcvr()
{
}


void
MBRecordRcvr::recvCommands(Byte *pb, u_int len, Bool isReply)
{
	PageId pageId;
	net2host(((Pkt_PageHdr*) pb)->pd_page, pageId);
	MBBaseRcvr::recvCommands(pb, len, isReply);
	MBRecordPage *pPage = (MBRecordPage*) DefinePage(pageId);
	assert(pPage && "Huh! pPage cannot be NULL at this point");

	MTrace(trcArchive, ("Trying to archive a record"));
	if (stream_!=NULL) stream_->Record(pPage);
}


int
MBRecordStream::init_file_header(int argc, const char * const *argv)
{
	BEGIN_PARSE_ARGS(argc,argv);
	END_PARSE_ARGS;

	memset(&hdr_, 0, sizeof(FileHeader));
	if (Invoke("media", NULL)==TCL_ERROR) {
		MTrace(trcArchive, ("Cannot determine media of stream %s",
		       name()));
		strcpy(hdr_.media, "mediaboard");
	} else {
		strcpy(hdr_.media, Tcl::instance().result());
	}

	strcpy(hdr_.protocol, PROTO_SRM);
	strcpy(hdr_.cname, "");
	strcpy(hdr_.name,  "");

	hdr_.start_sec = hdr_.start_usec = 0;
	hdr_.end_sec   = hdr_.end_usec   = 0;
	hdr_.privateLen = 0;
	return TCL_OK;
}


/*
 * Check if the mustRewriteHdr_ flag is set; rewrite the header only if the
 * flag is set
 */
int
MBRecordStream::write_headers(int argc, const char * const *argv)
{
	BEGIN_PARSE_ARGS(argc,argv);
	END_PARSE_ARGS;

	if (mustRewriteHdr_) {
		if (WriteHeaders()!=TCL_OK) return TCL_ERROR;
		mustRewriteHdr_ = false;
	}
	return TCL_OK;
}


int
MBRecordStream::WriteHeaders()
{
	FileHeader hdrNet;
	ArchiveFile *file;

	host2net(hdr_, hdrNet);
	// just make sure that the private length is zero
	hdrNet.privateLen = host2net((u_int32_t)0);

	strcpy(hdrNet.version, DATA_VERSION);
	file = DataFile_();
	if (file!=NULL) {
		if (file->Write(&hdrNet) < 0) return TCL_ERROR;
	}

	strcpy(hdrNet.version, INDEX_VERSION);
	file = IndexFile_();
	if (file!=NULL) {
		if (file->Write(&hdrNet) < 0) return TCL_ERROR;
	}

	return TCL_OK;
}


int
MBRecordStream::SeekToStartOfData()
{
	ArchiveFile *file;
	file = DataFile_();
	if (file!=NULL) {
		if (file->Seek(sizeof(FileHeader), SEEK_SET)==TCL_ERROR)
			return TCL_ERROR;
	}

	file = IndexFile_();
	if (file!=NULL) {
		if (file->Seek(sizeof(FileHeader), SEEK_SET)==TCL_ERROR)
			return TCL_ERROR;
	}

	return TCL_OK;

}


Bool
MBRecordStream::Record(MBRecordPage *pPage)
{
	ulong seqno = pPage->LastRecorded() + 1, maxSeqno;
	int packetLen, totalBytes;
	MBCmd *pCmd;
	Pkt_PageHdr *pHdr;
	DataFile  *dataFile;
	IndexFile *indexFile;
	struct IndexRecord idxHost, idxNet;
	timeval tv, now;
	u_int32_t ts;

	totalBytes = sizeof(Pkt_PageHdr);
	if (buffer_.alloc(totalBytes)==FALSE) {
		MTrace(trcArchive, ("Cannot allocate buffer space"));
		return FALSE;
	}


	maxSeqno = pPage->getMaxSeqno();
	MTrace(trcArchive|trcVerbose, ("Trying to access seqno %d of %d",seqno,
				    maxSeqno));
	while (seqno <= maxSeqno && (pCmd=pPage->getCmd(seqno))!=NULL) {
		packetLen = pCmd->getPacketLen();
		if (buffer_.alloc(totalBytes + packetLen)==FALSE) {
			MTrace(trcArchive, ("Cannot allocate buffer space"));
			return FALSE;
		}
		MTrace(trcArchive|trcVerbose,
		       ("Packetizing cmd %d: Timestamp %u",
			pCmd->getType(), pCmd->getTimeStamp()));
		pCmd->Packetize(buffer_.pb + totalBytes);
		totalBytes += packetLen;
		seqno++;
	}

	if (seqno > pPage->LastRecorded()+1) {
		// we have some data to write

		pHdr = (Pkt_PageHdr*) buffer_.pb;
		host2net(pPage->getId(), pHdr->pd_page);
		pHdr->pd_sseq = host2net(pPage->LastRecorded()+1);
		pHdr->pd_eseq = host2net(seqno - 1);

		pCmd = pPage->getCmd(pPage->LastRecorded()+1);

		dataFile = DataFile_();
		indexFile= IndexFile_();

		if (dataFile==NULL) {
			MTrace(trcArchive, ("No data file exists"));
			return FALSE;
		}

		gettimeofday(&now, NULL);
		tv = mb2logical(pCmd->getTimeStamp(), now);

		idxHost.recvTS_sec = now.tv_sec;
		idxHost.recvTS_usec= now.tv_usec;
		idxHost.sentTS_sec = tv.tv_sec;
		idxHost.sentTS_usec= tv.tv_usec;
		idxHost.seqno      = pCmd->getSeqno();
		idxHost.filePointer= dataFile->Tell();

		if (dataFile->Write(buffer_.pb, totalBytes) < 0) {
			MTrace(trcArchive, ("Error writing to data file"));
			return FALSE;
		}

		if (indexFile!=NULL) {
			host2net(idxHost, idxNet);
			if (indexFile->Write(&idxNet) < 0) {
				MTrace(trcArchive,
				       ("Error writing to index file"));
				return FALSE;
			}
		}

		// set a flag to remember to rewrite the header
		MustRewriteHdr();

		// notify all observers
		bytesWritten_ += totalBytes;
		Invokef("notify_observers bytes_rcvd %lu",
			(unsigned long) bytesWritten_);

		// increment LastRecorded
		pPage->LastRecorded(seqno-1);

		// set the start timestamp if required
		ts = pPage->getCmd(1)->getTimeStamp();
		if ( (hdr_.start_sec==0 && hdr_.start_usec==0) ||
		    (ts < logical2mb(hdr_.start_sec, hdr_.start_usec)) ) {
			tv = mb2logical(ts, now);
			hdr_.start_sec = tv.tv_sec;
			hdr_.start_usec= tv.tv_usec;
		}

		// set the end timestamp if required
		ts = pPage->getCmd(pPage->LastRecorded())->getTimeStamp();
		if ( (hdr_.end_sec==0 && hdr_.end_usec==0) ||
		     (ts > logical2mb(hdr_.end_sec, hdr_.end_usec)) ) {
			tv = mb2logical(ts, now);
			hdr_.end_sec = tv.tv_sec;
			hdr_.end_usec= tv.tv_usec;
		}
	}
	else {
		MTrace(trcArchive|trcVerbose,
		       ("No data yet to record for page %lu:%s",
			pPage->getId().uid, (char*)pPage->getId().sid));
	}
	return TRUE;
}



MBBaseRcvr *
MBRecordMgr::NewReceiver(const SrcId &srcId, Bool isLocal)
{
	MTrace(trcArchive, ("Creating a new receiver %s (local: %d)",
	       (char*)srcId, isLocal));
	if (isLocal==FALSE) {
		MBRecordStream *stream = (MBRecordStream*)
			TclObject::New("ArchiveStream/Record/Mediaboard",
				       name(), NULL);
		if (stream==NULL) {
			MTrace(trcArchive, ("'new ArchiveStream/Record/"
					    "Mediaboard' failed: %s",
					    Tcl::instance().result()));
		}
		stream->Invoke("bind",name(), NULL);


		MBRecordRcvr *rcvr = new MBRecordRcvr(this, srcId);
		if (stream!=NULL) {
			// link the RecordStream and MBRecordRcvr objects
			// together

			stream->attach(rcvr);
			rcvr->attach(stream);

			// write the headers
			FileHeader *hdr = stream->FileHeader_();
			strcpy(hdr->cname, (char*)(srcId));
			// FIXME: for now use rcvr time to set the start and end
			// times in the header
			/*timeval now;
			gettimeofday(&now, NULL);
			hdr->start_sec = now.tv_sec;
			hdr->start_usec= now.tv_usec;*/

			// FIXME: must check the return value
			stream->WriteHeaders();
			stream->SeekToStartOfData();
		}

		return rcvr;
	}
	else {
		return new MBRecordRcvr(this, srcId);
	}
}


int
MBRecordMgr::cname_update(int argc, const char * const *argv)
{
	TclObject *src;
	const char *cname;

	BEGIN_PARSE_ARGS(argc, argv);
	ARG(src);   // TclObject *
	ARG(cname); // char *
	END_PARSE_ARGS;

	Tcl &tcl = Tcl::instance();
	if (src==NULL) {
		tcl.resultf("invalid source object: '%s'", argv[2]);
		tcl.add_errorf("\ninvoked from within "
			       "MBRecordMgr::CnameUpdate()");
		return TCL_ERROR;
	}

	MBRecordRcvr *rcvr = (MBRecordRcvr*) ((SRM_Source*)src)->handler();
	if (rcvr!=NULL && rcvr->stream()!=NULL) {
		rcvr->stream()->Invoke("notify_observers","name",cname,NULL);
		strcpy(rcvr->stream()->FileHeader_()->name, cname);
		rcvr->stream()->MustRewriteHdr();
		//return rcvr->stream()->WriteHeaders();
		return TCL_OK;
	}
	else return TCL_OK;
}
