/*
 * postdct.h --
 *
 * Post-DCT Deblocking Algorithms.
 *
 * Derived from original code provided by Jim Chou. For more information see
 * http://www-wavelet.eecs.berkeley.edu/~jimchou/deblock.html. Published in:
 *
 * <J. Chou, M. Crouse & K. Ramchandran "A simple algorithm for
 * removing blocking artifacts in block-transform coded images,"
 * Proc. of International Conference on Image Processing, ICIP,
 * Chicago, IL, Oct. 1998.>
 *
 * <J. Chou, M. Crouse & K. Ramchandran "A simple algorithm for
 * removing blocking artifacts in block-transform coded images," IEEE
 * Signal Processing Letters, Feb. 1998, Vol. 5, No. 2., pp.33-35.>
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @(#) $Header: /usr/mash/src/repository/mash/mash-1/codec/postdct.h,v 1.3 2002/02/03 03:13:33 lim Exp $
 */

#ifndef MASH_POSTDCT_H
#define MASH_POSTDCT_H

/*
 * Define codec types.
 */
#define CODEC_ANY  0
#define CODEC_JPEG 1
#define CODEC_H261 2

/*
 * get_threshold returns the calculated image threshold value sqrt(T)
 * for the *image based on the 8x8 quantization table *quant. h_size
 * and v_size are the dimensions of the image.
 *
 * The second version of get_threshold returns a heuristic threshold
 * value based on the type of codec and the quality value it was
 * encoded at.
 */
double get_threshold(u_int8_t *image, int h_size, int v_size, const short *quant);
double get_threshold(int codec, int quality);

/*
 * deblock applies some calculations to 8x8 image blocks in the *image
 * provided. v_thresh, h_thresh, and vis_thresh are calculation
 * thresholds. h_size and v_size are the dimensions of the image.
 *
 * The second version of deblock is used when you only want to apply
 * the deblocking to a certain portion of the image. offset is how far
 * into the image we should go before starting to deblock things, and
 * strip_length is how far across a row we should deblock. h_size
 * should be the same as above, but v_size may be a variable amount
 * which indicates how many rows to deblock.
 */
void deblock(u_int8_t *image, double v_thresh, double h_thresh, double
	     vis_thresh, int h_size, int v_size);
void deblock(u_int8_t *image, double v_thresh, double h_thresh, double
	     vis_thresh, int h_size, int v_size, int offset, int
	     strip_length);

#endif // MASH_POSTDCT_H
