/*
 * drawboard.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef DRAWBOARD_CC
#define DRAWBOARD_CC
#include <assert.h>
#include "drawboard.h"

#ifndef lint
static const char rcsid[] = "$Header: /usr/mash/src/repository/mash/mash-1/misc/drawboard.cc,v 1.13 2002/02/03 04:11:42 lim Exp $";
#endif

int DrawBoardTools::st_MaxFontN_ = 20;

////////////////////////////////////////////////////////
//
// DrawBoardTools
//
////////////////////////////////////////////////////////
//
static class DrawBoardToolsClasss : public TclClass {
public:
        DrawBoardToolsClasss() : TclClass("DrawBoardTools") {}
        TclObject* create(int argc, const char*const* argv) {
                if (argc!=8) {
                        fprintf(stderr,
                                "Err creating DrawBoardTools: wrong # of args\n");
                        return (TclObject*)NULL;
                }
                Tcl_Interp* interp=Tcl::instance().interp();
                Tk_Window tkwin = Tk_NameToWindow(interp, (char *)argv[4],
                                                  Tk_MainWindow(interp));
                int w,h;
                if (!(str2int(argv[5],w) && str2int(argv[6], h))) {
                        fprintf(stderr,
                                "Err creating DrawBoardTools: invalid width/height\n");
                        return (TclObject*)NULL;
                }
                XColor* pBg = Tk_GetColor(interp, tkwin,
                                          Tk_GetUid((char*)argv[7]));
                return (new DrawBoardTools(tkwin, w, h, pBg));
        }
} class_DrawBoardTools;

DrawBoardTools::DrawBoardTools(Tk_Window tkwin, int w, int h,
                               XColor* pBg/*=NULL*/)
        :
        bx0_(0.0),  by0_(0.0), bx1_(0.0), by1_(0.0),
        tkwin_(tkwin), offscreen_(0), width_(w), height_(h), bg_gc_(NULL)
{
        matrix_.clear();
        setBackGround(pBg);
        arTkfonts_ = new Tk_Font[st_MaxFontN_];
        arFontGCs_ = new GC[st_MaxFontN_];
        for (int i=0; i<st_MaxFontN_; i++) {
                arTkfonts_[i] = 0;
                arFontGCs_[i] = 0;
        }
}

DrawBoardTools::~DrawBoardTools() {
        for (int i=0; i<st_MaxFontN_; i++) {
                if (arTkfonts_[i]) {
                        Tk_FreeFont(arTkfonts_[i]);
                        Tk_FreeGC(Tk_Display(tkwin_), arFontGCs_[i]);
                }
        }
        delete [] arFontGCs_;
        delete [] arTkfonts_;
}

void DrawBoardTools::setBackGround(XColor *pBg)
{
        XGCValues gcv;
        if (pBg) {
                gcv.background = gcv.foreground = pBg->pixel;
        } else {
                XColor* c = Tk_GetColor(Tcl::instance().interp(), tkwin_,
                                        Tk_GetUid("lightgrey"));
                if (c == 0) abort();    // should not happen
                gcv.background = gcv.foreground = c->pixel;
                bg_gc_ = Tk_GetGC(tkwin_, GCForeground|GCBackground, &gcv);
        }
}

//////////////////////////////////////////////////////////////////////////
// Changes the bounding bbox
// Side effect: recalcMatrix will be called if bbox changed
//
void DrawBoardTools::setBBox(double x0, double y0, double x1, double y1)
{
        if (bx0_==x0 && bx1_==x1 && by0_==y0 && by1_==y1) {
                return;
        }
        bx0_=x0; bx1_=x1; by0_=y0; by1_=y1;
        // the new bbox will cover a diff. area, so we must clear
        // off the old stuff
        clearWindow();
        recalcMatrix();
}


void
DrawBoardTools::clearOffScreen()
{
        XFillRectangle(Tk_Display(tkwin_), offscreen_, bg_gc_, 0, 0,
                       width_, height_);
}

void
DrawBoardTools::clearWindow()
{
        XFillRectangle(Tk_Display(tkwin_), Tk_WindowId(tkwin_), bg_gc_, 0, 0,
                       width_, height_);
}

void DrawBoardTools::fg2gc(XColor* pColor, GC& gc)
{
        XGCValues gcv;
        gcv.foreground = pColor->pixel;
        gcv.background = pColor->pixel;
        gcv.fill_style = FillSolid;
        gcv.fill_rule = EvenOddRule;
        gcv.line_width = 2;
        gc=Tk_GetGC(tkwin_, GCLineWidth | GCForeground | GCBackground |
                    GCFillStyle | GCFillRule, &gcv);
}

GC DrawBoardTools::getFontGCByIdx(int idx)
{
        assert( idx >= 0 && idx < st_MaxFontN_ );
        if (idx >= 0 && idx < st_MaxFontN_) {
                return arFontGCs_[idx];
        } else {
                return ((GC) NULL);
        }
}

Tk_Font DrawBoardTools::getFontByIdx(int idx)
{
        assert( idx >= 0 && idx < st_MaxFontN_ );
        if (idx >= 0 && idx < st_MaxFontN_) {
                return arTkfonts_[idx];
        } else {
                return ((Tk_Font) NULL);
        }
}

int DrawBoardTools::setFontByIdx(int idx, const char* szFont,
                                 const char* szColor)
{
        if (szFont && (idx < st_MaxFontN_) && (idx >= 0)) {
                Tk_Font pf =
                        Tk_GetFont(Tcl::instance().interp(), tkwin_,
				   Tk_GetUid((char*)szFont));
                XColor* pFgC = Tk_GetColor(Tcl::instance().interp(),
                                           tkwin_, Tk_GetUid((char*)szColor));
                if (pf && pFgC) {
                        XGCValues gcv;
                        gcv.foreground = pFgC->pixel;
                        gcv.font = Tk_FontId(pf);
                        GC gc=Tk_GetGC(tkwin_, GCFont|GCForeground, &gcv);
                        if (gc) {
                                arFontGCs_[idx] = gc;
                                arTkfonts_[idx] = pf;
                                return 1;
                        }
                }
        }
        return 0;                   // unsuccessful
}

void
DrawBoardTools::line(float x0, float y0, float x1, float y1, XColor* color)
{
        if (!offscreen_) return;

        float ax, ay, bx, by;
        matrix_.map(x0, y0, ax, ay);
        matrix_.map(x1, y1, bx, by);

        GC gc;
        fg2gc(color,gc);
        XDrawLine(Tk_Display(tkwin_), offscreen_, gc,
                  rint(ax), rint(ay), rint(bx), rint(by));
}

void
DrawBoardTools::rect(float x0, float y0, float x1, float y1, XColor* color)
{
        if (!offscreen_) return;

        float x, y, xx, yy;
        matrix_.map(x0, y0, x, y);
        matrix_.map(x1, y1, xx, yy);

        float w = xx - x;
        if (w < 0) {
                x = xx;
                w = -w;
        }
        float h = yy - y;
        if (h < 0) {
                h = -h;
                y = yy;
        }
        GC gc;
        fg2gc(color, gc);
        XDrawRectangle(Tk_Display(tkwin_), offscreen_, gc, rint(x),
                       rint(y), rint(w), rint(h));
}

void
DrawBoardTools::string(float fx, float fy, const char* str, int idx)
{
        GC gc = getFontGCByIdx(idx);
	Tk_Font tkfont = getFontByIdx(idx);
        if (!gc) {
                fprintf(stderr,
                        "DrawBoardTools: drawing string using unregistered font\n");
                return;
        }
        float x, y;
        matrix_.map(fx, fy, x, y);
        Tk_DrawChars(Tk_Display(tkwin_), offscreen_, gc, tkfont, str,
		     strlen(str), rint(x), rint(y));
}

// floats is in the order x0, y0, x1, y1, x2, y2, ... x_(n/2), y(n/2)
void
DrawBoardTools::poly(const float* pFloats, int n, XColor* color)
{
        if (!offscreen_) return;
        assert( (0==(n % 2)) && "must be even number of points" );
        const int cMaxPt=20;
        // use variables from stack if possible
        XPoint arPts[cMaxPt];
        XPoint *pPts = arPts;
        int npts = n/2 ;
        if (npts+1 > cMaxPt) {
                pPts = new XPoint[npts+1];
                if (!pPts) return;      // not enough memory, do nothing
        }
        float tx, ty;
        for (int i=0; i < n; i+=2) {
                matrix_.map(pFloats[i], pFloats[i+1], tx, ty);
                pPts[i>>1].x = rint(tx);
                pPts[i>>1].y = rint(ty);
        }
        pPts[npts] = pPts[0];        // wrap around

        GC gc;
        fg2gc(color, gc);
        XFillPolygon(Tk_Display(tkwin_), offscreen_, gc, pPts, npts+1, Convex,
                     CoordModeOrigin);
        /* draw the lines so that the polygon is fully filled */
        XDrawLines(Tk_Display(tkwin_), offscreen_, gc, pPts, npts+1,
                   CoordModeOrigin);

        // delete if not from stack
        if (pPts != arPts) delete[] pPts;
}

//#define DEBUG_DB
//#define DEBUG_REDRAW
#ifdef DEBUG_REDRAW
#include <sys/time.h>

#define GET_TIME()                              \
   timeval v;                                   \
   gettimeofday(&v, 0);

#define DIFF_TIME(str)                          \
   timeval v1;                                  \
   gettimeofday(&v1, 0);                        \
   fprintf(stderr, "[%f] "str" takes %fs\n",    \
           (v1.tv_sec%1000) + 1e-6*v.tv_usec,   \
           (v1.tv_sec - v.tv_sec) +             \
           1e-6*(v1.tv_usec - v.tv_usec));

#else //  DEBUG_REDRAW
#define GET_TIME()
#define DIFF_TIME(str)
#endif

//
// only copy stuff in the bbox
//
void
DrawBoardTools::draw(int isPartial)
{
        if (offscreen_) {
                if (isPartial) {
                        int x0, y0, x1, y1;
                        matrix_.map(bx0_, by0_, x0, y0);
                        matrix_.map(bx1_, by1_, x1, y1);
                        // the y's are flipped, thus the wierd mapping
                        GET_TIME();
                        XCopyArea(Tk_Display(tkwin_), offscreen_,
                                  Tk_WindowId(tkwin_),
                                  bg_gc_, x0, y1, x1-x0, y0-y1, x0, y1);
                        DIFF_TIME("xcopy");
                } else {
                        GET_TIME();
                        XCopyArea(Tk_Display(tkwin_), offscreen_,
                                  Tk_WindowId(tkwin_),
                                  bg_gc_, 0, 0, width_, height_, 0, 0);
                        DIFF_TIME("xcopy");
                }
        }
}

////////////////////////////////////////////////////////////////////
//
// Recalculates the new matrix based on the bounding box
//
// Called when width/height or bbox changes
//
// Pre-condition: bounding box for all the items is set
//
void DrawBoardTools::recalcMatrix()
{
        matrix_.clear();

        double x = 0;
        double y = 0;
        double w = width_;
        double h = height_;
        /*
         * Set up a transform that maps bb -> canvas.  I.e,
         * bb -> unit square -> allocation, but which retains
         * the aspect ratio.  Also, add a margin.
         */
        double nw = bx1_ - bx0_;
        double nh = by1_ - by0_;

        /*
         * Grow a margin.
         */
        double bbw = 1.1 * nw;
        double bbh = 1.1 * nh;
        double tx = bx0_ - 0.5 * (bbw - nw);
        double ty = by0_ - 0.5 * (bbh - nh);

        /*
         * move base coordinate system to origin
         */
        matrix_.translate(-tx, -ty);

        /*
         * flip vertical axis because X is backwards.
         */
        matrix_.scale(1.0, -1.);
        matrix_.translate(0.0, bbh);

        double ws = w / bbw;
        double hs = h / bbh;

        if (ws <= hs) {
                matrix_.scale(ws, ws);
                matrix_.translate(x, y + 0.5 * (h - ws * bbh));
        } else {
                matrix_.scale(hs, hs);
                matrix_.translate(x + 0.5 * (w - hs * bbw), y);
        }
}

////////////////////////////////////////////////////////////////////
//
// resize the target screen size to draw onto
//
// Pre-condition: bounding box for all the items is set
//
void
DrawBoardTools::resize(int width, int height)
{
        // REVIEW: for now we assume that the bbox does not change,
	// and ignore resize requests if there is no change in size
        if (width_ == width && height_ == height)
                return;

        if (offscreen_ != 0)
                Tk_FreePixmap(Tk_Display(tkwin_), offscreen_);
        width_ = width;
        height_ = height;
        recalcMatrix();
        offscreen_ = Tk_GetPixmap(Tk_Display(tkwin_), Tk_WindowId(tkwin_),
                                   width_, height_, Tk_Depth(tkwin_));
}

/*virtual from TclObject*/
int
DrawBoardTools::command(int argc, const char*const* argv)
{
        float x0,y0,x1,y1;
        int i1;
        XColor* pFgC;
        Tcl& tcl = Tcl::instance();
        if (!strcmp(argv[1],"rect") && argc==7) {
                // <obj> rect x0 y0 x1 y1 colorName
                if ( !(str2float(argv[2], x0) && str2float(argv[3], y0)
                       && str2float(argv[4], x1) && str2float(argv[5], y1)) ) {
                        goto ERR_ARGCONV;
                }
                pFgC = Tk_GetColor(tcl.interp(), tkwin_,
                                   Tk_GetUid((char*)argv[6]));
                rect(x0, y0, x1, y1, pFgC);
        } else if (!strcmp(argv[1], "line") && argc==7) {
                // <obj> line x0 y0 x1 y1 colorName
                if ( !(str2float(argv[2], x0) && str2float(argv[3], y0)
                       && str2float(argv[4], x1) && str2float(argv[5], y1)) ) {
                        goto ERR_ARGCONV;
                }
                pFgC = Tk_GetColor(tcl.interp(), tkwin_,
                                   Tk_GetUid((char*)argv[6]));
                line(x0, y0, x1, y1, pFgC);
        } else if (!strcmp(argv[1], "poly") && (argc>4) && (0==(argc-3)%2)) {
                pFgC = Tk_GetColor(tcl.interp(), tkwin_,
                                   Tk_GetUid((char*)argv[2]));
                if (!pFgC) goto ERR_ARGCONV;

                const int startArg = 3;
                float* pFloats = new float[argc - startArg];
                for (int i=0; i + startArg < argc; i++) {
                        if (!str2float(argv[i+startArg], pFloats[i])) {
                                delete[] pFloats;
                                goto ERR_ARGCONV;
                        }
                }

                poly(pFloats, argc - startArg, pFgC);
                delete[] pFloats;
        } else if (!strcmp(argv[1], "text") && argc==6) {
                // <obj> text x0 y0 fontIdx string
                if ( !(str2float(argv[2], x0) && str2float(argv[3], y0)
                       && str2int(argv[4], i1)) ) {
                        goto ERR_ARGCONV;
                }
                string(x0, y0, argv[5], i1);
        } else if (!strcmp(argv[1], "set_font") && argc==5) {
                // <classname> set-font idx font-name color
                if (!str2int(argv[2], i1))  goto ERR_ARGCONV;
                if (!setFontByIdx(i1, argv[3], argv[4])) {
                        Tcl_AddErrorInfo(tcl.interp(), "invalid font name");
                        return TCL_ERROR;
                }
        } else {
                Tcl_AddErrorInfo(tcl.interp(),
                                 "Argument mismatch or invalid command\n");
                return TCL_ERROR;
        }
        // okay by default
        return TCL_OK;

 ERR_ARGCONV:
        Tcl_AddErrorInfo(tcl.interp(),
                         "Argument conversion failed");
        return TCL_ERROR;
}

////////////////////////////////////////////////////////
//
// DrawBoard Widget
//
////////////////////////////////////////////////////////
//
static class DrawBoardWidgetClass : public TclClass {
public:
        DrawBoardWidgetClass() : TclClass("DrawBoardWidget") {}
        TclObject* create(int argc, const char*const* argv) {
                if (argc < 5 || argc > 6)
                        fprintf(stderr,
                                "wrong number of args creating DrawBoardWgt\n");
                if (argc == 5) {
                        // create default DrawBoardTools
                        return (new DrawBoardWidget(argv[4]));
                } else {                // user supplied drawtools
                        return (new DrawBoardWidget(argv[4],argv[5]));
                }
        }
} class_DrawBoardWgt;


DrawBoardWidget::DrawBoardWidget(const char* szPath,
                                 const char* szDrawTools /*=NULL*/)
        : pOTclPainter_(NULL)
{
	TkWidget::init(szPath,"DrawBd", 0, 0);
        if (szDrawTools) {          // user supplied drawtools
                pDrawBdTools_=(DrawBoardTools*)TclObject::lookup(szDrawTools);
        } else {                    // create default DrawBoardTools
                pDrawBdTools_ = new DrawBoardTools(tk_, width_, height_);
        }
}

/*virtual*/
DrawBoardWidget::~DrawBoardWidget()
{
        delete[] pOTclPainter_;
}

/*virtual from tkwidget*/
void
DrawBoardWidget::draw(int isPartial)
{
#ifdef DEBUG_DB
        fprintf(stderr, "d");
#endif
        pDrawBdTools_->clearOffScreen();
        if (pOTclPainter_) {
                GET_TIME();
                Tcl::instance().evalf("%s draw", pOTclPainter_);
                DIFF_TIME("painter draw ");
        }
        pDrawBdTools_->draw(isPartial);
}

/*virtual from tkwidget*/
void
DrawBoardWidget::resize()
{
#ifdef DEBUG_DB
        fprintf(stderr, "*R*");
#endif

        Tcl &tcl = Tcl::instance();
        tcl.evalf("%s bbox", pOTclPainter_);
        double x0, y0, x1, y1;
        sscanf(tcl.result(), "%lg %lg %lg %lg", &x0, &y0, &x1, &y1);

#ifdef DEBUG_DB
        fprintf(stderr, "have bbox: %g %g %g %g\n", x0, y0, x1, y1 );
#endif
        pDrawBdTools_->setBBox(x0, y0, x1, y1);
        pDrawBdTools_->resize(width_, height_);
}

/*virtual from tkwidget*/
void
DrawBoardWidget::update()
{
#ifdef DEBUG_DB
        fprintf(stderr, "*U*");
#endif
        // REVIEW: should just update a region
        draw(1);
}

/*virtual from TclObject*/
int
DrawBoardWidget::command(int argc, const char*const* argv)
{
        if (argv[1][0]=='a' &&
	    !strcmp(argv[1],"attach_painter") && argc==3) {
                delete[] pOTclPainter_;
                AllocNCopy(&pOTclPainter_, argv[2]);
                return TCL_OK;
        } else if (argv[1][0]=='r' &&
		   !strcmp(argv[1],"refresh") && argc==2) {
                redraw();
                return TCL_OK;
        } else if (argv[1][0]=='r' && !strcmp(argv[1],"resize") && argc==2) {
                resize();
                return TCL_OK;
        } else if (pDrawBdTools_) {
                return (pDrawBdTools_->command(argc, argv));
        }
        return TCL_ERROR;
}

#endif /* #ifdef DRAWBOARD_CC */
