/*
 * ps-file.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1992-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/ps/ps-file.cc,v 1.13 2002/02/03 04:14:18 lim Exp $";
#endif

#ifndef WIN32
#   include <sys/param.h>
#   include <sys/types.h>
#   include <sys/mman.h>
#   include <unistd.h>
#endif
#include <sys/stat.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <tclcl.h>

/*
 * A class for breaking up a PostScript file into preamble and pages.
 * The file must be DSC conformant (see Appendix G of The PosctScript
 * Language Reference Manual, 2nd Ed.), in which case PFfile::valid()
 * is true.  Otherwise, it is false and the behavior of all other methods
 * is undefined.
 */
class PSfile : public TclObject {
    private:
	struct pageinfo {
		off_t offset;
		int length;
	};
    public:
	PSfile();
	~PSfile();
	int command(int argc, const char*const* argv);
	inline int valid() const { return npage_ >= 0; }
	inline int npages() const { return npage_; }
	inline int preamble_length() const { return info_[0].length; }
	int page_length(int pageno);
	const char* page(int pageno, int& len);
	inline const char* preamble(int& len) { return (page(0, len)); }
	void release(const char*, int len);
    private:
	int init();
	int scanhdr(pageinfo&);
	void scaninfo(int pageno);
	int fsearch(const char* cp, int len, const char* target) const;
	int bsearch(const char* cp, int len, const char* target) const;
	int match(const char* nt, const char* s, int len) const;
	off_t nextpage(off_t prev);

	Tcl_Channel channel_;
	int filesize_;
	int lastnum_;
	int npage_;		/* number of pages */
	pageinfo* info_;
	int dumpPage(int pageNo, Tcl_Channel);
};

static class DSCParserClass : public TclClass {
    public:
	DSCParserClass() : TclClass("DSC_Parser") {}
	TclObject* create(int, const char*const*) {
		return (new PSfile);
	}
} dsc_parser_class;

#ifndef MIN
#define MIN(a, b) ((a) < (b) ? (a) : (b))
#endif

PSfile::PSfile()
{
	bind("npage_", &npage_);
	npage_ = -1;
	info_ = 0;
}

PSfile::~PSfile()
{
	if (info_)
		delete info_;
}

int PSfile::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if (argc == 3) {
		if (strcmp(argv[1], "set-file") == 0) {
			int mode;
			channel_ = Tcl_GetChannel(tcl.interp(),
						  (char*)argv[2], &mode);
			if (channel_ == 0) {
				tcl.resultf("bad channel %s", argv[2]);
				return (TCL_ERROR);
			}
			/*FIXME check for wrong type of channel.*/
			init();
			return (TCL_OK);
		}
	} else if (argc == 4) {
		if (strcmp(argv[1], "dump") == 0) {
			int pageNo = strtol(argv[2], 0, 0);
			if (pageNo > npage_) {
				tcl.resultf("page %d (of %d) too large",
					    pageNo, npage_);
				return (TCL_ERROR);
			}
			int mode;
			Tcl_Channel out = Tcl_GetChannel(tcl.interp(),
							 (char*)argv[3],
							 &mode);
			if (out == 0) {
				tcl.resultf("bad channel %s", argv[3]);
				return (TCL_ERROR);
			}
			dumpPage(pageNo, out);
			return (TCL_OK);
		}
	}

	return (TclObject::command(argc, argv));
}

int PSfile::init()
{
	(void)Tcl_Seek(channel_, 0, SEEK_END);
	filesize_ = Tcl_Tell(channel_);
	(void)Tcl_Seek(channel_, 0, SEEK_SET);

	pageinfo pi;
	npage_ = scanhdr(pi);
	if (npage_ < 0)
		return (-1);
	/*
	 * Allocate enough room for each page plus the preamble.
	 * It is convenient to number the premable 0, since pages
	 * are numbered starting from 1.
	 */
	info_ = new pageinfo[npage_ + 1];
	info_[0] = pi;
	for (int i = 1; i <= npage_; ++i) {
		info_[i].length = -1;
		info_[i].offset = -1;
	}
	return (0);
}

/*
 * Return true if null-terminated string, n, matches string s,
 * which is not necessarily null-terminated.
 * min(len, strlen(n)) bytes are examined.
 */
int PSfile::match(const char* nt, const char* s, int len) const
{
	while (--len >= 0) {
		if (*nt == 0)
			return (1);
		if (*nt++ != *s++)
			return (0);
	}
	return (1);
}

int PSfile::fsearch(const char* cp, int len, const char* target) const
{
	const char* base = cp;
	while (1) {
		if (match(target, cp, len))
			return (cp - base);
		do
			if (--len <= 0)
				return (-1);
		while (*cp++ != '\n');
	}
}

int PSfile::bsearch(const char* cp, int len, const char* target) const
{
	const char* base = cp;
	cp += len;
	while (cp > base) {
		len = 0;
		while (cp > base) {
			if (*--cp == '\n') {
				++cp;
				break;
			}
			++len;
		}
		if (len != 0 && match(target, cp, len))
			return (cp - base);
		--cp;
	}
	return (-1);
}

off_t PSfile::nextpage(off_t offset)
{
	char file[4096];
	while (1) {
		int len = MIN(filesize_ - offset, int(sizeof(file)));
		Tcl_Seek(channel_, offset, SEEK_SET);
		if (Tcl_Read(channel_, file, len) != len)
			return (-1);
		int k = fsearch(file, len, "%%Page:");
		if (k > 0) {
			int pnum = atoi(file + k + 8);
			if (pnum >= lastnum_) {
				lastnum_ = pnum;
				return (offset + k);
			}
		}
		if (offset + len >= filesize_)
			return (-1);
		offset +=  len - 8;
	}
}

/*
 * Check that file is DSC conformant.  Determine number of pages and
 * length of preamble by finding corresponding comment strings.
 * -1 is returned for bogus files.
 */
int PSfile::scanhdr(pageinfo& pi)
{
	int n, off;
	off_t poff;
	char* num;
	char file[4096];

	int len = MIN(filesize_, int(sizeof(file)));
	if (Tcl_Read(channel_, file, len) != len)
		return (-1);
	if (strncmp(file, "%!PS-Adobe-", 11) != 0) {
		/*
		 * the file isn't conforming -- if it's postscript,
		 * say it has one page.
		 */
		if (strncmp(file, "%!", 2) != 0)
			return (-1);

		goto one_page;
	}
	off = fsearch(file, len, "%%Pages:");
	if (off < 0)
		goto one_page;
	for (off += 8; file[off] == ' '; ++off) {
	}
	num = file + off;
	if (match("(atend)", num, 7)) {
		if (filesize_ > int(sizeof(file))) {
			off_t noff = filesize_ - len;
			Tcl_Seek(channel_, noff, SEEK_SET);
			if (Tcl_Read(channel_, file, len) != len)
				goto one_page;
		}
		n = bsearch(file, len, "%%Pages:");
		if (n < 0)
			goto one_page;
		for (n += 8; file[n] == ' '; ++n) {
		}
		num = file + n;
	}
	n = atoi(num);
	if (n <= 0)
		goto one_page;
	/*
	 * Look for start of first page.
	 * This gives us the size of the preamble.
	 */
	lastnum_ = 0;
	poff = nextpage(0);
	if (poff <= off)
		goto one_page;
	pi.offset = 0;
	pi.length = poff;
	return (n);
one_page:
	pi.offset = 0;
	pi.length = 2; /*FIXME*/
	return (1);
}

void PSfile::scaninfo(int pageno)
{
	/* check for special 'non-conforming ps' marker */
	if (info_[0].length == 2 && pageno == 1) {
		info_[1].offset = info_[0].length;
		info_[1].length = filesize_ - info_[0].length;
		return;
	}
	int i = pageno;
	while (info_[i].length < 0)
		--i;
	off_t offset = info_[i].offset + info_[i].length;
	while (++i <= pageno) {
		info_[i].offset = offset;
		if (i == npage_) {
			info_[i].length = filesize_ - offset;
			break;
		}
		off_t noff = nextpage(offset + 8);
		if (noff <= 0) {
			/* FIXME bogus document */
			do {
				info_[i].length = 0;
			} while (++i <= npage_);
			return;
		}
		info_[i].length = noff - offset;
		offset = noff;
	}
}

int PSfile::page_length(int pageno)
{
	register int l = info_[pageno].length;
	if (l < 0)
		scaninfo(pageno);
	l = info_[pageno].length;
	if (l > 0)
		l += info_[0].length;
	return (l);
}

/*FIXME*/
static char* pbuf;
int pbufsize;

const char* PSfile::page(int pageno, int& len)
{
	off_t o;
	int l;
	pageinfo& pi = info_[pageno];
	if (pi.length < 0)
		scaninfo(pageno);
	o = pi.offset;
	l = pi.length;
	if (pageno == 1) {
		/* adjust page 1 so it includes the preamble */
		l += pi.offset;
		o = 0;
	}
	if (l > pbufsize) {
		if (pbuf == 0)
			pbuf = (char*)malloc(l);
		else
			pbuf = (char*)realloc(pbuf, l);
		pbufsize = l;
	}
	Tcl_Seek(channel_, o, SEEK_SET);
	Tcl_Read(channel_, pbuf, l);
	len = l;
	return (pbuf);
}

int PSfile::dumpPage(int pageNo, Tcl_Channel out)
{
	int cc;
	const char* bp = page(pageNo, cc);
	(void)Tcl_Write(out, (char*)bp, cc);
	return (0);/*FIXME*/
}
