# agent-transcoder.tcl --
#
#       Allows you to transcode different video sources and output at
#       different bit rates and formats
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import AnnounceListenManager/AS/Service/Transcoder/Video
import Session/RTP/RTPGW
import AddressBlock
import MediaAgent/RTPGW

##############################################################################
#
# Code for CTranscoderAgent
#
##############################################################################
Class CTranscoderAgent

CTranscoderAgent instproc init { specAS iBW specService } {
    $self instvar m_almVideo
    $self instvar m_llTranscoderState

    $self instvar m_aSession
    $self instvar m_aTrancoder

    # create a new announce listen object and start it
    set m_almVideo [new AnnounceListenManager/AS/Service/Transcoder/Video \
	    $self $specAS $iBW $specService]

    $m_almVideo start
}

##############################################################################
#
# CTranscoderAgent instproc ReceiveMessage { msg } {
#
# Input:
# msg - the state message given by client
#
# Output:
# none
#
# Description:
# a new state is has just arrived and we need to update the state of the
# transcoder to reflect this new state.  This is done by first marking
# all sessions and transcoders with a clear bit which is later set when
# the session or transcoder is updated.  If any session or transcoder is
# in the new state but not in our current state, then they must be made.
# When all the state information has been process, go through the list
# of sessions and transcoders and make sure that their marked bit is
# set, if not then delete this session or transcoder
#
##############################################################################
CTranscoderAgent instproc ReceiveMessage { msg } {
    $self instvar m_aSession
    $self instvar m_aTranscoder
    $self instvar m_llTranscoderState

    # store away the list
    set m_llTranscoderState $msg

    # mark each session and transcoder with a clear bit
    foreach {sessionName sessionObject} [array get m_aSession] {
	set aMarkBit($sessionName) 0
    }
    foreach {transcoderName transcoderObject} [array get m_aTranscoder] {
	set aMarkBit($transcoderName) 0
    }



    # go through the state list and update or create
    foreach lTranscoderState $m_llTranscoderState {
	array set aTranscoderState $lTranscoderState

	set specInputSession $aTranscoderState(InputSession)
	set specOutputSession $aTranscoderState(OutputSession)

	# the creation will chec if it's there and return immediately if there
	$self CreateSession $specInputSession
	$self CreateSession $specOutputSession

	# set this session's mark bit
	set aMarkBit($specInputSession) 1
	set aMarkBit($specOutputSession) 1

	# ok now let's deal with the transcoders and sources
	set inputSession $m_aSession($specInputSession)
	set sourceManager [$inputSession sm]
	set lSource [$sourceManager set sources_]

	foreach source $lSource {
	    set iSourceId [$source srcid]

	    # get the bw and format information
	    if { [info exist aTranscoderState($iSourceId)] } {
		set transcoderInfo aTranscoderState($iSourceId)
	    } else {
		set transcoderInfo aTranscoderState(Default)
	    }

	    set outputSession $m_aSession($specOutputSession)
	    if {![info exists m_aTranscoder($specOutputSession,$iSourceId)]} {
		$self CreateTranscoder $outputSession $source $transcoderInfo
	    } else {
		$self UpdateTranscoder $outputSession $source $transcoderInfo
	    }

	    # set the mark bit
	    set aMarkbit($specOutputSession,$iSourceId) 1
	}
    }


    # finally delete all the unused transcoders and sessions
    # FIXME need to do this part


}


##############################################################################
#
# CTranscoderAgent instproc CreateSession { specSession } {
#
# Input:
# specSession - the spec (addr/port) of the session to be created
#
# Output:
# session - the created session or existing session with the same spec
#
# Description:
# Creates a session with the spec inputted.  If the session already exists
# this function will return with no effect.  If the session does not exist
# then one will be created and inserted info the list of sessions.
#
##############################################################################
CTranscoderAgent instproc CreateSession { specSession } {
    $self instvar m_aSession

    # check it already exists
    if { [info exist m_aSession($specSession)] } {
	return $m_aSession($specSession)
    }

    # otherwise we have to create it
    set session [new Session/RTP/RTPGW/Video $self]

    # this is copied from agent-rtpgw.tcl
    # FIXME delete this in destructor!
    $session buffer-pool [new BufferPool]
    # no local loopback
    $session loopback-layer -1
    $session set rate_control_ 1
    $session set txonly_ 0
    set sm [new MediaAgent/RTPGW $self]
    $sm site-drop-time [$self get_option siteDropTime]
    $session sm $sm

    if { $specSession != "none" } {
	set ab [new AddressBlock $specSession]
	$session reset $ab
	delete $ab
    }

    # Add this session to the array
    set m_aSession($specSession) $session

    # no local source
    $session set loopbackLayer_ 0
}


##############################################################################
#
# CTranscoderAgent instproc CreateTranscoder { outputSession source \
#	transcoderInfo } {
#
# Input:
# outputSession - the session to output to
# source - the source object that serves as input
# transcoderInfo - the list of attributes for the transcoder (bw, format, etc.)
#
# Output:
# transcoder that it created
#
# Description:
# Creates a transcoder with the information inputed.  Will return immediately
# if the transcoder already exists.
#
##############################################################################
CTranscoderAgent instproc CreateTranscoder { outputSession source \
	transcoderInfo } {
    $self instvar m_aTranscoder


    # get the source id and session spec
    set specOutputSession [$outputSession GetSpec]
    set iSourceId [$source srcid]

    set transcoder [new Transcoder/Null $source $outputSession]
    $transcoder set tname_ pass-thru


    # Control path is set up in base constructor since it is
    # identical for all transcoders.
    set i [$outputSession data-handler]
    set o [$transcoder data-handler]
    $o target $i

    # Add the transcoder to the array
    set m_aTranscoder($specOutputSession,$iSourceId) $transcoder

    return $transcoder
}


##############################################################################
#
# CTranscoderAgent instproc UpdateTranscoder { outputSession source \
#	transcoderInfo } {
#
# Input:
# outputSession - the session to output to
# source - the source object that serves as input
# transcoderInfo - the list of attributes for the transcoder (bw, format, etc.)
#
# Output:
# transcoder that it created
#
# Description:
# Creates a transcoder with the information inputed.  Will return immediately
# if the transcoder already exists.
#
##############################################################################
CTranscoderAgent instproc UpdateTranscoder { outputSession source \
	transcoderInfo } {
    # FIXME need to do this
}



##############################################################################
#
# CTranscoderAgent instproc register { src } {
#
# Input:
# src - the source object that's registered
#
# Output:
# none
#
# Description:
# Call back function that is dynamically created in agent-rtp.tcl.  The
# call originates from the source object.  We'll set up the control pass
# through for this source.
#
##############################################################################
CTranscoderAgent instproc register { src } {
    $self instvar m_llTranscoderState
    $self instvar m_aTranscoder
    $self instvar m_aSession

    #
    # set up pass through control path
    #
    set replicator [new Replicator/Packet/Copy]
    set inputSession [$src set session_]
    set specInputSession [$inputSession GetSpec]

    # with this session, attach all other sessions to the replicator
    foreach lTranscoderState $m_llTranscoderState {
	array set aTranscoderState $lTranscoderState

	# we only care if the inputsession is equal to this source's session
	if { $aTranscoderState(InputSession) != $specInputSession } {
	    continue
	}

	# get the output session and add it's ctrl-handler to the target
	set specOutputSession $aTranscoderState(OutputSession)
	set outputSession $m_aSession($specOutputSession)
	$replicator add-target [$outputSession ctrl-handler]
    }


    # finally set this replicator to this source
    $src ctrl-handler $replicator
}

##############################################################################
#
# CTranscoderAgent instproc unregister { src } {
#
# Input:
# src - the source object that's being unregistered
#
# Output:
# none
#
# Description:
# Call back function that is dynamically created in agent-rtp.tcl.  The
# call originates from the source object.  We'll tear down the control pass
# through for this source.
#
##############################################################################
CTranscoderAgent instproc unregister { src } {

    # just delete the ctrl-handler created during register
    delete [$src ctrl-handler]
}

##############################################################################
#
# CTranscoderAgent instproc activate { source } {
#
# Input:
# source - the source object that's being activated
#
# Output:
# none
#
# Description:
# Call back function that is dynamically created in agent-rtp.tcl.  The
# call originates from the source object.  This is where all the action
# happens.  The data and control replicators are created.  Transcoders are
# made.
#
##############################################################################
CTranscoderAgent instproc activate { source } {

puts ""
puts "CTranscoderAgent: activate called"
puts ""

    $self instvar m_llTranscoderState
    $self instvar m_aTranscoder
    $self instvar m_aSession

    set sourceSession [$source set session_]
    set specSourceSession [$sourceSession GetSpec]
    set iSourceId [$source srcid]

    #
    # Set up data path and redirect control path
    #
    set rep [new Replicator/Packet/Copy]
    $source data-handler $rep

    delete [$source ctrl-handler]
    set rep [new Replicator/Packet/Copy]
    $source ctrl-handler $rep

    # with this session, attach all other sessions to the replicator
    foreach lTranscoderState $m_llTranscoderState {
	array set aTranscoderState $lTranscoderState

	# we only care if the inputsession is equal to this source's session
	if { $aTranscoderState(InputSession) != $specSourceSession } {
	    continue
	}

	# just in case check if there's a transcoder already
	if { [info exists m_aTranscoder($specSourceSession,$iSourceId)] } {
	    return
	}

	# get the output session
	set specOutputSession $aTranscoderState(OutputSession)
	set outputSession $m_aSession($specOutputSession)

	# get the bw and format information
	if { [info exist aTranscoderState($iSourceId)] } {
	    set transcoderInfo aTranscoderState($iSourceId)
	} else {
	    set transcoderInfo aTranscoderState(Default)
	}

	# create the transcoder
	$self CreateTranscoder $outputSession $source $transcoderInfo
    }
}



##############################################################################
#
# CTranscoderAgent instproc set_maxchannel { n } {
#
# Input:
# n - the number of channels
#
# Output:
# none
#
# Description:
# Call back for the network manager object.  It's inform you of the max
# channels, currently we don't make use of this information
#
##############################################################################
CTranscoderAgent instproc set_maxchannel { n } {
	# If and when we have a layered transcoder - we'll have to
	# do something here.
}

##############################################################################
#
# Session/RTP/RTPGW instproc GetSpec { } {
#
# Input:
# none
#
# Output:
# spec (addr/port) of this session
#
# Description:
# returns the spec of this session
#
##############################################################################
Session/RTP/RTPGW instproc GetSpec { } {
    $self instvar agent_ netmgr_

    set net [$netmgr_ set net_(0)]
    set inetAddr "[$net set addr_]/[$net set port_]"

    return $inetAddr
}


# src is the source that we need as input
# sess is the session which we want this source to be transcoded into

Transcoder instproc init { src sess } {
    puts "transcoder: init called src-$src sess-$sess"

    $self next

    $self source $src

    set datarep [$src data-handler]
    set ctrlrep [$src ctrl-handler]

    # creates a new data handler and add it to the set of replicate targets
    set h [new Module/DataHandler]
    $self data-handler $h
    $datarep add-target $h

    set h [new Module/ControlHandler]
    $h target [$sess ctrl-handler]
    $self ctrl-handler $h
    $ctrlrep add-target $h
}

Transcoder/Null instproc rtcp-thumbnail n {}

Transcoder set bps_ 64000
Transcoder set opkts_ 0
Transcoder set obytes_ 0
Transcoder set ofrms_ 0
Transcoder set forward_ 0
Transcoder set txonly_ 0
