# agent-video.tcl --
#
#       VideoAgent object definition
#
# Copyright (c) 1996-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/net/agent-video.tcl,v 1.53 2002/02/11 02:28:54 lim Exp $


import RTPAgent RTP/Video \
	AnnounceListenManager/AS/Client/MeGa/Video

Module/VideoDecoder/PVH set maxChannel_ 0

#
# The VideoAgent class provides a coarse-grained interface
# that abstracts away all the details of networked video decoding.
# This agent is responsible for creating the underlying RTP
# session for the video channel, for opening and initializing
# any video decoding hardware, and for creating and deleting software
# decoder objects to process the compressed bit streams from the network.
# <p>
# Since VideoAgent is derived from the RTPAgent base class,
# the standard RTP observer API is supported.  Any ``interesting''
# events (e.g., the arrival of a new RTP flow) within the underlying
# RTP sessions are relayed to all the observers attached to this agent.
# (Note that the AudioAgent class catches and handles some of the
# events on its own.)
# <p>
# Typically, a mash script will create a separate user interface
# object and attach it to the VideoAgent as an Observer.
#
Class VideoAgent -superclass { RTPAgent RTP/Video } -configuration {
	megaVideoFormat h261
	# multicast is default
	megaRecvVideoPort 0
	megaVideoCtrl 224.4.5.24/50000/31
	megaVideoCtrlBW 20000
	videoServiceLocation urn:vgw
}

#
VideoAgent public init  { app spec {callback {}}} {
	$self set myhandler_ $app

	if { $spec != "" } {
		set ab [new AddressBlock $spec]
		set fmt [$ab fmt]
		if { $fmt != {} } { $self add_option videoFormat $fmt }
	} else {
		set ab ""
	}
	$self next $ab $callback
	if { $ab != "" } {
		delete $ab
	}

	$self site-drop-time [$self get_option siteDropTime]
	$self instvar decoders_
	#FIXME
	set decoders_ ""

	set localbw [$app get_option videoSessionBW]
	if { $localbw == "" } {
		set localbw [$app get_option maxVideoSessionBW]
	}
	$self sessionbw $localbw

	$self start_mega
}


#
# you should ensure nobody is using actively this agent (a VideoPipeline, 
# for example) before destroying it
#
VideoAgent public destroy { } {
	$self instvar al_
	if [info exists al_] { delete $al_ }
	$self next
}


VideoAgent public start_mega { } {
	$self instvar al_ myhandler_

	if [info exists al_] { delete $al_ }
	if { [$myhandler_ get_option megaVideoSession] != "" } {
		set sname [$myhandler_ get_option megaVideoSession]
		set sspec [$self get_option videoSessionSpec]
		set rportspec [$self get_option megaRecvVideoPort]
		set ofmt [$self get_option megaVideoFormat]

		set localbw [$myhandler_ get_option videoSessionBW]
		if { $localbw == "" } {
			set localbw [$myhandler_ get_option maxVideoSessionBW]
		}
		set bw [expr 0.02*$localbw]
		set megaspec [$self get_option megaVideoCtrl]
		set loc [$self get_option videoServiceLocation]

		set ab [new AddressBlock $sspec]
		set sspec [$ab addr]/[$ab sport]:[$ab rport]/[$ab ttl]
		delete $ab
		set al_ [new AnnounceListenManager/AS/Client/MeGa/Video \
				$self $megaspec $bw [Application name] video \
				$sname $sspec $rportspec $ofmt $loc]
	        $al_ start
	}
}


VideoAgent public video_handler {} { return [$self set myhandler_] }


#
# Used when we need to point the mega audio gateway we're controlling
# to listen to a new source.  This is kludgy, and needs a redesign.
#
VideoAgent public reset_mega {} {
	$self instvar al_
	if ![info exists al_] {
		$self start_mega
	} else {
		$al_ reset_spec [$self get_option videoSessionSpec]
	}
}


#
# Create a Session object that appropriate for this
# type of agent.  Called by the RTPAgent class when
# initializing the network state.
#
VideoAgent public create_session {} {
	return [new Session/RTP/Video]
}

#
# Handle an activate event on source <i>src</i>.
# Called from C++ (through Source/RTP) when we start
# actively receiving data pkts from the given source.
# Create the decoder and dispatch the method to the
# observers as normal.
# Extends method in RTPAgent.
#
VideoAgent public activate src {
	$self instvar decoders_
	set d [$self create_decoder $src]
	lappend decoders_ $d
	$src data-handler $d
	$self next $src
}

#
# Handle a deactivate event on source <i>src</i>.
# Called from C++ (through Source/RTP) when a source
# has left the session (either via an RTCP BYE message
# or via an expiration timer).  This method can also
# get called back if the Source object is explicitly
# deleted from the local program.
# Extends method in RTPAgent.
#
VideoAgent public deactivate src {
	$self instvar decoders_
	# Source/RTP handler is deprecated to differentiate control handler
	#	(ctrl-handler) and data handler (data-handler).
	set d [$src data-handler]
	set k [lsearch -exact $decoders_ $d]
	set decoders_ [lreplace $decoders_ $k $k]
	$self next $src
	delete $d
}

#
#called when the format changes and we need to delete the old decoder
#and create a new one
#
VideoAgent public reactivate src {
	$self instvar decoders_
	set d [$src data-handler]
	if {$d!=""} {
		delete $d
	}
	set k [lsearch -exact $decoders_ $d]
	set decoders_ [lreplace $decoders_ $k $k]
	set decoder [$self create_decoder $src]
	lappend decoders_ $decoder
	$src data-handler $decoder
	return $decoder
}

#
# Change the target bandwidth assumed to be in use by the aggregate
# underlying network session to <i>b</i> bits per second.
# Upon changing the bandwidth, each attached observer
# is notified.
#
VideoAgent public sessionbw b {
	$self set sessionbw_ $b
	$self notify_observers sessionbw $b
}

#
# Change the target bandwidth used by the local source
# with the underlying network session to <i>b</i> bits per second.
# Upon changing the bandwidth, each attached observer
# is notified.
#
VideoAgent public local_bandwidth b {
	[$self set session_] data-bandwidth $b
}

#
# Dispatch a decoder_changed event to all observers of
# the VideoAgent object associated with this Decoder.
# This method is from C++ the video stream state changes in a way that
# would affect the choice of renderer.  For example, when a jpeg stream
# changes from type-0 to type-1 we might have to revert from
# hardware to software decoding, or we might have to reallocate
# a 422 renderer as a 411 renderer.  This never needs to happen
# for most stream types (i.e., because the decimation factor is fixed).
#
Module/VideoDecoder public parameters_changed {} {
	$self instvar agent_ src_
	$agent_ notify_observers decoder_changed $src_
}

#
# Called by the underlying network object when the number
# of media layers expected across all sources changes.
# For example, an underlying network protocol might adjust
# the number of multicast channels received to carry out
# congestion control.  When the level of subscription changes,
# the codecs must be informed so that they do not wait
# unnecessarily for packets that will never arrive (because
# the corresponding layer is not present).
# <p>
# Extends method in RTPAgent.
#
VideoAgent public set_maxchannel n {
	$self instvar decoders_ channels_
	if [info exists decoders_] {
		foreach d $decoders_ {
			$d set maxChannel_ $n
		}
	}
	set channels_ $n
}

#
# Create and return a video decoder object that is capabale of decoding
# the RTP video stream from source <i>src</i>.  If the stream type
# is not supported, create and return a ``null'' decoder, which consumes
# packets and collects statistics but cannot decode the video.
# FIXME this method should be private, but currently the observer
# is currently responsible for creating new decoders on format changes.
#
VideoAgent public create_decoder src {
	set c [$self classmap [$src format_name]]
	set decoder [new Module/VideoDecoder/$c]
	if { $decoder == "" } {
		# don't support this format
		set decoder [new Module/VideoDecoder/Null]
	}
	#FIXME
	$decoder set agent_ $self
	$decoder set src_ $src

	$self instvar channels_
	$decoder set maxChannel_ $channels_

	return $decoder
}
