# al.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#  @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/net/al.tcl,v 1.24 2002/02/03 04:28:05 lim Exp $


import Network Timer


AnnounceListenManager public init { spec {mtu 1500} } {
	$self next $mtu

	$self instvar data_ snet_ rnet_
	set data_ ""
	set snet_ ""
	set rnet_ ""

	# check if this is just a port number
	if [regexp {^[0-9]*$} $spec] {
		# Unicast UDP server
		set rnet_ [new Network]
		$rnet_ open $spec
	} else {
		set ab [new AddressBlock/Simple $spec]
		set addr  [$ab addr]
		set sport [$ab sport]
		set rport [$ab rport]
		set ttl   [$ab ttl]
		delete $ab

		set snet_ [new Network]
		if [in_multicast $addr] {
			$snet_ open $addr $sport $rport $ttl
			set rnet_ $snet_
		} else {
			# unicast.  Open a unicast send connection and a UDP
			# server connection if needed.
			if { $rport != 0 } {
				set rnet_ [new Network]
				$rnet_ open $rport
			}
			$snet_ open $addr $sport 0 1
		}
	}


	if { $snet_ != "" } {
		$snet_ loopback 1
		$self send_network $snet_
	}
	if { $rnet_ != "" } {
		$self recv_network $rnet_
	}
}


AnnounceListenManager public destroy {} {
	$self instvar snet_ rnet_ timers_
	if { $rnet_==$snet_ } {
		delete $snet_
	} else {
		if { $snet_ != "" } {
			delete $snet_
		}
		if { $rnet_ != "" } {
			delete $rnet_
		}
	}
	if [info exists timers_] {
		foreach t [array names timers_] {
			delete $timers_($t)
		}
	}
	$self next
}


# Use this method to set the timer object(s) associated with the
# AnnounceListenManager object.
# <ul>
# <li> <tt>$alm timer [new Timer/Periodic]</tt>. This will set the timer
# for the default announcement; every time the timer expires, the
# <tt>send_announcement</tt> method is invoked without any arguments
# <li> You can also create more than one announcement, each with its own
# timer: <tt>$alm timer $data [new Timer/Periodic]</tt>. Every time the
# timer expires, the <tt>send_announcement</tt> method is invoked with
# <tt>$data</tt> as its argument. The default <tt>send_announcement</tt>
# will send <tt>$data</tt> as part of the announcement packet.
# </ul>
# The manager assumes control of the timer object;
# so the application must not try to delete the object itself.
# If the timer argument to this method is an empty string, it deletes the
# existing timer
AnnounceListenManager public timer {args} {
	$self instvar timers_
	if {[llength $args]==1} {
		set d __default_timer__
		set t [lindex $args 0]
		if {$t!={}} { $t proc timeout { } "$self send_announcement" }
	} else {
		set d [lindex $args 0]
		set t [lindex $args 1]
		if {$t!={}} { $t proc timeout { } \
				[list $self send_announcement $d] }
	}

	# remember whether the timer is currently active or not
	if [info exists timers_($d)] {
		set sched [$timers_($d) is_sched]
		delete $timers_($d)
	} else {
		set sched 0
	}

	if {$t!={}} {
		set timers_($d) $t
		if $sched {
			# reschedule the timer
			$t start
		}
	} else {
		catch {unset timers_($d)}
	}
	return $t
}


# Use this method to retrieve the timer object(s) associated with this
# object. If no argument is specified, the timer for the default announcement
# is returned; If an argument <tt>$data</tt> is specified, then the timer
# associated with the announcement for <tt>$data</tt> is returned. This
# method returns an empty string if there is no associated timer object
AnnounceListenManager public get_timer {args} {
	$self instvar timers_
	if {[llength $args]==0} {
		set d __default_timer__
	} else {
		set d [lindex $args 0]
	}

	if [info exists timers_($d)] { return $timers_($d) } else { return "" }
}


# Call this method to initiate the announcement operation.
# If this method is invoked without any arguments, it initiates
# the default announcement. When an argument is specified, a
# new announcement for that argument is initiated.
# If the application hasn't created a Timer object for this
# announcement (see AnnounceListenManager::timer),
# this method will create a default randomized Timer/Periodic object
# set to a timeout of 5 seconds
AnnounceListenManager public start {args} {
	if { [llength $args]==0 } {
		set t [$self get_timer]
	} else {
		set d [lindex $args 0]
		set t [$self get_timer $d]
	}

	if { $t=={} } {
		set t [new Timer/Periodic]
		$t randomize 1
		if [info exists d] { $self timer $d $t } else { $self timer $t}
	}

	if [info exists d] {$self send_announcement $d} \
			else {$self send_announcement}
	$t start
}


# Call this method to stop the AnnounceListenManager operation.
# If this method is invoked without any arguments, it stops
# all announcements. When an argument is specified,
# the announcement for that argument is stopped.
AnnounceListenManager public stop {args} {
	$self instvar timers_
	if {[llength $args]==0} {
		foreach d [array names timers_] {
			$timers_($d) cancel
		}
	} else {
		set d [lindex $args 0]
		$timers_($d) cancel
	}
}


# Subclass this method to handle the received announcement
AnnounceListenManager public recv_announcement { addr port data len } {
	puts "ALM::recv_announcement $addr/$port \[$len\]: $data"
}


# Subclass this method if you wish to construct and send the announcement
# in a different manner
AnnounceListenManager public send_announcement {args} {
	$self instvar data_
	#puts "sending announcement for $self"
	if {[llength $args]==0} {
		if { $data_!={} } { $self announce $data_ }
	} else {
		$self announce [lindex $args 0]
	}
}


# Call this method to set the default announcement that this object
# must send out
AnnounceListenManager public set_announcement { data } {
	$self set data_ $data
}


# Call this method to retrieve the current default announcement that is
# being sent out
AnnounceListenManager public get_announcement { } {
	return [$self set data_]
}

# Sets the announcement ttl
AnnounceListenManager public ttl {num} {
	$self instvar snet_
	$snet_ ttl $num
}

