# pane-manager.tcl --
#
#       Sets up vertical or horizontal panes.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#  @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/ui_tools/pane-manager.tcl,v 1.6 2002/02/03 04:30:23 lim Exp $


#
# Sets up vertical or horizontal panes.  Two frames, <i>f1</i> and
# <i>f2</i>, are placed inside another frame, which represents a grip
# that is used to adjust the boundary between the two panes.  The
# arguments, which are optional, are "-orient [vertical|horizontal]",
# "-percent [0-1]", and "-in [parent window]".
# This was copied almost directly from example 22-5 of "Practical Programming in Tcl and Tk" -Welch
#
Class PaneManager

#
# The code is set to work with either horizontal or vertical layouts.
# The pane_(D) variable is either <i>x</i>, for a horizontal layout, or
# <i>y</i>, for a vertical layout.  Remember the previous position in
# pane_(lastD) and use that to update the percentage split between the
# two contained panes.
#
PaneManager instproc pane_drag {master D} {
	$self instvar pane_

	if [info exists pane_(lastD)] {
		if ![info exists pane_(lastPercent)] {
			set pane_(lastPercent) $pane_(-percent)
		}

		set delta [expr double($pane_(lastD) - $D) \
				/ $pane_(size)]
		set pane_(lastPercent) [expr $pane_(lastPercent) - $delta]
		set pane_(-percent) $pane_(lastPercent)
		if {$pane_(-percent) < 0.0} {
			set pane_(-percent) 0.0
		} elseif {$pane_(-percent) > 1.0} {
			set pane_(-percent) 1.0
		}
		if {0} {
			$self pane_geometry $master
		} else {
			if {$pane_(D) == "X"} {
				place $pane_(divider) -relx $pane_(-percent)
				place $pane_(grip) -relx $pane_(-percent)
			} else {
				place $pane_(divider) -rely $pane_(-percent)
				place $pane_(grip) -rely $pane_(-percent)
			}
		}
	}
	set pane_(lastD) $D
}

#
PaneManager instproc pane_stop {master} {
	$self instvar pane_

	if {1} {
		$self pane_geometry $master
	}

	catch {unset pane_(lastD)}
	catch {unset pane_(lastPercent)}
}

#
# Adjust the positions of the frames.  Called when the main window is
# resized, so it updates pane_(size).  Also called as the user drags the
# grip.  For a vertical layout, the grip is moved by setting its
# relative Y position.  The size of the two contained frames is set with
# a relative height.  This is combined wit the fixed height of -1 to get
# some space between the two frames.
#
PaneManager instproc pane_geometry {master} {
	$self instvar pane_

	raise $pane_(divider)
	raise $pane_(grip)
	if {$pane_(D) == "X"} {
		place $pane_(1) -relwidth $pane_(-percent)
		place $pane_(2) -relwidth [expr 1.0 - $pane_(-percent)]
		place $pane_(divider) -relx $pane_(-percent)
		place $pane_(grip) -relx $pane_(-percent)
		set pane_(size) [winfo width $master]
	} else {
		place $pane_(1) -relheight $pane_(-percent)
		place $pane_(2) -relheight [expr 1.0 - $pane_(-percent)]
		place $pane_(divider) -rely $pane_(-percent)
		place $pane_(grip) -rely $pane_(-percent)
		set pane_(size) [winfo height $master]
	}
}


PaneManager public percent { {p {}} } {
	$self instvar pane_
	if { $p == {} } { return $pane_(-percent) }
	if { $p < 0.0 } { set p 0.0 }
	if { $p > 1.0 } { set p 1.0 }
	set pane_(-percent) $p
	$send pane_geometry
}


#
# Try this to see how PaneManager works.
#
proc pane_test {{p .p} {orient vert}} {
	catch {destroy $p}
	frame $p -width 200 -height 200
	button $p.1 -bg blue -text foo
	button $p.2 -bg green -text bar
	pack $p -expand true -fill both
	pack propagate $p off
	new PaneManager $p.1 $p.2 -in $p -orient $orient -percent 0.3
}

#
# Given two widgets and an optional set of parameters, sets them up as
# vertical/horizontal panes offering the user a grip with which to
# adjust the boundary between them.
#
PaneManager instproc init {f1 f2 args} {
	$self instvar pane_

	# Map optional arguments into array values
	set t(-orient) vertical
	set t(-percent) 0.5
	set t(-in) [winfo parent $f1]
	array set t $args

	# Keep state in an array associated with the master frame
	set master $t(-in)
	array set pane_ [array get t]

	# Create the grip and set placement attributes that
	# will not change.  A thin divider line is achieved by
	# making the two frames one pixel smaller in the
	# adjustable dimension and making the main frame black.

	set pane_(1) $f1
	set pane_(2) $f2
	if { [winfo toplevel $master] == $master } {
		set parent $master
	} else {
		set parent [winfo parent $master]
	}
	set pane_(divider) [frame $parent.divider$self -width 2 -height 2 \
			-bd 2 -relief groove]
	set pane_(grip) [frame $parent.grip$self -background gray50 \
			-width 10 -height 10 -bd 1 -relief raised \
			-cursor crosshair]
	if {[string match vert* $pane_(-orient)]} {
		set pane_(D) Y; # Adjust boundary in Y direction.
		place $pane_(1) -in $master -x 0 -rely 0.0 -anchor nw \
				-relwidth 1.0 -height -1
		place $pane_(2) -in $master -x 0 -rely 1.0 -anchor sw \
				-relwidth 1.0 -height -1
		place $pane_(divider) -in $master -anchor w -relx 0.0 \
				-relwidth 1.0
		place $pane_(grip) -in $master -anchor c -relx 0.8
	} else {
		set pane_(D) X; # Adjust boundary in X direction
		place $pane_(1) -in $master -relx 0.0 -y 0 -anchor nw \
				-relheight 1.0 -width -1
		place $pane_(2) -in $master -relx 1.0 -y 0 -anchor ne \
				-relheight 1.0 -width -1
		place $pane_(divider) -in $master -anchor n -rely 0.0 \
				-relheight 1.0
		place $pane_(grip) -in $master -anchor c -rely 0.8
	}
	#$master configure -background black

	# Set up bindings for resize, <Configure>, and
	# for dragging the grip.

	# The <Configure> event occurs when the containing frame is resized by the user.
	bind $master <Configure> [list $self pane_geometry $master]
	# When the user presses the mouse button over the grip and drags it,
	# there is a <ButtonPress-1> event, one or more <B1-Motion> events, and finally
	# a <ButtonRelease-1> event.
	bind $pane_(grip) <ButtonPress-1> \
			[list $self pane_drag $master %$pane_(D)]
	bind $pane_(grip) <B1-Motion> \
			[list $self pane_drag $master %$pane_(D)]
	bind $pane_(grip) <ButtonRelease-1> \
			[list $self pane_stop $master]

	# Do the initial layout
	set pane_(master) $master
	$self pane_geometry $master
}


PaneManager public destroy {} {
	$self instvar pane_
	destroy $pane_(divider)
	destroy $pane_(grip)

	if [winfo exists $pane_(1)] { place forget $pane_(1) }
	if [winfo exists $pane_(2)] { place forget $pane_(2) }
	if [winfo exists $pane_(master)] { bind $pane_(master) <Configure> "" }

	$self next
}

