#include "CORBAmodule.h"

GHashTable *poa_objects;

/***************************************************************************
	Wraps POAManager
	Methods handled:
		activate
*/


PyTypeObject POAManager_PyObject_Type = {
	PyObject_HEAD_INIT(NULL)
	0,       /*ob_size*/
	"PortableServer.POAManager",         /*tp_name*/
	sizeof(POAManager_PyObject),   /*tp_basicsize*/
	0,       /*tp_itemsize*/
	/* methods */
	(destructor)POAManager_PyObject__dealloc, /*tp_dealloc*/
	0,       /*tp_print*/
	(getattrfunc)POAManager_PyObject__getattr, /*tp_getattr*/
	(setattrfunc)POAManager_PyObject__setattr, /*tp_setattr*/
	0,       /*tp_compare*/
	0,       /*tp_repr*/
	0,       /*tp_as_number*/
	0,       /*tp_as_sequence*/
	0,       /*tp_as_mapping*/
	0,       /*tp_hash*/
};

POAManager_PyObject *POAManager_Object_to_PyObject(PortableServer_POAManager o)
{
	POAManager_PyObject *self;
	self = PyObject_NEW(POAManager_PyObject, &POAManager_PyObject_Type);
	if (!self)
		return NULL;
	CORBA_exception_init(&self->ev);
	self->obj = o;
	return self;
}

void POAManager_PyObject__dealloc(POAManager_PyObject *self)
{
	PyMem_DEL(self);
}

PyObject *POAManager_PyObject__activate(POAManager_PyObject *self, 
	                                     PyObject *args)
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	PortableServer_POAManager_activate(self->obj, &self->ev);
	if (!check_corba_ex(&self->ev))
		return NULL;
	Py_INCREF(Py_None);
	return Py_None;
}

PyMethodDef POAManager_PyObject_methods[] = {
	{ "activate", (PyCFunction)POAManager_PyObject__activate, 1 },
	{ NULL, NULL }
};

PyObject *POAManager_PyObject__getattr(POAManager_PyObject *self, char *name)
{
	return Py_FindMethod(POAManager_PyObject_methods, (PyObject *)self, name);
}

int POAManager_PyObject__setattr(POAManager_PyObject *self, char *name,
	                              PyObject *v)
{
	return -1;
}

/***************************************************************************
	Wraps POA
	Methods handled:
*/

PyTypeObject POA_PyObject_Type = {
	PyObject_HEAD_INIT(NULL)
	0,       /*ob_size*/
	"PortableServer.POA",         /*tp_name*/
	sizeof(POA_PyObject),   /*tp_basicsize*/
	0,       /*tp_itemsize*/
	/* methods */
	(destructor)POA_PyObject__dealloc, /*tp_dealloc*/
	0,       /*tp_print*/
	(getattrfunc)POA_PyObject__getattr, /*tp_getattr*/
	(setattrfunc)POA_PyObject__setattr, /*tp_setattr*/
	0,       /*tp_compare*/
	0,       /*tp_repr*/
	0,       /*tp_as_number*/
	0,       /*tp_as_sequence*/
	0,       /*tp_as_mapping*/
	0,       /*tp_hash*/
};

POA_PyObject *POA_Object_to_PyObject(PortableServer_POA object)
{
	POA_PyObject *self;
	CORBA_ORB_PyObject *orb;

	self = g_hash_table_lookup(poa_objects, object);

	if (self) {
		Py_INCREF(self);
		return self;
	}

	orb = g_hash_table_lookup(orb_objects, object->orb);
	if (!orb)
		return raise_system_exception(OPExc_BAD_PARAM, 0, CORBA_COMPLETED_NO,
		                              "POA object belongs to unknown ORB");
	
	self = PyObject_NEW(POA_PyObject, &POA_PyObject_Type);
	if (!self)
		return NULL;
	CORBA_exception_init(&self->ev);
	self->obj = object;
	self->orb = orb;
	Py_INCREF(orb);
	return self;
}

void POA_PyObject__dealloc(POA_PyObject *self)
{
	d_message(1, "POA_PyObject__dealloc: entered");
	Py_DECREF(self->orb);
	CORBA_Object_release((CORBA_Object)self->obj, &self->ev);
	CORBA_exception_free(&self->ev);
	PyMem_DEL(self);
}

PyObject *POA_PyObject__activate_object(POA_PyObject *self, PyObject *args)
{
	PortableServer_ObjectId *id;
	PyObject *instance; // servant
	Servant_PyClass_Glue *class_glue;
	Servant_PyInstance_Glue *instance_glue;

	if (!PyArg_ParseTuple(args, "O", &instance))
		return NULL;

	d_message(1, "activating %x", (guint)instance);

	// see if the instance_glue for this instance exists already
	// (i.e. if delegation was used)
	instance_glue = g_hash_table_lookup(servant_instance_glue, instance);
	if (!instance_glue) {
		class_glue = (Servant_PyClass_Glue *)
		                  get_class_glue_from_instance(instance);
		if (!class_glue)
			return raise_system_exception(OPExc_BAD_PARAM, 0, CORBA_COMPLETED_NO,
                                       "object not a servant");
		instance_glue = ORBit_Python_init_pserver(class_glue, instance);
	}
	else if (instance_glue->activated == CORBA_TRUE) 
		return raise_system_exception(OPExc_BAD_INV_ORDER, 0, CORBA_COMPLETED_NO, 
		                              "servant already activated");

	id = PortableServer_POA_activate_object(self->obj, instance_glue, &self->ev);
	if (!check_corba_ex(&self->ev))
		return NULL;

	Py_INCREF(instance_glue->impl);
	instance_glue->activated = CORBA_TRUE;
	instance_glue->poa = self;
	Py_INCREF(self);
	instance_glue->oid = id;
	Py_INCREF(instance_glue->servant);
	return Py_BuildValue("s#", id->_buffer, id->_length);

}

PyObject *POA_PyObject__activate_object_with_id(POA_PyObject *self,
	                                             PyObject *args)
{
	PyObject *instance;
	Servant_PyClass_Glue *class_glue;
	Servant_PyInstance_Glue *instance_glue;
	PortableServer_ObjectId *id;

	id = (PortableServer_ObjectId *)CORBA_sequence_octet__alloc();
	if (!PyArg_ParseTuple(args, "s#O", &id->_buffer, &id->_length, &instance)) {
		ORBit_free(id, CORBA_FALSE);
		return NULL;
	}
	id->_length++; // oid length includes NULL

	// see if the instance_glue for this instance exists already
	// (i.e. if delegation was used)
	instance_glue = g_hash_table_lookup(servant_instance_glue, instance);
	if (!instance_glue) {
		class_glue = (Servant_PyClass_Glue *)
		                  get_class_glue_from_instance(instance);
		if (!class_glue)
			return raise_system_exception(OPExc_BAD_PARAM, 0, CORBA_COMPLETED_NO,
                                       "object not a servant");
		instance_glue = ORBit_Python_init_pserver(class_glue, instance);
	}
	else if (instance_glue->activated == CORBA_TRUE) 
		return raise_system_exception(OPExc_BAD_INV_ORDER, 0, CORBA_COMPLETED_NO, 
		                              "servant already activated");

	PortableServer_POA_activate_object_with_id(self->obj, id, instance_glue,
	                                           &self->ev);

	ORBit_free(id, CORBA_FALSE);

	if (!check_corba_ex(&self->ev))
		return NULL;

	Py_INCREF(instance_glue->impl);
	instance_glue->activated = CORBA_TRUE;
	instance_glue->poa = self;
	Py_INCREF(self);
	instance_glue->oid = id;
	Py_INCREF(instance_glue->servant);

	Py_INCREF(Py_None);
	return Py_None;
}


PyObject *POA_PyObject__deactivate_object(POA_PyObject *self, PyObject *args)
{
	PyObject *instance;
	Servant_PyInstance_Glue *instance_glue;

	if (!PyArg_ParseTuple(args, "O", &instance))
		return NULL;

	instance_glue = g_hash_table_lookup(servant_instance_glue, instance);
	// FIXME: should raise ObjectNotActive
	if (!instance_glue)
		return raise_system_exception(OPExc_BAD_INV_ORDER, 0, CORBA_COMPLETED_NO,
                                    "object not an activated servant");
	if (!instance_glue->activated)
		return raise_system_exception(OPExc_BAD_INV_ORDER, 0, CORBA_COMPLETED_NO,
                                    "servant not activated");

	d_message(1, "Deactivating %x", (guint)instance);
	PortableServer_POA_deactivate_object(self->obj, instance_glue->oid, 
		                                     &self->ev);
	if (!check_corba_ex(&self->ev))
		return NULL;
	instance_glue->activated = CORBA_FALSE;
	Py_DECREF(instance_glue->impl);
	Py_DECREF(instance_glue->servant);
	Py_DECREF(self);
	ORBIT_OBJECT_KEY(instance_glue->_private)->object = NULL;
	CORBA_free(instance_glue->oid);

	d_message(1, "POA_PyObject__deactivate_object: deactivated %x", 
	          (guint)instance);
	Py_INCREF(Py_None);
	return Py_None;
}

PyObject *POA_PyObject__servant_to_reference(POA_PyObject *self, PyObject *args)
{
	CORBA_Object ref = CORBA_OBJECT_NIL;
	PyObject *servant;
	Servant_PyInstance_Glue *servant_glue;

	if (!PyArg_ParseTuple(args, "O", &servant))
		return NULL;

	servant_glue = g_hash_table_lookup(servant_instance_glue, servant);

	// ORBit will implicitly activate for us, but we need to prevent it
	// from doing that because our activate_object does some additional
	// stuff.
	// (XXX: should handle MULTIPLE_ID, but ORBit doesn't appear to support
	// it.  We don't support setting POA policies yet anyway.)
	if (self->obj->implicit_activation == PortableServer_IMPLICIT_ACTIVATION &&
		 (!servant_glue || !servant_glue->activated)) {
			PyObject *res = POA_PyObject__activate_object(self, args);
			if (!res)
				return NULL;

			Py_DECREF(res);
			servant_glue = g_hash_table_lookup(servant_instance_glue, servant);
	}
		
	if (!servant_glue)
		return raise_system_exception(OPExc_BAD_PARAM, 0, CORBA_COMPLETED_NO,
                                    "object not an activated servant");

	if (!servant_glue->activated)
		return raise_system_exception(OPExc_BAD_PARAM, 0, CORBA_COMPLETED_NO,
                                    "servant must be activated");
	
	ref = PortableServer_POA_servant_to_reference(self->obj, servant_glue,
	                                              &self->ev);
	if (!check_corba_ex(&self->ev))
		return NULL;
	if (ref == CORBA_OBJECT_NIL) {
		Py_INCREF(Py_None);
		return Py_None;
	}
	// finally, take care of returning reference -- creating it if it
	// doesn't exist 
	return (PyObject *)CORBA_Object_to_PyObject(ref);
}

PyObject *POA_PyObject__reference_to_servant(POA_PyObject *self, PyObject *args)
{
	CORBA_Object obj;
	Servant_PyInstance_Glue *instance_glue;
	PyObject *ref;
	CORBA_PyInstance_Glue *objglue;

	if (!PyArg_ParseTuple(args, "O", &ref))
		return NULL;

	objglue = g_hash_table_lookup(object_instance_glue, ref);
	if (!objglue)
		return raise_system_exception(OPExc_BAD_PARAM, 0, CORBA_COMPLETED_NO,
                                    "parameter 1 not a CORBA object");

	obj = (CORBA_Object)PortableServer_POA_reference_to_servant(self->obj, 
			              objglue->obj, &self->ev);
	if (!check_corba_ex(&self->ev))
		return NULL;
	instance_glue = (Servant_PyInstance_Glue *)obj;
	if (!instance_glue) {
		Py_INCREF(Py_None);
		return Py_None;
	}
	Py_INCREF(instance_glue->servant);
	return instance_glue->servant;
}

PyObject *POA_PyObject__get_the_POAManager(POA_PyObject *self, PyObject *args)
{
	PortableServer_POAManager m;
	m = PortableServer_POA__get_the_POAManager(self->obj, &self->ev);
	if (!check_corba_ex(&self->ev))
		return NULL;
	if (m == CORBA_OBJECT_NIL) {
		Py_INCREF(Py_None);
		return Py_None;
	}
	return (PyObject *)POAManager_Object_to_PyObject(m);
}


PyMethodDef POA_PyObject_methods[] = {
	{ "activate_object", (PyCFunction)POA_PyObject__activate_object, 1 },
	{ "activate_object_with_id", 
	           (PyCFunction)POA_PyObject__activate_object_with_id, 1 },
	{ "deactivate_object", (PyCFunction)POA_PyObject__deactivate_object, 1 },
	{ "servant_to_reference", 
	           (PyCFunction)POA_PyObject__servant_to_reference, 1 },
	{ "reference_to_servant", 
	           (PyCFunction)POA_PyObject__reference_to_servant, 1 },
	{ "_get_the_POAManager", 
	           (PyCFunction)POA_PyObject__get_the_POAManager, 1 },
	{ NULL, NULL }
};

PyObject *POA_PyObject__getattr(POA_PyObject *self, char *name)
{
	if (!strcmp(name, "the_POAManager"))
		return POA_PyObject__get_the_POAManager(self, NULL);

	return Py_FindMethod(POA_PyObject_methods, (PyObject *)self, name);
}

int POA_PyObject__setattr(POA_PyObject *self, char *name, PyObject *v)
{
	return -1;
}

void ORBit_Python_init_portable_server()
{
	servant_base = PyClass_New(NULL, PyDict_New(),
	                           PyString_FromString("Servant"));
	PyObject_SetAttrString(servant_base, "__module__",
	                       PyString_FromString("PortableServer"));
}

/* Local Variables: */
/* c-file-style: "python" */
/* End: */
