/*
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <Python.h>
#define NO_IMPORT_PYGOBJECT
#include <pygobject.h>
#include <gtk/gtk.h>
#include "streamtuner.h"
#include "pst-handler-field.h"
#include "pst-helpers.h"

/*** attributes **************************************************************/

static PyObject *pst_handler_field_get_id (PSTHandlerField *self, void *closure);
static PyObject *pst_handler_field_get_label (PSTHandlerField *self, void *closure);
static PyObject *pst_handler_field_get_type (PSTHandlerField *self, void *closure);
static PyObject *pst_handler_field_get_flags (PSTHandlerField *self, void *closure);
static PyObject *pst_handler_field_get_description (PSTHandlerField *self, void *closure);

static PyGetSetDef getsetters[] = {
  { "id", (getter) pst_handler_field_get_id, NULL },
  { "label", (getter) pst_handler_field_get_label, NULL },
  { "type", (getter) pst_handler_field_get_type, NULL },
  { "flags", (getter) pst_handler_field_get_flags, NULL },
  { "description", (getter) pst_handler_field_get_description, NULL },
  
  { NULL }
};

/*** type object *************************************************************/

static PyObject *pst_handler_field_new (PyTypeObject *type, PyObject *args, PyObject *keywords);

PyTypeObject PSTHandlerField_Type = {
  PyObject_HEAD_INIT(NULL)
  0,				/* ob_size */
  "ST.HandlerField",		/* tp_name */
  sizeof(PSTHandlerField),	/* tp_basicsize */
  0,				/* tp_itemsize */
  NULL,				/* tp_dealloc */
  NULL,				/* tp_print */
  NULL,				/* tp_getattr */
  NULL,				/* tp_setattr */
  NULL,				/* tp_compare */
  NULL,				/* tp_repr */
  NULL,				/* tp_as_number */
  NULL,				/* tp_as_sequence */
  NULL,				/* tp_as_mapping */
  NULL,				/* tp_hash */
  NULL,				/* tp_call */
  NULL,				/* tp_str */
  NULL,				/* tp_getattro */
  NULL,				/* tp_setattro */
  NULL,				/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
  NULL,				/* tp_doc */
  NULL,				/* tp_traverse */
  NULL,				/* tp_clear */
  NULL,				/* tp_richcompare */
  0,				/* tp_weaklistoffset */
  NULL,				/* tp_iter */
  NULL,				/* tp_iternext */
  NULL,				/* tp_methods */
  NULL,				/* tp_members */
  getsetters,			/* tp_getset */
  NULL,				/* tp_base */
  NULL,				/* tp_dict */
  NULL,				/* tp_descr_get */
  NULL,				/* tp_descr_set */
  0,				/* tp_dictoffset */
  NULL,				/* tp_init */
  NULL,				/* tp_alloc */
  pst_handler_field_new		/* tp_new */
};

/*** function declarations ***************************************************/

static int pst_handler_field_convert_type (PyObject *object, GType *type);

/*** type methods ************************************************************/

static int
pst_handler_field_convert_type (PyObject *object, GType *type)
{
  GType _type;

  _type = pyg_type_from_object(object);
  if (_type)
    {
      *type = _type;
      return 1;
    }
  else
    return 0;		/* exception raised by pyg_type_from_object() */
}

static PyObject *
pst_handler_field_new (PyTypeObject *type, PyObject *args, PyObject *keywords)
{
  PSTHandlerField *self;
  static char *keyword_list[] = { "id", "label", "type", "flags", "description", NULL };
  int id;
  const char *label;
  GType gtype;
  unsigned int flags = 0;
  const char *description = NULL;
  
  if (! PyArg_ParseTupleAndKeywords(args, keywords, "isO&|is", keyword_list,
				    &id,
				    &label,
				    pst_handler_field_convert_type, &gtype,
				    &flags,
				    &description))
    return NULL;

  self = (PSTHandlerField *) type->tp_alloc(type, 0);
  if (! self)
    return NULL;

  self->field = st_handler_field_new(id, label, gtype, flags);
  if (description)
    st_handler_field_set_description(self->field, description);

  return (PyObject *) self;
}

static PyObject *
pst_handler_field_get_id (PSTHandlerField *self, void *closure)
{
  return PyInt_FromLong(self->field->id);
}

static PyObject *
pst_handler_field_get_label (PSTHandlerField *self, void *closure)
{
  return pst_string_from_string_or_null(st_handler_field_get_label(self->field));
}

static PyObject *
pst_handler_field_get_type (PSTHandlerField *self, void *closure)
{
  return PyLong_FromUnsignedLong(st_handler_field_get_type(self->field));
}

static PyObject *
pst_handler_field_get_flags (PSTHandlerField *self, void *closure)
{
  return PyLong_FromUnsignedLong(st_handler_field_get_flags(self->field));
}

static PyObject *
pst_handler_field_get_description (PSTHandlerField *self, void *closure)
{
  return pst_string_from_string_or_null(st_handler_field_get_description(self->field));
}

/*** C API *******************************************************************/

gboolean
pst_handler_field_register (PyObject *module)
{
  PyTypeObject *ptr = &PSTHandlerField_Type;

  g_return_val_if_fail(module != NULL, FALSE);

  if (PyType_Ready(&PSTHandlerField_Type) < 0)
    return FALSE;

  Py_INCREF(&PSTHandlerField_Type);
  PyModule_AddObject(module, "HandlerField", (PyObject *) ptr);

  PyModule_AddIntConstant(module, "HANDLER_FIELD_VISIBLE", ST_HANDLER_FIELD_VISIBLE);
  PyModule_AddIntConstant(module, "HANDLER_FIELD_EDITABLE", ST_HANDLER_FIELD_EDITABLE);
  PyModule_AddIntConstant(module, "HANDLER_FIELD_VOLATILE", ST_HANDLER_FIELD_VOLATILE);
  PyModule_AddIntConstant(module, "HANDLER_FIELD_NO_DEDICATED_COLUMN", ST_HANDLER_FIELD_NO_DEDICATED_COLUMN);
  PyModule_AddIntConstant(module, "HANDLER_FIELD_START_HIDDEN", ST_HANDLER_FIELD_START_HIDDEN);

  return TRUE;
}

PyObject *
pst_handler_field_from_field (STHandlerField *field)
{
  PSTHandlerField *self;

  g_return_val_if_fail(field != NULL, NULL);

  self = PyObject_New(PSTHandlerField, &PSTHandlerField_Type);
  if (! self)
    return NULL;

  self->field = field;

  return (PyObject *) self;
}
