/* ==================================================== ======== ======= *
*
*  uuicon.cpp
*  Ubit Project [Elc][2003]
*  Author: Eric Lecolinet
*
*  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
*
*  (C) 1999-2003 Eric Lecolinet @ ENST Paris
*  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
*
* ***********************************************************************
* COPYRIGHT NOTICE :
* THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE
* IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
* YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU
* GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION;
* EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
* SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
* ***********************************************************************
*
* ==================================================== [Elc:03] ======= *
* ==================================================== ======== ======= */

//pragma ident	"@(#)uuicon.cpp	ubit:03.06.00"
#include <iostream>
#include <ubit/ubit.hpp>
#include <ubit/udir.hpp>
#include <ubit/ext/uicon.hpp>
#include <sys/stat.h>
using namespace std;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

const UFlag UTitlebox::BottomTitle, UTitlebox::LeftTitle;

UTitlebox::~UTitlebox() {}

UTitlebox::UTitlebox(const UStr& _name, const UArgs& args) {
  pname  = ustr(_name);
  pscale = uscale();
  pspane = uscrollpane(true, false);
  pcontent = uvbox(args);
  pcontent->addAttr(*pscale);  // tjrs en 1er

  UEdit& ed = uedit();
  ed.setEditable(false); // pour pouvoir scroller
  name_grp = &uhbox(ed + *pname + " ");

  alt_btn = uflatbutton
    (UMode::canSelect
     + "<>"
     + UOn::select  /ucall(this,true, &UTitlebox::showAltViewImpl)
     + UOn::unselect/ucall(this,false,&UTitlebox::showAltViewImpl)
     );
  
  zoom_grp = &ugroup
    (
     uhspacing(1) + UFont::bold   +   UFont::small
     + uflatbutton(" z " + ucall(this, -1, &UIconbox::incrScale))
     + uflatbutton(" Z " + ucall(this, +1, &UIconbox::incrScale))
     //+ uflatbutton(" = " + ucall(this,  0, &UIconbox::setScale))
     + *alt_btn
     );

  iconify_btn = &uflatbutton
    (
     UMode::canSelect
     + UOn::select   / ustr(" + ")
     + UOn::unselect / ustr(" - ")
     + UOn::select   / ushow(&pcontent, false)
     + UOn::unselect / ushow(&pcontent, true)
     );

  close_btn = &uflatbutton(" X " + ucall(this, 0, &UBox::close));
  
  ptitle = ulabel
    (
     *zoom_grp
     + " " + *name_grp
     + uhflex() + ulabel(" ") + uright() + iconify_btn + close_btn
     );
  ptitle->addAttr(UBgcolor::lightgrey);
  ptitle->addAttr(UColor::black);

  pspane->add(*pcontent);

  addlist(UBorder::etchedIn + UBorder::none //+ uhflex() + uvflex()
          + BottomTitle / ugroup(uhflex() + uvflex() + *pspane + utop() + *ptitle)
          + !BottomTitle / ugroup(uhflex() + utop() + *ptitle + uvflex() + *pspane));
}

/*
void UTitlebox::moveTitleToTop() {
  title_on_top = true;
  remove(0, false);
  remove(0, true);
  remove(0, false);
  remove(0, true);
  // addlist(utop() + *ptitle + uvflex() + *pspane);
  insert(0, utop(), false);
  insert(1, *ptitle), false;
  insert(2, uvflex(), false);
  insert(3, *pspane, false);
  //update();
}

void UTitlebox::moveTitleToBottom() {
  title_on_top = false;
  remove(0, false);
  remove(0, true);
  remove(0, false);
  remove(0, true);
  //addlist(uvflex() + *pspane + utop() + *ptitle);
  insert(0, uvflex(), false);
  insert(1, *pspane, false);
  insert(2, utop()), false;
  insert(3, *ptitle, false);
  //update();  
}
*/
/* ==================================================== ======== ======= */

void UTitlebox::showTitleName(bool state)  {
  name_grp->show(state);
}

void UTitlebox::showZoomButtons(bool state)  {
  zoom_grp->show(state);
}

void UTitlebox::showIconifyButton(bool state)  {
  iconify_btn->show(state);
}

void UTitlebox::showCloseButton(bool state)  {
  close_btn->show(state);
}

void UTitlebox::showAltView(bool state) {
  alt_btn->select(state);
}

void UTitlebox::showAltViewImpl(bool state) {
}

/* ==================================================== ======== ======= */

bool UTitlebox::isIconified() const {
  return iconify_btn->isSelected();
}

void UTitlebox::iconify(bool state) {
  if (state == isIconified()) return;
  iconify_btn->select(state);
}

/* ==================================================== ======== ======= */

const UStr& UTitlebox::getName() const {
  return *pname;
}

void UTitlebox::setName(const UStr& _name) {
  *pname = _name;
}

void UTitlebox::setScale(int val) {
  pscale->set(val);
  //rearrangeIcons();
}

void UTitlebox::incrScale(int val) {
  pscale->incr(val);
  //rearrangeIcons();
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UStyle *UIconbox::style = null;
const UStyle& UIconbox::getStyle(UContext*) const {
  if (!style) {
    style = new UStyle(UVbox::makeStyle());
    style->local.padding.set(3, 3);
  }
  return *style;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

const UFlag UIconbox::DontShowDirs, UIconbox::DontShowDocs;
//const UFlag IconBgcolor;

UIconbox::UIconbox(const UStr& _name, const UArgs& args) : UTitlebox(_name)
  //persp(scale),lastbar(null)
{
  ppath= new UStr();
  picon_width = new UWidth(70);
  picon_content_width = new UWidth(60);
  picon_content_height= new UHeight(60);

  icons = new UListbox();
  // NB: icons ne doit contenir que des UIcon sinon getIcon ne marchera pas
  //icons->addAttr(UBgcolor::white);
  icons->addAttr(UFlowView::style);
  icons->addAttr(uhspacing(6));
  icons->addAttr(uvspacing(6));
  icons->addAttr(show_dirs);
  icons->addAttr(show_docs);
  icons->addAttr(show_icon_names);
  icons->addAttr(show_icon_contents);
  icons->addlist(args);
  //icons->addAttr(upropdef(IconBgcolor, UBgcolor::red));
  pcontent->addlist(*icons);
}

UIconbox::~UIconbox() {
  //cerr<<"Inconbox " << this << "deleted" << endl; 
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

const UStr& UIconbox::getPath() const {
  return *ppath;
}

void UIconbox::setPath(const UStr& _name) {
  *ppath = _name;
}

void UIconbox::showDirIcons(bool state)  {
  if (state) show_dirs.set(null);
  else show_dirs.set(DontShowDirs);
  //update();
}

void UIconbox::showDocIcons(bool state)  {
  if (state) show_docs.set(null);
  else show_docs.set(DontShowDocs);
  //update();
}

void UIconbox::showIconNames(bool state)  {
  if (state) show_icon_names.set(null);
  else show_icon_names.set(UIcon::DontShowName);
  //update();
}

void UIconbox::showIconContents(bool state)  {
  if (state) show_icon_contents.set(null);
  else show_icon_contents.set(UIcon::DontShowContent);
  //update();
}

void UIconbox::showAltViewImpl(bool state) {
  showIconContents(!state);
}

/* ==================================================== ======== ======= */

void UIconbox::selectBehavior(UEvent& e) {
  e.setID(UEvent::select);
  e.setAux(e.getSource());  // new
  fire(e, UOn::select);
}

void UIconbox::okBehavior(UEvent& e) {
  e.setAux(e.getSource());  // new
  fire(e, UOn::action);
}

void UIconbox::cancelBehavior(UEvent& e) {
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UIcon* UIconbox::getSelectedIcon() const {
  UGroup* i = choice().getItem();
  return i ? dynamic_cast<UIcon*>(i) : null;
}

void UIconbox::selectIcon(class UIcon* i, bool update) {
  choice().setItem(i, update);
}

void UIconbox::selectIcon(const UStr& name, bool update) {
  UBrick* child = null;
  UListPos lpos;
  while ((child = icons->getChild(lpos)))  {
    UIcon * icon = dynamic_cast<UIcon*>(child);
    if (icon && icon->getName() == name) {
      choice().setItem(icon, update);
      return;
    }
  }
}

void UIconbox::selectPreviousIcon() {
  int index = choice().getIndex();
  if (index > 0) choice().setIndex(index-1);
}

UIcon* UIconbox::getPreviousIcon() const {
  int index = choice().getIndex();
  if (index > 0) return getIcon(index-1);
  else return getIcon(0);
}

void UIconbox::selectNextIcon() {
  int index = choice().getIndex();
  choice().setIndex(index+1);
  int new_index = choice().getIndex();
  // end of list
  if (new_index < 0) choice().setIndex(index);
}

UIcon* UIconbox::getNextIcon() const {
  int index = choice().getIndex();
  int last_index = choice().getLastIndex();
  if (index < last_index) return getIcon(index+1);
  else return getIcon(last_index);
}

UIcon* UIconbox::getIcon(int n) const {
  UBrick* i = icons->getChild(n);
  return i ? dynamic_cast<UIcon*>(i) : null;
}

UIcon* UIconbox::getIcon(UListPos& pos) const {
  UBrick* i = icons->getChild(pos);
  if (!i) return null;  
  return (i ? dynamic_cast<UIcon*>(i) : null);
}

/* ==================================================== ======== ======= */

void UIconbox::addIcon(UIcon& icon, bool upd) {

  icon.addlist   // add icon selector and callbacks
    (     
     //+ UOn::enter  / ucall(this, &UIconbox::selectBehavior)
      UOn::action / ucall(this, &UIconbox::okBehavior)
     );

  icons->add(icon, false);
  if (upd) update();
}

/* ==================================================== ======== ======= */

void UIconbox::removeAllIcons(bool delete_singles) {
  //persp.bar_container.removeAll(delete_singles);
  //iconlist.removeAll(delete_singles);
  icons->removeAll(delete_singles);
}

/* ==================================================== ======== ======= */

int UIconbox::readDir(const UStr& _name, const UStr& _pathname) {
  UStr prefix, filter;
  bool want_dotfiles = false;

  *pname = _name;
  *ppath = _pathname;

  UDir dir(_pathname, prefix, filter, want_dotfiles);
  // if (!found)  return  UFilestat::NotOpened;

  UIcon* icon = new UIcon(*picon_content_width, 
			  *picon_content_height, 
			  *picon_width + ucall(this, &UIconbox::okBehavior)
			  );
  icon->setContent("..", S_IFDIR);
  icons->add(!DontShowDirs / icon, false);

  for (u_int k = 0; k < dir.getFileCount(); k++) {
    UIcon* icon = 
      new UIcon(*picon_content_width,
		*picon_content_height, 
		*picon_width + ucall(this, &UIconbox::okBehavior)
		);
    
    icon->setContent(*dir.getFileEntry(k)->getName(),
		     dir.getFileEntry(k)->getMode());

    if (dir.getFileEntry(k)->isDir())
      icons->add(!DontShowDirs / icon, false);
    else 
      icons->add(!DontShowDocs / icon, false);
  }

  return UFilestat::Opened;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UStyle *UIcon::style = null;
const UStyle& UIcon::getStyle(UContext*) const {
  if (!style) {
    style = new UStyle(UVbox::makeStyle());
    style->orient = UOrient::VERTICAL;
    style->local.halign   = UHalign::CENTER;
    style->local.valign   = UValign::FLEX;
    style->local.padding.set(4, 4);
    /*
    style->bgcolors = UStyle::makeColors(&UBgcolor::none, &UBgcolor::white);
    style->setBgcolors(UOn::ENTERED, UBgcolor::darkgrey);
    style->fgcolors = UStyle::makeColors(&UColor::inherit, &UColor::black);
    style->setFgcolors(UOn::ENTERED, UColor::blue);
    */
  }
  return *style;
}

const UFlag UIcon::DontShowContent, UIcon::DontShowName;

UIcon::UIcon(UWidth& _content_width, UHeight& _content_height, 
	     const UArgs& args)
{
  pname = new UStr();
  content_width = _content_width;
  content_height = _content_height;

  contentbox.addAttr(_content_width);
  contentbox.addAttr(_content_height);
  contentbox.addAttr(uhcenter());
  contentbox.addAttr(UMode::ignoreEvents);

  namebox.addAttr(UBorder::none);
  namebox.addAttr(UFont::small);
  namebox.addAttr(UMode::ignoreEvents);
  namebox.addlist(*pname);

  addlist
    (
     args
     + UOrient::vertical
     + uhcenter()
     //+ upropval(IconBgcolor)
     + uvflex()  + !DontShowContent / contentbox
     + ubottom() + !DontShowName / namebox
     );
}

UIcon::~UIcon() {}

/* ==================================================== ======== ======= */

void UIcon::setContent(const UStr& _name, UGroup& _content) {
  *pname = _name;
  mode = 0;
  contentbox.removeAll(true, false);
  contentbox.add(_content);
  
  contentbox.update();    // %% pour layout
  update(UUpdate::paint); // %% pour affichage
}

void UIcon::setContent(const UStr& _name, UElem&  _content) {
  *pname = _name;
  mode = 0;
  contentbox.removeAll(true, false);
  contentbox.add(_content);

  contentbox.update();    // %% pour layout
  update(UUpdate::paint); // %% pour affichage
}

void UIcon::setContent(const UStr& _name, long _mode) {
  *pname = _name;
  mode   = _mode;
  contentbox.removeAll(true, false);
  contentbox.add(findIconImage());

  contentbox.update();    // %% pour layout
  update(UUpdate::paint); // %% pour affichage
}

void UIcon::clearContent() {
  *pname = "";
  mode = 0;
  contentbox.removeAll(true, true);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

int UIcon::readContent(const UStr& _name, const UStr& pathname) { 
  UStr fext = pathname.getFileSuffix();
  int stat = false;

  // on a selectionne une image
  // nb: compareTo(xxx, true) -> does not check case
  if (fext.compareTo("gif",true)==0
      || fext.compareTo("jpg",true)==0
      || fext.compareTo("xpm",true)==0) {
    
    uptr<UIma> full_ima = new UIma();

    if ((stat = full_ima->readFile(pathname)) <= 0) {
      delete full_ima;
      return stat;
    }

    // ATT: ne duplique pas toujours l'image!
    UIma* scaled_ima = 
      rescaleIma(full_ima, content_width->get(), content_height->get());
    
    setContent(_name, *scaled_ima);
    return UFilestat::Opened;
  }

  return UFilestat::UnknownType;
}

/* ==================================================== ======== ======= */

const UStr& UIcon::getName() const {return *pname;}

void UIcon::setName(const UStr& s) {*pname = s;}
void UIcon::setMode(long _mode)    {mode = _mode;}

bool UIcon::isDir()  const {return S_ISDIR(mode);}

bool UIcon::isFile() const {return S_ISREG(mode);}

bool UIcon::isLink() const {return S_ISLNK(mode);}

bool UIcon::isExec() const {
  return (mode & (S_IXUSR | S_IXGRP | S_IXOTH)) != 0;
}

/* ==================================================== ======== ======= */
// returns ima or a copy

UIma* UIcon::rescaleIma(UIma* ima,
			u_dim ima_max_width, u_dim ima_max_height){
  if (!ima || ima->getWidth()==0 || ima->getHeight()==0)
    return null;

  float xscale = (float)ima_max_width  / ima->getWidth();
  float yscale = (float)ima_max_height / ima->getHeight();
  float xyscale = min(xscale, yscale);   // preserve ratio

  // rescale if ima is too large
  if (xyscale < 1.) {
    UIma* ima2 = new UIma();
    ima2->copyRescaled(*ima, xyscale, xyscale);
    return ima2;
  }
  else return ima;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

static const char *textdoc_xpm[] = {
"    28    38      200            2",
".. c #4166a3",
".# c None s None",
".a c #45679f",
".b c #4668a0",
".c c #4669a1",
".d c #4769a0",
".e c #486aa0",
".f c #486aa2",
".g c #5f7aa5",
".h c #666666",
".i c #6a81a6",
".j c #6a86b2",
".k c #6b6b6b",
".l c #6c83a8",
".m c #6e6e6e",
".n c #6f6f6f",
".o c #7087ac",
".p c #7289af",
".q c #748bb0",
".r c #758cb1",
".s c #768db2",
".t c #778eb3",
".u c #788fb4",
".v c #797979",
".w c #798ba9",
".x c #7990b5",
".y c #7b93b7",
".z c #7f7f7f",
".A c #808080",
".B c #818ea3",
".C c #838383",
".D c #878787",
".E c #8a8a8a",
".F c #8c94a1",
".G c #8c9ab0",
".H c #8d8d8d",
".I c #8e99aa",
".J c #8f9aab",
".K c #909090",
".L c #909aac",
".M c #909bac",
".N c #919191",
".O c #919baa",
".P c #919daf",
".Q c #979797",
".R c #979fac",
".S c #989898",
".T c #999999",
".U c #9a9a9a",
".V c #9f9f9f",
".W c #a0a8b5",
".X c #a1a1a1",
".Y c #a2a3a2",
".Z c #a3a3a3",
".0 c #a4a4a4",
".1 c #a4acba",
".2 c #a5a5a5",
".3 c #a5adba",
".4 c #a6aebc",
".5 c #a7a7a7",
".6 c #a7a9ad",
".7 c #a8a8a8",
".8 c #a8b0be",
".9 c #a9a9a9",
"#. c #ababab",
"## c #adadad",
"#a c #aeaeae",
"#b c #afafaf",
"#c c #b0b0b0",
"#d c #b1b1b1",
"#e c #b2b2b2",
"#f c #b3b3b3",
"#g c #b3bbc7",
"#h c #b4b4b4",
"#i c #b4b5b6",
"#j c #b4bbc7",
"#k c #b5b5b5",
"#l c #b5bcc8",
"#m c #b6b6b6",
"#n c #b6bdc9",
"#o c #b7b6b7",
"#p c #b7b7b7",
"#q c #b7beca",
"#r c #b8b8b7",
"#s c #b8b8b8",
"#t c #b8bfcb",
"#u c #b9b9b9",
"#v c #b9c0cc",
"#w c #bababa",
"#x c #bac1cd",
"#y c #bbbbbb",
"#z c #bbc2ce",
"#A c #bcbbbb",
"#B c #bcbcbc",
"#C c #bcc3cf",
"#D c #bdbdbd",
"#E c #bdc4d0",
"#F c #bebebe",
"#G c #bec5d1",
"#H c #bfbfbf",
"#I c #bfbfc1",
"#J c #bfc6d2",
"#K c #c0c0c0",
"#L c #c1c1c1",
"#M c #c1c7d3",
"#N c #c1c8d4",
"#O c #c2c1c2",
"#P c #c2c2c2",
"#Q c #c2c9d5",
"#R c #c3c3c3",
"#S c #c3cad6",
"#T c #c4c3c3",
"#U c #c4c4c4",
"#V c #c4cbd7",
"#W c #c5c5c5",
"#X c #c6c6c6",
"#Y c #c7c6c6",
"#Z c #c7c7c7",
"#0 c #c8c8c8",
"#1 c #c9c9c9",
"#2 c #c9c9cb",
"#3 c #cacaca",
"#4 c #cbcbcb",
"#5 c #cbd1dc",
"#6 c #cccccc",
"#7 c #cdcdcd",
"#8 c #cecece",
"#9 c #cfcfcf",
"a. c #d0d0d0",
"a# c #d0d0d2",
"aa c #d1d1d1",
"ab c #d2d2d2",
"ac c #d3d3d3",
"ad c #d4d4d4",
"ae c #d4d4d6",
"af c #d5d5d5",
"ag c #d6d6d6",
"ah c #d7d7d7",
"ai c #d8d8d8",
"aj c #d8d8da",
"ak c #d9d9d9",
"al c #d9dada",
"am c #dadada",
"an c #dadbdc",
"ao c #dbdbdb",
"ap c #dcdcdc",
"aq c #dddddd",
"ar c #dddddf",
"as c #dedede",
"at c #dededf",
"au c #dfdfdf",
"av c #e0dfe2",
"aw c #e0e0e0",
"ax c #e1e1e1",
"ay c #e1e1e3",
"az c #e2e1e1",
"aA c #e2e2e2",
"aB c #e3e3e3",
"aC c #e3e4e3",
"aD c #e4e4e4",
"aE c #e4e4e6",
"aF c #e4e5e5",
"aG c #e5e5e4",
"aH c #e5e5e5",
"aI c #e5e5e7",
"aJ c #e6e6e6",
"aK c #e7e7e7",
"aL c #e7e7e9",
"aM c #e8e8e8",
"aN c #e9e9e9",
"aO c #e9e9eb",
"aP c #eaeaea",
"aQ c #eaeaec",
"aR c #ebebeb",
"aS c #ebebed",
"aT c #ececec",
"aU c #ececee",
"aV c #ededed",
"aW c #ededef",
"aX c #eeeeee",
"aY c #efefef",
"aZ c #f0f0f0",
"a0 c #f1f1f1",
"a1 c #f2f1f1",
"a2 c #f2f2f2",
"a3 c #f3f3f3",
"a4 c #f4f4f4",
"a5 c #f4f5f4",
"a6 c #f5f4f5",
"a7 c #f5f5f5",
"a8 c #f6f6f6",
"a9 c #f7f7f7",
"b. c #f8f8f8",
"b# c #f9f9f9",
"ba c #fafafa",
"bb c #fbfbfb",
"bc c #fcfcfc",
"bd c #fdfdfd",
"be c #fefefe",
"bf c #ffffff",
/* pixels */
".j.y.x.x.u.u.u.t.s.s.r.q.q.p.o.l.i.g...#.#.#.#.#.#.#.#.#",
"#xa1aPaPaNaMaKaHaDaAaxauaqamag#7#R#k.O.e.#.#.#.#.#.#.#.#",
"#gaNaxaAaAaxauasaqaoamaiagadab#8#U#m#b.R.d.#.#.#.#.#.#.#",
"#jaPaBaBaBaAaxawauaqaoamaiahafac#6#H.5#a.3.b.#.#.#.#.#.#",
"#jaPaBaJaDaDaBaBaxawauasapamahagab#Z##.Uaq.4.a.#.#.#.#.#",
"#laRax#1apaoauakaoaoaiacaaadadaaa.#Z#f.KasaT.1.a.#.#.#.#",
"#naVau###a#d#W#c#c#F#k#k#..Z.0###y.T.5.AaobbaN.1.b.#.#.#",
"#naVaF#9agawauasaqaxasaqagamacadab#L#y.QaBbfbfaY.8.b.#.#",
"#qaXaK#H#a#s#y.7#e#L#e#e#m#w#H#d#m.X.V.E.X#b#d#w#1.W.f.#",
"#taYaMaBaAauaiauauaoaqamapapagadaaa.#R.X.A.n.h.k.v.S.F.c",
"#va0aJ#d#K#F#P#F###.#s###1#h.9.Y.0.7#m.H.D.z.C.m.N.Q.2.B",
"#xa2aMagaAaxawafataAaDaHaKaHaAawawaqag#1#H#u#y#s#s#f#c.6",
"#za4ax#d#u#U.0.5#m#dasaRaPaPaPaNaKaHaxapahafacaca.#3#R#i",
"#Ca7aTaKaHaMaAaHaJagaKaRaRaRaRaPaNaMaHaBaxauaqaoakaga.#I",
"#Ea8aYa0a0aYaZaZaYaZaXaXaVaTaVaVaTaRaNaMaJaDaAawaqaoak#2",
"#Ga9aXasaAaMaJawaDaKaHapaMaMaBasaoaHaDaxaqauaoaoauasaqa#",
"#JbaaM#s#4#s#u#O#K#d#W#k#Haa#U#L#f#m#W#W#K#e#p#eaoazawae",
"#Mbba3aMaRaPaAaPaRauaVaHaGaTaNaRaPaMaJaBaPaAamauaBaAaCaj",
"#Nbda0#u#e#c#p#4#X#p#K#X#r#y#u#w#f#d#o#eaa#y#y###1#RaNan",
"#Qbea8aDagaKaJaKaTaBaHaXaMawaBaDaJaBaPaMaNaMaxaiaBaHaNar",
"#Sbfa4#e#h#B#K#k#0#w#B#7#u#b#p#paa#a#c#s#m#B#h#Z#PaXaRav",
"#Sbfa8a0aTaVa2aJaRaTaXaMaTaXaPaXaPaBaDaJaRaTaNaJaKaNaVay",
"#Sbfa4#H#s#yap#K#u#d#F#F#Wax#K#F#c#ua.#R#u#y#s#m#daoa0aE",
"#Sbfa9aVaXaKaXaXaTaTaTaXaRaVaRaPaRaRaTaKaRaMaAawaDaNa0aI",
"#Qbfa7#Pah#1#D#B#yas#B#K#y#L#F#1#8#Bab#m#D#K#F#u#s#Ha7aL",
"#Qbfa7asaNaPaJaHaNaPaAaMaPaPaHaMaMaPa3aXaTaXaXaYaRaVa4aO",
"#QbfaY#F#Aa.#w#mam#m#Ha.#e#8#K#T#F#9a7a9a8a8a8a7a7a4a7aQ",
"#Qbfbab#b.b.b.a9b.a8b.a2aTa9a9a8a6a8a8a8a8a8a8a7a7a4a8aS",
"#Qbfbbbdbcbbbbbbbabbbabbbcbab#b#b#b.b.b.b.b.a9a8a8a7a8aU",
"#Qbfb#aNa7a9a3a3a9a8a4a0aTa3a4a4a3a7a3aZaRaNaXa5a8a8a9aU",
"#Qbfb##H#s#Uaf#faa#3#K#3#0#D#D#P#Laa#y#m#e#F#uacbaa9b.aU",
"#QbfbaaPa7aXa0aPa0a7aTaRa8aYa4a4a4a0aTaZaHaZa2a0a2a4b#aU",
"#Sbfa2#H#W#b#Y#P#F#K#3#P#R#fak#P#F#paa#1ak#w#B#F#uaTbbaW",
"#Vbfbaa3aYaVaVa4a0aXb.a7a0aVa7a0aPa0aXaTa3aYa0a4aZa9bbaW",
"#Vbfa3#Za.#Kal#0#w#kaba.#H#F#6a.#k#R#0#D#3aTbdbbbbb#bbaW",
"#Vbfbda9b.a8b#b.a7a7a9b.a8a7a8a8aDa2a9a2a2b#bbbbbababbaW",
"#5bfbfbfbfbfbfbfbfbfbfbfbfbfbfbfbfbfbfbfbfbfbfbfbfbfbfa9",
".w.P.M.M.M.M.M.M.M.M.M.M.M.M.J.J.J.J.J.J.J.J.J.I.I.I.L.G"
};

static const char *folder_xpm[] = {
"38 38 195 2",
"  	c None",
". 	c None",
"+ 	c None",
"@ 	c #9C9E9D",
"# 	c #C4C2C4",
"$ 	c #CCCECC",
"% 	c #CCCACC",
"& 	c #F4F6F4",
"* 	c #F4F2F4",
"= 	c #ECEAEC",
"- 	c #CCC6C4",
"; 	c #ECF2F4",
"> 	c #ECEEEC",
", 	c #E4E6E4",
"' 	c #D4D2D4",
") 	c #D4CECC",
"! 	c #C4C6C4",
"~ 	c #ACAEAC",
"{ 	c #9C9A9C",
"] 	c #7C868C",
"^ 	c #74869C",
"/ 	c #748EAC",
"( 	c #94AECC",
"_ 	c #A4C2EB",
": 	c #D4D6D4",
"< 	c #949699",
"[ 	c #7C8A94",
"} 	c #748AA4",
"| 	c #7C96B4",
"1 	c #84A6CC",
"2 	c #94B6E4",
"3 	c #9CBEEC",
"4 	c #ACC6F4",
"5 	c #B4DAFC",
"6 	c #ACD6FC",
"7 	c #DCDADC",
"8 	c #E4E2DC",
"9 	c #DCDEDC",
"0 	c #84969C",
"a 	c #94B2E4",
"b 	c #A4C6F4",
"c 	c #ACCAF4",
"d 	c #B4D2F4",
"e 	c #ACD2F4",
"f 	c #8C9E9D",
"g 	c #9CBAD4",
"h 	c #ACCEEC",
"i 	c #B4D6FC",
"j 	c #BCD6FC",
"k 	c #ACCEF4",
"l 	c #9CC2F1",
"m 	c #9CBAEB",
"n 	c #949A9C",
"o 	c #8C9A9D",
"p 	c #B4CED4",
"q 	c #C4DEE4",
"r 	c #D4F6FC",
"s 	c #D4F2FC",
"t 	c #CCEEFC",
"u 	c #C4E6FC",
"v 	c #C4DEFC",
"w 	c #A4C6EC",
"x 	c #94BAEC",
"y 	c #9CB6E4",
"z 	c #949E9D",
"A 	c #CCDADC",
"B 	c #DCEEFC",
"C 	c #ECFEFC",
"D 	c #E4FEFC",
"E 	c #DCFAFC",
"F 	c #BCDAFC",
"G 	c #8CB2E9",
"H 	c #ACCEFC",
"I 	c #8C8E92",
"J 	c #F4FEFC",
"K 	c #C4E2FC",
"L 	c #A4CEF9",
"M 	c #DCFEFC",
"N 	c #CCEAFC",
"O 	c #848E94",
"P 	c #B4D2FC",
"Q 	c #8CAEE4",
"R 	c #A4CAF4",
"S 	c #8CB6EC",
"T 	c #94B6EC",
"U 	c #94BEEF",
"V 	c #9CC6F4",
"W 	c #D4FAFC",
"X 	c #BCE2FC",
"Y 	c #84AEE4",
"Z 	c #84AAE4",
"` 	c #CCF2FB",
" .	c #BCDEFC",
"..	c #ACD2FC",
"+.	c #9CCAFA",
"@.	c #94C6F4",
"#.	c #BCE6FC",
"$.	c #7CAAE4",
"%.	c #9CCEFC",
"&.	c #8C9294",
"*.	c #C4EAFC",
"=.	c #B4DEFC",
"-.	c #A4D2FC",
";.	c #84AEEC",
">.	c #848A8C",
",.	c #C4EEFC",
"'.	c #8CBAF0",
").	c #7CAEEC",
"!.	c #94CAFC",
"~.	c #94C6FC",
"{.	c #D4FEFC",
"].	c #CCF6FC",
"^.	c #7CAAEC",
"/.	c #74A2E4",
"(.	c #7CA6E6",
"_.	c #848E9C",
":.	c #C4F2FC",
"<.	c #B4E2FC",
"[.	c #ACDAFC",
"}.	c #84B2F4",
"|.	c #7CB2F3",
"1.	c #84B2EC",
"2.	c #84B6EC",
"3.	c #8CC6FC",
"4.	c #8CBEF4",
"5.	c #74A6EA",
"6.	c #6C9EE4",
"7.	c #84BAF4",
"8.	c #8CC2FC",
"9.	c #CCFAFC",
"0.	c #BCEAFC",
"a.	c #74AAEC",
"b.	c #84C2FC",
"c.	c #7C929C",
"d.	c #B4E6FC",
"e.	c #ACDEFC",
"f.	c #84B6F4",
"g.	c #6CA2EA",
"h.	c #649AE4",
"i.	c #7CB6F4",
"j.	c #849EAC",
"k.	c #C4ECF5",
"l.	c #C4F6FC",
"m.	c #BCEEFC",
"n.	c #649EE4",
"o.	c #7CBEFC",
"p.	c #8CA6B4",
"q.	c #B4BBC2",
"r.	c #B2D6ED",
"s.	c #A4DAFC",
"t.	c #9CD2FC",
"u.	c #84BEFA",
"v.	c #6CA6EA",
"w.	c #74AEEC",
"x.	c #74B2F4",
"y.	c #7CBAF7",
"z.	c #7CA6BC",
"A.	c #B4EAFC",
"B.	c #64A2EC",
"C.	c #74B6F4",
"D.	c #8CB2BC",
"E.	c #B2BBC2",
"F.	c #A4D6FC",
"G.	c #ACE2FC",
"H.	c #94CEFC",
"I.	c #8CCAFC",
"J.	c #74AEF4",
"K.	c #84BAFC",
"L.	c #9CD6FC",
"M.	c #6C92CC",
"N.	c #748AB8",
"O.	c #89B8E1",
"P.	c #A4DEFC",
"Q.	c #749EDC",
"R.	c #748EBC",
"S.	c #748AAC",
"T.	c #94D6FC",
"U.	c #7CAEF4",
"V.	c #6C8EC4",
"W.	c #9CDAFC",
"X.	c #8492AC",
"Y.	c #87B3D7",
"Z.	c #7496D4",
"`.	c #6C8ABF",
" +	c #7C8EAC",
".+	c #789FBB",
"++	c #8BC1F2",
"@+	c #94D2FC",
"#+	c #6C9AD4",
"$+	c #7486B4",
"%+	c #7CB2D0",
"&+	c #6C82B4",
"*+	c #7C8AAC",
"=+	c #6482AC",
"-+	c #8C9AB4",
". . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . . . . . + + @ . . . . . . ",
". . . . . . . @ + + + @ . . . . . . . . . . . . . @ + + @ @ # $ % # . . . . ",
". . . . . . # # & * = = # . . . . . . . . @ + + @ @ # # % $ % $ - @ @ @ . . ",
". . . . . @ # = ; * * > , # # @ + + +   @ @ # # ' $ ' ) ! ~ { ] ^ / ( _ . . ",
". . . . @ # * > * * * = = = = = # # # # : ' : ' # @ < [ } | 1 2 3 4 5 6 . . ",
". . . . @ 7 * * * = 8 , 9 7 ' ) $ ' : ' @ @ < [ 0 1 a _ b c b _ _ _ d e . . ",
". . . @ # = = = 9 7 ' % $ $ ' % @ @ + + f g h i 5 j e k 4 l m a a m e k . . ",
". . . + # , , $ $ ' $ - @ @ n o @ p q r s t u v i d w m x y 2 m m _ e k . . ",
". . . + # = : ! @ @ o z # A B C C C D E t u F k 4 _ _ _ _ 3 m 2 G y H k . . ",
". . . + $ $ I n @ # D D D J J J D E s u K F j i d c b m 2 a G a 2 l k L . . ",
". . . + $ @ @ D D D D J J J D M E r s t N K 5 k w 3 m m 3 m x x a m L L . . ",
". . . + O   J J J C C C D D C D D E s u F i P k k c b l x G Q Q G 3 L R . . ",
". . . + @ J D D D C C C J C D E r t N u u K 5 P R 3 x S 2 T x T T U R V . . ",
". . . + @ C D C C C C D D M E E E W r t X i k R b b V l m S Y Z Z x L V . . ",
". . . + # C D D M M D D C C C D r ` u X  . .5 ..R l x G Y G G S T l +.@.. . ",
". . . I # M E D D C C C D M E W r r s t #.5 H V U U x U U S G Z $.S %.@.. . ",
". . . &.# D D E E E E M M D D M E ` *.=.i ....-.L V x G ;.$.Z ;.S l +.@.. . ",
". . . >.# W r W E D D D D E r ` t ,.*.*.#.5 ..V x S S S '.S S ).$.S !.~.. . ",
". . . ] # {.M M M E W W r W {.M W ].*.=...L +.+.V V U S ;.^./.(.).U !.+.. . ",
". . . _.# W ` ` ` W E M D D W :.*.#.<.<.=.[.-.V '.}.).).|.1.2.1.^.'.3.!.. . ",
". . . ] # :.].W {.M W W ].].` ].].].*.<.-.+.U 4.U 4.4.S 1.5./.6./.7.8.!.. . ",
". . . 0 # ].:.t ,.:.].W {.{.9.].0.<.[.6 -.-.%.+.4.1.^.5.5.a.).).).'.b.%.. . ",
". . . c.# #.#.,.].W {.].` ,.0.0.#.d.d.e.-.~.4.f.|.f.|.f.|.a.g.h.h.i.b.%.. . ",
". . . j.# k.,.,.0.0.0.0.,.l.].m.0.e.-.+.!.~.~.8.4.i.a.g.n.n.g.a.a.7.o.%.. . ",
". . . p.q.r.e.=.0.,.:.l.:.m.<.e.s.[.[.s.t.!.u.|.5.v.a.w.w.a.g.h.h.x.y.%.. . ",
". . . z.# d.#.0.0.d.<.<.<.d.A.d.d.e.t.~.u.f.f.i.7.x.a.B.h.n.B.w.y.b.C.. . . ",
". . . D.E.[.F.F.s.G.A.m.m.A.e.s.H.!.!.I.!.8.y.J.B.v.v.x.y.u.u.K.7.f.v.. . . ",
". . . z.# L.F.G.<.d.G.s.F.L.L.s.L.t.I.u.x.J.x.y.u.b.b.u.K.i.|.^.M.N.. . . . ",
". . . j.O.s.t.%.H.H.t.s.P.P.L.!.b.u.K.b.3.H.I.3.u.K.i.^.Q.R.S.. . . . . . . ",
". . . j.O.3.~.H.L.s.L.t.I.8.3.!.T.L.L.I.3.u.i.U./.V.N.. . . . . . . . . . . ",
". . . j.O.H.I.3.u.u.3.I.L.P.P.W.H.b.K.|./.M.N.X.. . . . . . . . . . . . . . ",
". . . j.Y.C.i.u.!.W.P.P.L.I.K.|./.Z.`. +. . . . . . . . . . . . . . . . . . ",
". . . .+++@+W.W.L.I.8.i.(.#+`.$+. . . . . . . . . . . . . . . . . . . . . . ",
". . . %+W.H.b.f.a.Q.`.&+. . . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . #+|./.`.&+*+. . . . . . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . =+*+-+. . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . "};


static const char*anydoc_xpm[] = {
"    25    36      107            2",
".. c #6b6b6b",
".# c #6f6f6f",
".a c #707070",
".b c #7a7a7a",
".c c #868686",
".d c #969696",
".e c #979797",
".f c #989898",
".g c #999999",
".h c #9a9a9a",
".i c #9f9f9f",
".j c #a4a4a4",
".k c #a5a5a5",
".l c #a7a7a7",
".m c #a8a8a8",
".n c #aaaaaa",
".o c #adadad",
".p c #aeaeae",
".q c #afafaf",
".r c #b0b0b0",
".s c #b1b1b1",
".t c #b3b3b3",
".u c #b4b4b4",
".v c #b5b5b5",
".w c #b6b6b6",
".x c #b9b9b9",
".y c #bababa",
".z c #bbbbbb",
".A c #bcbcbc",
".B c #bebebe",
".C c #bfbfbf",
".D c #c1c1c1",
".E c #c3c3c3",
".F c #c4c4c4",
".G c #c7c7c7",
".H c #c8c8c8",
".I c #c9c9c9",
".J c #cacaca",
".K c #cbcbcb",
".L c #cccccc",
".M c #cdcdcd",
".N c #cfcfcf",
".O c #d0d0d0",
".P c #d1d1d1",
".Q c #d2d2d2",
".R c #d3d3d3",
".S c #d4d4d4",
".T c #d5d5d5",
".U c #d6d6d6",
".V c #d7d7d7",
".W c #d8d8d8",
".X c #d9d9d9",
".Y c #dadada",
".Z c #dbdbdb",
".0 c #dcdcdc",
".1 c #dddddd",
".2 c #dedede",
".3 c #dededf",
".4 c #dfdfdf",
".5 c #e0e0e0",
".6 c #e1e1e1",
".7 c #e2e2e2",
".8 c #e3e3e3",
".9 c #e4e4e4",
"#. c #e5e5e5",
"## c #e6e6e6",
"#a c #e7e7e7",
"#b c #e8e8e8",
"#c c #e8e9e8",
"#d c #e9e9e9",
"#e c #eaeaea",
"#f c #ebebeb",
"#g c #ececec",
"#h c #ecedec",
"#i c #ededec",
"#j c #ededed",
"#k c #eeeeee",
"#l c #efefef",
"#m c #f0f0f0",
"#n c #f0f1f1",
"#o c #f1f1f0",
"#p c #f1f1f1",
"#q c #f1f2f2",
"#r c #f2f1f1",
"#s c #f2f2f2",
"#t c #f3f3f3",
"#u c #f4f4f4",
"#v c #f5f5f5",
"#w c #f6f6f6",
"#x c #f6f7f6",
"#y c #f7f7f7",
"#z c #f7f8f7",
"#A c #f7f8f8",
"#B c #f8f7f8",
"#C c #f8f8f8",
"#D c #f8f9f8",
"#E c #f9f8f9",
"#F c #f9f9f9",
"#G c #faf9f9",
"#H c #faf9fa",
"#I c #fafafa",
"#J c #fbfbfb",
"#K c #fcfcfc",
"#L c #fcfdfc",
"#M c #fdfdfd",
"#N c #fefefe",
"#O c none",
/* pixels */
"#.#..9.9.9.8.7.7.6.5.5.3.0.W.V.4#s#O#O#O#O#O#O#O#O",
"#e#e#d#b#a#..9.7.6.4.1.Y.U.M.E.v.C#b#O#O#O#O#O#O#O",
".6.7.7.6.4.2.1.Z.Y.W.U.S.Q.M.F.w.q.B#d#O#O#O#O#O#O",
".8.8.8.7.6.5.4.1.Z.Y.W.V.T.R.L.B.l.p.L#d#O#O#O#O#O",
".8.8.8.8.8.7.5.4.2.0.Y.X.V.T.Q.G.o.h.1.M#b#O#O#O#O",
".9.9.9.9.9.9.8.6.5.4.1.Z.X.V.S.L.v.d.4#g.K#b#O#O#O",
"#.#.#.#.#.#..9.8.7.6.4.1.0.Y.V.O.z.d.6#J#d.K#d#O#O",
"#.#.#.#.#.#.#.#..9.7.6.5.2.1.Y.R.C.g.9#O#O#l.N#e#O",
"#a#a#a#a#a#######.#..8.7.5.4.0.U.E.i.m.p.s.y.I.H#f",
"#b#b#b#b#b#b#a#a#a###..9.7.6.4.X.H.n.c.a...#.b.f.u",
"#d#d#d#d#b#b#b#b#b#b#a###..9.6.0.O.y.j.e.d.d.d.d.k",
"#e#e#e#e#e#d#d#d#d#d#d#b#a##.9.5.X.M.D.A.A.z.x.t.r",
"#g#g#f#f#f#f#f#e#e#e#e#e#d#b#a.9.6.0.V.S.R.Q.O.J.F",
"#j#j#j#j#g#g#g#g#f#f#f#f#f#e#d#b#..8.6.4.1.Z.X.U.P",
"#k#k#k#k#k#j#j#j#j#g#g#g#g#f#f#e#b#a#..9.6.4.1.Z.X",
"#m#m#l#l#l#l#k#k#k#k#j#j#j#g#g#f#e#d#a##.9.7.5.2.1",
"#p#p#p#o#m#m#m#l#l#l#l#k#k#k#k#j#g#f#e#b##.9.8.5.5",
"#t#t#t#t#s#s#s#p#p#p#p#n#m#m#l#l#l#k#i#f#e#b##.9.8",
"#u#u#u#u#u#t#t#t#s#s#s#q#p#p#p#m#m#l#l#j#f#e#b####",
"#w#w#w#v#v#v#v#u#u#u#t#t#t#s#s#s#s#p#p#m#k#h#f#d#d",
"#y#y#w#w#w#w#v#v#v#u#u#u#t#t#t#t#s#s#s#p#m#l#j#f#f",
"#y#y#y#y#y#w#w#w#v#v#v#u#u#u#u#t#t#t#s#s#p#m#l#j#j",
"#C#C#C#y#y#y#y#w#w#w#w#v#v#u#u#u#t#t#t#t#s#s#m#l#l",
"#C#C#C#C#C#y#y#y#y#w#w#w#w#v#v#v#u#u#t#t#t#s#s#p#p",
"#F#F#F#C#C#C#C#A#y#y#y#w#w#w#v#v#v#v#u#u#t#t#t#s#t",
"#F#F#F#F#F#C#C#C#C#C#y#y#y#y#w#w#v#v#v#u#u#u#t#t#u",
"#I#I#I#I#F#F#F#F#C#C#C#C#y#y#y#w#w#w#w#v#v#v#u#u#v",
"#I#I#I#I#I#I#G#F#F#F#D#C#C#C#B#y#y#w#w#w#w#v#v#u#w",
"#J#J#J#I#I#I#I#I#F#F#F#F#C#C#C#C#z#y#y#w#w#w#w#v#w",
"#J#J#J#J#J#J#I#I#I#I#I#F#F#F#C#C#C#C#C#y#y#x#w#w#y",
"#K#K#J#J#J#J#J#J#I#I#I#I#I#F#F#F#C#C#C#C#y#y#y#w#C",
"#K#K#K#K#K#K#J#J#J#J#J#I#I#I#I#H#F#F#F#C#C#C#B#y#F",
"#M#M#M#M#M#M#L#K#K#K#K#K#J#J#J#J#J#I#I#I#F#F#F#C#I",
"#N#N#M#M#M#M#M#M#M#K#K#K#K#K#J#J#J#J#I#I#I#I#F#F#J",
"#N#N#N#N#M#M#M#M#M#M#M#K#K#K#K#K#J#J#J#J#I#I#I#F#J",
"#N#N#N#N#N#N#M#M#M#M#M#M#M#K#K#K#K#K#J#J#J#J#I#I#J"
};


static const char *imadoc_xpm[] = {
"    39    31      237            2",
".. c #0c1214",
".# c #4e72a2",
".a c #64a2bc",
".b c #24422c",
".c c #9ca4a1",
".d c #7c8664",
".e c #1c2a04",
".f c #8486c4",
".g c #5c4a34",
".h c #9cbac0",
".i c #6c8a9c",
".j c #242a3c",
".k c #9cacdc",
".l c #8cb3c3",
".m c #7c928c",
".n c #84a4b9",
".o c #18263c",
".p c #88b2cc",
".q c #7c9cb2",
".r c #546264",
".s c #648edc",
".t c #4c5a7c",
".u c #24191c",
".v c #3c4234",
".w c #7c98dc",
".x c #5c7ac4",
".y c #4c4854",
".z c #342d30",
".A c #84abbb",
".B c #84a4e9",
".C c #546294",
".D c #acb5df",
".E c #3c393f",
".F c #9cb0ea",
".G c #9caeb3",
".H c #bcc1e2",
".I c #3c4e3c",
".J c #6499b1",
".K c #1c2118",
".L c #8e9eda",
".M c #849ba1",
".N c #649dbf",
".O c #4c4634",
".P c #8cabe9",
".Q c #6c6a64",
".R c #24231a",
".S c #9cbaf0",
".T c #94abb8",
".U c #8c868c",
".V c #9cb6ec",
".W c #1c1214",
".X c #94a299",
".Y c #94a5b5",
".Z c #8c9dad",
".0 c #343a1c",
".1 c #6492b0",
".2 c #3c4448",
".3 c #829ee6",
".4 c #24262c",
".5 c #94b3eb",
".6 c #6086ac",
".7 c #443a64",
".8 c #7c8a90",
".9 c #6c7e7c",
"#. c #acbce1",
"## c #545254",
"#a c #8c9592",
"#b c #5c6a7c",
"#c c #556e9e",
"#d c #3c3e42",
"#e c #a1b6c4",
"#f c #2c2b2a",
"#g c #141628",
"#h c #8caad8",
"#i c #8ca5b7",
"#j c #748ec4",
"#k c #7c99e5",
"#l c #8cabbb",
"#m c #94a5e5",
"#n c #7c8ca2",
"#o c #7ca2b0",
"#p c #9cb6cc",
"#q c #849db0",
"#r c #445664",
"#s c #949da1",
"#t c #a4b1e6",
"#u c #9cb0bf",
"#v c #7497e4",
"#w c #94acea",
"#x c #2c2521",
"#y c #6486d8",
"#z c #a0aaa8",
"#A c #8caaac",
"#B c #645f67",
"#C c #7496ac",
"#D c #8c9b9a",
"#E c #343e24",
"#F c #b4c6e4",
"#G c #2c322a",
"#H c #141313",
"#I c #5c729c",
"#J c #8ca3a8",
"#K c #7c90e0",
"#L c #94bac4",
"#M c #444a2c",
"#N c #8ca4e9",
"#O c #8c9de7",
"#P c #749cbc",
"#Q c #6c6e6c",
"#R c #748ba6",
"#S c #90bae4",
"#T c #b4bbe1",
"#U c #a4b3da",
"#V c #9ca5ac",
"#W c #54484c",
"#X c #748c96",
"#Y c #342a3c",
"#Z c #94b3c2",
"#0 c #7c93a2",
"#1 c #2c4254",
"#2 c #446a84",
"#3 c #34566c",
"#4 c #6c768c",
"#5 c #686e8c",
"#6 c #6c83a1",
"#7 c #c0c6e4",
"#8 c #2c2c16",
"#9 c #343742",
"a. c #7c92ac",
"a# c #7c8284",
"aa c #5c66a0",
"ab c #342664",
"ac c #4c4e4c",
"ad c #6c82c4",
"ae c #94b4cc",
"af c #748bdf",
"ag c #34325c",
"ah c #94a6dc",
"ai c #6c7a84",
"aj c #6c76c4",
"ak c #cccae4",
"al c #241e1a",
"am c #4c4a4a",
"an c #94aaac",
"ao c #546694",
"ap c #243e4c",
"aq c #585a64",
"ar c #544e51",
"as c #5c6274",
"at c #707484",
"au c #343733",
"av c #446a94",
"aw c #4c7aa4",
"ax c #494e57",
"ay c #363233",
"az c #6a9eb4",
"aA c #1c322c",
"aB c #8e8e90",
"aC c #818684",
"aD c #2c323c",
"aE c #1c1a19",
"aF c #2c321c",
"aG c #4c4648",
"aH c #94a4ab",
"aI c #8c949e",
"aJ c #64627c",
"aK c #343654",
"aL c #8ca4db",
"aM c #44443a",
"aN c #9c9694",
"aO c #443a3c",
"aP c #746864",
"aQ c #94acd8",
"aR c #9c9a9c",
"aS c #6c84d5",
"aT c #748699",
"aU c #9caab7",
"aV c #749eb4",
"aW c #7492a8",
"aX c #7492e2",
"aY c #34323c",
"aZ c #6c76a4",
"a0 c #94bacc",
"a1 c #241e2c",
"a2 c #a4bac4",
"a3 c #a4b6ec",
"a4 c #443e42",
"a5 c #3c3e2f",
"a6 c #746e74",
"a7 c #5c6684",
"a8 c #5c4e54",
"a9 c #6ca3bf",
"b. c #5c5e5c",
"b# c #6c8bdd",
"ba c #849adb",
"bb c #b4b6e4",
"bc c #a4aeb4",
"bd c #445234",
"be c #6c9ab4",
"bf c #6c9bbc",
"bg c #a4bbec",
"bh c #6c8eac",
"bi c #444446",
"bj c #2c252d",
"bk c #848b91",
"bl c #5c5458",
"bm c #646c78",
"bn c #8499e7",
"bo c #848ea0",
"bp c #a4b0c0",
"bq c #949ee4",
"br c #84969f",
"bs c #34465c",
"bt c #847e7c",
"bu c #3c3a34",
"bv c #7ca4c0",
"bw c #4c6e9c",
"bx c #2c2e3c",
"by c #84a6c4",
"bz c #7c9ebc",
"bA c #84acc4",
"bB c #94969c",
"bC c #3c364c",
"bD c #9cb0f4",
"bE c #1c2124",
"bF c #242424",
"bG c #94adc4",
"bH c #94b4f4",
"bI c #acbaec",
"bJ c #2c2d34",
"bK c #8ca6c4",
"bL c #748ad4",
"bM c #8cabc4",
"bN c #9cb3dc",
"bO c #849ebc",
"bP c #94aef4",
"bQ c #8c9ca4",
/* pixels */
"#.#T#U#..D.D.DbNah.kah#.#7#t.wbqb#b#aSb#b#.sb##yb##yaSaSaS#xaEbFaE.j..a7bF#g#g",
".D#U#U.Da3bN#Ubb.H.H.H#T#7.Hak#F#.afafafb#b#bL.safb#b#afaSaM#H.WbJ#jaSagaZad#r",
"#U#U.k.Lahahah.kaL.k.D.H#T.HbNbb#T#taX#KaX#K#Kafb##K#Kbj#Q#f.R.KaEadbjaEaj.xaf",
"ah.k.kaL.Lbaba#k.w.3.3.3#tbb.k#k.L#v#v#K#kaX#v#k#v#K#va4.O.v.g.u.7#f..bFbEb#aE",
".Lba#K#Kba#O.w#kbn.w#kbn#k#kbn.3#k#k#k#kbnbnbn#k#k#k#k#H.WaFaMal.zbu#HbJao#1.o",
"babababa.3bn#O#N.3#O.B#N#N#N#m#O#N#N#N.3#N#O.B.B#Obnba#x.u#N.4al#YaY.4aJaL.oap",
"aQ.L.BahaL#m#NaQ#w#w#t.F#t.F.P#m.P#m#m.P#m#N#N.P.B#N#N#8#PabalbFaFaE#d.4ax#xbC",
"#haQaL#mah#N.P.F#w#w#w#wbP#w#wbPbDbPbPbP.P.PbP#N.P.PbP.KbO#6at#Hay.C.yas#mbK.K",
"#m#NaL#w.P#w.P#w.F#w.5bP.5bDbD#w#wbPbHbDbP#w.kbPbDbD#SbF.V.EaFalaubjaubEbmbsbO",
"aLahaL.P#h#w.5.5.5.5.VbD.V.VbD.V.V.FbPbDbHbPbHbHbHbD#N#pa4aram.KasaG#x#8.x.f#M",
"aQ.FbN.F.VbNa3a3.Fbg.Sa3.V.S.S.V.SbDbg.Vbgbg.VbD.S.S.V.VaJ#4.5a6.r.ta9a1aK#waa",
"aQ.5.FbN#S.V.V.VbgbI.SbibC#9#9#9a3.SbgbMbi.2biaxbibiaq#r#xaG#B#HbFambJ#falaM.K",
".k#U#p#ZaO.E#d.E#.#rbJbJaYaDaYbibi.2bibiaYbxbxaD#9a4.E.Ealax.zal#Bbj#fbj#xbJ.z",
"#h.Eauaybu#da5aObi.E#dbJaDaDbiaGa4#da4biaD#d#Wa4#daYar.ybjaY#xbCbi#WbE.4al.QaE",
"au.z.E#fbJay.Eaybubibi.EblaMacbibia4a4aMaO.E#9aGaGbibi#W.E#d.ublaGau#f.Eay.K.R",
".zayaya4a5#daGauayaGbi.4#W##bi.E#d#xaGaybJ.E#f#d.Eara8#W#fbJaBa5am#Q.E.vbJa4bd",
"ao#I#c#c#c#c#cbw#c#I#c#c#c#cbwbw.#.#avavbwaw.6bf.1#C.6#6axa7#n#GaB#Q#gbj.z#8..",
"#C.J.JbeaV.J.Jbebf.Na9a9aV.pa9bha9.a.N.N.Na9.Naza9be.N#0#R.1#2bra6#3#o.Rb..e.0",
".q.qa.#R#qbvaWbz.q.nbv.nbA.nbMbv.lbAbyaeby#P.pbMbvbO#qbAa0aebB.Q#QaA.q.8.d#H.R",
"#q#i#q#i.A#lbzbAbKbK.n#l.A#i.A.n.nbA.AbM#lbM#i#ibMbMa0ae.l#X.v#Ebi.vbr#5.K.Ra5",
"bK.nbKbGbMbM#lbK#ZbMbGbG#i#Z.TbG.n.pae#ZbM.lbMbA.haebG#Z#ibK##.b.I.8.lboa5aP.K",
".nbK.nbObK.qbAbybG.n#l.n.l#Z.n#l#pbGaeae#lbM#Z.naeaeaeae#ZbM.h#lbG.T.T#iaraq#H",
"#l.nbK#l.nbK#Z#iaebKbG#Z#i#q.AbG.lbGbGbG.la2#uae#ibGbG#L#Z.TbG#i#ebG#i.T#i#8.R",
".Y#i#La.#lbMbM#ZbM#l#Z.h#l#l#u#lbG#Z#i.h#Z#Z#Z.Y#Z#u.hbGaU.h#l.T#i#q#R#Rbr#s#g",
".n#i#ubG#q#u#i#l#l.l#i#Z.T.T#u#Z#u.T.Ta2a2bp#i#u#e.Y#e.T.Y#u.Z.i#Rbh#6.iaT#XaI",
".Z#q.n#i#u#i#i.TbG.T#u.T.T#ZaH#l.Y#JbG.Y#u.n#u#u#ubp#e.GaU#i#u#q#0#n#n#0#naTbQ",
"br.Z#q.M#q#i#J.M.T#i.T#l#A#0.T.Y#J#laU#u.T.Y#z.Y.Y#u.Y.GaT#i.Z.Y.MbraUaI.YaI.8",
"#b#J#X.Z#JbG#J.X.Z.GanaH.c.haU.G.GaH.Yan.TaH.T.TbG.TaUaUaH#V#a#s#JaI.ZaI.YaiaC",
"ai#0#J#J#JbQbQ#V.M#JaHaHbG#JaHan#A.GaH.TaUaHbc#z.MbcaHaH#saI#D.c.X.X.c#VaB#aaB",
"bk.Mbrbr#X#J.ZbQ#0.man#A.XbkaH.TaH#JaRa#an#z#z#J#VbQ.Y#s#sbrbB.cbcaRaBaC.U#Bar",
"#0aC.8.9br.M.MbraIanaH#s#JbQaHanbQbrbkaIbr#D#X.c#saBa##s#V#zaN#aaBaC.UaCbtbmat"
};

static const char *up_xpm[] = {
"    34    34       74            2",
".. c #64668c",
".# c #9c9aa4",
".a c #b4b6c4",
".b c #84869c",
".c c #a4aaa4",
".d c #7c7a94",
".e c #9ca29c",
".f c #6c6e8c",
".g c #8c8e9c",
".h c #b4aeac",
".i c #aca6a4",
".j c #7472a4",
".k c #847e9c",
".l c #a4a2b4",
".m c #9c9ab4",
".n c #746e94",
".o c #c4c2cc",
".p c #8c8a9c",
".q c #acb2ac",
".r c #8c8eb4",
".s c #acaaa4",
".t c #7c7eac",
".u c #a4a29c",
".v c #8482ac",
".w c #7476a4",
".x c #746e9c",
".y c #acaab4",
".z c #a4a2ac",
".A c #6c668c",
".B c #a49e9c",
".C c #bcbec4",
".D c #b4b2ac",
".E c #a49eb4",
".F c #948eb4",
".G c #8c86b4",
".H c #7c76ac",
".I c #9c9eac",
".J c #6c6e9c",
".K c #9496a4",
".L c #aca6bc",
".M c #847eac",
".N c #a4a6b4",
".O c #9c9ebc",
".P c #74728c",
".Q c #cccacc",
".R c #acaeac",
".S c #a4a6a4",
".T c #8486b4",
".U c #6c6a8c",
".V c #9492b4",
".W c #9c9e9c",
".X c #bcbac4",
".Y c #6c6e94",
".Z c #9492a4",
".0 c #b4aeb4",
".1 c #aca6ac",
".2 c #7472ac",
".3 c #a4a2bc",
".4 c #9c9abc",
".5 c #c4c6cc",
".6 c #8c8ab4",
".7 c #acaaac",
".8 c #a4a2a4",
".9 c #8482b4",
"#. c #7476ac",
"## c #74729c",
"#a c #acaeb4",
"#b c #a4a6ac",
"#c c #b4b2b4",
"#d c #7c7aac",
"#e c #847eb4",
"#f c #747294",
"#g c #6c6a94",
"#h c #9492bc",
/* pixels */
"#c.S.7#c.q.7.R.1.S.7.R.7#c.R.R#c.R#c.R.R.R#c.R.7.R#c.R#c.R.S.7.R.R.7",
".R.i.c#c.D.7.7.c.S.7.R.7#c.7.R#c.R.h.R.0.R.D.7.7.R#c.R.q.7.S.7.7.R.7",
".0.S.7#c#a.7.7.i.S.7.7.7.D.7#a.h.R#a.h.R.R#c.7.7.R.0.R.0.7.S.7.7.R.7",
".q.1.c.0.q.s.7#b.S.7.R.7#c.R.R.0.R.h#a.R.R#c.7.7.7.D.7.D.7.8.s.7.R.i",
".R.S.7.q.0.7.7.s.S.1.7.c#c.7.R.R.R#c.R.R.7#c.7.c.R.q.7#a.7.S.7.7.R.c",
"#c.S.7.R#c.c.7#b.S.c.R.7.q.7.R.0.R.0.R.0.R.q.R.7.7#c.R.R.R.8.c.R.R.1",
".0.S.7#c.q.7.7.c.S.1.7.7#c.7.R#c.K.K#c.R.R.0.7.7.7.q.7.0.7.S.7.7.R#b",
".q.S.7.R.0.c.7.1.S.S.R.7.q.R.R.l.Y.Y.#.R.R#c.c.R.7#c.R.q.7.S.1.R.R.i",
".0.S.7.D#c.s.7.c.S.1.R.7.D.7#c#..n.f.U#c.R#c.7.7.7.q.7.h.7.S.c.7.R#b",
"#c.S.7.R#a.S.7.1.8.i.R.7.0.c.m#d##.U#g.p.7.D.7.7.7#c.7.R.7.8.7.7.R.i",
".R.S.7.R#c.s.c#b.S.S.R.7#a#a.6.G.2.U.U.U.y.q.7.c.7.R.7.R.7.S.S.7.R#b",
".0.S.7#a.h#b.7.i.S.S.7.7#c.V.6#h#..Y.U.U.d#c.7.7.7.q.7.R.7.8.c.7.h.S",
".q.S.7.R#c.c.S.c.S.S.R.7#a.6#h.V#d.j.U...U.N.c.7.c.0.c.R.7.S.S.7.R.S",
".h.S.7.R.q.1.c.1.S.S.7.R#h.r#h.r#d.j.Y.f.U#f.R.7.7.q.7.R.c.S.S.R.R.S",
"#a.S.s.R#c.S.1.S.S.S.7.N.6.r.F.T.H#.##.U.U.U.K.7.S#a.7.R.1.8.S.7.R.8",
".h.S.7#a.h#b.c.i.8.z.q.6.F#h.6.9#.#d.j.Y.U.Y...R.7.R.s.7.c.S#b.7.R.8",
".q.S.7.R.R.c.S.c.S.8.4.r#h.r.9#d.2#.#..j.A.f.A.b.S.R.7.R.1.S.S.7.R.S",
".0.8.S.R.0#b.i#b.8.7.6.6.V.T#d#d#.#..H.j.Y.f...A#b#a.S.R.S.8.S.7.R.8",
".R.S.7.7.q.S.S#b.S.r.r.F.r#e#.#d.2.H#.#..x.Y.U.U.P.D#b.7.S.S.S.7.R.8",
".h#b.c.R.0.1.S.i.N.G.r#h.T#d#..H.j.H#..H.j##...U...E.S.R.1.S.8.7.R.8",
"#a.S.1.7.R.S.S.7.T.6.V.6.t#..2#d.2#..2.H.2.j.U.A...f.s.7.S.8.S.s.7.e",
".h.S.7.R.q.1.S.I.6.6.r.9#d.H#..H.j.H#..H.w.H.J...A.U.g.7.S.8.S.7.R.8",
"#a.S.S.7#a.S.7.6.6.6.6#d.H.2.2#d.2#..2.H.2#.##.Y...U...R.1.S.8.c#a.8",
".D.8.7.7.h.S.V.6.6.6.v.H#..j#.#..j.H#.#..w#..j.x.A.U...k.S.8.S.7.7.8",
"#a.S.S.7.R.y.6.r.r.v#d#..H.2.H.H.j.2.2.H.2.H.j.j#g.U...U#b.S.8.c.R.e",
".R.8.S.c#c.O.O.N.m.6.9.v.v.t.M.9.t.9.v.v.t.9.t#d.j.U...U.f.S.8.7.7.B",
".h.8.s.1.X.a.o.X.L.V.r.F#h.r.F#h.r.F.r#h.F.F.r.r.T.H.A.A...#.8.S.R.I",
".q.8#b.c.5.Q.o.a.4.r#h.r.F.r.r.V.6.r.F.r.r.F.r.6.6.v.J.f...P.S.i.R.B",
"#c.8.c.7.C.o.a.3.r.T.6.T.6.6.T.r.T.6.6.6.T.6.G.6.T.9.j.x#f.Z.S#b.R.W",
"#a.u.1.S.h.7#a.y.7.1#a.y#a.y.y#c.1#a.y#c.L#a.c.y.7.0#b.y.7.S.8.1.7.B",
".R.8.S.1.R.8.u.S.8.W.7.S.s.S.S.R.8.S.S.R.u#b.8.S.8.c.B.c#b.8.8.S.R.I",
".q.8.c.S.R.e.8.S.S.W.7.S.7.c.i#a.e.7.S.R.8.7.S.S.S.7.8.7.S.8.8.c.R.W",
".h.e.1.S.R.8.8.S.8.W.7.S.7.S.S.7.8.s.S.R.u.S.S.S.8.7.W.c.S.8.e.i.7.W",
".R.8.S.S.R.u.8.S.8.W.s.S.c.1.S.R.u.1.S.R.8.1.S.S.8.7.e.1.S.8.B.S.R.B"
};

static const char *left_xpm[] =
{
  "34 34 74 2",
  "AA c #646466668C8C",
  "BA c #9C9C9A9AA4A4",
  "CA c #B4B4B6B6C4C4",
  "DA c #848486869C9C",
  "EA c #A4A4AAAAA4A4",
  "FA c #7C7C7A7A9494",
  "GA c #9C9CA2A29C9C",
  "HA c #6C6C6E6E8C8C",
  "IA c #8C8C8E8E9C9C",
  "JA c #B4B4AEAEACAC",
  "KA c #ACACA6A6A4A4",
  "LA c #74747272A4A4",
  "MA c #84847E7E9C9C",
  "NA c #A4A4A2A2B4B4",
  "OA c #9C9C9A9AB4B4",
  "PA c #74746E6E9494",
  "AB c #C4C4C2C2CCCC",
  "BB c #8C8C8A8A9C9C",
  "CB c #ACACB2B2ACAC",
  "DB c #8C8C8E8EB4B4",
  "EB c #ACACAAAAA4A4",
  "FB c #7C7C7E7EACAC",
  "GB c #A4A4A2A29C9C",
  "HB c #84848282ACAC",
  "IB c #74747676A4A4",
  "JB c #74746E6E9C9C",
  "KB c #ACACAAAAB4B4",
  "LB c #A4A4A2A2ACAC",
  "MB c #6C6C66668C8C",
  "NB c #A4A49E9E9C9C",
  "OB c #BCBCBEBEC4C4",
  "PB c #B4B4B2B2ACAC",
  "AC c #A4A49E9EB4B4",
  "BC c #94948E8EB4B4",
  "CC c #8C8C8686B4B4",
  "DC c #7C7C7676ACAC",
  "EC c #9C9C9E9EACAC",
  "FC c #6C6C6E6E9C9C",
  "GC c #94949696A4A4",
  "HC c #ACACA6A6BCBC",
  "IC c #84847E7EACAC",
  "JC c #A4A4A6A6B4B4",
  "KC c #9C9C9E9EBCBC",
  "LC c #747472728C8C",
  "MC c #CCCCCACACCCC",
  "NC c #ACACAEAEACAC",
  "OC c #A4A4A6A6A4A4",
  "PC c #84848686B4B4",
  "AD c #6C6C6A6A8C8C",
  "BD c #94949292B4B4",
  "CD c #9C9C9E9E9C9C",
  "DD c #BCBCBABAC4C4",
  "ED c #6C6C6E6E9494",
  "FD c #94949292A4A4",
  "GD c #B4B4AEAEB4B4",
  "HD c #ACACA6A6ACAC",
  "ID c #74747272ACAC",
  "JD c #A4A4A2A2BCBC",
  "KD c #9C9C9A9ABCBC",
  "LD c #C4C4C6C6CCCC",
  "MD c #8C8C8A8AB4B4",
  "ND c #ACACAAAAACAC",
  "OD c #A4A4A2A2A4A4",
  "PD c #84848282B4B4",
  "AE c #74747676ACAC",
  "BE c #747472729C9C",
  "CE c #ACACAEAEB4B4",
  "DE c #A4A4A6A6ACAC",
  "EE c #B4B4B2B2B4B4",
  "FE c #7C7C7A7AACAC",
  "GE c #84847E7EB4B4",
  "HE c #747472729494",
  "IE c #6C6C6A6A9494",
  "JE c #94949292BCBC",
  "EENCGDCBNCEEGDCBGDEENCGDCBJACEJACBGDNCJACEJACEPBCENCJACBEECENCCBJANC",
  "OCKAOCHDOCOCOCOCOCOCOCOCOCOCOCOCOCODOCDEOCOCOCODOCODODODODGBODODGAOD",
  "NDEANDEANDNDNDNDNDNDNDNDNDNDEBNDNDOCNDEAHDNDOCNDOCOCEBDEEAHDOCEAHDOC",
  "EEEEEEGDCBNCEENCPBNCNCCENCNCNCCENCNCNDNCNDNCNDNDNDEAHDEANDOCHDOCOCOC",
  "CBPBCECBGDEECBGDEECEEEJAEECBEEJANCGDCBGDNCCBCEJANCEEDDLDOBJANCNCNCNC",
  "NDNDNDEBNDEANDEAEBOCEBDEEAHDOCDEEADEOCHDOCHDOCOCKBKCCAMCABNDODGAODGB",
  "NCNDNDNDNDNDNDNDNDNDEANDOCEAHDEAOCKAOCOCOCOCNDBDMDKCABABCACEGBODODOD",
  "HDEAKADEEBDEEAHDEAHDDEKAEAHDOCKAEADEDEKANDECMDMDDBJCDDCAJDKBOCOCOCOC",
  "OCOCOCOCOCOCOCOCOCODOCOCOCOCOCODOCODOCJCPCMDMDMDDBOAHCKDDBNDODOCODOD",
  "NDNDNDNDHDEAHDOCHDKAOCOCOCOCOCLBODNDDBCCMDMDMDMDHBMDBDDBPCHDCDCDCDCD",
  "NCNCNDNCNDNCNDNCNCNCNCNDNCNDNDCBKDMDDBDBBDDBMDHBFEPDDBJEMDCENDNDNDEB",
  "NDNDNDNDEANDNDNDNDNDNDNDNDNCJCMDDBMDBCJEMDPDFEDCAEHBBCDBPCKBOCOCOCOC",
  "EEEEPBEEEECBEECBPBGDCEEECEJEMDBCJEBDDBPCFBFEDCAEDCHBJEBCMDCEEBNDNDEA",
  "NCNDNDNCNDNDNDNCNDEACEBDMDDBDBJEDBPCGEFEAEDCIDLAIDFBDBDBMDKBOCEAOCHD",
  "NCNCCENCNCNCNCNCEEOAMDMDJEJEBCMDPDFEAEAEIDAEIDAEDCICBCDBPCKBOCKAOCOC",
  "EEEEJAGDNCGDEENAAEFECCJEBDDBPCPDFEFEFEDCFEDCFEAEDCPDJEBDDBEENCCENDNC",
  "NCNCNCNCNCNCGCEDPABEIDAEFEFEDCAEIDAEIDLAIDLAIDLALAFBDBMDPCHDODGAODGB",
  "EEJACEJAEEGDGCEDHAADADEDLALAAEFEAEAEDCDCAEDCAEDCIDPDBCDBMDCEOCNDEBHD",
  "NCNCJACENCNCEEBAADIEADADADEDBELAAEDCAEAEIDAEIDAEIDHBDBBCMDKBOCOCOCOC",
  "NCGDNCNCNCGDNCNCEEBBADADAAHAADEDLALAAEDCDCDCDCAEDCHBJEDBMDEENCNCNCNC",
  "NCNCNCNCNDNCNCNCNCNDKBFAADADADADMBEDJBLAIDIBIDIBIDFBBCDBPCHCGBODGBOD",
  "EEPBEEEEEECBGDEEEEPBCBEEJCHEADEDHAHAEDBELADCAEAEDCPDBCBCMDCEDENDOCHD",
  "NCNDNDNDNDNCNDEANDNDNDNDEANCGCAAMBAAADAAADFCBELALAFBDBDBCCEAODOCOCOC",
  "NDNDNDNDEANDNDNCNDNDEANDNDNDNDNCDAMBADADMBAAEDJBLAFEDBMDMDKBOCOCOCOC",
  "NCNCNCNDNCNDNDNDNDNDNDNDEANDOCNDOCDELCAAAAMBAAMBIELAPCMDPCNDODOCODOD",
  "EEEEGDPBCBEECBEECBEENCCBGDCBCENCNCCEPBACHAADADADADADDCHBPDGDEANDNDND",
  "NCNCNCNDNDNCNDNCNDNDNDNDEANDNDEBNDOCDEOCEBIAAAAAAAAAMBFCLADENBODCDGA",
  "EECBGDPBCENCGDCBJANCNCNCNCNCNCNDNCNCNDNCNDNDNCMAADADMBHAJBKBEANDEAHD",
  "NCNDNDNDNDNCNDNDNDNDNDNDNDEAHDEAHDOCOCHDOCOCHDOCDEHAAAAAHENDDEOCOCOC",
  "OCOCOCODOCODOCOCOCODOCODOCOCODOCOCODOCOCODODOCODOCOCBALCFDOCODODODOD",
  "NDNDNDEBNDEANDHDEANDOCEAOCOCOCDEOCOCOCODOCOCODOCODODODOCOCODODODGANB",
  "NCNDNDNDNDNCNDNCNDNDNDNDNDNCNDNDNDNDNDNDEBNDEANDEANDOCKADEHDOCEAKAOC",
  "NCNCNCNCNCNCNCNCNCNCNCJANCNCNCNCNCNCNCNCNDNCCENDNCNDNCNCNCNDNCNCNDNC",
  "NDNDNDKAEAHDDEKADEKADEOCOCOCODODOCODODODGAODODODGANBECNBCDNBECCDCDNB",
};

static const char *right_xpm[] =
{
  "34 34 74 2",
  "AA c #646466668C8C",
  "BA c #9C9C9A9AA4A4",
  "CA c #B4B4B6B6C4C4",
  "DA c #848486869C9C",
  "EA c #A4A4AAAAA4A4",
  "FA c #7C7C7A7A9494",
  "GA c #9C9CA2A29C9C",
  "HA c #6C6C6E6E8C8C",
  "IA c #8C8C8E8E9C9C",
  "JA c #B4B4AEAEACAC",
  "KA c #ACACA6A6A4A4",
  "LA c #74747272A4A4",
  "MA c #84847E7E9C9C",
  "NA c #A4A4A2A2B4B4",
  "OA c #9C9C9A9AB4B4",
  "PA c #74746E6E9494",
  "AB c #C4C4C2C2CCCC",
  "BB c #8C8C8A8A9C9C",
  "CB c #ACACB2B2ACAC",
  "DB c #8C8C8E8EB4B4",
  "EB c #ACACAAAAA4A4",
  "FB c #7C7C7E7EACAC",
  "GB c #A4A4A2A29C9C",
  "HB c #84848282ACAC",
  "IB c #74747676A4A4",
  "JB c #74746E6E9C9C",
  "KB c #ACACAAAAB4B4",
  "LB c #A4A4A2A2ACAC",
  "MB c #6C6C66668C8C",
  "NB c #A4A49E9E9C9C",
  "OB c #BCBCBEBEC4C4",
  "PB c #B4B4B2B2ACAC",
  "AC c #A4A49E9EB4B4",
  "BC c #94948E8EB4B4",
  "CC c #8C8C8686B4B4",
  "DC c #7C7C7676ACAC",
  "EC c #9C9C9E9EACAC",
  "FC c #6C6C6E6E9C9C",
  "GC c #94949696A4A4",
  "HC c #ACACA6A6BCBC",
  "IC c #84847E7EACAC",
  "JC c #A4A4A6A6B4B4",
  "KC c #9C9C9E9EBCBC",
  "LC c #747472728C8C",
  "MC c #CCCCCACACCCC",
  "NC c #ACACAEAEACAC",
  "OC c #A4A4A6A6A4A4",
  "PC c #84848686B4B4",
  "AD c #6C6C6A6A8C8C",
  "BD c #94949292B4B4",
  "CD c #9C9C9E9E9C9C",
  "DD c #BCBCBABAC4C4",
  "ED c #6C6C6E6E9494",
  "FD c #94949292A4A4",
  "GD c #B4B4AEAEB4B4",
  "HD c #ACACA6A6ACAC",
  "ID c #74747272ACAC",
  "JD c #A4A4A2A2BCBC",
  "KD c #9C9C9A9ABCBC",
  "LD c #C4C4C6C6CCCC",
  "MD c #8C8C8A8AB4B4",
  "ND c #ACACAAAAACAC",
  "OD c #A4A4A2A2A4A4",
  "PD c #84848282B4B4",
  "AE c #74747676ACAC",
  "BE c #747472729C9C",
  "CE c #ACACAEAEB4B4",
  "DE c #A4A4A6A6ACAC",
  "EE c #B4B4B2B2B4B4",
  "FE c #7C7C7A7AACAC",
  "GE c #84847E7EB4B4",
  "HE c #747472729494",
  "IE c #6C6C6A6A9494",
  "JE c #94949292BCBC",
  /* pixels */
  "NCJACBNCCEEECBJANCCEPBCEJACEJANCGDCBJACEJACBGDNCEEGDCBGDEENCCBGDNCEE",
  "ODGAODODGBODODODODOCODOCOCOCDEOCODOCOCOCOCOCOCOCOCOCOCOCOCOCHDOCKAOC",
  "OCHDEAOCHDEADEEBOCOCNDOCNDHDEANDOCNDNDEBNDNDNDNDNDNDNDNDNDNDEANDEAND",
  "OCOCOCHDOCNDEAHDEANDNDNDNCNDNCNDNCNCCENCNCNCCENCNCPBNCEENCCBGDEEEEEE",
  "NCNCNCNCJAOBLDDDEENCJACECBNCGDCBGDNCJAEECBEEJAEECEEEGDCBEEGDCBCEPBCB",
  "GBODGAODNDABMCCAKCKBOCOCHDOCHDOCDEEADEOCHDEADEEBOCEBEANDEANDEBNDNDND",
  "ODODODGBCECAABABKCMDBDNDOCOCOCOCKAOCEAHDEAOCNDEANDNDNDNDNDNDNDNDNDNC",
  "OCOCOCOCKBJDCADDJCDBMDMDECNDKADEDEEAKAOCHDEAKADEHDEAHDEADEEBDEKAEAHD",
  "ODODOCODNDDBKDHCOADBMDMDMDPCJCOCODOCODOCOCOCOCOCODOCOCOCOCOCOCOCOCOC",
  "CDCDCDCDHDPCDBBDMDHBMDMDMDMDCCDBNDODLBOCOCOCOCOCKAHDOCHDEAHDNDNDNDND",
  "EBNDNDNDCEMDJEDBPDFEHBMDDBBDDBDBMDKDCBNDNDNCNDNCNCNCNCNDNCNDNCNDNCNC",
  "OCOCOCOCKBPCDBBCHBAEDCFEPDMDJEBCMDDBMDJCNCNDNDNDNDNDNDNDNDEANDNDNDND",
  "EANDNDEBCEMDBCJEHBDCAEDCFEFBPCDBBDJEBCMDJECEEECEGDPBCBEECBEEEEPBEEEE",
  "HDOCEAOCKBMDDBDBFBIDLAIDDCAEFEGEPCDBJEDBDBMDBDCEEANDNCNDNDNDNCNDNDNC",
  "OCOCKAOCKBPCDBBCICDCAEIDAEIDAEAEFEPDMDBCJEJEMDMDOAEENCNCNCNCNCCENCNC",
  "NCNDCENCEEDBBDJEPDDCAEFEDCFEDCFEFEFEPDPCDBBDJECCFEAENAEEGDNCGDJAEEEE",
  "GBODGAODHDPCMDDBFBLALAIDLAIDLAIDAEIDAEDCFEFEAEIDBEPAEDGCNCNCNCNCNCNC",
  "HDEBNDOCCEMDDBBCPDIDDCAEDCAEDCDCAEAEFEAELALAEDADADHAEDGCGDEEJACEJAEE",
  "OCOCOCOCKBMDBCDBHBIDAEIDAEIDAEAEDCAELABEEDADADADIEADBAEENCNCCEJANCNC",
  "NCNCNCNCEEMDDBJEHBDCAEDCDCDCDCAELALAEDADHAAAADADBBEENCNCGDNCNCNCGDNC",
  "ODGBODGBHCPCDBBCFBIDIBIDIBIDLAJBEDMBADADADADFAKBNDNCNCNCNCNDNCNCNCNC",
  "HDOCNDDECEMDBCBCPDDCAEAEDCLABEEDHAHAEDADHEJCEECBPBEEEEGDCBEEEEEEPBEE",
  "OCOCOCODEACCDBDBFBLALABEFCADAAADAAMBAAGCNCEANDNDNDNDEANDNCNDNDNDNDNC",
  "OCOCOCOCKBMDMDDBFELAJBEDAAMBADADMBDANCNDNDNDNDEANDNDNCNDNDEANDNDNDND",
  "ODODOCODNDPCMDPCLAIEMBAAMBAAAALCDEOCNDOCNDEANDNDNDNDNDNDNDNCNDNCNCNC",
  "NDNDNDEAGDPDHBDCADADADADADHAACPBCENCNCCECBGDCBNCEECBEECBEECBPBGDEEEE",
  "GACDODNBDELAFCMBAAAAAAAAIAEBOCDEOCNDEBNDNDEANDNDNDNDNCNDNCNDNDNCNCNC",
  "HDEANDEAKBJBHAMBADADMANCNDNDNCNDNCNCNDNCNCNCNCNCNCJACBGDNCCEPBGDCBEE",
  "OCOCOCDENDHEAAAAHADEOCHDOCOCHDOCOCHDEAHDEANDNDNDNDNDNDNDNCNDNDNDNDNC",
  "ODODODODOCFDLCBAOCOCODOCODODOCOCODOCOCODOCOCODOCODOCOCOCODOCODOCOCOC",
  "NBGAODODODOCOCODODODOCODOCOCODOCOCOCDEOCOCOCEAOCNDEAHDNDEANDEBNDNDND",
  "OCKAEAOCHDDEKAOCNDEANDEANDEBNDNDNDNDNDNDNCNDNDNDNDNDNCNDNCNDNDNDNDNC",
  "NCNDNCNCNDNCNCNCNDNCNDCENCNDNCNCNCNCNCNCNCNCJANCNCNCNCNCNCNCNCNCNCNC",
  "NBCDCDECNBCDNBECNBGAODODODGAODODODOCODODOCOCOCDEKADEKADEHDEAKANDNDND",
};

//UPix *anydoc, *textdoc, *imadoc, *folder, *parent_folder;

UPix UIcon::anydoc(anydoc_xpm, UMode::UCONST);
UPix UIcon::textdoc(textdoc_xpm, UMode::UCONST);
UPix UIcon::imadoc(imadoc_xpm, UMode::UCONST);
UPix UIcon::folder(folder_xpm, UMode::UCONST);
UPix UIcon::up(up_xpm, UMode::UCONST);
UPix UIcon::left(left_xpm, UMode::UCONST);
UPix UIcon::right(right_xpm, UMode::UCONST);

/* ==================================================== ======== ======= */

UIma* UIcon::findIconImage() { 
  if (isDir()) {
    if (*pname == "..") {
      //if (!up) up = new UPix(parent_folder_xpm, UMode::UCONST);
      return &up;
    }
    else {
      //if (!folder) folder = new UPix(folder_xpm, UMode::UCONST);
      return &folder;
    }
  }

  if (isFile() || isLink()) {
    UStr ext = pname->getFileSuffix();

    if (ext.compareTo("gif", true) == 0
	|| ext.compareTo("jpg", true) == 0
	|| ext.compareTo("jpeg", true) == 0
	|| ext.compareTo("tif", true) == 0
	|| ext.compareTo("tiff", true) == 0
	|| ext.compareTo("png", true) == 0
	|| ext.compareTo("xpm", true) == 0
	|| ext.compareTo("xbm", true) == 0
	|| ext.compareTo("ras", true) == 0
	) {
      //if (!imadoc) imadoc = new UPix(imadoc_xpm, UMode::UCONST);
      return &imadoc;
    }

    if (ext.compareTo("txt", true) == 0
	|| ext.compareTo("htm", true) == 0
	|| ext.compareTo("html", true) == 0
	|| ext.compareTo("xml", true) == 0
	|| ext.compareTo("pdf", true) == 0
	|| ext.compareTo("doc", true) == 0
	|| ext.compareTo("ppt", true) == 0
	|| ext.compareTo("xls", true) == 0
	|| ext.compareTo("c", true) == 0
	|| ext.compareTo("h", true) == 0
	|| ext.compareTo("cc", true) == 0
	|| ext.compareTo("hh", true) == 0
	|| ext.compareTo("cpp", true) == 0
	|| ext.compareTo("hpp", true) == 0
	|| ext.compareTo("java", true) == 0
	) {
      //if (!textdoc) textdoc = new UPix(textdoc_xpm, UMode::UCONST);
      return &textdoc;
    }
  }

  if (isExec()) {
    return &UPix::ray;
  }

  //if (!anydoc) anydoc = new UPix(anydoc_xpm, UMode::UCONST);
  return &anydoc;	 // default
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */

