/* ==================================================== ======== ======= *
 *
 *  uprop.hpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _uprop_hpp_
#define	_uprop_hpp_
//pragma ident	"@(#)uprop.hpp	ubit:03.06.04"
#include <ubit/unumber.hpp>


/** Base class for Graphical Properties.
 * Notes: 
 * - use the onChange() method to add a callback to this object.
 *   This callback will be fired when the object's value changes.
 * - the 'UOn::propChange' callbacks of the object's parents will
 *   also be fired (after the object's callbacks)
 */
class UProp: public UBrick {
  friend class UBox;
public:
  UProp(u_modes b_modes = 0) : UBrick(b_modes) {}
  ///< constructor.

  virtual class UProp* propCast() {return this;}
  ///< dynamic cast

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // callbacks

  virtual void onChange(UCall&);
  ///< adds callbacks that are activated when the objects' value changes.

  virtual void changed(bool update_now);
  /**< called when object's content is changed.
   * This function:
   * - updates grahics (if arg is true)
   * - then fires object's UOn::change callbacks
   * - then fires parents' UOn::propChange callbacks
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation
  
  virtual void update() = 0;
  ///< updates graphics.

  virtual void putProp(class UContext*, class UCtrl*) = 0;
  ///< [impl] changes corresponding value in the UContext
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Scale property.
 *  Scale properties can be cascaded in the instance graph.
 *  set*() and get*() methods are inherited from Uintg.
 */
class UScale : public UIntgBase, public UProp {
public:
  UScale(int value = 0);
  UScale(const UIntgBase& value);

  friend UScale& uscale(int val = 0)           {return *new UScale(val);}
  friend UScale& uscale(const UIntgBase& val)  {return *new UScale(val);}
  ///< larger than normal size if 'value' is positive; smaller if negative.

  // inherited: set(), incr(), decr(), equals()

  float getXScale() const;
  float getYScale() const;
  void  getXYScale(float& xscale, float& yscale) const;

  static float getXScale(int lscale);
  static float getYScale(int lscale);
  static void  getXYScale(int lscale, float& xscale, float& yscale);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void changed(bool update_now);
  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Alpha blending property.
 *  Note: name subjet to change.
 */
class UAlpha : public UProp {
  float value;
public:
  UAlpha(float value = 1.0);
  friend UAlpha& ualpha(float val = 1.0) {return *new UAlpha(val);}
  ///< value must be in range [0.0, 1.0] : 1.0 : opaque object; 0.0 : fully transparent object.

  float get() const {return value;}
  ///< returns current alpha value.
  
  void set(float value);
  void set(float value, bool update_now);
  ///< changes alpha value. must be in interval [0.0, 1.0].

  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Background property.
 * See also: UBgcolor
 */
class UBackground : public UProp {
  const class UColor  *bgcolor;
  const class UIma    *bgima;
  const class UHalign *halign;
  const class UValign *valign;
public:
  UBackground();
  UBackground(const UIma& bgima);
  UBackground(const UColor& bgcolor);

  friend UBackground& ubackground(const UColor& bgcolor);
  friend UBackground& ubackground(const UIma& bgima);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  void set(const UColor& bgcolor);
  ///< sets an uniform background color.

  void set(const UIma& bgima);
  /**< sets a tiled background from an image.
   * Note: the background image can be a transparent UPix */

  void set(const UIma& bgima, const UColor& bgcolor);
  /**< sets a tiled background with a background color.
   * Note: useful if the image is transparent
   * or if the image is centered instead of being tiled */

  void setLayout(const UHalign&, const UValign&);
  /**< if null, the bg image is tiled in this direction
   * if &UHalign::center, it is centered in the horizontal direction, etc.*/

  const class UColor*  getColor()  const {return bgcolor;}
  const class UIma*    getIma()    const {return bgima;}
  const class UHalign* getHalign() const {return halign;}
  const class UValign* getValign() const {return valign;}

  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/*
class UShape : public UProp {
  friend class UView;
  const UPix *pix;
public:
  // The shape of the component will adapt to the shape of the UPix argument
  UShape(const UPix&);
  friend UShape& ushape(UPix&);

  void set(const UPix*);
  void set(const UPix&);

  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
};
*/

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Position property.
 *  Locates an UBox (or subclass) object at an arbitrary position.
 *  - coordinates are relative to parent.
 *  - !Warning: does not work with hardwin UWin objects (use UWin::move())
 */
class UPos : public UProp {
public:
  UPos();
  UPos(const UPos&);
  UPos(u_pos x, u_pos y);

  friend UPos& upos(const UPos& p)     {return *new UPos(p);}
  friend UPos& upos(u_pos x, u_pos y)  {return *new UPos(x, y);}

  virtual ~UPos() {destructs();}  // necessaire car removingFrom specifique

  UPos& operator=(const UPos&);
  void set(const UPos&, bool update = true);
  
  void set(u_pos x, u_pos y, bool update = true);
  void setX(u_pos x);
  void setY(u_pos y);

  u_pos getX() const {return x;}
  u_pos getY() const {return y;}

  virtual void addingTo(class ULink *selflink, UGroup *parent);
  virtual void removingFrom(class ULink *selflink, UGroup *parent);
  ///< NOTE that this function require a specific destructor.
  
  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
private:
  u_pos x, y;
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Width property.
 * Usage:
 * - uwidth(int) with int >= 0  : size fixed by argument.
 *   horizontal size won't change except if the box is uhflex()
 * - uwidth(UWidth::KEEP_SIZE)   : keep initial size.
 *   size is computed automatically the first time, then does not change.
 * - uwidth(UWidth::AUTO_RESIZE) : size automatically adapts
 *   size is re-computed automatically each time 
 */
class UWidth : public UProp {
public:
  static const u_dim AUTO_RESIZE, KEEP_SIZE;

  UWidth(u_dim value = 0);
  UWidth(const UWidth&);
  friend UWidth& uwidth(u_dim value = 0);

  u_dim get() const {return value;}
  void set(u_dim);
    
  UWidth& operator=(u_dim _val)         {set(_val); return *this;}
  UWidth& operator=(const UWidth& _val) {set(_val.get()); return *this;}

  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
private:
  u_dim value;
};

/* ==================================================== ======== ======= */
/** Height property.
 * Usage: see UWidth.
 */
class UHeight : public UProp {
public:
  static const u_dim AUTO_RESIZE, KEEP_SIZE;

  UHeight(u_dim value = 0);
  UHeight(const UHeight&);
  friend UHeight& uheight(u_dim value = 0);

  u_dim get() const {return value;}
  void  set(u_dim);

  UHeight& operator=(u_dim _val)          {set(_val); return *this;}
  UHeight& operator=(const UHeight& _val) {set(_val.get()); return *this;}

  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
private:
  u_dim value;
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Orientation property.
 *  Can be: UOrient::vertical or  UOrient::horizontal
 */
class UOrient : public UProp {
public:
  enum {VERTICAL, HORIZONTAL, INHERIT};
  static UOrient vertical, horizontal, inherit;

  UOrient(char value = VERTICAL, u_modes = 0);
  UOrient(const UOrient&);
  friend UOrient& uorient(const UOrient&);

  void set(const UOrient&);
  char get() const {return value;}

  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
private:
  char value;
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Vertical Alignment property.
 *  Can be: top, bottom, center, flex ("flex" means "flexible object")
 *  - rather use functions: utop(), ubottom(), uvcenter(), uvflex() 
 *    that are equivalent but improve readability
 */
class UValign : public UProp {
  char value;
public:
  enum {TOP, BOTTOM, FLEX, CENTER};
  static UValign top, bottom, flex, center;

  UValign(char value = TOP, u_modes = 0);
  UValign(const UValign&);
  friend UValign& uvalign(const UValign&);

  friend UValign& utop()     {return UValign::top;}
  friend UValign& ubottom()  {return UValign::bottom;}
  friend UValign& uvcenter() {return UValign::center;}
  friend UValign& uvflex()   {return UValign::flex;}

  char get() const {return value;}
  void set(const UValign&);
  
  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Horizontal Alignment property.
 *  Can be: left, right, center, flex ("flex" means "flexible object")
 *  - rather use functions: uleft(), uright(), uhcenter(), uhflex() 
 *    that are equivalent but improve readability
 */
class UHalign : public UProp {
  char value;
public:
  enum {LEFT, RIGHT, FLEX, CENTER};
  static UHalign left, right, flex, center;

  UHalign(char value = LEFT, u_modes = 0);
  UHalign(const UHalign&);
  friend UHalign& uhalign(const UHalign&);

  friend UHalign& uleft()    {return UHalign::left;}
  friend UHalign& uright()   {return UHalign::right;}
  friend UHalign& uhcenter() {return UHalign::center;}
  friend UHalign& uhflex()   {return UHalign::flex;}

  char get() const {return value;}
  void set(const UHalign&);
  
  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Vertical Spacing property.
 */
class UVspacing : public UProp {
public:
  UVspacing(int = 0);
  friend UVspacing& uvspacing(int);

  int get() const {return value;}
  void set(int);
  UVspacing& operator=(int i) {set(i); return *this;}
  
  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
private:
  int value;
};

/* ==================================================== ======== ======= */

/** Horizontal Spacing property.
 */
class UHspacing : public UProp {
public:
  UHspacing(int = 0);
  friend UHspacing& uhspacing(int);

  int get() const {return value;}
  void set(int);
  UHspacing& operator=(int i) {set(i); return *this;} 

  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
private:
  int value;
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Vertical Margin property.
 */
class UVmargin : public UProp {
public:
  UVmargin(int = 0);
  friend UVmargin& uvmargin(int);

  int get() const {return value;}
  void set(int);
  UVmargin& operator=(int i) {set(i); return *this;} 

  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
private:
  int value;
};

/* ==================================================== ======== ======= */

/** Horizontal Margin property.
 */
class UHmargin : public UProp {
public:
  UHmargin(int = 0);
  friend UHmargin& uhmargin(int);

  int get() const {return value;}
  void set(int);
  UHmargin& operator=(int i) {set(i); return *this;} 

  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
private:
  int value;
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/* Tool Tip.
 */
class UTip: public UProp {
  uptr<UStr> value;
public:

  UTip(const char* value);
  ///< contructor (value is duplicated internally).

  UTip(UStr& value);
  /**< contructor (value is NOT duplicated internally and must NOT be deleted)
   * Note: 'value' can be shared and directly modified by clients.
   * It should not be deleted nor allocated in the stack.
   * It will be automatically deleted when not pointed by any uptr.
   */

  friend UTip& utip(const char* _value) {return *(new UTip(_value));}
  friend UTip& utip(UStr& _value)       {return *(new UTip(_value));}
  ///< creator shortcuts.

  virtual void set(const UStr& value);
  virtual const UStr& get() const {return *value;}

  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*) {}
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/* Window Title.
 */
class UTitle: public UProp {
  uptr<UStr> value;

public:
  UTitle(const char* value);
  UTitle(const UStr& value);
  ///< contructor (value is duplicated internally).

  UTitle(UStr& value);
  /**< contructor (value is NOT duplicated internally and must NOT be deleted)
   * Note: 'value' can be shared and directly modified by clients.
   * It should not be deleted nor allocated in the stack.
   * It will be automatically deleted when not pointed by any uptr.
   */

  friend UTitle& utitle(const char* _value)  {return *new UTitle(_value);}
  friend UTitle& utitle(const UStr& _value)  {return *new UTitle(_value);}
  friend UTitle& utitle(UStr& _value)        {return *new UTitle(_value);}
  ///< creator shortcuts.

  virtual void set(const UStr& value);
  virtual const UStr& get() const {return *value;}

  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*) {}
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/* Comment (not displayed)
 */
class UComment: public UProp {
  char *value;

public:
  UComment(const char* = null);
  friend UComment& ucomment(const char *s);

  const char* get() const {return value;}
  void set(const char*);
  
  virtual void update();
  virtual void putProp(class UContext*, class UCtrl*);
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Flag Definition property.
*  See: UFlag.
*/
class UFlagdef : public UProp {
public:
  UFlagdef();
  UFlagdef(const UFlag&);

  friend UFlagdef& uflagdef() {return *new UFlagdef();}
  friend UFlagdef& uflagdef(const class UFlag& f) {return *new UFlagdef(f);}

  virtual void set(const UFlag&);
  virtual void clear();

  const UFlag* getFlag() const {return flag;}
  
  virtual void update();
  virtual void putProp(UContext*, class UCtrl*);

protected:
  const class UFlag* flag;
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

class UPropdef : public UFlagdef {
public:
  UPropdef();
  UPropdef(const UFlag&);
  UPropdef(const UFlag&, UProp&);
  UPropdef(const UFlag&, UProp*);
  friend UPropdef& upropdef(const class UFlag& f, UProp&);
  friend UPropdef& upropdef(const class UFlag& f, UProp*);

  virtual void set(const UFlag&);
  virtual void set(UProp&);
  virtual void set(UProp*);
  virtual void set(const UFlag&, UProp&);
  virtual void set(const UFlag&, UProp*);
  virtual void clear();

  UProp* getProp() const {return prop;}
  virtual void putProp(UContext*, class UCtrl*);

private:
  uptr<UProp> prop;
};

/* ==================================================== ======== ======= */

class UPropval : public UProp {
public:
  UPropval(const UFlag&);
  friend UPropval& upropval(const class UFlag& f);

  const UFlag* getFlag() const {return flag;}

  virtual void update();
  virtual void putProp(UContext*, class UCtrl*);

private:
  const class UFlag* flag;
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
