/* ==================================================== ======== ======= *
 *
 *  uunatappli.cpp  [Native Layer: platform dependent implementation]
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uunatappli.cpp	ubit:03.06.04"
#include <unistd.h>       // darwin
#include <iostream>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <ubrick.hpp>
#include <ucall.hpp>
#include <uerror.hpp>
#include <unatappli.hpp>
#include <ugraph.hpp>
using namespace std;

UNatAppli::UNatAppli(UAppli& app, UNatDisp& nd) : 
  appli(app),
  natdisp(nd) {
}

UNatAppli::~UNatAppli() {
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UNatAppli::setProperties(int argc, char** argv) {
  // fait par XmbSetWMProperties
  // XSetCommand(natdisp.xdisplay, natdisp.xwin, argv, argc);

  XClassHint* class_hints = XAllocClassHint();
  class_hints->res_name  = const_cast<char*>(appli.getCommandName());
  class_hints->res_class = const_cast<char*>(appli.getCommandName());

  XmbSetWMProperties(natdisp.xdisplay, natdisp.xwin,
		     appli.getCommandName(),   // window_name,
		     appli.getCommandName(),   // icon_name, 
		     argv, argc,
                     null,       // normal_hints,
		     null,       // wm_hints, 
		     class_hints);  
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

static inline void FIX_TIME(timeval& time) {
  while (time.tv_usec >= 1000000) {
    time.tv_usec -= 1000000;                  
    time.tv_sec++;                            
  }
  while (time.tv_usec < 0) {
    if (time.tv_sec > 0) {
      time.tv_usec += 1000000;
      time.tv_sec--;
    }
    else {         
      time.tv_usec = 0;
      break;
    }      
  }
}

void UNatAppli::getTime(timeval& time) {
  gettimeofday(&time, 0);
  FIX_TIME(time);
}

u_time UNatAppli::convertToMilliSec(timeval& time) {
  gettimeofday(&time, 0);
  FIX_TIME(time);
  return time.tv_sec * 1000 + time.tv_usec / 1000;
}

static void add_time(struct timeval& time, unsigned long millisec_delay) {
  time.tv_sec  += millisec_delay / 1000;
  time.tv_usec += (millisec_delay % 1000) * 1000;
  FIX_TIME(time);
  //cerr << "add_time "<< time.tv_sec << " " << time.tv_usec <<endl;
}

static void min_time(struct timeval& mintime, struct timeval& time) {
  if (mintime.tv_sec > time.tv_sec) {
    mintime.tv_sec  = time.tv_sec;
    mintime.tv_usec = time.tv_usec;
  }
  else if (mintime.tv_sec == time.tv_sec && mintime.tv_usec > time.tv_usec)
    mintime.tv_usec = time.tv_usec;
  //cerr << "min_time "<< mintime.tv_sec << " " << mintime.tv_usec <<endl;
}

static bool check_time(struct timeval& timeout, struct timeval& time){
  if (timeout.tv_sec < time.tv_sec)
    return true;
  else if (timeout.tv_sec == time.tv_sec && timeout.tv_usec <= time.tv_usec)
    return true;

  return false; //si pas passe par return 
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

bool UNatAppli::resetTimers(UGroup* timers, struct timeval& delay) {
  struct timeval time, mintime;
  getTime(time);
  mintime.tv_sec  = LONG_MAX;
  mintime.tv_usec = LONG_MAX;
  
  bool found = false;
  UListPos lpos;
  UBrick* child;

  while ((child = timers->getChild(lpos)))  {
    UTimer* t = dynamic_cast<UTimer*>(child);
    if (t) {
      if (t->ntimes == 0) {   // can be -1(for always) or > 0
	t->close();
      } // endif (t->ntimes == 0)

      else {   // this is an active timeout
	found = true;
	if (t->must_rearm) {
	  t->must_rearm = false;
	  *(t->timeout) = time; //pas tt a fait correct mais bon...
	  add_time(*t->timeout, t->delay);
	}
	min_time(mintime, *t->timeout);
      }
    }
  }

  // is no active timeout was found
  if (!found) return false;

  if (check_time(mintime, time)) { // is mintime <= time ?
    delay.tv_sec  = 0;
    delay.tv_usec = 0;
  }
  else {
    delay.tv_sec  = mintime.tv_sec  - time.tv_sec;
    if (mintime.tv_usec >= time.tv_usec)
      delay.tv_usec = mintime.tv_usec - time.tv_usec;
    else {
      delay.tv_sec--;
      delay.tv_usec = 1000000 - time.tv_usec + mintime.tv_usec;
    }
  }
  //cerr << "delay "<< delay.tv_sec << " " << delay.tv_usec <<endl;

  //NB: delay can be (0,0)
  return (delay.tv_sec != LONG_MAX || delay.tv_usec != LONG_MAX);
}

/* ==================================================== ======== ======= */

void UNatAppli::fireTimers(UGroup* timers) {
  struct timeval time;
  getTime(time);
  UListPos lpos;
  UBrick* child;

  while ((child = timers->getChild(lpos)))  {
    UTimer* t = dynamic_cast<UTimer*>(child);
    if (t 
	&& t->ntimes != 0   // can be -1(for always) or > 0
	&& check_time(*t->timeout, time)
	) {
      
      UEvent e(UEvent::action, null, null, NULL);  // No InputFlow !!
      e.setTime(convertToMilliSec(time));
      e.setAux(t);
      t->fire(e, UOn::action);
      t->must_rearm = true;
      if (t->ntimes > 0) t->ntimes--;  //nb: ntimes < 0 means always
    }
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UNatAppli::resetInputs(UGroup* inputs, fd_set& read_set, int& maxfd) {
  for (ULink *l = inputs->getChildLinks(); l != null; l = l->getNext()) {
    int fd = 0;
    UInput* i = dynamic_cast<UInput*>(l->getChild());

    if (i && ((fd = i->getSource()) >= 0)) {
      FD_SET(fd, &read_set);
      maxfd = std::max(maxfd, fd);
    }
  }
}

/* ==================================================== ======== ======= */

void UNatAppli::cleanInputs(UGroup* inputs) {
  struct stat statbuf;

  ULink* prevlink = null;
  ULink *l = inputs->getChildLinks();

  while (l != null) {
    int fd = 0;
    UInput* i = dynamic_cast<UInput*>(l->getChild());
    if (i
	|| (fd = i->getSource()) <= 0
	|| fstat(fd, &statbuf) < 0)
      {
	inputs->removeImpl(UGroup::ELEM_LIST, i, prevlink, 
			   UGroup::AUTO_DEL,  // delete if no other parents
			   false, null);         // no update
	l = prevlink ? prevlink->getNext() : inputs->getChildLinks();
      }
    else {prevlink = l; l = l->getNext();}
  }
}

/* ==================================================== ======== ======= */

void UNatAppli::fireInputs(UGroup* inputs, fd_set& read_set) {
  struct timeval time;
  getTime(time);
  UListPos lpos;
  UBrick* child;

  while ((child = inputs->getChild(lpos)))  {
    int fd = 0;
    UInput* i = dynamic_cast<UInput*>(child);
 
    if (i
	&& ((fd = i->getSource()) >= 0)
	&& FD_ISSET(fd, &read_set)
	) {

      UEvent e(UEvent::action, null, null, NULL);  // No InputFlow !!
      e.setTime(convertToMilliSec(time));
      e.setAux(i); 
      i->fire(e, UOn::action);
    }
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

static UNatDisp* pending(UDispList& displist) {
  for (unsigned int k = 0; k < displist.size(); k++) {
    if (XPending(displist[k]->getNatDisp()->getXDisplay()))
      return displist[k]->getNatDisp();
  }
  return null;
}

void UNatAppli::eventLoop(int &loop_status) {
  XEvent xev;

  while (loop_status < 0) {
#ifdef WITH_GL
    glFlush(); // necessaire ici for some reason...
#endif

#ifdef MONODISP
    // ** X main loop
    while (loop_status < 0 && XPending(natdisp.xdisplay) ) {
      XNextEvent(natdisp.xdisplay, &xev);   //get next X event
      natdisp.dispatchEvents(&xev);
    }
#else
    // ** X main loop
    UDispList& displist = appli.displist;
    UNatDisp* nd = null;

    while (loop_status < 0 && (nd = pending(displist))) {
      XNextEvent(nd->xdisplay, &xev);    //get next X event
      nd->dispatchEvents(&xev);
    }
#endif

    if (loop_status >= 0) break;

#ifdef WITH_GL
    glFlush(); // necessaire apres dispatch
#endif

    // ** inputs and timers
    fd_set read_set;
    FD_ZERO(&read_set);

#ifdef MONODISP
    FD_SET(natdisp.xconnection, &read_set);
    int maxfd = natdisp.xconnection;
#else
    int maxfd = 0;
    for (unsigned int k = 0; k < displist.size(); k++) {
      int xconnection = displist[k]->getNatDisp()->xconnection;
      FD_SET(xconnection, &read_set);
      maxfd = std::max(maxfd, xconnection);
    }
#endif

    if (appli.inputs) resetInputs(appli.inputs, read_set, maxfd);

    struct timeval delay;
    bool has_timeout = false;
    //NB: delay can be (0,0)
    if (appli.timers) has_timeout = resetTimers(appli.timers, delay);

    // bloquer tant que: 
    // rien sur xconnection, rien sur inputs, timeouts pas atteints

    // tst: delay.tv_sec  = 0; delay.tv_usec = 1;

    int has_input = select(maxfd+1,
			   &read_set, //read
			   null,      //write
			   null,      //except
			   (has_timeout ? &delay : null));

    if (has_input < 0) {
      if (errno == EINTR || errno == EAGAIN) errno = 0;
      UError::error("warning@UNatAppli::eventLoop", "error in select()");
      cleanInputs(appli.inputs); // remove invalid inputs
    }

    else {
      if (has_input > 0) {	// input event
	if (appli.inputs) fireInputs(appli.inputs, read_set);
      }
      // pas de else!
      if (has_timeout) {	// timeout event
	if (appli.timers) fireTimers(appli.timers);
      }
    }

    UAppli::processSafeDeleteRequests();    
  }
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
