// $Id: CommandLineSettings.cc,v 1.7 2003/02/20 15:19:56 flaterco Exp $
/*  CommandLineSettings  Settings from command line.

    Copyright (C) 1998  David Flater.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.hh"

// Fodder for a next generation getopt.

CommandLineSettings::Argdesc switches[] = {

  // XTide switches

  {"b",  CommandLineSettings::temporal}, // Begin time
  {"e",  CommandLineSettings::temporal}, // End time
  {"f",  CommandLineSettings::format},   // Format
  {"l",  CommandLineSettings::text},     // Location
  {"m",  CommandLineSettings::mode},     // Mode
  {"ml", CommandLineSettings::number},   // Mark level (with units appended)
  {"o",  CommandLineSettings::text},     // Output file
  {"s",  CommandLineSettings::temporal}, // Step size

  {"el", CommandLineSettings::boolean},  // Extra lines
  {"fe", CommandLineSettings::boolean},  // Flat Earth
  {"in", CommandLineSettings::boolean},  // Infer constituents
  {"nf", CommandLineSettings::boolean},  // No fill
  {"ns", CommandLineSettings::boolean},  // No sun moon
  {"tl", CommandLineSettings::boolean},  // Top lines
  {"v",  CommandLineSettings::boolean},  // Print version & exit
  {"z",  CommandLineSettings::boolean},  // Zulu time
  {"suck", CommandLineSettings::boolean}, // Undocumented test mode

  {"bg", CommandLineSettings::color},    // Background color
  {"bc", CommandLineSettings::color},    // Button color
  {"cw", CommandLineSettings::number},   // Clock width
  {"df", CommandLineSettings::text},     // Date format
  {"Dc", CommandLineSettings::color},    // Datum color
  {"dc", CommandLineSettings::color},    // Daytime color
  {"ec", CommandLineSettings::color},    // Ebb color
  {"fc", CommandLineSettings::color},    // Flood color
  {"fg", CommandLineSettings::color},    // Foreground color
  {"ga", CommandLineSettings::number},   // Graph aspect
  {"gh", CommandLineSettings::number},   // Graph height
  {"gl", CommandLineSettings::number},   // Globe center longitude
  {"gw", CommandLineSettings::number},   // Graph width
  {"hf", CommandLineSettings::text},     // Hour format
  {"lw", CommandLineSettings::number},   // Line width
  {"mc", CommandLineSettings::color},    // Mark color
  {"Mc", CommandLineSettings::color},    // Mean Tide Level color
  {"nc", CommandLineSettings::color},    // Nighttime color
  {"tf", CommandLineSettings::text},     // Time format
  {"th", CommandLineSettings::number},   // TTY height
  {"tw", CommandLineSettings::number},   // TTY width
  {"u",  CommandLineSettings::unit},     // Units

  // Switches passed through to X11
  // Most of these others are unsupported, discouraged, deprecated,
  // untested, considered harmful...  You get the picture.

  // This no-op is used to clobber -geometry in xxContext.
  {"XX", CommandLineSettings::number},

  {"foreground", CommandLineSettings::color},    // X background color
  {"background", CommandLineSettings::color},    // X background color
  {"display", CommandLineSettings::text},        // X display
  {"fn", CommandLineSettings::text},             // X font
  {"font", CommandLineSettings::text},           // X font
  {"bd", CommandLineSettings::color},            // X border color
  {"bordercolor", CommandLineSettings::color},   // X border color
  {"bw", CommandLineSettings::number},           // X border width
  {"borderwidth", CommandLineSettings::number},  // X border width
  {"iconic", CommandLineSettings::boolean},      // X start iconic
  {"name", CommandLineSettings::text},           // X window name
  {"rv", CommandLineSettings::boolean},          // X reverse video
  {"reverse", CommandLineSettings::boolean},     // X reverse video
  {"selectionTimeout", CommandLineSettings::number},  // X timeout
  {"synchronous", CommandLineSettings::boolean}, // X synchronous mode
  {"title", CommandLineSettings::text},          // X window title
  {"xnllanguage", CommandLineSettings::text},    // X window language
  {"xrm", CommandLineSettings::text},            // X resource string

  {NULL, CommandLineSettings::boolean} // Stop
};

#define duplicated_code_block {        \
      if (temprest) {                  \
	if (rest1) {                   \
          while (rest1) {              \
            CommandLineSettings::arglist *t = rest1->next;  \
            delete rest1;              \
            rest1 = t;                 \
          }                            \
          while (temprest) {           \
            CommandLineSettings::arglist *t = temprest->next;  \
            delete temprest;           \
            temprest = t;              \
          }                            \
	  return NULL;                 \
	} else {                       \
	  rest1 = temprest;            \
	  mycookedswitch = name;       \
	}                              \
      }                                \
}

// The goal is to disambiguate the argument string.
// Inputs:
//   int argc, char **argv   The usual
//   int argi                Index to the argument now being looked at
// Returns:
//   arglist*  Disambiguated argument list if valid.  Null if not.

CommandLineSettings::arglist *CommandLineSettings::disambiguate
(int argc, char **argv, int argi) {

  char *argii = argv[argi];
  if (*argii != '-' && *argii != '+')
    return NULL;
  int isplus = (*argii == '+');
  argii++;

  Dstr mycookedswitch;
  CommandLineSettings::arglist *rest1=NULL, *temprest;
  for (unsigned settingslooper = 0; switches[settingslooper].name != NULL;
       settingslooper++) {
    if (isplus && switches[settingslooper].typ != CommandLineSettings::boolean)
      continue;
    char *name = switches[settingslooper].name;
    if (!strncmp (argii, name, strlen(name))) {
      // Try 1:  concatenated argument
      temprest = check_arg (argc, argv, argi, argii + strlen(name),
			    switches[settingslooper].typ);
      duplicated_code_block;
      // Try 2:  separate argument
      if ((*(argii + strlen(name)) == '\0') && (argi + 1 < argc)) {
	temprest = check_arg (argc, argv, argi+1, argv[argi+1],
			      switches[settingslooper].typ);
        duplicated_code_block;
      }
    }
  }

  if (rest1) {
    // check_arg is supposed to allocate one for us with a null
    // switch.
    assert (rest1->svitch.isNull());
    if (isplus)
      mycookedswitch *= '+';
    else
      mycookedswitch *= '-';
    rest1->svitch = mycookedswitch;
    if (isplus) {
      if (rest1->arg == "y")
	rest1->arg = "n";
      else if (rest1->arg == "n")
	rest1->arg = "y";
      else
	assert (0);
    }
  }
  return rest1; // Too easy.
}

// The goal is still to disambiguate the argument string.
// char *argii   Index to the current character position in argv[argi].
// argtype t     The type of argument we expect to find there.

CommandLineSettings::arglist *CommandLineSettings::check_arg
(int argc, char **argv, int argi, char *argii, CommandLineSettings::argtype t)
{
  Dstr mycookedarg;
  switch (t) {

  case CommandLineSettings::boolean:
    switch (*argii) {
    case '\0':
    case 'y':
    case 'Y':
      mycookedarg = "y";
      break;
    case 'n':
    case 'N':
      mycookedarg = "n";
      break;
    default:
      return NULL;
    }
    break;

  case CommandLineSettings::temporal:
  case CommandLineSettings::number:
    {
      float a;
      if (sscanf (argii, "%f", &a) == 1) {
        mycookedarg = argii;
        break;
      }
    }
    return NULL;

  case CommandLineSettings::mode:
    if (strlen (argii) != 1)
      return NULL;
    switch (*argii) {
    case 'b':
    case 'c':
    case 'C':
    case 'g':
    case 'l':
    case 'm':
    case 'p':
    case 'r':
    case 's':
      mycookedarg = argii;
      break;
    default:
      return NULL;
    }
    break;

  case CommandLineSettings::format:
    if (strlen (argii) != 1)
      return NULL;
    switch (*argii) {
    case 'h':
    case 'p':
    case 't':
      mycookedarg = argii;
      break;
    default:
      return NULL;
    }
    break;

  case CommandLineSettings::unit:
    if (!strcmp (argii, "ft") ||
        !strcmp (argii, "m") ||
        !strcmp (argii, "x")) {
      mycookedarg = argii;
      break;
    }
    return NULL;

  case CommandLineSettings::color:
    // Parsecolor wants to print warnings...  not worth fixing.
    if (isalpha(*argii) || *argii == '#') {
      mycookedarg = argii;
      break;
    }
    return NULL;

  case CommandLineSettings::text:
    if (*argii == '\0')
      return NULL; // They probably didn't mean that.
    mycookedarg = argii;
    break;

  default:
    assert (0);
  }

  CommandLineSettings::arglist *rest = NULL;
  if (++argi < argc) {
    rest = disambiguate (argc, argv, argi);
    if (!rest)
      return NULL;
  }
  CommandLineSettings::arglist *myarg = new CommandLineSettings::arglist;
  // myarg->svitch intentionally left blank
  myarg->arg = mycookedarg;
  myarg->next = rest;
  return myarg; // Easy.
}

CommandLineSettings::CommandLineSettings (int argc, char **argv) {
  static CommandLineSettings::arglist *args = NULL;
  if (!args)
    if (argc > 1) {
      args = disambiguate (argc, argv, 1);
      if (!args)
        barf (BAD_OR_AMBIGUOUS_COMMAND_LINE);
    }

  if (findarg ("-v", args)) {
    Dstr line;
    version_string (line);
    fprintf (stderr, "%s\n", line.aschar());
    exit (0);
  }
  {
    int a;
    for (a=0; a<numcolors; a++) {
      Dstr aname ("-");
      aname += colorarg[a];
      colors[a] = findarg (aname.aschar(), args);
    }
  }

  el = findarg("-el",args);
  fe = findarg("-fe",args);
  u = findarg("-u",args);
  z = findarg("-z",args);
  df = findarg("-df",args);
  hf = findarg("-hf",args);
  tf = findarg("-tf",args);
  tl = findarg("-tl",args);
  nf = findarg("-nf",args);
  ns = findarg("-ns",args);
  in = findarg("-in",args);

  b = findarg("-b",args);
  e = findarg("-e",args);
  f = findarg("-f",args);
  m = findarg("-m",args);
  s = findarg("-s",args);
  o = findarg("-o",args);

  l_len = 0;
  {
    struct lnode *templ, *last;
    l = last = NULL;
    for (CommandLineSettings::arglist *i = args; i; i=i->next) {
      if (i->svitch == "-l") {
        l_len++;
	templ = new lnode;
	templ->name = i->arg;
	templ->next = NULL;
        if (last) {
          last->next = templ;
          last = templ;
        } else {
  	  l = last = templ;
        }
      }
    }
  }
  if (l_len == 0) {
    char *defloc = getenv("XTIDE_DEFAULT_LOCATION");
    if (defloc) {
      l = new lnode;
      l->name = defloc;
      l->next = NULL;
      l_len = 1;
    }
  }

  char *t;
  if ((t = findarg ("-ga", args))) {
    ga_isnull = 0;
    ga = getposdouble (t);
  }
  if ((t = findarg ("-gl", args))) {
    gl_isnull = 0;
    gl = getgldouble (t);
  }
  if ((t = findarg ("-lw", args))) {
    lw_isnull = 0;
    lw = getposdouble (t);
  }
  if ((t = findarg ("-gh", args))) {
    gh_isnull = 0;
    gh = max (mingheight, getposint (t));
  }
  if ((t = findarg ("-gw", args))) {
    gw_isnull = 0;
    gw = max (mingwidth, getposint (t));
  }
  if ((t = findarg ("-th", args))) {
    th_isnull = 0;
    th = max (minttyheight, getposint (t));
  }
  if ((t = findarg ("-tw", args))) {
    tw_isnull = 0;
    tw = max (minttywidth, getposint (t));
  }
  if ((t = findarg ("-cw", args))) {
    cw_isnull = 0;
    cw = max (mingwidth, getposint (t));
  }

  ml_isnull = 1;
  if ((t = findarg ("-ml", args))) {
    double v;
    Dstr uts;
    ml_isnull = 0;
    int i;
    for (i=0; i<strlen(t); i++) {
      char c = t[i];
      if (!((c >= '0' && c <= '9') || c == '+' || c == '-' || c == '.'
      || c == 'e' || c == 'E'))
        break;
    }
    uts = t + i;
    if (sscanf (t, "%lf", &v) != 1) {
      Dstr details ("The offending input was '");
      details += t;
      details += "'.";
      barf (NOT_A_NUMBER, details);
    }
    ml = PredictionValue (PredictionValue::Unit (uts), v);
  }
}

CommandLineSettings::~CommandLineSettings () {
  while (l) {
    struct lnode *templ = l;
    l = l->next;
    delete templ;
  }
}

char *
CommandLineSettings::findarg (char *arg, CommandLineSettings::arglist *args) {
  while (args) {
    if (args->svitch == arg)
      return args->arg.aschar();
    args = args->next;
  }
  return NULL;
}

char
*CommandLineSettings::settingsid() {
  return "command line";
}
