/**********************************************************************************************
    Copyright (C) 2018 Oliver Eichler oliver.eichler@gmx.de

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

**********************************************************************************************/

#include "CMainWindow.h"
#include "canvas/CCanvas.h"
#include "helpers/CGdalFile.h"

#include <gdal_priv.h>
#include <ogr_spatialref.h>

#include <QtWidgets>

CGdalFile::CGdalFile()
{
}


void CGdalFile::unload()
{
    isValid = false;
    if(dataset != nullptr)
    {
        GDALClose(dataset);
    }
    dataset = nullptr;
}


void CGdalFile::load(const QString& filename)
{
    CCanvas * canvas = CMainWindow::self().getCanvas();

    dataset = (GDALDataset*)GDALOpenShared(filename.toUtf8(),GA_ReadOnly);

    if(nullptr == dataset)
    {
        QMessageBox::warning(canvas, tr("Error..."), tr("Failed to load file: %1").arg(filename));
        return;
    }

    projcs = dataset->GetProjectionRef();
//    qDebug() << projcs;

    GDALRasterBand *pBand = dataset->GetRasterBand(1);

    if(nullptr == pBand)
    {
        GDALClose(dataset);
        dataset = nullptr;
        QMessageBox::warning(canvas, tr("Error..."), tr("Failed to load file: %1").arg(filename));
        return;
    }
    hasOverviews = pBand->GetOverviewCount();
    qDebug() << "hasOverviews" << hasOverviews;

    // ------- setup color table ---------
    rasterBandCount = dataset->GetRasterCount();
    if(rasterBandCount == 1)
    {
        if(pBand->GetColorInterpretation() ==  GCI_PaletteIndex )
        {
            GDALColorTable * pct = pBand->GetColorTable();
            for(int i=0; i < pct->GetColorEntryCount(); ++i)
            {
                const GDALColorEntry& e = *pct->GetColorEntry(i);
                colortable << qRgba(e.c1, e.c2, e.c3, e.c4);
            }
        }
        else if(pBand->GetColorInterpretation() ==  GCI_GrayIndex )
        {
            for(int i=0; i < 256; ++i)
            {
                colortable << qRgba(i, i, i, 255);
            }
        }
        else
        {
            GDALClose(dataset);
            dataset = nullptr;
            QMessageBox::warning(canvas, tr("Error..."), tr("File must be 8 bit palette or gray indexed."));
            return;
        }

        int success = 0;
        qreal idx = pBand->GetNoDataValue(&success);
        if(success)
        {
            if((idx >= 0) && (idx < colortable.size()))
            {
                QColor tmp(colortable[idx]);
                tmp.setAlpha(0);
                colortable[idx] = tmp.rgba();
                hasNoData = idx;
            }
            else
            {
                qDebug() << "Index for no data value is out of bound";
                return;
            }
        }
    }
    qDebug() << "hasNoData" << hasNoData;

    xsize_px = dataset->GetRasterXSize();
    ysize_px = dataset->GetRasterYSize();

    qreal adfGeoTransform[6];
    dataset->GetGeoTransform( adfGeoTransform );

    xscale  = adfGeoTransform[1];
    yscale  = adfGeoTransform[5];
    xrot    = adfGeoTransform[4];
    yrot    = adfGeoTransform[2];

    trFwd = QTransform();
    trFwd.translate(adfGeoTransform[0], adfGeoTransform[3]);
    trFwd.scale(adfGeoTransform[1], adfGeoTransform[5]);

    if(adfGeoTransform[4] != 0.0)
    {
        trFwd.rotate(qAtan(adfGeoTransform[2]/adfGeoTransform[4]));
    }

    trInv = trFwd.inverted();

    QPointF pt1(0,0);
    pt1 = trFwd.map(pt1);
    pt1 = trInv.map(pt1);

    isValid = true;
}

QString CGdalFile::getInfo() const
{
    QString str;
    QTextStream out(&str);

    if(projcs.isEmpty())
    {
        out << "no projection" << endl;
    }
    else
    {
        char str[1025] = {0};
        strncpy(str, projcs.toLatin1().data(), sizeof(str) - 1);
        OGRSpatialReference oSRS;
        char *wkt = str;
        oSRS.importFromWkt(&wkt);

        char *proj4 = nullptr;
        oSRS.exportToProj4(&proj4);
        out << proj4 << endl;
        free(proj4);

        out << "xscale: " << xscale << "px/m\tyscale: " << yscale << "px/m" << endl;
    }



    out << "num. bands:\t" << rasterBandCount << " ";
    switch(rasterBandCount)
    {
    case 1:
        out << tr("(color table)");
        break;

    case 3:
        out << tr("(RGB)");
        break;

    case 4:
        out << tr("(RGBA)");
        break;

    default:
        out << tr("(unknown)");
    }

    out << endl;

    out << "has overviews:\t" << hasOverviews << endl;

    if((rasterBandCount != 4) && (hasNoData != -1))
    {
        out << "has no data:\t" << hasNoData << endl;
    }


    return str;
}
