#ifndef _RHEO_FIELD_ELEMENT_H
#define _RHEO_FIELD_ELEMENT_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// for checking the spectral convergence exp(-ak)
// on the reference element
//
// author: Pierre.Saramito@imag.fr
//
// date: 1 october 2017
//
#include "rheolef/skit.h"
#include "rheolef/quadrature.h"
#include "rheolef/basis.h"
#include "equispaced.icc"

namespace rheolef {
// ---------------------------------------------------
struct space_element {
// ---------------------------------------------------
  space_element (std::string name, reference_element hat_K)
   : _b(name), _hat_K(hat_K) {}
  space_element (const basis& b, reference_element hat_K)
   : _b(b), _hat_K(hat_K) {}
// data:
  basis          _b;
  reference_element _hat_K;
};
// ---------------------------------------------------
struct field_element {
// ---------------------------------------------------
  field_element (const space_element& V)
   : _V(V), _dof(), _phi()
  {
    _dof.resize (_V._b.size (_V._hat_K));
  }
  template <class Function> 
  void interpolate (const Function& f)
  {
    _V._b.compute_dof (_V._hat_K, f, _dof);
  }
  Float operator() (const point& hat_x) const
  {
    _V._b.eval(_V._hat_K, hat_x, _phi);
    arma::Mat<Float> value = trans(_dof)*_phi;
    return value(0,0);
  }
// data:
  space_element             _V;
  arma::Col<Float>          _dof;
  mutable arma::Col<Float>  _phi;
};
// ---------------------------------------------------
// error_l2
// ---------------------------------------------------
template <class Function>
Float
error_l2 (const field_element& uk, const Function& u, size_t qorder = size_t(0)) {
  if (qorder == size_t(0)) qorder = max(size_t(5), 2*uk._V._b.degree() + 2);
  quadrature_option qopt;
  qopt.set_order (qorder);;
  quadrature<Float> quad (qopt);
  Float err = 0;
  for (typename quadrature<Float>::const_iterator iter_q = quad.begin(uk._V._hat_K),
        last_q = quad.end(uk._V._hat_K); iter_q != last_q; iter_q++) {
    const point& hat_xq = (*iter_q).x;
    Float        hat_wq = (*iter_q).w;
    err += sqr(uk(hat_xq) - u(hat_xq))*hat_wq;
  }
  return sqrt(err/measure(uk._V._hat_K));
}
// ---------------------------------------------------
// error_linf
// ---------------------------------------------------
template <class Function>
Float
error_linf (const field_element& uk, const Function& u, size_t nsub = 0) {
  if (nsub == 0) nsub = max(size_t(5), 2*uk._V._b.degree() + 2);
  std::vector<point_basic<Float> >  hat_xnod;
  pointset_lagrange_equispaced (uk._V._hat_K, nsub, hat_xnod);
  Float err = 0;
  for (size_t loc_inod = 0, loc_nnod = hat_xnod.size(); loc_inod < loc_nnod; ++loc_inod) {
    err = max (err, abs(uk(hat_xnod[loc_inod]) - u(hat_xnod[loc_inod])));
  }
  return err;
}

} // namespace rheolef
#endif // _RHEO_FIELD_ELEMENT_H
