/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* writerperfect
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2006 Ariya Hidayat (ariya@kde.org)
 * Copyright (C) 2006-2007 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

#include <limits>
#include <stdio.h>
#include <string.h>

#ifdef ENABLE_EOT
extern "C"
{
#include <libeot/libeot.h>
}
#endif

#include <libmspub/libmspub.h>
#include <libodfgen/libodfgen.hxx>

#include "OutputFileHelper.hxx"
#include "StringDocumentHandler.hxx"
#include "UsageHelper.hxx"

#define TOOLNAME "pub2odg"

class OdgOutputFileHelper : public OutputFileHelper
{
public:
	OdgOutputFileHelper(const char *outFileName, const char *password) :
		OutputFileHelper(outFileName, password) {}
	~OdgOutputFileHelper() {}

	bool convertDocument(librevenge::RVNGInputStream &input, bool isFlat)
	{
		OdgGenerator collector;
#ifdef ENABLE_EOT
		collector.registerEmbeddedImageHandler("application/vnd.ms-fontobject", &handleEmbeddedEOTFont);
#endif
		StringDocumentHandler stylesHandler, contentHandler, manifestHandler, settingsHandler;
		if (isFlat)
			collector.addDocumentHandler(&contentHandler, ODF_FLAT_XML);
		else
		{
			collector.addDocumentHandler(&contentHandler, ODF_CONTENT_XML);
			collector.addDocumentHandler(&manifestHandler, ODF_MANIFEST_XML);
			collector.addDocumentHandler(&settingsHandler, ODF_SETTINGS_XML);
			collector.addDocumentHandler(&stylesHandler, ODF_STYLES_XML);
		}
		if (!libmspub::MSPUBDocument::parse(&input, &collector))
			return false;
		if (isFlat)
		{
			printf("%s\n", contentHandler.cstr());
			return true;
		}

		const char s_mimetypeStr[] = "application/vnd.oasis.opendocument.graphics";
		if (!writeChildFile("mimetype", s_mimetypeStr, (char)0) ||
		        !writeChildFile("META-INF/manifest.xml", manifestHandler.cstr()) ||
		        !writeChildFile("content.xml", contentHandler.cstr()) ||
		        !writeChildFile("settings.xml", settingsHandler.cstr()) ||
		        !writeChildFile("styles.xml", stylesHandler.cstr()))
			return false;

		librevenge::RVNGStringVector objects=collector.getObjectNames();
		for (unsigned i=0; i<objects.size(); ++i)
		{
			StringDocumentHandler objectHandler;
			if (collector.getObjectContent(objects[i], &objectHandler))
				writeChildFile(objects[i].cstr(), objectHandler.cstr());
		}
		return true;
	}
	bool isSupportedFormat(librevenge::RVNGInputStream &input)
	{
		return libmspub::MSPUBDocument::isSupported(&input);
	}
private:

#ifdef ENABLE_EOT
	static bool handleEmbeddedEOTFont(const librevenge::RVNGBinaryData &input, librevenge::RVNGBinaryData &output)
	{
		EOTMetadata metadata;
		uint8_t *ttf = 0;
		unsigned ttfSize = 0;

		if (input.size() > std::numeric_limits<unsigned>::max()) // can't convert
			return false;

		const EOTError err = EOT2ttf_buffer(input.getDataBuffer(), unsigned(input.size()), &metadata, &ttf, &ttfSize);
		const bool success = err == EOT_SUCCESS;

		if (success)
		{
			output.clear();
			output.append(ttf, ttfSize);
		}

		EOTfreeMetadata(&metadata);
		EOTfreeBuffer(ttf);

		return success;
	}
#endif
};

int printUsage(const char *name)
{
	UsageHelper usage(name, "converts MS Publisher documents to ODF.");
	usage.addToDescription("If OUTPUT is omitted, the result is printed as Flat ODF to standard output.\n");
	usage.addStdoutOption();
	return usage.printUsage();
}

int main(int argc, char *argv[])
{
	if (argc < 2)
		return printUsage(TOOLNAME);

	char *szInputFile = 0;
	char *szOutFile = 0;
	bool stdOutput = false;

	for (int i = 1; i < argc; i++)
	{
		if (!strcmp(argv[i], "--stdout"))
			stdOutput = true;
		else if (!strcmp(argv[i], "--help"))
			return printUsage(TOOLNAME);
		else if (!strcmp(argv[i], "--version"))
			return UsageHelper::printVersion(TOOLNAME);
		else if (!szInputFile && strncmp(argv[i], "--", 2))
			szInputFile = argv[i];
		else if (szInputFile && !szOutFile && strncmp(argv[i], "--", 2))
			szOutFile = argv[i];
		else
			return printUsage(TOOLNAME);
	}

	if (!szInputFile)
		return printUsage(TOOLNAME);

	if (szOutFile && stdOutput)
		szOutFile = 0;

	OdgOutputFileHelper helper(szOutFile, 0);
	librevenge::RVNGFileStream input(szInputFile);
	if (!helper.isSupportedFormat(input))
	{
		fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Microsoft Publisher Document.\n");
		return 1;
	}

	if (!helper.convertDocument(input, szOutFile==0))
	{
		fprintf(stderr, "ERROR : Couldn't write convert the document\n");
		return 1;
	}

	return 0;
}

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
