/*----------------------------------------------------------------------------

   libtunepimp -- The MusicBrainz tagging library.  
                  Let a thousand taggers bloom!
   
   Copyright (C) Robert Kaye 2003
   
   This file is part of libtunepimp.

   libtunepimp is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   libtunepimp is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with libtunepimp; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   $Id: ov_trm.cpp,v 1.9 2004/03/09 23:27:55 robert Exp $

----------------------------------------------------------------------------*/
#include "../config.h"
#ifdef HAVE_OGGVORBIS

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ov_trm.h"

#include <musicbrainz/mb_c.h>

#include "tunepimp.h"
#include <vorbis/vorbisfile.h>

const int decodeSize = 8192;

/* The callback functions were used since the ov_open call simply crashed and burned */
size_t ReadFunc(void *ptr, size_t size, size_t nmemb, void *datasource)
{
	return fread(ptr, size, nmemb, (FILE *)datasource);
}

int SeekFunc(void *datasource, ogg_int64_t offset, int whence)
{
	return fseek((FILE *)datasource, (int)offset, whence);
}


int CloseFunc(void *datasource)
{
	return fclose((FILE *)datasource);
}

long TellFunc(void *datasource)
{
	return ftell((FILE *)datasource);
}

TRMResult TRMGeneratorOggVorbis::generate(const string &fileName, string &trmsig,
                                          unsigned long &durationArg)
{
   OggVorbis_File   vf;
   vorbis_info     *vi;
   int              section, ret = 0, count = 0;
   char            *buffer;
   FILE            *in;
   trm_t            trm;
   bool             done;
   char             sig[17];
   ov_callbacks     callbacks;
   ogg_int64_t      duration;
   char             ascii_sig[37];
   TRMResult        result = eOtherError;
   string           proxyServer;
   short            proxyPort;

   in = fopen(fileName.c_str(), "rb");
   if (in == NULL)
      return eFileNotFound;

   trm = trm_New();

   tunepimp->getProxy(proxyServer, proxyPort);
   if (proxyServer.size() > 0 && proxyPort != 0)
   	 trm_SetProxy(trm, (char *)proxyServer.c_str(), proxyPort);

   callbacks.read_func = ReadFunc;
   callbacks.seek_func = SeekFunc;
   callbacks.close_func = CloseFunc;
   callbacks.tell_func = TellFunc;

   memset(&vf, 0, sizeof(vf));
   if (ov_open_callbacks(in, &vf, NULL, 0, callbacks) < 0)
       return eDecodeError;

   vi = ov_info(&vf, -1);
   trm_SetPCMDataInfo(trm, vi->rate, vi->channels, 16);

   if (durationArg == 0)
   {
       duration = ov_pcm_total(&vf, -1);
       if (duration > 0)
       {
          durationArg = (unsigned long)((duration * 1000) / vi->rate);
          duration = duration / vi->rate;
       }
   }
   else
       duration = durationArg / 1000;

   if (duration > 0)
   {
       trm_SetSongLength(trm, (long)duration);
   }

   buffer = new char[decodeSize * 2];
   done = false;
   while(!done)
   {
      ret = ov_read(&vf, buffer, decodeSize, 0, 2, 1, &section);
      if (ret == 0)
      {
          break;
      }
      if (ret < 0)
      {
          continue;
      }

      count += ret;
      done = (bool)trm_GenerateSignature(trm, buffer, ret);
   }

   ov_clear(&vf);

   if (ret >= 0)
   {
      ret = trm_FinalizeSignature(trm, sig, NULL);
      result = (ret == 0) ? eOk : eCannotConnect;
   }
   else
	  result = eDecodeError;

   trm_ConvertSigToASCII (trm, sig, ascii_sig);
   trmsig = string(ascii_sig);

   trm_Delete(trm);
   delete buffer;

   return result;
}

#endif
