//
// File:        ImplSource.java
// Package:     gov.llnl.babel.backend.c
// Revision:    @(#) $Revision: 4454 $
// Date:        $Date: 2005-03-22 14:18:30 -0800 (Tue, 22 Mar 2005) $
// Description: generate C server code to be edited by the developer
//
// Copyright (c) 2000-2004, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.c;

import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.CodeSplicer;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.c.C;
import gov.llnl.babel.backend.c.StubSource;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import java.util.Iterator;

/**
 * <p>
 * Class <code>ImplSource</code> generates a C implementation source to a
 * language writer output stream.  The constructor takes a language writer
 * stream and method <code>generateCode</code> generates the implementation
 * header file for the specified symbol to the output stream.  The language
 * writer stream is not closed by this object.
 * </p>
 * <p>
 * The implementation source contains definitions of the non-abstract
 * methods. Unlike most files generated by <code>babel</code>, this file
 * is intended to be edited by the developer. The developer needs to insert
 * the implementation of the class in this file or insert calls to something
 * that implements the class.
 * </p>
 * <p>
 * The code splicer is used to maintain code the developer has written when
 * overwriting an implementation source file. The splicer stores code
 * fragments from the previous file that can be insert into the new file.
 * </p>
 */
public class ImplSource {
  /**
   * The splicer is used when replacing an existing file. Sections from the
   * previous file are stored in the splicer and can be grafted back into
   * the file being generated.
   */
  private CodeSplicer        d_splicer;

  /**
   * The writer is the output stream for the new file. The content of the
   * new file is writtern to this writer.
   */
  private LanguageWriterForC d_writer;

  /**
   * Create a <code>ImplSource</code> object that will write symbol
   * information to the provided output language writer stream.  The
   * code splicer object may be used to save state between the old
   * source file and the new source file.  If the code splicer object
   * is null, then no state is transferred to the new source file.
   *
   * @param writer  the output writer for the new implementation source
   *                file. This writer will not be closed by this method.
   * @param splicer If not <code>null</code>, this splicer contains
   *                code fragments from a implementation source file
   *                that is being overwritten. The new implementation
   *                will try to incorporate those fragments into the
   *                new file.
   */
  public ImplSource(LanguageWriterForC writer, CodeSplicer splicer) {
    d_splicer = splicer;
    d_writer  = writer;
  }

  /**
   * This is a convenience utility function that writes the symbol
   * source information into the provided language writer output stream.
   * The output stream is not closed on exit.
   *
   * @param cls     the class whose implementation source is to be created.
   * @param writer  the output writer with which the new file is created.
   * @param splicer If not <code>null</code>, the splicer contains
   *                code fragments from a previous implementation source
   *                file that is being overwritten. The new implementation
   *                will incorporate those fragments into the new file.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   *
   */
  public static void generateCode(Class cls, LanguageWriterForC writer,
                                  CodeSplicer splicer)
    throws CodeGenerationException
  {
    ImplSource source = new ImplSource(writer, splicer);
    source.generateCode(cls);
  }

  /**
   * Start the source file with a large banner.
   *
   * @param cls     the class whose implementation source is to be created.
   */
  private void writePrologue(Class cls) {
    final SymbolID id = cls.getSymbolID();
    String header = C.getImplSourceFile(id);
    d_writer.writeBanner(cls, header, true, CodeConstants.C_DESC_IMPL_PREFIX 
      + id.getFullName());
    d_writer.beginBlockComment(false);
    d_writer.println("DEVELOPERS ARE EXPECTED TO PROVIDE IMPLEMENTATIONS");
    d_writer.println("FOR THE FOLLOWING METHODS BETWEEN SPLICER PAIRS.");
    d_writer.endBlockComment(false);
    d_writer.writeComment(cls, false);
  }

  /**
   * Output the include for the main header file.  Include a code
   * splicer block where the developer can include additional header files.
   *
   * @param cls     the class whose implementation source is to be created.
   */
  private void writeIncludeSection(Class cls) {
    final SymbolID id = cls.getSymbolID();
    String includes = id.getFullName() + "." + "_includes";
    d_writer.generateInclude(C.getImplHeaderFile(id), false);
    d_writer.println();
    //If this class overwrites parent methods, we need to generate
    //ways to access those super methods.

    d_splicer.splice(includes, d_writer, "includes and arbitrary code");
    d_writer.println();
  }

  /**
   * Write the body of an implementation method using code from the splicer
   * if available; otherwise, write a splicer block for the developer to
   * replace.
   *
   * @param id     the name of the symbol who owns the method.
   * @param name   the formal name of the method.
   * @param desc   a terse human grokable name for the method.
   */
  private void writeMethodBody(SymbolID id, String name, String desc) {
    String splicerTag = id.getFullName() + '.' + name;
    d_splicer.splice(splicerTag, d_writer, desc + " method");
  }

  /**
   * Write a method definition with a body from the previous implementation
   * or a code splicer block where the developer can insert the
   * implementation.
   *
   * @param m      the method description
   * @param id     the name of the symbol who owns the method
   * @param desc   a human grokable description of the method
   * @param suffix the method (interceptor) suffix, "" if none
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void writeMethodShell(Method m, SymbolID id, String desc,
                                String suffix)
    throws CodeGenerationException
  {
    boolean  isOrig    = suffix.equals("");
    
    String name        = m.getLongMethodName() + suffix;
    String method_name = C.getMethodImplName(id, name);

    if (isOrig) {
      d_writer.writeComment(m, false);
    } else {
      d_writer.writeComment(method_name + ": Interceptor implementation.", 
                            false);
    }

    d_writer.printlnUnformatted("#undef __FUNC__");
    d_writer.printlnUnformatted("#define __FUNC__ \"" + method_name + "\"");
    d_writer.println();
    ImplHeader.writeMethodSignature(d_writer, m, suffix, Utilities.s_self,
                                    false, false, true, id);
    d_writer.println();
    d_writer.backTab();
    d_writer.println("{");
    d_writer.tab();
    writeMethodBody(id, name, desc);
    d_writer.backTab();
    d_writer.println("}");
  }

  /**
   * Write the implementation shell for the constructor or destructor. These
   * methods are implicitly defined.
   *
   * @param id         the name of the <code>Class</code> whose
   *                   constructor/destructor is being written.
   * @param methodName the name of the method being written.
   * @param methodDesc a terse human grokable description of the method
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void writeBuiltinMethods(Class cls) throws CodeGenerationException {
    final SymbolID id = cls.getSymbolID();
    StubSource.generateSupers(cls.getOverwrittenClassMethods(), cls, d_writer);
    writeMethodShell(IOR.getBuiltinMethod(IOR.LOAD, id), id, 
                     "static class initializer","");
    writeMethodShell(IOR.getBuiltinMethod(IOR.CONSTRUCTOR, id), id, 
                     "constructor","");
    d_writer.println();
    writeMethodShell(IOR.getBuiltinMethod(IOR.DESTRUCTOR, id), id, "destructor",
                     "");
    if (IOR.supportAssertions(cls)) {
      if (cls.hasStaticMethod(true)) {
        d_writer.println();
        writeMethodShell(
          IOR.getBuiltinMethod(IOR.CHECK_ERROR,id,true), id, "check_error" 
            + IOR.s_static_suffix, "");
      }
      d_writer.println();
      writeMethodShell(IOR.getBuiltinMethod(IOR.CHECK_ERROR, id), id, 
                       "check_error", "");
    }
  }

  /**
   * Write definitions for the non-<code>abstract</code> methods of the
   * class.
   *
   * @param cls      the class whose implementation source is to be
   *                 created.
   * @param isStatic <code>true</code> means write only static methods;
   *                 <code>false</code> means write only non-static
   *                 methods.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void writeMethods(Class cls, boolean isStatic)
    throws CodeGenerationException
  {
    final SymbolID id = cls.getSymbolID();
    Iterator i = null;
    if (isStatic) {
      i = cls.getStaticMethods(false).iterator();
    } else {
      i = cls.getNonstaticMethods(false).iterator();
    }

    while (i.hasNext()) {
      Method m = (Method) i.next();
      if (!m.isAbstract()) {
        d_writer.println();
        if (IOR.supportInterceptors(cls)) {
          writeMethodShell(m, id, m.getLongMethodName(), "_pre");
          d_writer.println();
          writeMethodShell(m, id, m.getLongMethodName(), "");
          d_writer.println();
          writeMethodShell(m, id, m.getLongMethodName(), "_post");
        } else {
          writeMethodShell(m, id, m.getLongMethodName(), "");
        }
      }
    }
  }

  /**
   * Write C implementation source information for the provided class
   * to the language writer output stream provided in the constructor.
   * This method does not close the writer output stream.
   *
   * @param cls     the class whose implementation source is to be created.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  public void generateCode(Class cls) throws CodeGenerationException {
    final SymbolID id = cls.getSymbolID();
    writePrologue(cls);
    writeIncludeSection(cls);
    writeBuiltinMethods(cls);
    writeMethods(cls, true);    // static methods
    writeMethods(cls, false);   // object methods

    if (d_splicer.hasUnusedSymbolEdits()) {
      d_writer.println();
      d_writer.printlnUnformatted("#error File has unused splicer blocks.");
      d_writer.beginBlockComment(true);
      d_writer.println(CodeConstants.C_BEGIN_UNREFERENCED_METHODS);
      d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT1);
      d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT2);
      d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT3);
      d_writer.endBlockComment(true);
      d_splicer.outputUnusedSymbolEdits(d_writer.getPrintWriter());
      d_writer.writeCommentLine(CodeConstants.C_END_UNREFERENCED_METHODS);
    }
  }
}
