//
// File:        PythonClientHeader.java
// Package:     gov.llnl.babel.backend.python
// Revision:    @(#) $Revision: 4434 $
// Date:        $Date: 2005-03-17 09:05:29 -0800 (Thu, 17 Mar 2005) $
// Description: Write Python extension header file for a BABEL extendable
// 
// This is typically directed by GenPythonClient.
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.python;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.python.Python;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Interface;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.backend.IOR;

/**
 * Create and write a header for a Python C extension class to wrap a 
 * BABEL extendable in a Python object. The header has to expose a 
 * function to create a wrapped IOR, a function to check if a 
 * <code>PyObject</code> is an instance of this extension type, and
 * an import macro.
 */
public class PythonClientHeader {
  private Extendable d_ext = null;
  private LanguageWriterForC d_lw = null;
  private int d_numMethods;
  private String [] d_methodNames = null;
  private String [] d_methodPrototypes = null;
  private String [] d_methodReturn = null;
  private boolean d_isBaseInterface;


  /**
   * Create an object capable of generating the header file for a
   * BABEL extendable.
   *
   * @param ext   an interface or class symbol that needs a header
   *              file for a Python C extension class.
   */
  public PythonClientHeader(Extendable ext)
  {
    final SymbolID id = ext.getSymbolID();
    final String iorType = IOR.getObjectName(id);
    final String arrayType = IOR.getArrayName(id);
    d_ext = ext;
    d_isBaseInterface = BabelConfiguration.getBaseInterface().
      equals(ext.getFullName());
    d_numMethods = d_isBaseInterface ? 10 : 9;
    
    d_methodNames = new String[d_numMethods];
    d_methodPrototypes = new String[d_numMethods];
    d_methodReturn = new String[d_numMethods];

    d_methodNames[0] = Python.getExtendableWrapper(d_ext);
    d_methodReturn[0] = "PyObject *";
    d_methodPrototypes[0] = "(" + iorType + " *sidlobj)";

    d_methodNames[1] = Python.getExtendableConverter(d_ext);
    d_methodReturn[1] = "int";
    d_methodPrototypes[1] = "(PyObject *obj, " + iorType + " **sidlobj)";

    d_methodNames[2] = Python.getBorrowArrayFromPython(new Type(id));
    d_methodReturn[2] = "int";
    d_methodPrototypes[2] = "(PyObject *obj, " + arrayType + " **sidlarray)";

    d_methodNames[3] = Python.getBorrowArrayFromSIDL(new Type(id));
    d_methodReturn[3] = "PyObject *";
    d_methodPrototypes[3] = "(struct sidl__array *sidlarray)";

    d_methodNames[4] = Python.getExtendableBorrow(d_ext);
    d_methodReturn[4] = "PyObject *";
    d_methodPrototypes[4] = "(" + iorType + " *sidlobj)";

    d_methodNames[5] = Python.getExtendableDeref(d_ext);
    d_methodReturn[5] = "void";
    d_methodPrototypes[5] = "(" + iorType + " *sidlobj)";

    d_methodNames[6] = Python.getExtendableNewRef(d_ext);
    d_methodReturn[6] = "PyObject *";
    d_methodPrototypes[6] = "(" + iorType + " *sidlobj)";

    d_methodNames[7] = Python.getExtendableAddRef(d_ext);
    d_methodReturn[7] = "void";
    d_methodPrototypes[7] = "(" + iorType + " *sidlobj)";

    d_methodNames[8] = Python.getExtendableType(d_ext);
    d_methodReturn[8] = "PyTypeObject *";
    d_methodPrototypes[8] = "(void)";

    if (d_isBaseInterface) {
      d_methodNames[9] = Python.getBorrowArrayFromSIDL(null);
      d_methodReturn[9] = "PyObject *";
      d_methodPrototypes[9] = "( struct sidl__array *sidlarray)";
    }
  }

  /**
   * Generate the header file for the extendable with which this object was
   * created.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception for problems during the code
   *    generation phase.
   */
  public synchronized void generateCode() 
    throws CodeGenerationException 
  {
    String guard = Python.getIncludeGuard(d_ext, "MODULE");
    try {
      d_lw = Python.createCHeader
        (d_ext, "Module", 
         "expose a constructor for the Python wrapper");
      explainExtHeader();
      d_lw.openHeaderGuard(guard);
      d_lw.printlnUnformatted("#include <Python.h>");
      d_lw.printlnUnformatted("#include \"babel_config.h\"");
      d_lw.println();
      d_lw.openCxxExtern();
      d_lw.println("struct sidl__array;");
      addSharedDefs();
      d_lw.printUnformatted("#ifdef ");
      d_lw.printlnUnformatted(Python.getInternalGuard(d_ext));
      addInternalDecls();
      d_lw.printlnUnformatted("#else");
      addClientMacroDefs();
      d_lw.printlnUnformatted("#endif");
      d_lw.println();
      d_lw.closeCxxExtern();
      d_lw.println();
      d_lw.closeHeaderGuard();
    }
    finally {
      if (d_lw != null) {
        d_lw.close();
        d_lw = null;
      }
    }
  }

  private boolean isException() 
    throws CodeGenerationException
  {
    final Symbol symClass     = 
      Utilities.lookupSymbol(BabelConfiguration.getBaseExceptionClass());
    final Symbol symInterface = 
      Utilities.lookupSymbol(BabelConfiguration.getBaseExceptionInterface());
    SymbolID symCID = null;
    SymbolID symIID = null;
    if (symClass != null) {
      symCID = symClass.getSymbolID();
    }
    if (symInterface != null) {
      symIID = symInterface.getSymbolID();
    }
    if ((symCID != null) && (symIID != null)) {
      if (d_ext instanceof Class) {
        return (  (symCID.equals(d_ext.getSymbolID()))
               || d_ext.hasParentInterface(symIID, true)
               || ((Class)d_ext).hasParentClass(symCID, true)  );
      } else if (d_ext instanceof Interface) {
        return (  (symIID.equals(d_ext.getSymbolID()))
               || d_ext.hasParentInterface(symIID, true)  );
      }
    }
    return false;
  }

  private void addClientMacroDefs()
    throws CodeGenerationException
  {
    String apivar = Python.getAPIVarName(d_ext);
    d_lw.println();
    d_lw.print("static void **");
    d_lw.print(apivar);
    d_lw.println(";");
    d_lw.println();
    d_lw.disableLineBreak();
    for(int i = 0; i < d_numMethods; ++i) {
      d_lw.print("#define ");
      d_lw.print(d_methodNames[i]);
      d_lw.println(" \\");
      d_lw.print("  (*((");
      d_lw.print(d_methodNames[i]);
      d_lw.println("_RETURN (*) \\");
      d_lw.print("  ");
      d_lw.print(d_methodNames[i]);
      d_lw.println("_PROTO) \\");
      d_lw.print("  (");
      d_lw.print(apivar);
      d_lw.println(" \\");
      d_lw.print("  [");
      d_lw.print(d_methodNames[i]);
      d_lw.println("_NUM])))");
      d_lw.println();
    }
    if (isException()) {
      String exType = Python.getExceptionType(d_ext);
      d_lw.print("#define ");
      d_lw.print(exType);
      d_lw.println(" \\");
      d_lw.print("  ((PyObject *)(");
      d_lw.print(apivar);
      d_lw.print("[");
      d_lw.print(exType);
      d_lw.println("_NUM]))");
      d_lw.println();
    }
    d_lw.print("#define ");
    d_lw.print(Python.getExtendableImport(d_ext));
    d_lw.println("() \\");
    d_lw.println("{ \\");
    d_lw.print("  PyObject *module = PyImport_ImportModule(\"");
    d_lw.print(d_ext.getFullName());
    d_lw.println("\"); \\");
    d_lw.println("  if (module != NULL) { \\");
    d_lw.println("    PyObject *module_dict = PyModule_GetDict(module); \\");
    d_lw.println("    PyObject *c_api_object = \\");
    d_lw.println("      PyDict_GetItemString(module_dict, \"_C_API\"); \\");
    d_lw.println("    if (c_api_object && PyCObject_Check(c_api_object)) { \\");
    d_lw.print("      ");
    d_lw.print(apivar);
    d_lw.println(" = \\");
    d_lw.println("        (void **)PyCObject_AsVoidPtr(c_api_object); \\");
    d_lw.println("    } \\");
    d_lw.printUnformatted("    else { fprintf(stderr, \"babel: ");
    d_lw.printUnformatted(Python.getExtendableImport(d_ext));
    d_lw.printlnUnformatted(" failed to lookup _C_API (%p %p %s).\\n\", c_api_object, c_api_object ? c_api_object->ob_type : NULL, c_api_object ? c_api_object->ob_type->tp_name : \"\"); }\\");
    d_lw.println("    Py_DECREF(module); \\");
    d_lw.printUnformatted("  } else { fprintf(stderr, \"babel: ");
    d_lw.printUnformatted(Python.getExtendableImport(d_ext));
    d_lw.printlnUnformatted(" failed to import its module.\\n\"); }\\");
    d_lw.println("}");
    d_lw.enableLineBreak();
    d_lw.println();
  }

  private void addInternalDecls()
    throws CodeGenerationException
  {
    d_lw.println();
    d_lw.beginBlockComment(false);
    d_lw.println("This declaration is not for clients.");
    d_lw.endBlockComment(false);
    for(int i = 0 ; i < d_numMethods; ++i) {
      d_lw.print("static ");
      d_lw.print(d_methodNames[i]);
      d_lw.println("_RETURN");
      d_lw.println(d_methodNames[i]);
      d_lw.print(d_methodNames[i]);
      d_lw.println("_PROTO;");
      d_lw.println();
    }
    if (isException()) {
      String exType = Python.getExceptionType(d_ext);
      d_lw.println("static PyObject *");
      d_lw.print(exType);
      d_lw.println(";");
      d_lw.println();
    }
  }

  private void addSharedDefs()
    throws CodeGenerationException 
  {
    int i;
    d_lw.println();
    d_lw.writeCommentLine("Forward declaration of IOR structure");
    d_lw.print(IOR.getObjectName(d_ext.getSymbolID()));
    d_lw.println(";");
    d_lw.print(IOR.getArrayName(d_ext.getSymbolID()));
    d_lw.println(";");
    d_lw.println();
    d_lw.disableLineBreak();
    for(i = 0; i < d_numMethods; ++i) {
      d_lw.print("#define ");
      d_lw.print(d_methodNames[i]);
      d_lw.print("_NUM ");
      d_lw.println(Integer.toString(i));
      d_lw.print("#define ");
      d_lw.print(d_methodNames[i]);
      d_lw.print("_RETURN ");
      d_lw.println(d_methodReturn[i]);
      d_lw.print("#define ");
      d_lw.print(d_methodNames[i]);
      d_lw.print("_PROTO ");
      d_lw.println(d_methodPrototypes[i]);
      d_lw.println();
    }
    if (isException()) {
      d_lw.print("#define ");
      d_lw.print(Python.getExceptionType(d_ext));
      d_lw.print("_NUM ");
      d_lw.println(Integer.toString(i++));
      d_lw.println();
    }
    d_lw.print("#define ");
    d_lw.print(Python.getAPIVarName(d_ext));
    d_lw.print("_NUM ");
    d_lw.println(Integer.toString(i));
    d_lw.enableLineBreak();
    d_lw.println();
  }

   private void explainExtHeader()
     throws CodeGenerationException
   {
      d_lw.beginBlockComment(false);
      d_lw.println("THIS CODE IS AUTOMATICALLY GENERATED BY THE BABEL");
      d_lw.println("COMPILER. DO NOT EDIT THIS!");
      d_lw.println();
      d_lw.println("External clients need an entry point to wrap a pointer");
      d_lw.print("to an instance of ");
      d_lw.print(d_ext.getFullName());
      d_lw.println(".");
      d_lw.println("This header files defines two methods that such clients");
      d_lw.println("will need.");
      d_lw.print("    ");
      d_lw.println(Python.getExtendableImport(d_ext));
      d_lw.println("        This should be called in the client's init");
      d_lw.println("        module method.");
      d_lw.print("    ");
      d_lw.println(Python.getExtendableWrapper(d_ext));
      d_lw.println("        This will wrap an IOR in a Python object.");
      if (isException()) {
        d_lw.println("This object can be used as an exception. It exports");
        d_lw.println("a Python exception type that may be needed as well.");
        d_lw.print("    ");
        d_lw.println(Python.getExceptionType(d_ext));
        d_lw.println("        A Python exception type corresponding to");
        d_lw.println("        this object type.");
        d_lw.println("Here is the pattern for throwing an exception:");
        d_lw.print("  PyObject *obj = ");
        d_lw.print(Python.getExtendableWrapper(d_ext));
        d_lw.println("(ex);");
        d_lw.print("  PyErr_SetObject(");
        d_lw.print(Python.getExceptionType(d_ext));
        d_lw.println(", obj);");
        d_lw.println("  Py_XDECREF(obj);");
      }
      d_lw.endBlockComment(false);
   }
}
