//
// File:        PythonServerCSource.java
// Package:     gov.llnl.babel.backend.python
// Revision:    @(#) $Revision: 4462 $
// Date:        $Date: 2005-03-23 11:29:24 -0800 (Wed, 23 Mar 2005) $
// Description: Generate skeleton's to link to a Python implementation
// 
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.python;

import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.LevelComparator;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.python.Python;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.BabelConfiguration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * The purpose of this class is to generate the C skeleton code to support
 * sidl objects implemented in Python. The skeleton code makes the link
 * between the independent object representation (IOR) and the Python
 * implementation of the sidl class.
 *
 * The skeleton must map datatypes in C to datatypes in Python. It must
 * also provide C functions to populate the static and object entry
 * point vectors for the IOR.
 */
public class PythonServerCSource {
  /**
   * A writer for the C source file.
   */
  private LanguageWriterForC d_lw            = null;

  private Set                d_includedFiles = new HashSet();

  private Class d_cls = null;

  public PythonServerCSource(Class cls) {
    d_cls = cls;
  }

  private Collection argSymbols() throws CodeGenerationException {
    final String exceptionFundamentalName = BabelConfiguration.
                                                   getBaseExceptionType();
    HashSet references = new HashSet();
    for(Iterator i = d_cls.getMethods(false).iterator(); i.hasNext() ; ){
      Method m = (Method)i.next();
      references.addAll(m.getSymbolReferences());
      if (!m.getThrows().isEmpty()) {
        references.add(Utilities.lookupSymbol(
                                   exceptionFundamentalName).getSymbolID());
      }
    }
    references.remove(d_cls.getSymbolID());
    return references;
  }

  /**
   * Write a comment explaining the contents of the source file to 
   * anyone who might happen to read it.
   */
  private void explainSkelSource() {
    d_lw.beginBlockComment(false);
    d_lw.println("THIS CODE IS AUTOMATICALLY GENERATED BY THE BABEL");
    d_lw.println("COMPILER. DO NOT EDIT THIS!");
    d_lw.println();
    d_lw.println("This file contains skeleton code to provide an interface");
    d_lw.println("between the sidl independent object representation and");
    d_lw.println("the Python implementation of a class.  This translates");
    d_lw.println("method calls to the IOR to method class in Python.");
    d_lw.println("This file is for the sidl type " + 
                 d_cls.getFullName() + ".");
    d_lw.endBlockComment(false);
    d_lw.println();
  }

  private void addInclude(String filename, boolean useGuard) {
    if (!d_includedFiles.contains(filename)) {
      d_includedFiles.add(filename);
      d_lw.generateInclude(filename, useGuard);
    }
  }

  /**
   * Add <code>#include</code> lines for all the system headers and
   * the referenced types.
   */
  private void includeHeaderFiles() throws CodeGenerationException {
    d_lw.printlnUnformatted("#include <Python.h>");
    addInclude(IOR.getHeaderFile(d_cls.getSymbolID()), true);
    addInclude("sidlObjA.h", false);
    addInclude("sidlPyArrays.h", false);
    addInclude("Numeric/arrayobject.h", false);
    addInclude("sidl_header.h", true);
    addInclude("sidl_Python.h", true);
    addInclude("sidl_BaseInterface_Module.h", false);
    addInclude(Python.getCHeaderPath(d_cls, "Module"), false);
    Iterator i = Utilities.convertIdsToSymbols(argSymbols()).iterator();
    while (i.hasNext()) {
      Symbol sym = (Symbol)i.next();
      if (Symbol.ENUM != sym.getSymbolType()) {
        addInclude(Python.getCHeaderPath(sym, "Module"), false);
      }
    }
    d_lw.printlnUnformatted("#include <stdlib.h>");
    d_lw.printlnUnformatted("#include <string.h>");
    d_lw.println();
  }
  
  private void writeMethodSignature(Method m) throws CodeGenerationException {
    SymbolID id = d_cls.getSymbolID();
    Iterator arguments = m.getArgumentList().iterator();
    boolean needComma = false;
    d_lw.println("static " + IOR.getReturnString(m.getReturnType(), true, 
                                                 false));
    d_lw.println(Python.getSkelMethod(id, m) + "(");
    if (!m.isStatic() || arguments.hasNext() || !m.getThrows().isEmpty()) {
      d_lw.tab();
      if (!m.isStatic()) {
        d_lw.print("/* in */ " + IOR.getObjectName(id) + " *self");
        needComma = true;
      }
      while (arguments.hasNext()) {
        Argument arg = (Argument)arguments.next();
        if (needComma) {
          d_lw.println(",");
        }
        d_lw.print("/* " + arg.getModeString() + " */ ");
        d_lw.print(IOR.getArgumentWithFormal(arg, true, false, false));
        needComma = true;
      }
      if (!m.getThrows().isEmpty()) {
        if (needComma) d_lw.println(",");
        d_lw.println(IOR.getExceptionFundamentalType() + " *_exception");
        needComma = true;
      }
    } else {
      /* if no argument put void */
      d_lw.print("void");
    }
    d_lw.println(")");
    if (needComma) {
      d_lw.backTab();
    }
  }

  private void declareLocalVariables() {
    d_lw.println("PyObject *_context;");
    d_lw.println("PyObject *_pfunc;");
    d_lw.println("PyObject *_args;");
    d_lw.println("PyObject *_result;");
  }

  private void getFunction(boolean isStatic, String  methodName) {
    if (isStatic) {
      d_lw.println("if ((_context = PyImport_ImportModule(\"" 
        + d_cls.getFullName() + "_Impl" + "\"))) {");
    } else {
      d_lw.println("if ((_context = (self ? self->d_data : NULL))) {");
    }
    d_lw.tab();
    d_lw.println("if ((_pfunc = PyObject_GetAttrString(_context, \"" 
      + methodName + "\"))) {");
    d_lw.tab();
  }

  private void generateInit() throws CodeGenerationException {
    boolean usesGeneric = false;
    d_lw.println("static void _importModules(void) {");
    d_lw.tab();
    d_lw.println("static int _import = 1;");
    d_lw.println("if (_import) {");
    d_lw.tab();
    d_lw.println("_import = 0;");
    d_lw.println("import_SIDLObjA();");
    d_lw.println("import_SIDLPyArrays();");
    d_lw.println(Python.getExtendableImport(d_cls) + "();");
    for(Iterator i = Utilities.convertIdsToSymbols(argSymbols()).iterator(); 
        i.hasNext(); ) {

      Symbol sym = (Symbol)i.next();
      if (Symbol.ENUM != sym.getSymbolType()) {
        d_lw.println(Python.getExtendableImport(sym) + "();");
      }
    }
    for(Iterator i = d_cls.getMethods(false).iterator(); i.hasNext() ; ){
      Method m = (Method)i.next();
      ArrayList args = m.getArgumentList();
      for(Iterator argIt = args.iterator(); argIt.hasNext();) {
        Argument arg = (Argument)argIt.next();
        Type argType = arg.getType();
        if(argType.getType() == Type.ARRAY && argType.getArrayType() == null)
           usesGeneric = true;
      }
    }
    if(usesGeneric == true)
      d_lw.println(Python.getImport(BabelConfiguration.getBaseInterface())
        + "();");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  private void handleExceptions(Method m, Set throwTypes)
    throws CodeGenerationException
  {
    Object [] exceptions = throwTypes.toArray();
    /*
     * Be sure to use the fundamental base exception type here.
     */
    final String baseType = IOR.getObjectName
      (Utilities.lookupSymbol(BabelConfiguration.getBaseExceptionType()).
                                                                 getSymbolID());
    Arrays.sort(exceptions, new LevelComparator(SymbolTable.getInstance()));
    int i;
    d_lw.println("else {");
    d_lw.tab();
    d_lw.println("if (PyErr_Occurred()) {");
    d_lw.tab();
    d_lw.println("PyObject *_exType, *_exValue, *_exTrace;");
    d_lw.println("PyErr_Fetch(&_exType, &_exValue, &_exTrace);");
    for(i = 0; i < exceptions.length ; ++i){
      SymbolID id = (SymbolID)exceptions[i];
      Symbol sym = Utilities.lookupSymbol(id);
      if (i > 0) {
        d_lw.print("else ");
      }
      d_lw.println("if (PyErr_GivenExceptionMatches(_exType, " + 
                   Python.getExceptionType(sym) + ")) {");
      d_lw.tab();
      d_lw.println(IOR.getObjectName(id) + " *_local_exception;");
      d_lw.println("(void)" + Python.getExtendableConverter(sym) + 
                   "(_exValue , &_local_exception);");
      d_lw.println("if (_local_exception) {");
      d_lw.tab();
      d_lw.println(Python.getExtendableAddRef(sym) + "(_local_exception);");
      d_lw.println("*_exception = (" + baseType + " *)_local_exception;");
      d_lw.backTab();
      d_lw.println("}");
      d_lw.backTab();
      d_lw.println("}");
    }
    d_lw.println("else {");
    d_lw.tab();
    d_lw.println("PyErr_Restore(_exType, _exValue, _exTrace);");
    d_lw.println("_exType = NULL;");
    d_lw.println("_exValue = NULL;");
    d_lw.println("_exTrace = NULL;");
    d_lw.println("PyErr_Print();");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("Py_XDECREF(_exType);");
    d_lw.println("Py_XDECREF(_exValue);");
    d_lw.println("Py_XDECREF(_exTrace);");
    d_lw.println("PyErr_Clear();");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
  }

  private void cleanupReferences(Method m) throws CodeGenerationException {
    Set throwTypes = m.getThrows();
    d_lw.println("Py_DECREF(_result);");
    d_lw.backTab();
    d_lw.println("}");
    if (!throwTypes.isEmpty()) {
      handleExceptions(m, throwTypes);
    }
    d_lw.println("if (PyErr_Occurred()) {");
    d_lw.tab();
    d_lw.println("PyErr_Print();");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("Py_DECREF(_args);");
    d_lw.backTab();
    d_lw.println("} else if (PyErr_Occurred()) {");
    d_lw.tab();
    d_lw.println("PyErr_Print();");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("Py_DECREF(_pfunc);");
    d_lw.backTab();
    d_lw.println("} else if (PyErr_Occurred()) {");
    d_lw.tab();
    d_lw.println("PyErr_Print();");
    d_lw.backTab();
    d_lw.println("}");
    if (m.isStatic()) {
      d_lw.println("Py_DECREF(_context);");
    }
    d_lw.backTab();
    d_lw.println("}");
    if (m.isStatic()) {
      d_lw.println("else if (PyErr_Occurred()) {");
      d_lw.tab();
      d_lw.println("PyErr_Print();");
      d_lw.backTab();
      d_lw.println("}");
    }
  }

  private void declareReturnVariable(Method m) throws CodeGenerationException {
    final Type t = m.getReturnType();
    if (t.getDetailedType() != Type.VOID) {
      d_lw.print(IOR.getReturnString(t) + " " + TranslateArguments.RETURN_VAR);
      if (Utilities.isPointer(t)) {
        d_lw.print(" = NULL");
      } else if (t.getDetailedType() == Type.DCOMPLEX) {
        d_lw.print(" = { 0.0, 0.0 }");
      } else if (t.getDetailedType() == Type.FCOMPLEX) {
        d_lw.print(" = { 0.0, 0.0 }");
      } else {
        d_lw.print(" = (" + IOR.getReturnString(t) + ") 0");
      }
      d_lw.println(";");
    }
  }

  private void initializeOutVariables(Method m) {
    Iterator i = m.getArgumentList().iterator();
    while (i.hasNext()) {
      Argument arg = (Argument)i.next();
      if ((arg.getMode() == Argument.OUT) && 
          Utilities.isPointer(arg.getType())){
        d_lw.println('*' + arg.getFormalName() + " = NULL;");
      }
    }
    if (!m.getThrows().isEmpty()) {
      d_lw.println("*_exception = NULL;");
    }
  }

  private void releaseInOutParameters(Method m) throws CodeGenerationException {
    Extendable ext;
    Iterator i = m.getArgumentList().iterator();
    while (i.hasNext()) {
      Argument arg = (Argument)i.next();
      if (Argument.INOUT == arg.getMode()) {
        final Type t = arg.getType();
        switch (t.getDetailedType()) {
        case Type.STRING:
          d_lw.println("free((void *)*" + arg.getFormalName() + ");");
          d_lw.println("*" + arg.getFormalName() + " = NULL;");
          break;
        case Type.INTERFACE:
        case Type.CLASS:
          ext = (Extendable)
            Utilities.lookupSymbol(t.getSymbolID());
          d_lw.println(Python.getExtendableDeref(ext) + "(*" +
                       arg.getFormalName() + ");");
          d_lw.println("*" + arg.getFormalName() + " = NULL;");
          break;
        case Type.ARRAY:
          if (!t.isRarray()) {
            d_lw.println(Python.getDestroyArray(t.getArrayType()) + 
                         "((struct sidl__array *)(*" + 
                         arg.getFormalName() + "));");
            d_lw.println("*" + arg.getFormalName() + " = NULL;");
          }
          break;
        }
      }
    }
  }

  /**
   * Write the functions in C that will expose a Python method or
   * free function to the IOR.
   * 
   * @exception gov.llnl.babel.backend.CodeGeneration
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void convertMethod(Method m) throws CodeGenerationException {
    TranslateArguments trans = new TranslateArguments(d_lw, m, false, false);
    writeMethodSignature(m);
    trans.setConvertIncoming(true);
    trans.setBorrowArrays(true);
    d_lw.println("{");
    d_lw.tab();
    declareLocalVariables();
    declareReturnVariable(m);
    initializeOutVariables(m);
    getFunction(m.isStatic(), m.getLongMethodName());
    trans.declareProxies();
    trans.convertIncomingArguments(true);
    d_lw.print("_args =");
    trans.convertSidlToPython();
    d_lw.println("if (_args) {");
    d_lw.tab();
    d_lw.println("_result = PyObject_CallObject(_pfunc, _args);");
    d_lw.println("if (_result) {");
    d_lw.tab();
    trans.setConvertIncoming(false);
    trans.declareProxy(m.getReturnType(), TranslateArguments.RETURN_VAR,
                       Argument.OUT);
    d_lw.println("int _okay;");
    if (TranslateArguments.extractOut
        (m.getReturnType(), m.getArgumentList()).size() == 1) {
      d_lw.println("PyObject *_tmp = PyTuple_New(1);");
      d_lw.println("(void)PyTuple_SetItem(_tmp, 0, _result);");
      d_lw.println("_result = _tmp;");
    }
    releaseInOutParameters(m);
    d_lw.print("_okay = ");
    trans.setBorrowArrays(false);
    trans.convertPythonToSidl("_result");
    d_lw.println("if (_okay) {");
    d_lw.tab();
    trans.convertOutgoingArguments(false);
    d_lw.backTab();
    d_lw.println("}");
    cleanupReferences(m);
    if (m.getReturnType().getDetailedType() != Type.VOID) {
      d_lw.println("return " + TranslateArguments.RETURN_VAR + ";");
    }
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  /**
   * Write the functions in C that will expose Python object methods or
   * Python free functions to the IOR. This converts each method in turn.
   * 
   * @exception gov.llnl.babel.backend.CodeGeneration
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void convertMethods(Collection methods) throws CodeGenerationException
  {
    Iterator i = methods.iterator();
    while (i.hasNext()) {
      Method m = (Method)i.next();
      if (!m.isAbstract()) {
        convertMethod(m);
      }
    }
  }

  private void writeInitializeSEPV(Collection methods) {
    SymbolID id = d_cls.getSymbolID();
    Iterator i = methods.iterator();
    d_lw.openCxxExtern();
    d_lw.println("void");
    d_lw.println(Python.getSetSEPVName(id));
    d_lw.tab();
    d_lw.println("(" + IOR.getSEPVName(id) + " *sepv)");
    d_lw.backTab();
    d_lw.println("{");
    d_lw.tab();
    d_lw.println("_importModules();");
    while (i.hasNext()) {
      Method m = (Method)i.next();
      if (m.isStatic()) {
        d_lw.println("sepv->" + IOR.getVectorEntry(m.getLongMethodName()) 
          + " = " + Python.getSkelMethod(id, m) + ";");
      }
    }
    d_lw.backTab();
    d_lw.println("}");
    d_lw.closeCxxExtern();
    d_lw.println();
  }

  private void writeInitializeEPV(Collection methods) {
    SymbolID id = d_cls.getSymbolID();
    Iterator i = methods.iterator();
    d_lw.openCxxExtern();
    d_lw.println("void");
    d_lw.println(Python.getSetEPVName(id));
    d_lw.tab();
    d_lw.println("(" + IOR.getEPVName(id) + " *epv)");
    d_lw.backTab();
    d_lw.println("{");
    d_lw.tab();
    d_lw.println("_importModules();");
    while (i.hasNext()) {
      Method m = (Method)i.next();
      switch(m.getDefinitionModifier()) {
      case Method.FINAL:
      case Method.NORMAL:
        d_lw.println("epv->" + IOR.getVectorEntry(m.getLongMethodName()) + " = "
          + Python.getSkelMethod(id, m) + ";");
        break;
      case Method.ABSTRACT:
        d_lw.println("epv->" + IOR.getVectorEntry(m.getLongMethodName()) 
          + " = NULL;");
        break;
      default:
        /* do nothing */
        break;
      }
    }
    d_lw.backTab();
    d_lw.println("}");
    d_lw.closeCxxExtern();
    d_lw.println();
  }

  /**
   * Add the implicit builtin methods to the list of methods. There are
   * implicit functions for the constructor and destructor.
   */
  private Collection extendMethods(Collection localMethods) 
    throws CodeGenerationException 
  {
    final SymbolID id = d_cls.getSymbolID();
    ArrayList  extendedMethods = new ArrayList(localMethods.size()+2);
    extendedMethods.add(IOR.getBuiltinMethod(IOR.CONSTRUCTOR, id));
    extendedMethods.add(IOR.getBuiltinMethod(IOR.DESTRUCTOR, id));
    extendedMethods.addAll(localMethods);
    return extendedMethods;
  }

  private void writeConstructor() throws CodeGenerationException {
    final SymbolID id = d_cls.getSymbolID();
    Method ctor = IOR.getBuiltinMethod(IOR.CONSTRUCTOR, id);
    writeMethodSignature(ctor);
    d_lw.println("{");
    d_lw.tab();
    declareLocalVariables();
    getFunction(true, id.getShortName());
    d_lw.println("_args = Py_BuildValue(\"(O&)\", (void *)");
    d_lw.tab();
    d_lw.println(Python.getExtendableBorrow(d_cls) + ", self);");
    d_lw.backTab();
    d_lw.println("if (_args) {");
    d_lw.tab();
    d_lw.println("_result = PyInstance_New(_pfunc, _args, NULL);");
    d_lw.println("if (_result) {");
    d_lw.tab();
    d_lw.println("self->d_data = _result;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("Py_DECREF(_args);");
    d_lw.backTab();
    d_lw.println("} else {");
    d_lw.tab();
    d_lw.println("if (PyErr_Occurred()) PyErr_Print();");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("Py_DECREF(_pfunc);");
    d_lw.backTab();
    d_lw.println("} else {");
    d_lw.tab();
    d_lw.println("if (PyErr_Occurred()) PyErr_Print();");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("Py_DECREF(_context);");
    d_lw.backTab();
    d_lw.println("} else {");
    d_lw.tab();
    d_lw.println("if (PyErr_Occurred()) PyErr_Print();");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  private void writeDestructor() throws CodeGenerationException {
    final SymbolID id = d_cls.getSymbolID();
    Method dtor = IOR.getBuiltinMethod(IOR.DESTRUCTOR, id);
    writeMethodSignature(dtor);
    d_lw.println("{");
    d_lw.tab();
    d_lw.writeCommentLine("Remove only reference to Python object");
    d_lw.println("Py_XDECREF(((PyObject *)self->d_data));");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  /**
   * Generate the C skeleton source file for a Python implementation of
   * a sidl class.
   * 
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this a catch all exception for problems during the code
   *    generation phase.
   */
  public synchronized void generateCode() throws CodeGenerationException {
    try {
      Collection methods = d_cls.getMethods(false);
      Collection extendedMethods = extendMethods(methods);
      d_lw = Python.createSkel
        (d_cls, "skeletons to link IOR to Python implementation");
      explainSkelSource();
      includeHeaderFiles();
      generateInit();
      convertMethods(methods);
      writeConstructor();
      writeDestructor();
      writeInitializeEPV(extendedMethods);
      if (d_cls.hasStaticMethod(false)) {
        writeInitializeSEPV(extendedMethods);
      }
    } finally {
      if (d_lw != null) {
        d_lw.close();
        d_lw = null;
      }
    }
  }
}
