//
// File:        PythonServerLaunch.java
// Package:     gov.llnl.babel.backend.python
// Revision:    @(#) $Revision: 4434 $
// Date:        $Date: 2005-03-17 09:05:29 -0800 (Thu, 17 Mar 2005) $
// Description: Initialize Python before loading the real implementation
// 
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.python;

import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Method;

public class PythonServerLaunch {
  LanguageWriterForC d_lw = null;
  Class d_cls;
  boolean d_hasStatic;

  PythonServerLaunch(Class cls)
  {
    d_cls = cls;
    d_hasStatic = d_cls.hasStaticMethod(false);
  }

  private void writeIncludes()
  {
    d_lw.generateInclude("sidl_Python.h", true);
    d_lw.generateInclude("sidl_Loader.h", true);
    d_lw.printlnUnformatted("#include <stdio.h>");
    d_lw.println();
  }

  private void writeForwardDecls()
  {
    final SymbolID id = d_cls.getSymbolID();
    d_lw.println(IOR.getEPVName(id) + ";");
    if (d_hasStatic) {
      d_lw.println(IOR.getSEPVName(id) + ";");
    }
    d_lw.println();
  }

  private void writeStaticVars()
  {
    final SymbolID id = d_cls.getSymbolID();
    d_lw.println("static void (*s_getEPV)");
    d_lw.tab();
    d_lw.println("(" + IOR.getEPVName(id) + " *);");
    d_lw.backTab();
    if (d_hasStatic) {
      d_lw.println("static void (*s_getSEPV)");
      d_lw.tab();
      d_lw.println("(" + IOR.getSEPVName(id) + " *);");
      d_lw.backTab();
    }
    d_lw.println();
  }
  
  private void writeInit()
  {
    final SymbolID id = d_cls.getSymbolID();
    d_lw.println("static void");
    d_lw.println("initImplementation(void) {");
    d_lw.tab();
    d_lw.println("static int notInitialized = 1;");
    d_lw.println("if (notInitialized) {");
    d_lw.tab();
    d_lw.println("sidl_DLL dll = sidl_Loader_findLibrary(\"" + 
                 id.getFullName() + "\",");
    d_lw.tab();
    d_lw.println("\"python/impl\", sidl_Scope_SCLSCOPE,");
    d_lw.println("sidl_Resolve_SCLRESOLVE);");
    d_lw.backTab();
    d_lw.println("if (dll) {");
    d_lw.tab();
    d_lw.println("notInitialized = 0;");
    d_lw.println("sidl_Python_Init();");
    d_lw.println("s_getEPV = (void (*)(" + IOR.getEPVName(id) + " *))");
    d_lw.tab();
    d_lw.println("sidl_DLL_lookupSymbol(dll, \"" +
                 Python.getSetEPVName(d_cls.getSymbolID()) + "\");");
    d_lw.backTab();
    if (d_hasStatic) {
      d_lw.tab();
      d_lw.println("s_getSEPV = (void (*)(" + IOR.getSEPVName(id) + " *))");
      d_lw.println("sidl_DLL_lookupSymbol(dll, \"" +
                   Python.getSetSEPVName(d_cls.getSymbolID()) + "\");");
      d_lw.backTab();
    }
    d_lw.println("sidl_DLL_deleteRef(dll);");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  private void writeGetEPV()
  {
    final SymbolID id = d_cls.getSymbolID();
    d_lw.openCxxExtern();
    d_lw.println("void");
    d_lw.println(IOR.getSetEPVName(id) + "(" +
                 IOR.getEPVName(id) + " *epv) {");
    d_lw.tab();
    d_lw.println("initImplementation();");
    d_lw.println("if (s_getEPV) (*s_getEPV)(epv);");
    d_lw.println("else {");
    d_lw.tab();
    d_lw.println("fputs(\"\\");
    d_lw.printlnUnformatted("Babel: Fatal error loading stage 2 Python implementation for " +
                            id.getFullName() + "\",");
    d_lw.println("stderr);");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.closeCxxExtern();
    d_lw.println();
  }

  private void writeGetSEPV()
  {
    final SymbolID id = d_cls.getSymbolID();
    d_lw.openCxxExtern();
    d_lw.println("void");
    d_lw.println(IOR.getSetSEPVName(id) + "(" +
                 IOR.getSEPVName(id) + " *sepv) {");
    d_lw.tab();
    d_lw.println("initImplementation();");
    d_lw.println("if (s_getSEPV) (*s_getSEPV)(sepv);");
    d_lw.println("else {");
    d_lw.tab();
    d_lw.println("fputs(\"\\");
    d_lw.printlnUnformatted("Babel: Fatal error loading stage 2 Python implementation for " +
                            id.getFullName() + "\",");
    d_lw.println("stderr);");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.closeCxxExtern();
    d_lw.println();
  }

  private void writeCallLoad()
    throws CodeGenerationException
  {
    final SymbolID id = d_cls.getSymbolID();
    Method load = IOR.getBuiltinMethod(IOR.LOAD, id);
    d_lw.println("void " + IOR.getSymbolName(id) + "__call_load(void) { ");
    d_lw.tab();
    d_lw.println("static int i=0;");
    d_lw.writeCommentLine("FIXME: Not implemented for Python yet");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  public synchronized void generateCode()
    throws CodeGenerationException
  {
    try {
      d_lw = Python.createLaunch(d_cls, "launch the implementation");
      writeIncludes();
      writeForwardDecls();
      writeStaticVars();
      writeInit();
      writeGetEPV();
      if (d_hasStatic) {
        writeGetSEPV();
      }
      writeCallLoad();
    }
    finally {
      if (d_lw != null) {
        d_lw.close();
        d_lw = null;
      }
    }
  }
}
