
/**************************************************************************
 *                                                                        *
 *  BTools - Miscellaneous Java utility classes                           *
 *                                                                        *
 *  Copyright (c) 1998-2001, Ben Burton                                   *
 *  For further details contact Ben Burton (benb@acm.org).                *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or         *
 *  modify it under the terms of the GNU General Public License as        *
 *  published by the Free Software Foundation; either version 2 of the    *
 *  License, or (at your option) any later version.                       *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful, but   *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *  General Public License for more details.                              *
 *                                                                        *
 *  You should have received a copy of the GNU General Public             *
 *  License along with this program; if not, write to the Free            *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,        *
 *  MA 02111-1307, USA.                                                   *
 *                                                                        *
 **************************************************************************/

/* end stub */

package org.gjt.btools.image;

import java.awt.Toolkit;
import javax.swing.ImageIcon;

/**
 * Represents a standard image that may be requested multiple times.
 * The same <tt>ImageIcon</tt> will be returned for every request; thus
 * only one copy is ever loaded at a time.
 * <p>
 * The internal <tt>ImageIcon</tt> will not be created until it is first
 * requested through <tt>image()</tt>.  Each following request will
 * return the same <tt>ImageIcon</tt>, thus saving time and memory.
 * Calling <tt>finalImage()</tt> or <tt>discardImage()</tt>
 * will cause the internal <tt>ImageIcon</tt> to be discarded; this will
 * save memory but a subsequent call to <tt>getImage()</tt> will require
 * a new <tt>ImageIcon</tt> to be created.
 */
public class StandardImage {
    /**
     * The location of the image that this object represents.
     */
    private String location;

    /**
     * A class whose class loader will be used to locate the
     * corresponding image resource.
     * If this member is <tt>null</tt>, the system class loader will be
     * used instead.
     */
    private Class loaderClass;

    /**
     * The <tt>ImageIcon</tt> that will be returned to future requests
     * for this image, or <tt>null</tt> if the image is not loaded.
     */
    private ImageIcon internalImage;

    /**
     * Creates a new standard image representing the image at the
     * specified location.
     * The system class loader will be used to locate the image
     * resource.
     *
     * @param location the location of the image this object will
     * represent.
     */
    public StandardImage(String location) {
        this.location = location;
        this.loaderClass = null;
        internalImage = null;
    }

    /**
     * Creates a new standard image representing the image at the
     * specified location.
     * The class loader for the given class will be used to locate the image
     * resource.
     *
     * @param location the location of the image this object will
     * represent.
     * @param loaderClass the class whose class loader will be used to locate
     * the image resource.
     */
    public StandardImage(String location, Class loaderClass) {
        this.location = location;
        this.loaderClass = loaderClass;
        internalImage = null;
    }

    /**
     * Returns the <tt>ImageIcon</tt> containing this image.  If the
     * image has not already been loaded, it will be loaded now.  The
     * <tt>ImageIcon</tt> will be stored so the same <tt>ImageIcon</tt>
     * can be returned for future requests.
     *
     * @return the <tt>ImageIcon</tt> containing this image.
     */
    public ImageIcon image() {
        if (internalImage == null) {
            if (loaderClass == null)
                internalImage = makeImageIcon(location);
            else
                internalImage = makeImageIcon(location, loaderClass);
        }
        return internalImage;
    }

    /**
     * Returns and discards the <tt>ImageIcon</tt> containing this image.
     * This routine should be used when it is known that this image will
     * not be requested again.  The image returned will no longer be
     * stored by this object.  Thus memory will be saved, but
     * any future request for this image will require a new
     * <tt>ImageIcon</tt> to be created.
     *
     * @return the <tt>ImageIcon</tt> containing this image.
     */
    public ImageIcon finalImage() {
        if (internalImage == null) {
            if (loaderClass == null)
                return makeImageIcon(location);
            else
                return makeImageIcon(location, loaderClass);
        }

        ImageIcon ans = internalImage;
        internalImage = null;
        return ans;
    }

    /**
     * Discards the internally stored <tt>ImageIcon</tt> representing this
     * image.
     * This routine should be used when it is known that this image will
     * not be requested again.  Memory will be saved, but
     * any future request for this image will require a new
     * <tt>ImageIcon</tt> to be created.
     */
    public void discardImage() {
        internalImage = null;
    }

    /**
     * Creates a new image icon representing the image resource at the
     * given location.
     * The system class loader will be used to locate the image
     * resource.
     *
     * @param location the location of the image resource.
     * @return the new image icon.
     */
    public static ImageIcon makeImageIcon(String location) {
        try {
            return new ImageIcon(ClassLoader.getSystemResource(location));
            //return new ImageIcon(Toolkit.getDefaultToolkit().
                //createImage(location));
        } catch (Throwable th) {
            System.err.println("Error loading image [" + location + "]:");
            th.printStackTrace();
            return null;
        }
    }

    /**
     * Creates a new image icon representing the image resource at the
     * given location.
     * The class loader for the given class will be used to locate the image
     * resource.
     *
     * @param location the location of the image resource.
     * @param loaderClass the class whose class loader will be used to locate
     * the image resource.
     * @return the new image icon.
     */
    public static ImageIcon makeImageIcon(String location, Class loaderClass) {
        try {
            return new ImageIcon(loaderClass.getClassLoader().
                getResource(location));
        } catch (Throwable th) {
            System.err.println("Error loading image [" + location + "]:");
            th.printStackTrace();
            return null;
        }
    }
}

