/**************************************************************
*   
*   Creation Date: <97/07/19 17:20:37 samuel>
*   Time-stamp: <2001/06/27 16:35:19 samuel>
*   
*	<mmu_cmds.c>
*	
*	MMU-commands
*   
*   Copyright (C) 1997, 1999, 2000, 2001 Samuel Rydh
*
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation;
*
**************************************************************/

#include "mol_config.h"
#include <asm/mmu.h> 

#include "molcpu.h"
#include "memory.h"
#include "extralib.h"
#include "mmu_contexts.h"
#include "wrapper.h"
#include "nub.h"

static int 	cmd_tea( int, char**);		/* translate data ea */
static int 	cmd_itea( int, char**);		/* translate inst ea */
static int 	cmd_tear( int, char**);		/* translate data ea-range */
static int 	cmd_itear( int, char**);	/* translate inst ea-range */
static int 	cmd_lpte( int, char**);		/* get linux PTE */
static int 	cmd_lpter( int, char**);	/* get linux PTE */
static int 	cmd_mpter( int, char**);	/* display mac PTEs */
static int 	cmd_eatop( int, char**);	/* ea to physical */
static int 	cmd_lvtop( int, char**);	/* lv to physical */
static int 	cmd_tlbia( int, char**);	/* flush hash-table */
static int 	cmd_tlbie( int, char**);	/* flush hash-table */
static int 	cmd_mmucc( int, char**);	/* check MMU-consistency */
static int 	cmd_sdae( int, char**);		/* data access exception */
static int 	cmd_siae( int, char**);		/* data access exception */
static int 	cmd_dbtr( int, char **);	/* block translations */
static int 	cmd_romwi( int, char **);	/* write ROM-image */

static int 	cmd_setw( int, char **);	/* write word to memory */
static int 	cmd_find( int, char **);	/* Find string */
static int 	cmd_findl( int, char **);	/* Find ulong */


static int 	context_from_char( char ch );
static char 	char_from_context( int context );
static int 	print_pte( ulong ea, int context );

static int 	do_cmd_tea( int, char**, int data_access);
static int 	do_cmd_tear( int numargs, char **args, int data_access );

/************************************************************************/
/*	FUNCTIONS							*/
/************************************************************************/

void 
add_mmu_cmds( void ) 
{
	add_cmd( "tea", "tea ea [XSU] \ntranslate DATA effective address\n", -1, cmd_tea );
	add_cmd( "itea", "itea ea [XSU] \ntranslate INST effective address\n", -1, cmd_itea );
	add_cmd( "tear", "tear start end [SU] \ntranslate DATA range ea->mphys\n", -1, cmd_tear );
	add_cmd( "itear", "itear start end [SU] \ntranslate INST range ea->mphys\n", -1, cmd_itear );
	add_cmd( "lpte", "lpte ea [XSU] \ndisplay linux PTE\n", -1, cmd_lpte );
	add_cmd( "lpter", "lpter start end [XSU] \ndisplay linux PTE\n", -1, cmd_lpter );
	add_cmd( "mpter", "mpter \ndisplay mac PTEs\n", -1, cmd_mpter );
	add_cmd( "eatop", "eatop ea [XSU] \neffective address to physical\n", -1, cmd_eatop );
	add_cmd( "lvtop", "lvtop ea \nlinux virtual  address to physical\n", -1, cmd_lvtop );
	add_cmd( "tlbia", "tlbia \nflush linux hash table\n",-1, cmd_tlbia);
	add_cmd( "tlbie", "tlbie ea [end]\nflush entry from hash\n",-1, cmd_tlbie);
	add_cmd( "mmucc", "mmucc [XSU]\ncheck MMU consistency\n",-1, cmd_mmucc);
	add_cmd( "siae", "siae \nsimulate instruction access exception\n",-1, cmd_siae);
	add_cmd( "sdae", "sdae [ea]\nsimulate data access exception\n",-1, cmd_sdae);
	add_cmd( "dbtr", "dbtr [ea]\ndisplay block translations\n",-1, cmd_dbtr);
	add_cmd( "romwi", "romwi filename\nWrite ROM-image to file\n",-1, cmd_romwi );

	add_cmd( "setw", "setw addr data\n Write data to memory at addr\n", -1,cmd_setw );
	add_cmd( "find", "find string \nFind string [in RAM]\n", -1, cmd_find );
	add_cmd( "findl", "find ulong \nFind value [in RAM]\n", -1, cmd_findl );
}

void 
mmu_cmds_cleanup( void ) 
{
}


/**************************************************************
*  context_from_char / char_from_context
*    
**************************************************************/

static int
context_from_char( char ch ) 
{
	switch( toupper(ch) ) {
	case 'X':	/* unmapped */
		return kContextUnmapped;
	case 'S':	/* supervisor mode, mapped */
		return kContextMapped_S;
	case 'U':	/* user mode, mapped */
		return kContextMapped_U;
	}
	printm("'%c' - No such context!\n",ch );
	return 0;
}

static char 
char_from_context( int context ) 
{
	switch( context ) {
	case kContextUnmapped:
		return 'X';
	case kContextMapped_U:
		return 'U';
	case kContextMapped_S:
		return 'S';
	}
	printm("%d - No such context number!\n",context );
	return 0;
}


/**************************************************************
*  print_pte
*    
**************************************************************/

static int 
print_pte( ulong ea, int context ) 
{
	int 		num;
	mol_PTE_t	pte;
	
	num = _get_PTE( context, ea, (ulong*) &pte );
	if( num ) {
		printm("%08lX -> %08X ",ea&0xfffff000, pte.rpn<<12 );

		printm("[%c%c] [%c%c%c] %c%c",
		       pte.r? 'R' : ' ',
		       pte.c? 'C' : ' ',
		       pte.w? 'W' : ' ',
		       pte.i? 'I' : ' ',
		       pte.m? 'M' : ' ',
		       pte.g? 'G' : ' ',
		       pte.h? 'H' : ' ');
		printm(" %02lX",hexbin(pte.pp) );
		printm(" VSID: %08x\n", pte.vsid );
		if( num!=1 ) {
			printm("\nINTERNAL ERROR! %d valid PTE:s exists!\n",num);
		}
		return 0;
	}
	return 1;
}


/* display linux virtual address of ea*/
static int 
cmd_itea( int numargs, char **args ) 
{
	return do_cmd_tea( numargs, args, 0 );
}
/* display linux virtual address of ea*/
static int 
cmd_tea( int numargs, char **args ) 
{
	return do_cmd_tea( numargs, args, 1 );
}

static int
do_cmd_tea( int numargs, char **args, int data_translation )
{
	ulong	mac_ea;
	ulong	mphys;
	int	context;
	char	ch;

	context = get_data_context();
	
	if( numargs!=2 && numargs!=3 )
		return 1;
	if( numargs==3 && !(context = context_from_char(args[2][0])))
		return 1;

	ch = char_from_context( context );
	mac_ea = string_to_ulong(args[1]);

	if( ea_to_mphys( mac_ea, context, &mphys, data_translation )) {
		printm("No translation found for ea %08lX (%c)\n",mac_ea,ch );
		return 0;
	}
	printm("(%c) %s  EA: %08lX --> %08lX\n", ch,
	       data_translation? "Data" : "Inst", mac_ea, mphys );
	return 0;
}


/* display linux PTE of corresponding to ea */
static int 
cmd_lpte( int numargs, char **args ) 
{
	ulong	mac_ea;
	int	context;
	char	ch;
	
	context = get_data_context();
	
	if( numargs!=2 && numargs!=3 )
		return 1;
	if( numargs==3 && !(context = context_from_char(args[2][0])))
		return 1;

	mac_ea = string_to_ulong(args[1]);

	ch = char_from_context(context);
	if(!ch)
		return 0;
	printm("%c: ",ch );
	if( print_pte( mac_ea, context ) )
		printm("No translation found\n");
	return 0;
}

/* display linux PTE of corresponding to ea */
static int 
cmd_lpter( int numargs, char **args ) 
{
	ulong	ea, start, end, oldea;
	int	skip_printed=0, context;
	char	ch;
	int	segindex = 0;
	
	context = get_data_context();
	start = 0;
	end = 0xffffffff;
	
	if( numargs<1 || numargs>4 )
		return 1;

	if( numargs==2 )	/* lpter XSU */
		segindex = 1;
	if( numargs==3 || numargs==4 ) {   /* lpter start end */
		start = string_to_ulong(args[1]);
		end = string_to_ulong(args[2]);		
	}
	if( numargs==4 )
		segindex = 3;
	
	if( segindex && !(context = context_from_char(args[segindex][0])))
		return 1;

	ch = char_from_context(context);
	if(!ch)
		return 0;
	printm("HASH TABLE: (%c)\n",ch );

	oldea = start;
	for( ea=start ; ea<=end && oldea <=ea ; ea+=0x1000 ) {
		if( print_pte( ea, context ) ) {
			if( !(skip_printed % 0x10000 ))
				printm("...\n");
			skip_printed++;
		} else {
			skip_printed=0;
		}
		oldea = ea;
	}
	return 0;
}

static int
cmd_tear( int numargs, char **args )
{
	return do_cmd_tear( numargs, args, 1 );
}

static int
cmd_itear( int numargs, char **args )
{
	return do_cmd_tear( numargs, args, 0 );
}

/* translate range, (ea to linux virtual) */
static int 
do_cmd_tear( int numargs, char **args, int data_access ) 
{
	ulong	ea, ea2, mphys;
	char	ch;
	int	run, segindex = 0;
	ulong	start = 0;
	ulong	end = 0xfffff000;
	int	context = get_data_context();
	int	dots;
	
	switch( numargs ){
	case 1:
		break;
	case 2:
		segindex=1;
		break;
	case 4:
		segindex = 3;
		/* fall through */
	case 3:
		start = string_to_ulong(args[1]) & ~0xfff;
		end = string_to_ulong(args[2]) & ~0xfff;
		break;
	default:
		return 1;
	}
	if( segindex && !(context = context_from_char(args[segindex][0])))
		return 1;

	if( context == kContextUnmapped )
		context = kContextMapped_S;

	if( !(ch = char_from_context(context) ))
		return 1;

	printm("%s ", data_access ? "Data" : "Inst" );
	printm("effective address -> Mac physical: (%c)\n",ch );
	for( dots=0, run=1, ea=start; run ; ea+=0x1000 ) {
		if( ea == end )
			run = 0;
		if( ea_to_mphys( ea, context, &mphys, data_access ) ) {
			if( !(ea & 0x0fffffff) ) {
				printm("...\n");
				dots=0;
			} else if( dots==1 )
				dots=2;
			continue;
		}
		/* find end of block... */
		for( ea2=ea+0x1000 ; ea2!=end+0x1000; ea2+=0x1000 ){
			ulong mphys2;
			if( !(ea2 & 0x0fffffff) )
				break;
			if( ea_to_mphys( ea2, context, &mphys2, data_access ) )
				break;
			if( mphys2 != mphys + ea2-ea )
				break;
		}
		if( dots==2 )
			printm("...\n");
		dots=1;
		printm("EA: %08lX - %08lX  -->  MPHYS: %08lX - %08lX\n",
		       ea,ea2-1,  mphys, mphys+ea2-ea-1 );
		ea=ea2-0x1000;
		if( ea == end )
			run = 0;
	}
	return 0;
}


/* translate ea to physical address */
static int 
cmd_eatop( int numargs, char **args ) 
{
	ulong	mac_ea, mphys;
	char	*lvptr;
	int	context;
	char	ch;
	linux_page_t page;
	int	not_found=0;
	
	context = get_data_context();
	
	if( numargs!=2 && numargs!=3 )
		return 1;
	if( numargs==3 && !(context = context_from_char(args[2][0])))
		return 1;

	mac_ea = string_to_ulong(args[1]);

	ch = char_from_context(context);
	if(!ch)
		return 0;

	if( ea_to_mphys( mac_ea, context, &mphys,1 /* data*/ )){
		printm("DATA EA %08lX is unmapped\n", mac_ea );
		not_found=1;
	}
	if( ea_to_mphys( mac_ea, context, &mphys, 0 /*inst */ )){
		printm("INST EA %08lX is unmapped\n", mac_ea );
		not_found=1;
	}
	if( not_found )
		return 0;
	if( mphys_to_lvptr( mphys, &lvptr) < 0) {
		printm("MPHYS %08lX is not within RAM/ROM\n",mphys);
		return 0;
	}

	if( _dbg_get_physical_page( (ulong)lvptr, &page ) )
		return 0;
	printm("(%c) EA: %08lX   LV: %08lX  PPN: ", ch,
	       mac_ea&0xfffff000, (ulong)lvptr&0xfffff000 );

	if( page.mflags & M_PAGE_PRESENT )
		printm("%08lX\n", page.phys );
	else
		printm("unmapped\n");
	
	return 0;
}


/* translate lv to physical address */
static int 
cmd_lvtop( int numargs, char **args ) 
{
	char	*lvptr;
	int	context;
	linux_page_t page;
	int	flags;

	context = get_data_context();
	
	if( numargs!=2 && numargs!=3 )
		return 1;
	if( numargs==3 && !(context = context_from_char(args[2][0])))
		return 1;

	lvptr = (char*)string_to_ulong(args[1]);

	/* we do a physical read to ensure the pages are present */
	if( lvptr >= ram.lvbase && lvptr < ram.lvbase + ram.size ) {
		char tmp = *((volatile char*)lvptr);
		tmp++;	/* just to avoid GCC warnings... */
	}

	if( _dbg_get_physical_page( (ulong)lvptr, &page ) ) {
		printm("No linux page found\n");
		return 0;
	}
	flags = page.mflags;
	printm("LV: %08lX  ", (ulong)lvptr & ~0xfff );
	printm( (flags & M_PAGE_PRESENT)? "[P" : "[p");
	printm( (flags & M_PAGE_HASHPTE)? "H": "h");
	printm( (flags & M_PAGE_USER)? " U" : " u");
	printm( (flags & M_PAGE_RW)? "W" : "w");
	printm( (flags & M_PAGE_GUARDED)? "G" : "g");
	printm( (flags & M_PAGE_ACCESSED)? " R" :  " r");
	printm( (flags & M_PAGE_DIRTY)? "C" :  "c");
	printm( (flags & M_PAGE_WRITETHRU)? " W" :  " w");
	printm( (flags & M_PAGE_NO_CACHE)? "I" :  "i");
	printm( (flags & M_PAGE_COHERENT)? "M]" :  "m]");
	printm("  PPN: %08lX\n", page.phys );
	
	return 0;
}

static int 
cmd_tlbia( int numargs, char **args ) 
{
	printm("All (S) and (U) hash entries flushed\n");
	_emulate_tlbia();
	return 0;
}

static int 
cmd_tlbie( int numargs, char **args ) 
{
	ulong	ea, end;
	if( numargs!=2 && numargs!=3 )
		return 1;

	ea = string_to_ulong( args[1] );
	end = numargs==3 ? string_to_ulong( args[2] ) : ea;
	ea &= ~0xfff;
	end &= ~0xfff;

	if( ea == end )
		printm("Entries with page-index %08lX flushed\n",ea );
	else
		printm("Pages %08lX - %08lX flushed\n", ea, end );
	for( ; ea <= end ; ea += 0x1000 )
		_emulate_tlbie(ea);

	return 0;
}

/* mmu consistency check. Things has changed since this function was written - 
 * it might or might not do a good job.
 */
static int 
cmd_mmucc( int numargs, char **args ) 
{
	linux_page_t page;
	ulong 	mac_ea=0;
	ulong	i, num;
	ulong 	pte[2];
	int	context;
	char	*lvptr;
	
	context = get_data_context();
	
	if( numargs!=1 && numargs!=2 )
		return 1;
	if( numargs==2 && !(context = context_from_char(args[1][0])))
		return 1;
	
	for( i=0; i<0xfffff; i++ ) {
		mac_ea = i<<12;
		/* ea_to_lvptr first uses the mac-MMU to convert ea to
		 * mphys, then a mphys -> lvptr is performed
		 */
		if( ea_to_lvptr( mac_ea, context, &lvptr, 0 ))
			continue;
		/* _dbg_get_physical_page obtains the physical address from linux page tables */
		if( _dbg_get_physical_page( (ulong)lvptr, &page ) )
			continue;
		/* _get_PTE examines what's actually in the hash table */
		num = _get_PTE( context, mac_ea, pte );
		if( num>1 )
			printm("Duplicate PTE, CONTEXT: %d, EA: %08lx\n",context,mac_ea);
		if( num==0)
			continue;

		if( (pte[1]>>12) != (page.phys>>12) ) {
			printm("Different physical addresses! EA: %08lx LVPTR: %p PH: %08lX CORR_PH: %08lX\n",
			       mac_ea, lvptr, pte[1]&0xfffff000, page.phys );
		}
	}
	return 0;
}


/* siae - instruction access exception */
static int 
cmd_siae( int argv, char **args ) 
{
	ulong 	mac_ea=mregs->nip;
	char	*dummy;
	
	if( argv != 1 )
		return 1;

	if( !ea_to_lvptr( mac_ea, get_inst_context(), &dummy, 0/*inst*/ ) )
		if( !yn_question("There is already a mapping present. Proceed anyway? [y/N]",0) )
			return 0;

	printm("Simulation an instruction access exception - UNIMPLEMENTED\n");
	refresh_debugger_window();
	return 0;
}


/* sdae - data access exception */
static int 
cmd_sdae( int argv, char **args ) 
{
	ulong 	mac_ea=mregs->nip;
	char	*dummy;
	
	if( argv != 1 && argv != 2 )
		return 1;

	if( argv == 2 )
		mac_ea = string_to_ulong( args[1] );

	if( !ea_to_lvptr( mac_ea, get_inst_context(), &dummy, 1/*data trans*/ ) )
		if( !yn_question("There is already a mapping present. Proceed anyway? [y/N]",0) )
			return 0;
	
	refresh_debugger_window();

	printm("Simulation a data access exception - UNIMPLEMENTED\n");
	return 0;
}

/* dbtr - display block translations*/
static int 
cmd_dbtr( int argc, char **argv ) 
{
#if 1
	printm("Obsolete... block translations are handled differently now\n");
	return 0;
#else
	int i;
	struct blocktrans *p = mregs->blk_trans;       

	if( argc!=1 )
		return 1;
	for(i=0; i<mregs->num_blktrans; i++,p++ ){
		if( !p->valid )
			continue;
		printm("%08lX ---> %08lX  (size: %08lX)  ",
		       p->virt, p->phys, p->size );
		printm("%c%c%c%c  PP=%d, Ks=%d, Ku=%d, Vs=%d, Vp=%d\n", 
		       (p->wimg & 0x8)? 'W' : 'w',
		       (p->wimg & 0x4)? 'I' : 'i',
		       (p->wimg & 0x2)? 'M' : 'm',
		       (p->wimg & 0x1)? 'G' : 'g',
		       p->pp,p->ks,p->ku,p->vs,p->vp );
	}
	return 0;
#endif
}

/* write romimage */
static int 
cmd_romwi( int argc, char **argv )
{
	int fd;
	if( argc!=2 )
		return 1;
	if( (fd = open( argv[1], O_RDWR | O_CREAT | O_EXCL, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH )) == -1 ) {
		perrorm("open");
		return 0;
	}
	if( write( fd, rom.lvbase, rom.size ) != rom.size ) {
		printm("An error occured\n");
	} else { 
		printm("Wrote ROM-image '%s'.\n", argv[1] );
	}
	close(fd);
	return 0;
}


/*
 * Write word to memory...
 * XXX: THIS SHOULD BE GENERALIZED
 */
static int 
cmd_setw( int argc, char **argv )
{
	ulong val, mphys;
	ulong *lvptr;
	
	if( argc != 3 )
		return -1;

	mphys = string_to_ulong(argv[1]);
	val = string_to_ulong(argv[2]);
	if( mphys_to_lvptr( mphys, (char**)&lvptr) != 0) {
		printm("MPHYS %08lX is not within RAM\n",mphys);
		return 0;
	}
	*lvptr = val;
	return 0;
}


/* Find (in RAM) */
static int /* Syn. f string */
cmd_find( int argc, char **argv  )
{
	char *p1,*p2;
	size_t i;
	int num=0;
	
	if(argc!=2 )
		return 1;
	p1 = ram.lvbase;
	p2 = argv[1];

	printm("Searching for string '%s'\n", p2 );

	for(i=0; i<ram.size ; p1++, i++ ) {
		if( !*p2 ) {
			printm("Match at %08lx\n", ram.mbase + i );
			p2 = argv[1];
			if( num++ > 10 )
				break;
			continue;
		}
		if( *p2 == *p1 )
			p2++;
		else
			p2 = argv[1];
	}
	return 0;
}

static int /* Syn. find ulongkey */
cmd_findl( int argc, char **argv  )
{
	ulong *p1,key;
	int num=0;
	
	if( argc != 2 )
		return 1;
	key = string_to_ulong( argv[1] );
	printm("Seraching for %08lx\n",key );
	
	p1 = (ulong*)ram.lvbase;
	while( (char*)p1 < ram.lvbase+ram.size-3 ){
		if( *p1 == key ) {
			printm("Match at %p\n", 
			       (char*)p1 - (ulong)ram.lvbase + (ulong)ram.mbase );
			if( num++ > 10 )
				break;
		}
		
		p1=(ulong*)((char*)p1+2);
	}
	return 0;
}

static mol_PTE_t *
lookup_mpte( ulong vsid, ulong ea )
{
	ulong	phash, cmp, pteg, *p;
	ulong	v, mask, hash_mask;
	int	i;
	char	*hash_base;
	
	v = mregs->spr[S_SDR1];
	hash_mask = ((v & 0x1ff)<<16) | 0xffff;
	mphys_to_lvptr( (v & ~hash_mask), &hash_base );

	// we are only interested in the page index
	ea &= 0x0ffff000;
	mask = hash_mask>>6;

	// calculate primary hash function
	phash = (ea >> 12) ^ (vsid & 0x7ffff);
	pteg = ((phash & mask) << 6);

	// construct compare word
	cmp = BIT(0) | (vsid <<7) | ((ea&0x0fffffff)>>22);

	// look in primary PTEG
	p=(ulong*)((ulong)hash_base + pteg);
	for(i=0; i<8; i++, p+=2 )
		if( cmp == *p )
			return (mol_PTE_t*)p;
				
	// look in secondary PTEG
	p = (ulong*)( (ulong)hash_base + (pteg ^ (mask << 6)) );
	cmp |= BIT(25);

	for(i=0; i<8; i++,p+=2 )
		if( cmp == *p )
			return (mol_PTE_t*)p;
	return NULL;
}

/* display mac PTE of corresponding to ea */
static int 
cmd_mpter( int numargs, char **args ) 
{
	ulong	ea, start, end, oldea;
	
	start = 0;
	end = 0xffffffff;
	
	if( numargs!=1 )
		return 1;

	printm("MAC HASH TABLE: \n" );

	oldea = start;
	for( ea=start ; ea<=end && ea >= oldea; ea+=0x1000 ) {
		mol_PTE_t *pte;
		oldea = ea;
		if( !(pte = lookup_mpte( mregs->segr[ea>>28], ea )) )
			continue;
		printm("EA: %08lX --> MPHYS %08lX   ", ea, (ulong)(pte->rpn << 12) );
		printm("[%c%c] [%c%c%c] %c%c",
		       pte->r? 'R' : ' ',
		       pte->c? 'C' : ' ',
		       pte->w? 'W' : ' ',
		       pte->i? 'I' : ' ',
		       pte->m? 'M' : ' ',
		       pte->g? 'G' : ' ',
		       pte->h? 'H' : ' ');
		printm(" %02lX",hexbin(pte->pp) );
		printm(" VSID: %08x\n", pte->vsid );
	}
	return 0;
}
