/* 
 *   Creation Date: <2001/01/27 14:38:03 samuel>
 *   Time-stamp: <2001/08/12 18:00:06 samuel>
 *   
 *	<rvec.c>
 *	
 *	
 *   
 *   Copyright (C) 2000, 2001 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"

#include "mol_assert.h"
#include "rvec.h"
#include "mac_registers.h"
#include "async.h"
#include "timer.h"
#include "res_manager.h"
#include "session.h"

#include "molcpu.h"

// gRVECtable is also accessed mainloop.S
rvec_table_t gRVECtable[NUM_RVECS];

// Assembly exports (mainloop.S)
extern void mainloop_asm_init( void );
extern int mainloop( int rvec, mac_regs_t *m, ulong session_magic );

static int rvec_bad_vector( int vnum );
static int rvec_internal_error( int vnum, int err );
static int rvec_debugger( int vnum, int num );
static int rvec_interrupt( int vnum );

static void print_rvec_stats( int is_cleanup );

static int cmd_rvecs( int, char ** );
static int cmd_ks( int, char ** );

/************************************************************************/
/*	F U N C T I O N S						*/
/************************************************************************/

void
mainloop_init( void )
{
	int i;

	CLEAR( gRVECtable );
	for(i=0; i<NUM_RVECS; i++ )
		set_rvector( i, rvec_bad_vector, NULL, kFPU_Safe );

	set_rvector( RVEC_INTERNAL_ERROR, rvec_internal_error, "Internal Error", kFPU_Unsafe );
	set_rvector( RVEC_DEBUGGER, rvec_debugger, "Debugger", kFPU_Unsafe );
	set_rvector( RVEC_INTERRUPT, rvec_interrupt, "Interrupt", kFPU_Unsafe );

	add_cmd( "rvecs", "rvecs \nShow RVEC statistics\n", -1, cmd_rvecs );
	add_cmd( "ks", "ks \nShow kernel performance statistics\n", -1, cmd_ks );

	mainloop_asm_init();
}

void
mainloop_cleanup( void )
{
	// print_rvec_stats(1);
}

void 
set_rvector( int vnum, void *vector, char *dbg_str, int fpu_safe )
{
	assert( (vnum & RVEC_MASK) < NUM_RVECS && (vnum & RVEC_MASK) >= 0 );
	assert( (fpu_safe == kFPU_Safe) == !(vnum & RVF_FPU_UNSAFE) );
	
	vnum &= RVEC_MASK;
	gRVECtable[vnum].rvec = vector;
	gRVECtable[vnum].dbg_name = dbg_str;
}


/************************************************************************/
/*	Mainloop Control						*/
/************************************************************************/

static int __stop;
enum{ kGo=0, kStop=1, kQuit=-1, kSaveSession=-2 };

void
stop_emulation( void )
{
	if( debugger_attached() ) {
		__stop = kStop;
		interrupt_emulation();
	}
}
void
save_session( void )
{
	__stop = kSaveSession;
}
void 
resume_emulation( void ) 
{
	__stop = kGo;
}
void
quit_emulation( void )
{
	__stop = kQuit;
}

void
mainloop_start( void )
{
	int err;
	_spr_changed();

	/* RVEC_INITIALIZE initializes the low-level kernel parts */
	mainloop( RVEC_INITIALIZE, mregs, g_session_magic );

	__stop = (get_bool_res("debug_stop") == 1);

	while( __stop != kQuit ) {
		switch( __stop ){
		case kGo:
			mainloop( RVEC_NOP, mregs, g_session_magic );
			assert( __stop );
			break;

		case kStop:
			refresh_debugger();
			while( __stop == kStop ) {
				while( mregs->interrupt )
					rvec_interrupt( RVEC_INTERRUPT );
				usleep(1);
			}
			break;

		case kSaveSession:
			printm("Saving the session...\n");
			if( !(err = save_session_now()) ) {
				printm("The session was saved successfully\n");
				__stop = kQuit;
			} else if( err == 1 ){
				static int tries=0;
				__stop = kGo;
				if( ++tries < 200 ) {
					schedule_save_session(1000);
				} else {
					printm("Did not reach a state where the session could be saved\n");
					tries=0;
				}
			} else {
				__stop = kGo;
			}
			break;
		}
	}
	printm("exiting...\n");
}


/************************************************************************/
/*	emulation interrupts						*/
/************************************************************************/

void 
interrupt_emulation( void )
{
	// printm("interrupt_emulation()\n");

	assert( mregs );

	mregs->interrupt = 1; 
	abort_doze();

	// XXX: On SMP we must signal the main thread (this presumably 
	// triggers an IPI which will cause MOL to stop processing 
	// mac instructions).
}

/* We come here as soon as mregs->interrupt is set (or by the kernel
 * code returning RVEC_INTERRUPT). This mechanism is used to handle 
 * a lot of things.
 */
static int
rvec_interrupt( int dummy_rvec )
{
	int flags;
	mregs->interrupt = 0;
	flags = mregs->flag_bits;

	if( mregs->kernel_dbg_stop ) {
		printm("Kernel stop emulation (kernel_dbg_stop = %08lX)\n", mregs->kernel_dbg_stop );
		mregs->kernel_dbg_stop = 0;
		__stop = kStop;
	}

	clear_abort_doze();
	do_async_io();
	do_timer_stuff( (flags & fb_TimerINT) );

	if( (mregs->msr & MSR_EE) ) {
		if( mregs->irq )
			irq_exception();
		else if( flags & fb_DecINT ) {
			//printm("DEC %08lX %08lX\n", mregs->nip, mregs->dec_stamp - get_tbl());
			dec_exception();
		}
	}
	return __stop ? kRVecExit : 0;
}

/************************************************************************/
/*	Misc Return Vectors						*/
/************************************************************************/

static int
rvec_bad_vector( int rvec )
{
	printm("Uninitialized rvector %d occured (%X)\n", rvec & RVEC_MASK, rvec );
	quit_emulation();
	return kRVecExit;
}

static int
rvec_debugger( int dummy_rvec, int num )
{
	printm("RVEC-DEBUGGER <%x>\n", num );
	stop_emulation();
	return kRVecExit;
}

static int
rvec_internal_error( int dummy_rvec, int err )
{
	if( err == 0x1700 ){
		printm("==================================================\n"
		       "A problem related to TAU-interrupts has occured.\n"
		       "This is probably a hardware problem - please turn\n"
		       "off the kernel config option\n"
		       "    Interrupt driven TAU driver (CONFIG_TAU_INT)\n"
		       "\nor\n"
		       "    Thermap Management Support (CONFIG_TAU)\n"
		       "(a kernel recompilation is necessary...)\n"
		       "==================================================\n");
	}
	printm("RVEC Internal Error %x\n", err );
	quit_emulation();
	return kRVecExit;
}


/************************************************************************/
/*	Debugger							*/
/************************************************************************/

static void
print_rvec_stats( int is_cleanup )
{
#ifdef COLLECT_RVEC_STATISTICS
	rvec_table_t *p = gRVECtable;
	int i;	

	for(i=0; i<NUM_RVECS; i++, p++ ) {
		if( p->rvec == rvec_bad_vector )
			continue;
		printm("RVEC %02d %-26s: %d\n",i, p->dbg_name ? p->dbg_name : "----", p->dbg_count );
	}
#else
	if( !is_cleanup ) {
		printm("No RVEC statistics is collected\n");
		printm("MOL was not compiled with COLLECT_RVEC_STATISTICS\n");
	}
#endif

}

static int 
cmd_rvecs( int argc, char **argv )
{
	if( argc != 1 )
		return 1;
	print_rvec_stats(0);
	return 0;
}

static int 
cmd_ks( int argc, char **argv )
{
	int c,i=0;
	char buf[200];

	while( !_get_performance_info(i++,buf,sizeof(buf),&c) )
		printm("%-30s: %d\n", buf, c );
	return 0;
}
