/* 
** Mount the davfs file system.
 */

/*
 *  This file is part of davfs2.
 *
 *  davfs2 is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  davfs2 is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with davfs2; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#ifndef __MOUNT_DAVFS_H__
#define __MOUNT_DAVFS_H__


/* Data Types */
/*============*/

/* This data structure holds almost everything davfs gathers while reading and
   checking command line and configuration files. (See comment for data origin;
   highest precedence first.)
   Some data will be copied into global or local variables to be available in
   daemon mode. The rest will be freed when forking into daemon mode. */
typedef struct {
    gid_t dav_group;          /* System config file */
    char *conf;               /* Command line */
    /* Mount options */
    int user;                 /* Command line */
    unsigned long int mopts;  /* Command line */
    char *kernel_fs;          /* User config file, system config file */
    size_t buf_size;          /* User config file, system config file */
    /* File mode */
    uid_t uid;                /* Command line */
    gid_t gid;                /* Command line */
    mode_t dir_umask;
    mode_t file_umask;
    mode_t dir_mode;          /* Command line */
    mode_t file_mode;         /* Command line */
    /* WebDAV-resource */
    char *scheme;             /* Command line */
    char *host;               /* Command line */
    int port;                 /* Command line */
    char *path;               /* Command line */
    char *secrets;            /* User config file */
    char *username;           /* User secrets file, system secrets file */
    char *password;           /* User secrets file, system secrets file */
    char *p_host;             /* User config file, sys conf f., environment */
    int p_port;               /* User config file, sys conf f., environment */
    char *p_user;             /* User secrets file, system secrets file */
    char *p_passwd;           /* User secrets file, system secrets file */
    int useproxy;             /* User config file, sys conf f., command line */
    int askauth;              /* User config file, sys conf f., command line */
    int displayname;          /* User config file, system config file */
    int locks;                /* User config file, sys conf f., command line */
    time_t read_timeout;      /* User config file, system config file */
    /* Cache */
    char *sys_cache;          /* System config file */
    char *cache_dir;          /* User config file */
    char *backup_dir;         /* User config file, system config file */
    size_t cache_size;        /* User config file, system config file */
    size_t table_size;        /* User config file, system config file */
    time_t idle_time;         /* User config file, system config file */
    time_t expire;            /* User config file, system config file */
    time_t get_wait;          /* User config file, system config file */
    time_t retry;             /* User config file, system config file */
    time_t max_retry;         /* User config file, system config file */
    time_t lock_timeout;      /* User config file, system config file */
    time_t lock_refresh;      /* User config file, system config file */
    char * lock_owner;        /* User config file, system config file */
} dav_args;


/* Public function. Just main(). */
/*===============================*/

/* Main launches a daemon program that runs a directory and file cache and
   is connected to the WbDAV resource and the kernel file system module.
   It must run setuid root. After forking into  daemon mode it releases root
   permissions permanently. The daemon runs with the uid of the user that owns
   the file system. (If invoked by root and the mounted file system is owned
   by root, the daemon runs as root. This should be avoided.)
   Launching the daemon (and stopping) is done in 5 steps.
   Step 1:
   - Gathering information from command line, configuration files and
     environment.
   - Checking this information for consistency and any errors that would
     prevent successful running of the daemon.
   - Checking whether the the user has permissions to mount.
   - Checking whether the neccessary files and directories for running the
     daemon are available.
   Step 2:
   - The modules for connecting to the kernel, connecting to the WebDAV resource
     and for caching are initialised.
   If an error accurs during step 1 or step 2 an error message is printed and
   the program dies immediately. Clean up is left to the operating system.
   Step 3:
   - Forking into daemon mode.
   - While the daemon (child) writes the pid file and starts reading upcalls
     from the kernel in an endless loop, the parent process tries to mount the
     file system and write an entry into mtab (_PROC_MOUNTS).
   - If an error occurs in one of the processes it sends SIGTERM to the other.
     While the parent just dies, the daemon will run its normal exit code
     (see step 5). In rare cases this might nevertheless leave stale pid files
     or entries in mtab that must be cleaned manually by the administrator.
   - If mounting is successful the parent process exits with success.
   Step 4:
   - Running as daemon.
   Step 5:
   - Terminating.
   - The daemon has set a signal handler for SIGTERM and SIGHUP. If it gets one
     of these signals it tries to unmount the file system and resets the global
     variable keep_on_running. This will terminate the message loop gracefully.
   - If the file system is unmounted (by the umount programm), the message
     loop will terminate gracefully.
   - The close functions of the modules are called, that will clean up the
     cache, save cached information if neccessary and close the connections. */
int main(int argc, char *argv[]);

#endif
