/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/file_test_data_source.cpp
 *
 * @brief [LEVEL: alpha] Implementation @ref diagnostics::unittest::File_Test_Data_Source class
 *
 * $Id: file_test_data_source.cpp,v 1.7 2005/06/23 09:54:25 esdentem Exp $
 * 
 * @author Christian Schallhart
 */

#include <diagnostics/unittest/test_system/file_test_data_source.hpp>

#include <diagnostics/util/marshaling.hpp>

#include <diagnostics/unittest/test_system_exception.hpp>

#include <fstream>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

File_Test_Data_Source::File_Test_Data_Source() 
    : m_initialized(false)
{
}


File_Test_Data_Source::~File_Test_Data_Source() 
{
}

void File_Test_Data_Source::init(::std::string const & file_name) 
{
    m_data=Data_t();
    m_initialized=true;
    m_modified=false;
    m_file_name=file_name;
}

bool File_Test_Data_Source::is_initialized() const 
{
    return m_initialized;
}


void File_Test_Data_Source::load(::std::string const & file_name) 
{
    using namespace ::diagnostics::internal;
    
    ::std::ifstream file(file_name.c_str());
    if(!file.good()) throw Test_System_Exception("Could not open " + file_name);

    int const header_length=512;
    char header[header_length];
    for(int i=0;i<512;++i) header[i]='c';
    file.read(header,header_length);
    if(!file.good()) throw Test_System_Exception("Error while reading " + file_name);

    Data_t new_data;
    if(!read(file,new_data)) throw Test_System_Exception("Error while reading " + file_name);

    // nothing changed in this about this line -- no exceptions down here
    m_initialized=false;
    m_file_name=file_name;
    ::std::swap(m_data,new_data);
    m_initialized=true;
    m_modified=false;
}



void File_Test_Data_Source::save(::std::string const & file_name) 
{
	save(file_name,false);
}


void File_Test_Data_Source::save(::std::string const & file_name,
								 bool const force) 
{
    using namespace ::diagnostics::internal;

    if(!m_initialized) throw Test_System_Exception("Attempt to save an uninitialized File_Test_Data_Source");

    // unmodified file -- no force
    if(!m_modified && file_name==m_file_name && !force) return;
    
    ::std::ofstream file(file_name.c_str());
    if(!file.good()) throw Test_System_Exception("Could not open " + file_name);

    int const header_length=512;
    char header[header_length];
    for(int i=0;i<512;++i) header[i]='c';
    file.write(header,header_length);
    if(!file.good()) throw Test_System_Exception("Error while writing " + file_name);

    write(file,m_data);

    if(!file.good()) throw Test_System_Exception("Error while writing " + file_name);

    // nothing changed in this about this line -- no exceptions down here
    m_file_name=file_name;
    m_modified=false;
}

bool File_Test_Data_Source::is_modified() const 
{ 
    if(!m_initialized) throw Test_System_Exception("Uninitialized File_Test_Data_Source");
    return m_modified; 
}

::std::string const & File_Test_Data_Source::file_name() const 
{ 
    if(!m_initialized) throw Test_System_Exception("Uninitialized File_Test_Data_Source");
    return m_file_name; 
}


bool File_Test_Data_Source::exists_entry(::std::string const & id) const 
{
    if(!m_initialized) throw Test_System_Exception("Uninitialized File_Test_Data_Source");
    
    return m_data.end()!=m_data.find(id);
}


::std::string const & File_Test_Data_Source::get_entry(::std::string const & id) const 
{
    if(!m_initialized) throw Test_System_Exception("Uninitialized File_Test_Data_Source");

    Data_t::const_iterator iter(m_data.find(id));    
    if(iter==m_data.end()) throw Test_System_Exception("Entry with ID: "+id+" does not exist");
    return iter->second;
}

void File_Test_Data_Source::set_entry(::std::string const & id, ::std::string const & value) 
{
    if(!m_initialized) throw Test_System_Exception("Uninitialized File_Test_Data_Source");
    m_data[id]=value;
    m_modified=true;
}





UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

// vim:ts=4:sw=4
