/* consound.c */

#include <unistd.h>
#include <stdio.h>
#include <linux/kd.h>
#include <sys/timeb.h>
#include <signal.h>

#include "consound.h"
#include "ditty.h"

// The following is a magic number taken from /usr/src/linux/drivers/char/vt.h
// It is the rate of a standard timer board, but actualy, even if the timer
// were at a different rate, this number is used - it is hard-wired into the
// kernel, and in fact the value passed via ioctl is automatically divided
// by this number
#define	CONSOUND_RATE	1193180


void handleTerm (int sig)
     // this is a signal handler
     // to prevent someone from hitting ^C
     // and having a non-stop sound emit from
     // the console.
{
  ioctl (1, KIOCSOUND, 0);
  exit (sig);
}

void
consoundWait (int duration)
     // wait for duration milliseconds to expire
{
  struct timeb		tb;
  unsigned long long	start, end;

  ftime (&tb);
  start = (((unsigned long long)tb.time) * 1000ULL) +
    (unsigned long long) tb.millitm;

  do
    {
      ftime (&tb);
      end = (((unsigned long long)tb.time) * 1000ULL) +
	(unsigned long long) tb.millitm;
    }
  while (end < (start + ((unsigned long long)duration)));
}

void
consoundPlayToneProc (int duration, int pitch)
{
  int arg;

  if (pitch == 0)
    {
      consoundWait (duration);
      return;
    }
  
  arg = (CONSOUND_RATE/pitch);

  ioctl (1, KIOCSOUND, arg);

  consoundWait (duration);

  ioctl (1, KIOCSOUND, 0);
}


int
consoundInit (void)
{
  if (!isatty(1))	// assumes 1 as stdout's file descriptor
    {
      fputs ("Standard output is not a tty\n", stderr);
      return -1;
    }

  // even though the sound rate is actually 1.19 MHz,
  // we set this to 1000 because we need all duration values
  // to be specified in milliseconds.
  dittyRate = 1000;

  playTone = consoundPlayToneProc;
  playRest = consoundWait;

  if (signal (SIGINT, handleTerm) == SIG_ERR)
    return -1;

  return 0;
}
