/* ditty.c */

/* Copyright (C) 2000  Micah John Cowan
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *
 * micah@cowanbox.com */


#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <errno.h>
#include <string.h>

#include "ditty.h"

#ifdef USE_CON_SOUND
#include "consound.h"
#endif

#ifdef USE_DEV_DSP
#include "devdspsound.h"
#endif


#define DEFAULT_TEMPO	120
#define DEFAULT_OCTAVE	3


long		dittyRate;
int		tempo;
static char	inited;		// the initialization has been called if
                                // this is non-zero.

dittyInitProc		dittyInit;  //should return -1 if an error occurs.
dittyPlayToneProc	playTone;
dittyPlayRestProc	playRest;

// octave 5 frequencies, from C to B
int	noteFreqs[] = {
  1046, 1109, 1175, 1244, 1318, 1397,
  1480, 1568, 1661, 1760, 1865, 1975
};


// These two are set whenever specified, and used as the default
// whenever not specified in the note.  In otherwords, if we've
// been using quarter-notes in octave 3, ditty will assume we're
// continuing to use quarter-notes in octave 3 until specifically
// told otherwise.

int	octave;		// what octave register we are playing in
int	duration;	// what note lengths we are using


// This macro translates a (possibly fractional) number of beats
// into a duration value that can be used by playTone() and playRest()
#define BEATS2DUR(beats)	(beats * (((double)dittyRate) * 60.0/((double)tempo)))

// This macro takes a frequency (should be one of noteFreqs[]) and
// adjusts it based on what octaveit should be played in.
#define ADJUST_FOR_OCTAVE(freq, octv)	(freq / (pow (2.0, 5.0 - (double)octv)))

int
doInit (void)
     // If initialization has not yet occured, it calls dittyInit()
     // and exits if it's return value is not 0.  If initialization
     // has already occured, it returns 0.
{
  int i;
  
  if (inited)
    return 0;

  inited = 1;
  i = dittyInit();
  if (i)
    exit (i);
}


int
getIndexFromNote (char *note)
     // based on the first 1 or two characters
     // of a string, returns an index into noteFreqs[]
     // which corresponds to that note-name.
{
  int	i;
  
  switch (note[0])
    {
    case 'c':
      i = 0;
      break;
    case 'd':
      i = 2;
      break;
    case 'e':
      i = 4;
      break;
    case 'f':
      i = 5;
      break;
    case 'g':
      i = 7;
      break;
    case 'a':
      i = 9;
      break;
    case 'b':
      i = 11;
      break;
    default:
      i = -5;	// This will be noticed as an error.
                // This is a somewhat arbitrary number,
                // but it must be less than -1, because
                // that will be wrapped around to 11.
    }

  switch (note[1])
    {
    case '#':
      i++;
      break;
    case 'b':
      i--;
      break;
    }

  // Magic numbers, yes, I know.  But
  // there will always be 12 semi-tones
  // in an octave, and that is not likely to
  // change soon.
  if (i == -1)
    i = 11;
  if (i == 12)
    i = 0;

  return i;
}

int
getDuration (char *dur)
     // returns a duration based on the tail end
     // of the note string.
{
  char	*ptr;
  int	d, last;

  d = 0;
  ptr = dur;

  while (*ptr != '\0')
    {
      switch (*ptr)
	{
	case 'w':
	  d += BEATS2DUR(4);
	  break;
	case 'h':
	  d += BEATS2DUR(2);
	  break;
	case 'q':
	  d += BEATS2DUR(1);
	  break;
	case 'e':
	  d += BEATS2DUR(0.5);
	  break;
	case 's':
	  d += BEATS2DUR(0.25);
	  break;
	case 't':
	  d += BEATS2DUR(0.125);
	  break;
	}
      ptr++;
    }

  return d;
}

void
dittyPlayNote (char *note)
{
  char	*ptr;
  int 	dur, pitch, i, o;

  dur = duration;
  o = octave;

  i = getIndexFromNote (note);

  ptr = note + 1;

  if (*ptr == '#' || *ptr  == 'b')
    ptr++;

  if (*ptr == '\0')
    goto play_note_finish;	// who *says* I can't use goto >:)

  if (isdigit (*ptr))
    {
      o = (*ptr - '0');
      if ( o == 0 || o > 5 )
	o = octave;
      ptr++;
    }

  if (*ptr == '\0')
    goto play_note_finish;

  dur = getDuration (ptr);
  
 play_note_finish:
  if (i < 0)
    {
      playRest (dur);
    }
  else
    {
      pitch = ADJUST_FOR_OCTAVE( (noteFreqs[i]), o );
      playTone (dur, pitch);
    }
    
  duration = dur;
  octave = o;
}


#define PROC_FILE_BUFFER_INCR	10

void
processInputFile (char *filename)
     // reads an input file in the same format
     // as arguments.  "-" means read from
     // stdin.
{
  ssize_t	err;
  char		*buffer;
  int		bufSize, index, fd;

  if (strcmp (filename, "-"))
    {
      fd = open (filename, O_RDONLY);
      if (fd == -1)
	{
	  fprintf (stderr, "Couldn't open file '%s': %s\n",
		   filename, strerror (errno));
	  exit (fd);
	}
    }
  else
    {
      fd = 0;		// assumes 0 is stdin's fd
    }
  
  bufSize = 0;
  buffer = NULL;	// realloc() acts like malloc() when
                        // this is NULL
  
  index = 0;

  do
    {
      do
	{
	  if ( (index+1) >= bufSize ) // the +1 is so we can easily
	                              // add a '\0' at the end
	    {
	      bufSize += PROC_FILE_BUFFER_INCR;
	      buffer = realloc (buffer, bufSize);
	      
	      if (buffer == NULL)
		{
		  fprintf (stderr, "Out of memory\n");
		  free (buffer);
		  close (fd);
		  exit (EXIT_FAILURE);
		}
	    }

	  err = read (fd, (buffer + index), 1);
	  if (err == -1)
	    {
	      fprintf (stderr, "Couldn't read from file '%s': %s\n",
		       filename, strerror (errno));
	      free (buffer);
	      close (fd);
	      return;
	    }

	  if (!(err == 0 || isspace(buffer[index])))
	    index++;
	}
      while ( !(isspace(buffer[index]) || (err == 0)) );

      buffer[index] = '\0';
      dittyPlayNote (buffer);

      index = 0;
    }
  while (err != 0);

  free (buffer);
  close (fd);
}


void
usage (FILE *fs)
{
  // Sorry if this looks wierd to you.  It just changes the usage
  // output based on what features were enabled at compile-time.
  // GCC will combine these adjacent strings into one big long string.
  // Note too that GCC allows newlines within strings.
  // Who cares about other compilers?  ;)
  
  fputs ("\n" PACKAGE " [options] notes\n", fs);
  fputs ("
OPTIONS:
    -v  Display current version and exit.

    -h  Display usage information.

    -k  Use the console (built-in) speaker."
#ifdef USE_CON_SOUND
"  This is the default."
#endif
"\n"
	 
#ifndef USE_CON_SOUND
"        (Note that the use of this option was disabled at compile-time).\n"
#endif

"
    -d  Use the soundcard (via /dev/dsp) instead of the console speaker.
        This is useful on terminals which don't support sound controls.\n"
#ifndef USE_DEV_DSP
"        (Note that the use of this option was disabled at compile-time).\n"
#endif
"
    -t  Sets the tempo at which the melody shall be played, in beats
        per minute.

    -f filename
        Reads notes from FILENAME.  This option may be specified multiple
        times, to play multiple files.  Additionally, the -t option may be
        used before each instance to set the tempo for each file.

See the manpage for details.

", fs);
  
}


int
main (int argc, char **argv)
{
  int i;
  
  octave = 3;
  tempo = DEFAULT_TEMPO;

#ifdef USE_CON_SOUND
  dittyInit = consoundInit;
#else
  dittyInit = devdspInit;
#endif

  do
    {
      i = getopt (argc, argv, "vt:kdf:h");
      if (i == -1) break;
      switch (i)
	{
	case 'v':
	  printf ("Version: %s %s\n", PACKAGE, VERSION);
	  return 0;
	  break;	// hardly necessary
	case 't':
	  tempo = strtol (optarg, NULL, 10);
	  if (tempo <= 0)
	    tempo = DEFAULT_TEMPO;
	  break;
#ifdef USE_CON_SOUND
	case 'k':
	  dittyInit = consoundInit;
	  break;
#endif
#ifdef USE_DEV_DSP
	case 'd':
	  dittyInit = devdspInit;
	  break;
#endif
	case 'f':
	  doInit();
	  
	  processInputFile (optarg);
	  break;
	case 'h':
	  usage (stdout);
	  break;
	case '?':
	  usage (stderr);
	  break;
	}
    }
  while (i != -1);

  if (argc == optind)
    return 0;

  doInit();
    
  for (i=optind; i<argc; i++)
    dittyPlayNote (argv[i]);
  
  return 0;
}
