;;; wl-mode.el -- for helping Dutch wordlist lovers to edit WL files.
;;; Copyright (C) 1996 Nederlandstalige TeX gebruikersgroep.
;;; Erick Branderhorst <branderh@iaehv.nl>, 1996

;; This file is part of NTG dutch.

;; NTG dutch is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; NTG dutch is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with NTG dutch; see the file COPYING.  If not, write to the
;; Free Software Foundation, 59 Temple Place - Suite 330, Boston,
;; MA 02111-1307, USA.

;;; This package provides the tools meant to help editing WL files,
;;; as documented in the NTG dutch user's manual.  See this manual
;;; for user documentation, which is not repeated here.

;;; To install, merely put this file somewhere GNU Emacs will find it,
;;; then add the following lines to your .emacs file:
;;;
;;;   (setq auto-mode-alist
;;;         (cons (cons "\\.wl[xf]?\\'\\|\\.wl\\." 'wl-mode) auto-mode-alist))
;;;   (autoload 'wl-mode "wl-mode")
;;;
;;; You may also adjust some customizations variables, below, by defining
;;; them in your .emacs file.

(defvar wl-next-goes-up t
  "*Next entry will go in the direction of the bottom of the wordlist.")

;;; WL mode variables and constants (usually not to customize).

(or (fboundp 'gettext) (defsubst gettext (string) string))
(defsubst _ (string) (gettext string))
(defsubst N_ (string) string)

(defun wl-mode-version ()
  "Show Emacs WL mode version."
  (interactive)
  (message (_"Emacs WL mode, version %s") (substring "$Revision: 1.1.1.1 $" 11 -2)))

(defvar wl-help-display-string
  (_"\
This is the help with wl-mode (beta beta beta).

Key    Entry               Key    Action
i, <   mark in             E      edit mode
o, >   mark out            C      next goes other direction
n      next unmarked       =      statistics
p      previous unmarked

Be aware that this is still beta software.
"))

;;; Mode activation.

(defvar wl-mode-map nil
  "Keymap for WL mode.")
(if wl-mode-map
    ()
  ;; The following line because (make-sparse-keymap) does not work on Demacs.
  (setq wl-mode-map (make-keymap))
  (suppress-keymap wl-mode-map)
  (define-key wl-mode-map " " 'wl-next-entry)
  (define-key wl-mode-map "?" 'wl-mark-entry-doubtfull)
  (define-key wl-mode-map "#" 'wl-mark-entry-missing)
  (define-key wl-mode-map "." 'wl-current-entry)
  (define-key wl-mode-map "<" 'wl-mark-entry-included)
  (define-key wl-mode-map "=" 'wl-statistics)
  (define-key wl-mode-map ">" 'wl-mark-entry-excluded)
  (define-key wl-mode-map "h" 'wl-help)
  (define-key wl-mode-map "k" 'wl-remove-entry)
  (define-key wl-mode-map "C" 'wl-next-goes-other-direction)
  (define-key wl-mode-map "n" 'wl-next-entry)
  (define-key wl-mode-map "i" 'wl-mark-entry-included)
  (define-key wl-mode-map "o" 'wl-mark-entry-excluded)
  (define-key wl-mode-map "p" 'wl-previous-entry)
  (define-key wl-mode-map "v" 'wl-mode-version)
  (define-key wl-mode-map "E" 'wl-edit-out-full)
  )

(defun wl-edit-out-full ()
  "Go into recursive edit for editing the WL file in fundamental mode."
  (interactive)
  (if (y-or-n-p (_"Should I let you edit the whole WL file? "))
      (progn
	(message (_"Type `C-c C-c' once done"))
	(let ((buffer-read-only nil))
	  (fundamental-mode)
	  (use-local-map wl-subedit-mode-map)
	  (recursive-edit)
	  (use-local-map nil)
	  (wl-mode)))))

(defvar wl-subedit-mode-map nil
  "Keymap while editing a WL mode entry (or the full WL file).")
(if wl-subedit-mode-map
    ()
  (setq wl-subedit-mode-map (make-sparse-keymap))
  (define-key wl-subedit-mode-map "\C-c\C-c" 'exit-recursive-edit))

(defun wl-mode ()
  "Major mode for translators when they edit WL files.
Special commands:\\{wl-mode-map}
Turning on WL mode calls the value of the variable `wl-mode-hook',
if that value is non-nil.  Behaviour may be adjusted with
variables wl-next-goes-up and ...."

  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'wl-mode)
  (setq mode-name "WL")
  (use-local-map wl-mode-map)
  (setq buffer-read-only t)

  ;; A WORK-BUFFER is associated with this WL file, for edition and other
  ;; various tasks.  WORK-BUFFER-LOCK indicates that the work buffer
  ;; is already in use, most probably editing some string through Emacs
  ;; recursive edit.  In this case, one cannot modify the buffer.

  (make-local-variable 'wl-work-buffer)
  (make-local-variable 'wl-work-buffer-lock)
  (setq wl-work-buffer
	(generate-new-buffer (concat "*Edit " (buffer-name nil) "*")))
  (setq wl-work-buffer-lock nil)

  ;; ENTRY-TYPE classifies the entry.

  (make-local-variable 'wl-entry-type)

  ;; A few counters are usefully shown in the Emacs mode line.

  (make-local-variable 'wl-plus-counter)
  (make-local-variable 'wl-in-counter)
  (make-local-variable 'wl-out-counter)
  (make-local-variable 'wl-todo-counter)
  (make-local-variable 'wl-equal-counter)
  (make-local-variable 'wl-questionable-counter)
  (make-local-variable 'wl-attention-counter)

  (make-local-variable 'wk-mode-line-string)
  (setq wl-mode-flag t)			; made local in `Window' page below

;;  (cal-variable 'wl-marker-stack)
  (setq wl-marker-stack nil)

  ;; SEARCH path contains a list of directories where files may be found,
  ;; in a format suitable for read completion.  Each directory includes
  ;; its trailing slash.  WL mode starts with "./" and "../".

  (make-local-variable 'wl-search-path)
  (setq wl-search-path '(("./") ("../")))

  ;; Message will be shown on screen to remind user that help available.

  (message (_"You may type `h' for a short WL mode reminder."))
  (run-hooks 'wl-mode-hook)
  (provide 'wl-mode))

(defun wl-mode-destroy-cleanup ()
  "When destroying a WL mode buffer, kill editing buffer as well."
  (and (string= mode-name "WL")
       (bufferp wl-work-buffer)
       (kill-buffer wl-work-buffer)))

(if (boundp 'kill-buffer-hook)
    (add-hook 'kill-buffer-hook 'wl-mode-destroy-cleanup))

;;; Regexps
(defvar wl-word-regexp "\\([^-+#=?<>\n]*\\)"
  "Returns true when matching a word")

(defvar wl-last-word-regexp
  (concat wl-word-regexp "\n")
  "Regexp being true after a word and an end of line matched.")

(defvar wl-entry-regexp
  "^[ \t\n]*\\([-+#=?]\\)[ \t\n]*\\([^ <>\t\n]*\\)\\([<>]*\\)[ \t\n]*$"
  "Regexp being true when matching a wl entry with optional whitespace.")

(defvar wl-done-entry-regexp
  "^\\([-+#=?]\\)\\([^ <>\t\n]*\\)[<>]$"
  "Regexp being true when matching a wl entry with optional whitespace.")

(defvar wl-identical-entry-regexp
  (concat "^=" wl-last-word-regexp)
  "Regexp being true after a = and a word match.")

(defvar wl-entry-with-replacement-regexp
  (concat "^-" wl-word-regexp ":" wl-last-word-regexp)
  "Regexp being true when matching -word:replacement\n.")

(defvar wl-entry-with-replacements-regexp
  (concat "^-" wl-word-regexp ":" wl-word-regexp "|" wl-last-word-regexp)
  "Regexp being true when matching -word:repl1|repl2\n.")

(defvar wl-questionable-entry-regexp
  (concat "^?" wl-last-word-regexp)
  "Regexp being true when matching questionable word.")

(defvar wl-missing-entry-regexp
  (concat "^#" wl-last-word-regexp)
  "Regexp being true when matching missing word.")

(defvar wl-added-entry-regexp
  (concat "^+" wl-last-word-regexp)
  "Regexp being true when matching an `added' word (I.e. a word not in GB).")

(defvar wl-excluded-entry-regexp
  (concat "^>" wl-last-word-regexp)
  "Regexp being true when matching a word which is marked `excluded'.")

(defvar wl-included-entry-regexp
  (concat "^<" wl-last-word-regexp)
  "Regexp being true when matching a word which is marked `included'.")

;;; Functions 

(defun wl-find-span-of-entry () 
  "Find the extent of the wl file entry where the cursor is.  Set
WL-ENTRY-TYPE to meaninful value."
  (let ((here (point)))
    (beginning-of-line)

    ;; Classify the entry.
    (setq wl-entry-type
	  (if (eq (following-char) =)
	      'identical))

    ;; Put the cursor back where it was.
    (goto-char here))

  ;; Put the cursor at the end of the line so that the next regexp
  ;; search will not find the entry at the current line.  It
  ;; wl-next-goes-up is nil the search will go in the direction of
  ;; the beginning of the file so we put the cursor at the beginning
  ;; of the line so that this entry isn't found.
  (and wl-next-goes-up
       (end-of-line)
       (beginning-of-line)))

(defun wl-next-word-with-regexp (regexp wrap)
  "Will search for the next word matching the regexp given as argument.
The direction can be either up- or downwards depending on the value of
wl-next-goes-up."
  (interactive)
  (progn (next-line 1))
  (message (_"wl-next-word-with-regexp"))
  (and wl-next-goes-up))

(defun wl-next-word ()
  "Will go to the next unmarked word in the list."
  (interactive)
  (beginning-of-line)
  (setq buffer-read-only nil)
  (if (looking-at wl-entry-regexp)
      (replace-match 
       (concat 
	(match-string 1) (match-string 2))))
  (setq buffer-read-only t)
  (if wl-next-goes-up (next-line 1) (previous-line 1)))

(defun wl-next-entry ()
  "Will go to the next unmarked word in the list."
  (interactive)
  (beginning-of-line)
  (if (not wl-next-goes-up) (next-line 1) (previous-line 1))
  (while (looking-at wl-done-entry-regexp)
    (if (not wl-next-goes-up) (next-line 1) (previous-line 1))))

(defun wl-previous-entry ()
  "Will go to the previous unmarked word in the list."
  (interactive)
  (beginning-of-line)
  (if (not wl-next-goes-up) (previous-line 1) (next-line 1))
  (while (looking-at wl-done-entry-regexp)
    (if (not wl-next-goes-up) (previous-line 1) (next-line 1))))

(defun wl-next-goes-other-direction ()
  "Will change the direction next goes."
  (interactive)
  (setq wl-next-goes-up (not wl-next-goes-up)))

(defun wl-mark-entry-included ()
  "Will mark this entry included."
  (interactive)
  (wl-change-entry-type "+" "<"))

(defun wl-mark-entry-excluded ()
  "Will mark this entry excluded."
  (interactive)
  (wl-change-entry-type "+" ">"))

(defun wl-mark-entry-similar ()
  "Will mark this entry similar."
  (interactive)
  (wl-change-entry-type "+" "="))

(defun wl-mark-entry-questionable ()
  "Will mark this entry similar."
  (interactive)
  (wl-change-entry-type "+" "?"))

(defun wl-mark-entry-missing ()
  "Will mark this entry similar."
  (interactive)
  (wl-change-entry-type "+" "#"))

(defun wl-change-entry-type (ot nt)
  "Will change entry type and remove obsolete whitespace as well."
  (interactive (let (nt (read-string "c" ))))
  (beginning-of-line)
  (setq buffer-read-only nil)
  (if (looking-at (concat "^\\(" ot "\\)[ \t\n]*\\([^ ><\t\n]*\\)[ ><\t\n]*$"))
      (replace-match (concat (match-string 1) (match-string 2) nt)))
  (setq buffer-read-only t)
  (wl-next-entry))
;;       (if (and (> (length (match-string 2)) wl-min-length-for-search)
;;		(< (length (match-string 2)) wl-max-length-for-search))
;;	   (wl-search-for (match-string 2)))

(defun wl-search-for (search-for)
  (while (or (and (wl-next-goes-up 
		   (re-search-backward 
		    (concat "^+[ \t\n]*\\(.*" search-for ".*\\)[ \t\n]*$"))))
	     (and (not (wl-next-goes-up) 
		       (re-search-forward
			(concat "^+[ \t\n]*\\(.*" search-for ".*\\)[ \t\n]*$")))))
    (message match-string 1)))

;; Miscellaneous features.

(defun wl-help ()
  "Provide an help window for WL mode."
  (interactive)
;;;  (wl-check-lock)
;;;  (pop-to-buffer wl-work-buffer)
  (save-window-excursion
    (switch-to-buffer wl-work-buffer)
    (erase-buffer)
    (insert wl-help-display-string)
    (delete-other-windows)
    (goto-char (point-min))
    (message (_"Type any character to continue"))
    (read-char))
  (bury-buffer wl-work-buffer))

(defun wl-compute-counters (flag)
  "Prepare counters for mode line display.  If FLAG, also echo entry position."
  (setq wl-plus-counter 0)
  (setq wl-in-counter 0)
  (setq wl-out-counter 0)
  (setq wl-todo-counter 0)
  (setq wl-equal-counter 0)
  (setq wl-questionable-counter 0)
  (setq wl-attention-counter 0)

  (let ((position 0) (total 0) here)
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward wl-entry-regexp nil t)
	(and (= (% total 20) 0)
	     (message (_"Position %d") total))
	(setq here (point))
	(setq total (1+ total))
	(and flag (setq position total))
	(goto-char (match-beginning 1))
	(cond ((eq (following-char) ?+)
	       (setq wl-plus-counter (1+ wl-plus-counter))
	       (goto-char (match-beginning 3))
	       (cond ((eq (following-char) ?<)
		      (setq wl-in-counter (1+ wl-in-counter)))
		     ((eq (following-char) ?>)
		      (setq wl-out-counter (1+ wl-out-counter)))
		     ((setq wl-todo-counter (1+ wl-todo-counter)))))
	      ((eq (following-char) ?=)
	       (setq wl-equal-counter (1+ wl-equal-counter)))
	      ((eq (following-char) ?-)
	       (setq wl-attention-counter (1+ wl-attention-counter)))
	      ((eq (following-char) ??)
	       (setq wl-questionable-counter (1+ wl-questionable-counter))))
	(goto-char here))

    ;; Push the results out.
    (if flag
	(message (_"\
Total %d; %d markable (%d/%d/%d in/out/un), %d attention")
		 total wl-plus-counter wl-in-counter 
		 wl-out-counter wl-todo-counter wl-attention-counter)
      (message "")))))

(defun wl-statistics ()
  "Say how many entries in each category, and the current position."
  (interactive)
  (wl-compute-counters t))

(defun wl-current-entry ()
  "Say how many entries in each category, and the current position."
  (interactive))

;; Window management
(defun wl-check-lock ()
  "Ensure that GNU Emacs is not currently in recursive edit for WL mode."
  (if wl-work-buffer-lock
      (progn
	(pop-to-buffer wl-work-buffer)
	(if (y-or-n-p (_"Here is your current edit.  Do you wish to abort it? "))
	    (abort-recursive-edit)
	  (error (_"Type `C-c C-c' once done"))))))
