#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <ctype.h>

#ifdef DVB_IN_KERNEL
#  include <linux/ost/frontend.h>
#  include <linux/ost/dmx.h>
#else
#  include <ost/frontend.h>
#  include <ost/dmx.h>
#endif


#define FRONTEND_DEV "/dev/ost/frontend0"
#define DEMUX_DEV "/dev/ost/demux0"

#define CHANNEL_FILE "/.tzap/channels.conf"

#define ERROR(x...)               \
   do {                           \
      fprintf(stderr, "ERROR: "); \
      fprintf(stderr, x);         \
      fprintf (stderr, "\n");     \
   } while (0)


typedef struct {
   char *name;
   int value;
} Param;

static const Param si_list [] = { { "INVERSION_AUTO", INVERSION_AUTO },
                                  { "INVERSION_OFF", INVERSION_OFF },
                                  { "INVERSION_ON", INVERSION_ON } };

static const Param bw_list [] = { { "BANDWIDTH_6_MHZ", BANDWIDTH_6_MHZ },
                                  { "BANDWIDTH_7_MHZ", BANDWIDTH_7_MHZ },
                                  { "BANDWIDTH_8_MHZ", BANDWIDTH_8_MHZ } };

static const Param fec_list [] = { { "FEC_1_2", FEC_1_2 },
                                   { "FEC_2_3", FEC_2_3 },
                                   { "FEC_3_4", FEC_3_4 },
                                   { "FEC_5_6", FEC_5_6 },
                                   { "FEC_7_8", FEC_7_8 },
                                   { "FEC_AUTO", FEC_AUTO },
                                   { "FEC_NONE", FEC_NONE } };

static const Param guard_list [] = {
   { "GUARD_INTERVAL_1_16", GUARD_INTERVAL_1_16 },
   { "GUARD_INTERVAL_1_32", GUARD_INTERVAL_1_32 },
   { "GUARD_INTERVAL_1_4", GUARD_INTERVAL_1_4 },
   { "GUARD_INTERVAL_1_8", GUARD_INTERVAL_1_8 }
};

static const Param hierarchy_list [] = { { "HIERARCHY_1", HIERARCHY_1 },
                                         { "HIERARCHY_2", HIERARCHY_2 },
                                         { "HIERARCHY_4", HIERARCHY_4 },
                                         { "HIERARCHY_NONE", HIERARCHY_NONE } };

static const Param constellation_list [] = { { "QPSK", QPSK },
	                                     { "QAM_128", QAM_128 },
                                             { "QAM_16", QAM_16 },
                                             { "QAM_256", QAM_256 },
                                             { "QAM_32", QAM_32 },
                                             { "QAM_64", QAM_64 } };

static const Param transmissionmode_list [] = {
   { "TRANSMISSION_MODE_2K", TRANSMISSION_MODE_2K },
   { "TRANSMISSION_MODE_8K", TRANSMISSION_MODE_8K },
};

#define LIST_SIZE(x) sizeof(x)/sizeof(Param)


static
int parse_param (int fd, const Param *plist, int list_size, int *param)
{
   char c;
   int character = 0;
   int index = 0;

   while (1) {
      if (read (fd, &c, 1) < 1)
         return -1;           /*  EOF? */

      if ((c == ':' || c == '\n') && plist->name[character] == '\0')
         break;

      while (toupper(c) != plist->name[character]) {
         index++;
	 plist++;
	 if (index >= list_size)
            return -2;        /*  parse error, no valid parameter name found */
      }

      character++;
   }

   *param = plist->value;
   
   return 0;
}


static 
int parse_int (int fd, int *val)
{
   char number [11];       /* 2^32 needs 10 digits... */
   int character = 0;

   while (1) {
      if (read (fd, &number[character], 1) < 1)
         return -1;           /*  EOF? */

      if (number[character] == ':' || number[character] == '\n') {
         number[character] = '\0';
         break;
      }

      if (!isdigit(number[character]))
         return -2;           /*  parse error, not a digit... */      

      character++;

      if (character > 10)
         return -3;           /*  overflow, number too big to fit in 32 bit */
   };

   *val = strtol (number, NULL, 10);
   
   return 0;
}


static
int find_channel (int fd, const char *channel)
{
   int character = 0;

   while (1) {
      char c;

      if (read (fd, &c, 1) < 1)
         return -1;                   /*  EOF! */

      if (c == ':' && channel[character] == '\0')
         break;

      if (toupper(c) == toupper(channel[character]))
         character++;
      else
         character = 0;
   };

   return 0;
}


static
int try_parse_int (int fd, int *val, const char *pname)
{
   int err;

   err = parse_int (fd, val);

   if (err)
      ERROR ("error while parsing %s (%s)!", pname,
             err == -1 ? "end of file" :
             err == -2 ? "not a number" : "number too big");
   
   return err;
}


static
int try_parse_param (int fd, const Param *plist, int list_size, int *param,
                     const char *pname)
{
   int err;

   err = parse_param (fd, plist, list_size, param);

   if (err)
      ERROR ("error while parsing %s (%s)!", pname,
             err == -1 ? "end of file" : "syntax error");

   return err;
}


int parse (const char *fname, const char *channel, FrontendParameters *frontend,
           int *vpid, int *apid)
{
   int fd;
   int err;

   if ((fd = open (fname, O_RDONLY | O_NONBLOCK)) < 0) {
      ERROR ("could not open file '%s'!", fname);
      return -1;
   }

   if (find_channel (fd, channel) < 0) {
      ERROR ("could not find channel '%s' in channel list!", channel);
      return -2;
   }

   if ((err = try_parse_int (fd, &frontend->Frequency, "Frequency")))
      return -3;

   if ((err = try_parse_param (fd, si_list, LIST_SIZE(si_list), 
                               (int*) &frontend->Inversion,
                               "SpectralInversion")))
      return -4;

   if ((err = try_parse_param (fd, bw_list, LIST_SIZE(bw_list), 
                               (int*) &frontend->u.ofdm.bandWidth,
                               "bandWidth")))
      return -5;

   if ((err = try_parse_param (fd, fec_list, LIST_SIZE(fec_list), 
                               (int*) &frontend->u.ofdm.HP_CodeRate,
                               "HP_CodeRate")))
      return -6;

   if ((err = try_parse_param (fd, fec_list, LIST_SIZE(fec_list), 
                               (int*) &frontend->u.ofdm.LP_CodeRate,
                               "LP_CodeRate")))
      return -7;

   if ((err = try_parse_param (fd, constellation_list,
                               LIST_SIZE(constellation_list), 
                               (int*) &frontend->u.ofdm.Constellation,
                               "Constellation")))
      return -8;

   if ((err = try_parse_param (fd, transmissionmode_list,
                               LIST_SIZE(transmissionmode_list), 
                               (int*) &frontend->u.ofdm.TransmissionMode,
                               "TransmissionMode")))
      return -9;

   if ((err = try_parse_param (fd, guard_list, LIST_SIZE(guard_list), 
                               (int*) &frontend->u.ofdm.guardInterval,
                               "guardInterval")))
      return -10;

   if ((err = try_parse_param (fd, hierarchy_list,
                               LIST_SIZE(hierarchy_list), 
                               (int*) &frontend->u.ofdm.HierarchyInformation,
                               "HierarchyInformation")))
      return -11;

   if ((err = try_parse_int (fd, vpid, "Video PID")))
      return -12;

   if ((err = try_parse_int (fd, apid, "Audio PID")))
      return -13;

   close (fd);

   return 0;
}



static
int setup_demux (int video_pid, int audio_pid)
{
   int vfd, afd;
   struct dmxPesFilterParams pesfilter;

   if ((vfd = open (DEMUX_DEV, O_RDWR)) < 0) {
      fprintf (stderr, "failed opening '%s' !!!\n", DEMUX_DEV);
      return -1;
   }

   if ((afd = open (DEMUX_DEV, O_RDWR)) < 0) {
      fprintf (stderr, "failed opening '%s' !!!\n", DEMUX_DEV);
      return -1;
   }

   pesfilter.pid = video_pid;
   pesfilter.input = DMX_IN_FRONTEND;
   pesfilter.output = DMX_OUT_DECODER;
   pesfilter.pesType = DMX_PES_VIDEO;
   pesfilter.flags = DMX_IMMEDIATE_START;

   if (ioctl (vfd, DMX_SET_PES_FILTER, &pesfilter) < 0) {
      fprintf (stderr, "ioctl(DMX_SET_PES_FILTER) for Video PID failed !\n");
      return -1;
   }
   pesfilter.pid = audio_pid;
   pesfilter.input = DMX_IN_FRONTEND;
   pesfilter.output = DMX_OUT_DECODER;
   pesfilter.pesType = DMX_PES_AUDIO;
   pesfilter.flags = DMX_IMMEDIATE_START;

   if (ioctl (afd, DMX_SET_PES_FILTER, &pesfilter) < 0) {
      fprintf (stderr, "ioctl(DMX_SET_PES_FILTER) for Audio PID failed !\n");
      return -1;
   }

   close (vfd);
   close (afd);
   return 0;
}

static
int setup_frontend (FrontendParameters *frontend)
{
   int fd;

   if ((fd = open (FRONTEND_DEV, O_RDWR)) < 0) {
      fprintf (stderr, "failed opening '%s' !!!\n", FRONTEND_DEV);
      return -1;
   }

   ioctl (fd, FE_SET_FRONTEND, frontend);

   close (fd);
   return 0;
}


static
int check_frontend (void)
{
   int fd;
   FrontendStatus status;

   if ((fd = open (FRONTEND_DEV, O_RDWR)) < 0) {
      fprintf (stderr, "failed opening '%s' !!!\n", FRONTEND_DEV);
      return -1;
   }

   do {
      ioctl (fd, FE_READ_STATUS, &status);
      printf (" status == 0x%02x: ", status);
      if (status & FE_HAS_LOCK)
         printf ("FE_HAS_LOCK");
      printf ("\n");
      usleep (1000000);
   } while (!(status & FE_HAS_LOCK));

   close (fd);
   return 0;
}


int main (int argc, char **argv)
{
   FrontendParameters frontend;
   int vpid, apid;
   char *confname;
   char *homedir = getenv ("HOME");

   if (argc != 2) {
      printf ("\n\tusage: %s <channel name>\n\n", argv[0]);
      exit (-1);
   }

   if (!homedir)
      ERROR("$HOME not set!");

   confname = malloc (strlen(homedir) + strlen(CHANNEL_FILE) + 1);
   memcpy (confname, homedir, strlen(homedir));
   memcpy (confname + strlen(homedir), CHANNEL_FILE, strlen(CHANNEL_FILE) + 1);

   memset (&frontend, 0, sizeof(FrontendParameters));

   if (parse (confname, argv[1], &frontend, &vpid, &apid))
      return -1; 

   setup_frontend (&frontend);
   setup_demux (vpid, apid);

   return check_frontend ();
}

