/*
 * dvb.c: driver for the Siemens Fujitsu DVB PCI card
 *
 * Copyright (C) 1999-2001 Ralph  Metzler 
 *                       & Marcus Metzler for convergence integrated media GmbH
 *
 * originally based on code by:
 *
 * Copyright (C) 1998,1999 Christian Theiss <mistert@rz.fh-augsburg.de>
 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 * 

 * The author can be reached at ralph@convergence.de, 

 * the project's page is at http://linuxtv.org/dvb/
 */

#define NEW_CI 1

#define __KERNEL_SYSCALLS__
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kmod.h>
#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/timer.h>
#include <linux/unistd.h>
#include <linux/byteorder/swabb.h>
#include <linux/poll.h>
#include <linux/slab.h>
#include <linux/smp_lock.h>
#include <stdarg.h>

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/pci.h>
#include <asm/system.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <asm/semaphore.h>
#include <linux/init.h>
#include <linux/vmalloc.h>

#include <linux/netdevice.h>
#include <linux/inetdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>


#include "tuner.h"
#include "dvb.h"


#include "saa7146_core.h"
#include "saa7146_v4l.h"
#include "saa7146_defs.h"

#ifdef MODULE
MODULE_DESCRIPTION("");
MODULE_AUTHOR("Ralph Metzler, Marcus Metzler, others");
#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif
MODULE_PARM(debug,"i");
MODULE_PARM(fdebug,"i");
MODULE_PARM(vidmem,"l");
MODULE_PARM(vidlow,"l");
MODULE_PARM(vidmode,"i");
MODULE_PARM(init_chan,"i");
MODULE_PARM(pids_off,"i");
#ifdef INC_FIRM
MODULE_PARM(readfirm,"i");
#endif
MODULE_PARM(adac,"i");
#endif

static int AV_StartPlay(struct dvb_struct *dvb, int av);
static void restart_feeds(struct dvb_struct *dvb);
static int bootarm(struct dvb_struct *dvb);
static inline int i2c_writereg(struct dvb_struct *dvb, u8 id, u8 reg, u8 val);
static int secSetTone(struct dvb_struct *dvb, secToneMode mode);
static int tune(struct dvb_struct *dvb, FrontendParameters *para);
static int secSetVoltage(struct dvb_struct *dvb, secVoltage voltage);
static inline u8 i2c_readreg(struct dvb_struct *dvb, u8 id, u8 reg);
static int  outcom(struct dvb_struct *dvb, int type, int com, int num, ...);
static void SetMode(struct dvb_struct *dvb, int mode);
static void ChangePIDs(struct dvb_struct *dvb, u16 vpid, u16 apid, u16 ttpid, 
                       u16 subpid, u16 pcrpid);
static void SetFEPIDs(struct dvb_struct *dvb);

void pes_to_ts(u8 const *buf, long int length, u16 pid, p2t_t *p);
void p_to_t(u8 const *buf, long int length, u16 pid, u8 *counter, dvb_demux_feed_t *feed);

static u32 vidmem	= 0;
static u32 vidlow	= 0;

static int debug = 0;
#define dprintk	if (debug) printk
static int fdebug = 0;
#define fprintk	if (fdebug) printk

static int vidmode=CVBS_RGB_OUT;
static int init_chan=0;
static int pids_off=1;
#ifdef INC_FIRM
static int readfirm=0;
#endif
static int adac=DVB_ADAC_TI;

static int num_dvb = 0;

#define saacomm(x,y) dvb->saa->command(dvb->i2cbus, (x), (y))

struct dvb_struct dvbs[MAX_NUM_DVB];


/****************************************************************************
 * General helper functions
 ****************************************************************************/

/* taken from linux/drivers/sound/sound_firmware.c */
/* Maybe the kernel should offer this somewhere else, not just in soundcore */

static int errno;

static int do_firmread(const char *fn, char **fp)
{
        int fd;
	loff_t l;
	char *dp;
       
	fd = open(fn, 0, 0);
	if (fd == -1) {
                dprintk(KERN_INFO "Unable to load '%s'.\n", fn);
		return 0;
	}
	l = lseek(fd, 0L, 2);
	if (l <= 0 || l > 300*1024) {
	        printk(KERN_INFO "Firmware too large '%s'\n", fn);
		sys_close(fd);
		return 0;
	}
	lseek(fd, 0L, 0);
	*fp= dp = vmalloc(l);
	if (dp == NULL)	{
		printk(KERN_INFO "Out of memory loading '%s'.\n", fn);
		sys_close(fd);
		return 0;
	}
	if (read(fd, dp, l) != l) {
		printk(KERN_INFO "Failed to read '%s'.\n", fn);
		vfree(dp);
		sys_close(fd);
		return 0;
	}
	sys_close(fd);
	*fp = dp;
	return (int) l;
}

static
int firmread(const char *fn, char **fp)
{
	int r;
	mm_segment_t fs = get_fs();
        
	set_fs(get_ds());
	r = do_firmread(fn, fp);
	set_fs(fs);
	return r;
}

static inline void ddelay(int i) 
{
        current->state=TASK_INTERRUPTIBLE;
        schedule_timeout((HZ*i)/100);
}


/****************************************************************************
 * GPIO and DEBI functions
 ****************************************************************************/

#define saaread(adr)            saa7146_read(saamem,(adr))
#define saawrite(dat,adr) 	saa7146_write(saamem,(adr),(dat))

inline static void 
setgpio(struct dvb_struct *dvb, int port, u32 data)
{
        void *saamem=dvb->saa_mem;
        u32 val;

        val=saaread(GPIO_CTRL);
        val&=~(0xff << (8*(port)));
        val|=(data)<<(8*(port));
        saawrite(val, GPIO_CTRL);
}

/* This DEBI code is based on the Stradis driver 
   by Nathan Laredo <laredo@gnu.org> */

static int debiwait_maxwait = 5000;

static int wait_for_debi_done(struct dvb_struct *dvb)
{
        void *saamem=dvb->saa_mem;
	int i;

	/* wait for registers to be programmed */
	for (i = 0; i < 100000; i++)
                if (saaread(MC2) & 2)
                        break;;
	/* wait for transfer to complete */
	for (i = 0; i < 500000 &&
	     (saaread(PSR) & SPCI_DEBI_S); i++)
		saaread(MC2);
	if (i > debiwait_maxwait)
		printk("wait-for-debi-done maxwait: %d\n",
			debiwait_maxwait = i);
	
	if (i == 500000)
		return -1;
	return 0;
}

static int debiwrite(struct dvb_struct *dvb, u32 config, 
                     int addr, u32 val, int count)
{
        void *saamem=dvb->saa_mem;
	u32 cmd;

	if (count <= 0 || count > 32764)
		return -1;
	if (wait_for_debi_done(dvb) < 0)
		return -1;
	saawrite(config, DEBI_CONFIG);
	if (count <= 4)		/* immediate transfer */
		saawrite(val, DEBI_AD);
	else			/* block transfer */
		saawrite(dvb->debi_bus, DEBI_AD);
	saawrite((cmd = (count << 17) | (addr & 0xffff)), DEBI_COMMAND);
	saawrite((2 << 16) | 2, MC2);
	return 0;
}

static u32 debiread(struct dvb_struct *dvb, u32 config, int addr, int count)
{
        void *saamem=dvb->saa_mem;
	u32 result = 0;

	if (count > 32764 || count <= 0)
		return 0;
	if (wait_for_debi_done(dvb) < 0)
		return 0;
	saawrite(dvb->debi_bus, DEBI_AD);
	saawrite((count << 17) | 0x10000 | (addr & 0xffff),
		 DEBI_COMMAND);
	saawrite(config, DEBI_CONFIG);
	saawrite((2 << 16) | 2, MC2);
	if (count > 4)	
		return count;
	wait_for_debi_done(dvb);
	result = saaread(DEBI_AD);
        result &= (0xffffffffUL >> ((4-count)*8));
	return result;
}

/* DEBI during interrupt */

static inline void 
iwdebi(struct dvb_struct *dvb, u32 config, int addr, u32 val, int count)
{
        if (count>4 && val)
                memcpy(dvb->debi_virt, (char *) val, count);
        debiwrite(dvb, config, addr, val, count);
}

static inline u32 
irdebi(struct dvb_struct *dvb, u32 config, int addr, u32 val, int count)
{
        u32 res;

        res=debiread(dvb, config, addr, count);
        if (count<=4) 
                memcpy(dvb->debi_virt, (char *) &res, count);
        return res;
}

/* DEBI outside interrupts, only for count<=4! */

static inline void 
wdebi(struct dvb_struct *dvb, u32 config, int addr, u32 val, int count)
{
        unsigned long flags;

        spin_lock_irqsave(&dvb->debilock, flags);
        debiwrite(dvb, config, addr, val, count);
        spin_unlock_irqrestore(&dvb->debilock, flags);
}

static inline u32 
rdebi(struct dvb_struct *dvb, u32 config, int addr, u32 val, int count)
{
        unsigned long flags;
        u32 res;

        spin_lock_irqsave(&dvb->debilock, flags);
        res=debiread(dvb, config, addr, count);
        spin_unlock_irqrestore(&dvb->debilock, flags);
        return res;
}


static inline char 
chtrans(char c)
{
        if (c<32 || c>126)
                c=0x20;
        return c;
}


/* handle mailbox registers of the dual ported RAM */

static inline void 
ARM_ResetMailBox(struct dvb_struct *dvb)
{
        unsigned long flags;

        spin_lock_irqsave(&dvb->debilock, flags);
        debiread(dvb, DEBINOSWAP, IRQ_RX, 2);
        //printk("dvb: IRQ_RX=%d\n", debiread(dvb, DEBINOSWAP, IRQ_RX, 2));
        debiwrite(dvb, DEBINOSWAP, IRQ_RX, 0, 2);
        spin_unlock_irqrestore(&dvb->debilock, flags);
}

static inline void 
ARM_ClearMailBox(struct dvb_struct *dvb)
{
        iwdebi(dvb, DEBINOSWAP, IRQ_RX, 0, 2);
}

static inline void 
ARM_ClearIrq(struct dvb_struct *dvb)
{
	irdebi(dvb, DEBINOSWAP, IRQ_RX, 0, 2);
}

static void 
reset_arm(struct dvb_struct *dvb)
{
        setgpio(dvb, RESET_LINE, GPIO_OUTLO);

        /* Disable DEBI and GPIO irq */
        saa7146_write(dvb->saa_mem, IER, 
                      saa7146_read(dvb->saa_mem, IER) & 
                      ~(MASK_19 | MASK_03));
        saa7146_write(dvb->saa_mem, ISR, (MASK_19 | MASK_03));

        mdelay(800);
        setgpio(dvb, RESET_LINE, GPIO_OUTHI);

        ARM_ResetMailBox(dvb); 
        saa7146_write(dvb->saa_mem, ISR, (MASK_19 | MASK_03));
        saa7146_write(dvb->saa_mem, IER, 
                      saa7146_read(dvb->saa_mem, IER) | MASK_03 );

        dvb->arm_ready=1;
        printk("dvb: ARM RESET\n");
}

static void 
recover_arm(struct dvb_struct *dvb)
{
        init_chan=0;
        if (current->files)
                bootarm(dvb);
        else {
                printk("OOPS, no current->files\n");
                reset_arm(dvb);
        }
        ddelay(10); 
        dvb_frontend_demod_command(&dvb->frontend, FE_INIT, 0); 

	memcpy (&dvb->frontend.new_param, 
                &dvb->frontend.param, sizeof(FrontendParameters));
        dvb_frontend_tune(&dvb->frontend, &dvb->frontend.new_param);
        restart_feeds(dvb);
}

static void 
arm_error(struct dvb_struct *dvb)
{
        dvb->arm_errors++;
        dvb->arm_ready=0;
        recover_arm(dvb);
}

static int arm_thread(void *data)
{
	struct dvb_struct *dvb = data;
        u16 newloops;
    
	lock_kernel();
#if 0
	daemonize();
#else
        exit_mm(current);
        current->session=current->pgrp=1;
#endif
	sigfillset(&current->blocked);
	strcpy(current->comm, "arm_mon");
	dvb->arm_thread = current;
	unlock_kernel();

	for (;;) {
                interruptible_sleep_on_timeout(&dvb->arm_wait, 5*HZ);
		if (dvb->arm_rmmod || signal_pending(current))
			break;
                if (!dvb->arm_ready)
                        continue;
                newloops=rdebi(dvb, DEBINOSWAP, STATUS_LOOPS, 0, 2);
                if (newloops==dvb->arm_loops) {
                        printk("dvb%d: ARM crashed!\n", dvb->num);
                        arm_error(dvb);
                        newloops=rdebi(dvb, DEBINOSWAP, STATUS_LOOPS, 0, 2)-1;
                }
                dvb->arm_loops=newloops;
	}
	dvb->arm_thread = NULL;
	return 0;
}


static int
record_cb(pes2ts_t *p2t, u8 *buf, size_t len)
{
        dvb_demux_feed_t *dvbdmxfeed=(dvb_demux_feed_t *) p2t->priv;

        if (!(dvbdmxfeed->ts_type & TS_PACKET)) 
                return 0;
	if (buf[3]==0xe0)        // video PES do not have a length in TS
                buf[4]=buf[5]=0;
        if (dvbdmxfeed->ts_type & TS_PAYLOAD_ONLY) 
                return dvbdmxfeed->cb.ts(buf, len, 0, 0, 
                                         &dvbdmxfeed->feed.ts, DMX_OK); 
        else
                return pes2ts(p2t, buf, len);
}

static int 
pes2ts_cb(void *priv, unsigned char *data)
{
        dvb_demux_feed_t *dvbdmxfeed=(dvb_demux_feed_t *) priv;
        
        dvbdmxfeed->cb.ts(data, 188, 0, 0,
                          &dvbdmxfeed->feed.ts,
                          DMX_OK); 
        return 0;
}

static int 
AV_StartRecord(struct dvb_struct *dvb, int av,
               dvb_demux_feed_t *dvbdmxfeed)
{
        dvb_demux_t *dvbdmx=dvbdmxfeed->demux;
  
        if (dvb->playing||(dvb->rec_mode&av))
                return -EBUSY;
        outcom(dvb, COMTYPE_REC_PLAY, __Stop, 0);
        dvbdmx->recording=1;
        dvb->rec_mode|=av;

        switch (dvb->rec_mode) {
        case RP_AUDIO:
                pes2ts_init(&dvb->p2t[0], dvbdmx->pesfilter[0]->pid,
                            pes2ts_cb, (void *)dvbdmx->pesfilter[0]);
                outcom(dvb, COMTYPE_REC_PLAY, __Record, 2, AudioPES, 0);
                break;
        case RP_VIDEO:
                pes2ts_init(&dvb->p2t[1], dvbdmx->pesfilter[1]->pid,
                            pes2ts_cb, (void *)dvbdmx->pesfilter[1]);
                outcom(dvb, COMTYPE_REC_PLAY, __Record, 2, VideoPES, 0);
                break;
        case RP_AV:
                pes2ts_init(&dvb->p2t[0], dvbdmx->pesfilter[0]->pid,
                            pes2ts_cb, (void *)dvbdmx->pesfilter[0]);
                pes2ts_init(&dvb->p2t[1], dvbdmx->pesfilter[1]->pid,
                            pes2ts_cb, (void *)dvbdmx->pesfilter[1]);
                outcom(dvb, COMTYPE_REC_PLAY, __Record, 2, AV_PES, 0);
                break;
        }
        return 0;
}

static int 
AV_StartPlay(struct dvb_struct *dvb, int av)
{
        if (dvb->rec_mode)
                return -EBUSY;
        if (dvb->playing&av)
                return -EBUSY;

        outcom(dvb, COMTYPE_REC_PLAY, __Stop, 0);

        if (dvb->playing == RP_NONE) {
                reset_ipack(&dvb->ipack[0]);
                reset_ipack(&dvb->ipack[1]);
        }

        dvb->playing|=av;
        switch (dvb->playing) {
        case RP_AUDIO:
                outcom(dvb, COMTYPE_REC_PLAY, __Play, 2, AudioPES, 0);
                break;
        case RP_VIDEO:
                outcom(dvb, COMTYPE_REC_PLAY, __Play, 2, VideoPES, 0);
                dvb->sinfo=0;
                break;
        case RP_AV:
                dvb->sinfo=0;
                outcom(dvb, COMTYPE_REC_PLAY, __Play, 2, AV_PES, 0);
                break;
        }
        return dvb->playing;
}

static void 
AV_Stop(struct dvb_struct *dvb, int av)
{
        if (!(dvb->playing&av) && !(dvb->rec_mode&av))
                return;

        outcom(dvb, COMTYPE_REC_PLAY, __Stop, 0);
        if (dvb->playing) {
                dvb->playing&=~av;
                switch (dvb->playing) {
                case RP_AUDIO:
                        outcom(dvb, COMTYPE_REC_PLAY, __Play, 2, AudioPES, 0);
                        break;
                case RP_VIDEO:
                        outcom(dvb, COMTYPE_REC_PLAY, __Play, 2, VideoPES, 0);
                        break;
                case RP_NONE:
                        SetMode(dvb, dvb->vidmode);
                        break;
                }
        } else {
                dvb->rec_mode&=~av;
                switch (dvb->rec_mode) {
                case RP_AUDIO:
                        outcom(dvb, COMTYPE_REC_PLAY, __Record, 2, AudioPES, 0);
                        break;
                case RP_VIDEO:
                        outcom(dvb, COMTYPE_REC_PLAY, __Record, 2, VideoPES, 0);
                        break;
                case RP_NONE:
                        break;
                }
        }
}

/****************************************************************************
 * Buffer handling
 ****************************************************************************/

static inline void 
ring_buffer_flush(ring_buffer_t *rbuf)
{
        spin_lock_irq(&rbuf->lock);
        rbuf->pwrite=rbuf->pread;
        spin_unlock_irq(&rbuf->lock);
        wake_up(&rbuf->queue);
}

static inline void 
ring_buffer_init(ring_buffer_t *rbuf, u8 *data, int len)
{
        rbuf->pread=rbuf->pwrite=0;
        rbuf->data=data;
        rbuf->size=len;
        init_waitqueue_head(&rbuf->queue);
        spin_lock_init(&(rbuf->lock));
        rbuf->lock=SPIN_LOCK_UNLOCKED;
        sema_init(&(rbuf->sema), 1);
}

static inline 
int ring_buffer_empty(ring_buffer_t *rbuf)
{
        return (rbuf->pread==rbuf->pwrite);
}

static inline 
int ring_buffer_free(ring_buffer_t *rbuf)
{
        int free;
  
        free=rbuf->pread - rbuf->pwrite;
        if (free<=0)
                free+=rbuf->size;
        return free;
}

static inline 
int ring_buffer_avail(ring_buffer_t *rbuf)
{
        int avail;
  
        avail=rbuf->pwrite - rbuf->pread;
        if (avail<0)
                avail+=rbuf->size;
        return avail;
}

#if 0
static void
ring_buffer_block(ring_buffer_t *rbuf, unsigned long count)
{
        if (ring_buffer_free(rbuf)>=count)
                return;
        while (!wait_event_interruptible(rbuf->queue,
                                         (ring_buffer_free(rbuf)>=count)));
}
#endif

static long 
ring_buffer_write(ring_buffer_t *rbuf, 
              const char *buf, unsigned long count, 
              int nonblock, int usermem)
{
        unsigned long todo = count;
        int free, split;
    
        dprintk ("function : %s\n", __FUNCTION__);
        while (todo > 0) {
                if (ring_buffer_free(rbuf)<=2048) {
                        if (nonblock)
                                return count-todo;
                        if (wait_event_interruptible(rbuf->queue,
                                                     (ring_buffer_free(rbuf)>2048)))
                        	return count-todo;
                }   
                dprintk ("function: %s pread=%08x pwrite=%08x\n", __FUNCTION__,
                        rbuf->pread, rbuf->pwrite);
                //mdelay(2);
                free = rbuf->pread - rbuf->pwrite;
                split=rbuf->size;
                if (free<=0) {
                        free+=rbuf->size;
                        split-=rbuf->pwrite;
                }
                if (free > todo)
                        free = todo;
                        
                if (split < free) {
                        if (!usermem) 
                                memcpy(rbuf->data+rbuf->pwrite, buf, split);
                        else
                                if (copy_from_user(rbuf->data+rbuf->pwrite,
                                                   buf, split)) 
                                        return -EFAULT;
                        buf += split;
                        todo -= split;
                        free -= split;
                        rbuf->pwrite = 0;
                }
                if (!usermem) 
                        memcpy(rbuf->data+rbuf->pwrite, buf, free);
                else
                        if (copy_from_user(rbuf->data+rbuf->pwrite, buf, free)) 
                                return -EFAULT;
                rbuf->pwrite = (rbuf->pwrite + free)%rbuf->size;
                todo -= free;
                buf += free;
        }

	return count-todo;
}

#if 0
static void
ring_buffer_put(ring_buffer_t *db, u8 *buf, int len)
{
        int split, fsize;

        fsize=db->pread - db->pwrite;
        if (fsize <= 0) {
                fsize+=db->size;
                split=db->size-db->pwrite;
        } else 
                split=0;
        if (len>=fsize) {
                dprintk("buffer overflow\n");
                return;
        }
        if (split>=len)
                split=0;
        if (split) {
                memcpy(db->data + db->pwrite, buf, split);
                len-=split;
                db->pwrite=0;
        }
        memcpy(db->data + db->pwrite, split + buf, len);
        db->pwrite=(db->pwrite+len)%db->size;
}
#endif


/****************************************************************************
 * TT budget / WinTV Nova
 ****************************************************************************/

static int 
TTBStop(struct dvb_struct *dvb)
{
        if (--dvb->feeding)
                return dvb->feeding;
        saa7146_write(dvb->saa_mem, MC1, MASK_20); // DMA3 off
        saa7146_write(dvb->saa_mem, MC1, MASK_28); // RPS0 off
        saa7146_write(dvb->saa_mem, IER, 
                      saa7146_read(dvb->saa_mem, IER) & ~MASK_10 );
        saa7146_write(dvb->saa_mem, IER, 
                      saa7146_read(dvb->saa_mem, IER)& ~MASK_07);
        return 0;
}

#define TS_WIDTH  (4*188)
#define TS_HEIGHT (1024/4)
static int
TTBStart(struct dvb_struct *dvb)
{
        struct saa7146 *saa=dvb->saa;

        //printk ("function : %s\n", __FUNCTION__);
        if (dvb->feeding++)
                return dvb->feeding;

      	saa7146_write(saa->mem, MC1, MASK_20); // DMA3 off

        memset(saa->grabbing, 0x00, TS_HEIGHT*TS_WIDTH);

        saa7146_write(saa->mem, PCI_BT_V1, 0x001c0000);

        dvb->tsf=0;
        saa7146_write(dvb->saa->mem, DD1_INIT, 0x02000680);
        saa7146_write(dvb->saa->mem, MC2, 
                      (MASK_09 | MASK_25 | MASK_10 | MASK_26));

        saa7146_write(saa->mem, BRS_CTRL, 0x60000000);	
      	saa7146_write(saa->mem, MC2, (MASK_08 | MASK_24));
        mdelay(10);

        saa7146_write(saa->mem, BASE_ODD3, 0);
        saa7146_write(saa->mem, BASE_EVEN3, TS_WIDTH*TS_HEIGHT/2);
        saa7146_write(saa->mem, PROT_ADDR3, TS_WIDTH*TS_HEIGHT);	
        saa7146_write(saa->mem, BASE_PAGE3, virt_to_bus(saa->page_table[0])|ME1|0xb0);
        saa7146_write(saa->mem, PITCH3, TS_WIDTH);	

        saa7146_write(saa->mem, NUM_LINE_BYTE3, ((TS_HEIGHT/2)<<16)|TS_WIDTH);
      	saa7146_write(saa->mem, MC2, (MASK_04 | MASK_20));

        // VPE
        saa7146_write(saa->mem, IER, saa7146_read(saa->mem, IER)|MASK_10);

     	saa7146_write(saa->mem, MC1, (MASK_04 | MASK_20)); // DMA3 on

        // FIDB
        saa7146_write(saa->mem, IER, saa7146_read(saa->mem, IER)|MASK_07);
        return dvb->feeding;
}

static void (*irc_handler)(u32);

void dvb_register_irc_handler(void (*func)(u32)) 
{
        //dprintk("registering %08x\n",func);
        irc_handler = func;
}

void dvb_unregister_irc_handler(void (*func)(u32)) 
{
        //dprintk("unregistering %08x\n",func);
        irc_handler = NULL;
}

EXPORT_SYMBOL(dvb_register_irc_handler);
EXPORT_SYMBOL(dvb_unregister_irc_handler);


void run_handlers(unsigned long ircom) 
{
        if (irc_handler != NULL)
                (*irc_handler)((u32) ircom);
}

DECLARE_TASKLET(irtask,run_handlers,0);

void IR_handle(struct dvb_struct *dvb, u32 ircom)
{
        printk("dvb: ircommand = %08x\n", ircom);
        irtask.data = (unsigned long) ircom;
        tasklet_schedule(&irtask);
}

/****************************************************************************
 * IRQ handling
 ****************************************************************************/

void CI_handle(struct dvb_struct *dvb, u8 *data, u16 len) 
{
        //CI_out(dvb, data, len);

        if (len<3)
                return;
        switch (data[0]) {
        case CI_MSG_CI_INFO:
                if (data[2]!=1 && data[2]!=2)
                        break;
                switch (data[1]) {
                case 0:
                        dvb->ci_slot[data[2]-1].flags=0;
                        break;
                case 1:
                        dvb->ci_slot[data[2]-1].flags|=CA_CI_MODULE_PRESENT;
                        break;
                case 2:
                        dvb->ci_slot[data[2]-1].flags|=CA_CI_MODULE_READY;
                        break;
                }
                break;
        case CI_SWITCH_PRG_REPLY:
                //dvb->ci_stat=data[1];
                break;
        default:
                break;
        }

}

static inline int
DvbDmxFilterCallback(u8 * buffer1, size_t buffer1_len,
                     u8 * buffer2, size_t buffer2_len,
                     dvb_demux_filter_t *dvbdmxfilter,
                     dmx_success_t success,
                     struct dvb_struct *dvb)
{
        if (!dvbdmxfilter->feed->demux->dmx.frontend)
                return 0;
        if (dvbdmxfilter->feed->demux->dmx.frontend->source==DMX_MEMORY_FE)
                return 0;
        
        switch(dvbdmxfilter->type) {
        case DMX_TYPE_SEC:
                if ((((buffer1[1]<<8)|buffer1[2])&0xfff)+3!=buffer1_len)
                        return 0;
                return dvbdmxfilter->feed->cb.sec(buffer1, buffer1_len,
						  buffer2, buffer2_len,
						  &dvbdmxfilter->filter,
						  DMX_OK); 
        case DMX_TYPE_TS:
                if (!(dvbdmxfilter->feed->ts_type & TS_PACKET)) 
                        return 0;
                if (dvbdmxfilter->feed->ts_type & TS_PAYLOAD_ONLY) 
                        return dvbdmxfilter->feed->cb.ts(buffer1, buffer1_len,
                                                         buffer2, buffer2_len,
                                                         &dvbdmxfilter->feed->feed.ts,
                                                         DMX_OK); 
                else
                        pes_to_ts(buffer1, buffer1_len, 
                                  dvbdmxfilter->feed->pid, 
                                  &dvb->p2t_filter[dvbdmxfilter->index]);
	default:
	        return 0;
        }
}


u8 pshead[0x26] = {
        0x00, 0x00, 0x01, 0xba, 0x5f, 0xff, 0xfe, 0xe6, 
        0xc4, 0x01, 0x01, 0x89, 0xc3, 0xf8, 0x00, 0x00,
        0x01, 0xbb, 0x00, 0x12, 0x80, 0xc4, 0xe1, 0x00,
        0xe1, 0xff, 0xb9, 0xe0, 0xe8, 0xb8, 0xc0, 0x20,
        0xbd, 0xe0, 0x44, 0xbf, 0xe0, 0x02,
};								


static void vpeirq(struct saa7146* saa, void *data)
{
        struct dvb_struct *dvb=(struct dvb_struct *) data;
  
        if (dvb->card_type!=DVB_CARD_TT_BUDGET)
                return;
        //printk("vpeirq %08x\n", saa7146_read(dvb->saa_mem, PCI_VDP3));
}

static void fidbirq(struct saa7146* saa, void *data)
{
        struct dvb_struct *dvb=(struct dvb_struct *) data;
        u8 *mem;
  
        if (dvb->card_type!=DVB_CARD_TT_BUDGET)
                return;
        
        mem=(dvb->tsf ? TS_HEIGHT*TS_WIDTH/2 :0)+(u8 *)dvb->saa->grabbing;

        dvb->tsf^=1;
        saa7146_write(dvb->saa_mem, DD1_INIT, 0x02000600|(dvb->tsf ? 0x40:0x80));
        saa7146_write(dvb->saa_mem, MC2, 
                      (MASK_09 | MASK_25 | MASK_10 | MASK_26));

        // FIXME: use bottom half or tasklet
        if (dvb->feeding && mem[0]==0x47)
                DvbDmxSWFilterPackets(&dvb->demux, mem, 512);
}

//#define DEBUG_TIMING
inline static void 
print_time(char *s)
{
#ifdef DEBUG_TIMING
        struct timeval tv;
        do_gettimeofday(&tv);
        printk("%s: %d.%d\n", s, (int)tv.tv_sec, (int)tv.tv_usec);
#endif
}

static void 
ci_get_data(ring_buffer_t *cibuf, u8 *data, int len)
{
        int free, split=0, pread=cibuf->pread;
        
        free=pread-cibuf->pwrite;
        if (free<=0)  
                free+=cibuf->size;
        if (free<=len+2) 
                return;
        cibuf->data[cibuf->pwrite]=(len>>8);   
        cibuf->data[(cibuf->pwrite+1)%cibuf->size]=(len&0xff); 
        cibuf->pwrite=(cibuf->pwrite+2)%cibuf->size;

        if (pread<=cibuf->pwrite)
                split=cibuf->size-cibuf->pwrite;
        if (split && split<len) {
                memcpy(cibuf->data + cibuf->pwrite, data, split);
                memcpy(cibuf->data, data+split, len-split);
        } else 
                memcpy(cibuf->data + cibuf->pwrite, data, len);
        cibuf->pwrite=(cibuf->pwrite+len)%cibuf->size;
        
        wake_up_interruptible(&cibuf->queue);
}

static void 
debiirq(struct saa7146* saa, void *data)
{
        struct dvb_struct *dvb=(struct dvb_struct *) data;
        int type=dvb->debitype;
        int handle=(type>>8)&0x1f;
        
        print_time("debi");
        saa7146_write(dvb->saa_mem, IER, 
                      saa7146_read(dvb->saa_mem, IER) & ~MASK_19 );
        saa7146_write(dvb->saa_mem, ISR, MASK_19 );

        if (type==-1) {
                printk("DEBI irq oops\n");
                ARM_ClearMailBox(dvb);
                ARM_ClearIrq(dvb);
                return;
        }
        dvb->debitype=-1;

        switch (type&0xff) {

        case DATA_TS_RECORD:
                DvbDmxSWFilterPackets(&dvb->demux, (const u8 *)dvb->debi_virt, 
                                      dvb->debilen/188);
                spin_lock(&dvb->debilock);
                iwdebi(dvb, DEBINOSWAP, RX_BUFF, 0, 2);
                ARM_ClearMailBox(dvb);
                spin_unlock(&dvb->debilock);
                break;

        case DATA_PES_RECORD:
                if (dvb->demux.recording) 
                        record_cb(&dvb->p2t[handle], 
                                  (u8 *)dvb->debi_virt,
                                  dvb->debilen);
                spin_lock(&dvb->debilock);
                iwdebi(dvb, DEBINOSWAP, RX_BUFF, 0, 2);
                ARM_ClearMailBox(dvb);
                spin_unlock(&dvb->debilock);
                return;

        case DATA_IPMPE:
        case DATA_FSECTION:
        case DATA_PIPING:
                if (dvb->handle2filter[handle]) 
                        DvbDmxFilterCallback((u8 *)dvb->debi_virt, 
                                             dvb->debilen, 0, 0, 
                                             dvb->handle2filter[handle], DMX_OK, dvb); 
                spin_lock(&dvb->debilock);
                iwdebi(dvb, DEBINOSWAP, RX_BUFF, 0, 2);
                ARM_ClearMailBox(dvb);
                spin_unlock(&dvb->debilock);
                return;

        case DATA_CI_GET:
        {
                u8 *data=dvb->debi_virt;

                if ((data[0]<2) && data[2]==0xff) {
                        int flags=0;
                        if (data[5]>0) 
                                flags|=CA_CI_MODULE_PRESENT;
                        if (data[5]>5) 
                                flags|=CA_CI_MODULE_READY;
                        dvb->ci_slot[data[0]].flags=flags;
                } else
                        ci_get_data(&dvb->ci_rbuffer, dvb->debi_virt, dvb->debilen);
                spin_lock(&dvb->debilock);
                iwdebi(dvb, DEBINOSWAP, RX_BUFF, 0, 2);
                ARM_ClearMailBox(dvb);
                spin_unlock(&dvb->debilock);
                return;
        }

        case DATA_COMMON_INTERFACE:
                CI_handle(dvb, (u8 *)dvb->debi_virt, dvb->debilen);
#if 1
        {
                int i;

                printk("dvb%d: ", dvb->num);
                printk("%02x ", *(u8 *)dvb->debi_virt);
                printk("%02x ", *(1+(u8 *)dvb->debi_virt));
                for (i=2; i<dvb->debilen; i++)
                  printk("%02x ", (*(i+(unsigned char *)dvb->debi_virt)));
                for (i=2; i<dvb->debilen; i++)
                  printk("%c", chtrans(*(i+(unsigned char *)dvb->debi_virt)));

                printk("\n");
        }
#endif
                spin_lock(&dvb->debilock);
                iwdebi(dvb, DEBINOSWAP, RX_BUFF, 0, 2);
                ARM_ClearMailBox(dvb);
                spin_unlock(&dvb->debilock);
                return;

        case DATA_DEBUG_MESSAGE:
                ((s8*)dvb->debi_virt)[Reserved_SIZE-1]=0;
                printk("%s\n", (s8 *)dvb->debi_virt);
                spin_lock(&dvb->debilock);
                iwdebi(dvb, DEBINOSWAP, RX_BUFF, 0, 2);
                ARM_ClearMailBox(dvb);
                spin_unlock(&dvb->debilock);
                return;

        case DATA_CI_PUT:
        case DATA_MPEG_PLAY:
        case DATA_BMP_LOAD:
                spin_lock(&dvb->debilock);
                iwdebi(dvb, DEBINOSWAP, TX_BUFF, 0, 2);
                ARM_ClearMailBox(dvb);
                spin_unlock(&dvb->debilock);
                return;
        default:
                break;
        }
        spin_lock(&dvb->debilock);
        ARM_ClearMailBox(dvb);
        spin_unlock(&dvb->debilock);
}

static int
pes_play(void *dest, ring_buffer_t *buf, int dlen)
{
        int len, split=0;
        u32 sync;
        u16 blen;

        dprintk ("function : %s\n", __FUNCTION__);
        if (!dlen) {
                wake_up(&buf->queue);
                return -1;
        }
        while (1) {
                if ((len=ring_buffer_avail(buf)) < 6)
                        return -1;
                sync=(buf->data[buf->pread])<<24;
                sync|=(buf->data[(buf->pread+1)%buf->size]<<16);
                sync|=(buf->data[(buf->pread+2)%buf->size]<<8);
                sync|=buf->data[(buf->pread+3)%buf->size];
                
                if (((sync&~0x1f)==0x000001e0) ||
                    ((sync&~0x1f)==0x000001c0) ||
                    (sync==0x000001bd))
                        break;
                printk("resync\n");
                buf->pread=(buf->pread+1)%buf->size;
        }
        blen=(buf->data[(buf->pread+4)%buf->size]<<8);
        blen|=buf->data[(buf->pread+5)%buf->size];
        blen+=6;
        if (len<blen || blen > dlen) {
                printk("buffer empty\n");
                wake_up(&buf->queue);
                return -1;
        }
/*        if (blen>2048) {
                buf->pread=(buf->pread+blen)%buf->size;
                printk("packet too large\n");
                return -1;
        }
*/
        len=blen;
        if (buf->pread + len > buf->size)
                split=buf->size-buf->pread;
        if (split>0) {
                memcpy(dest, buf->data+buf->pread, split);
                buf->pread=0;
                len-=split;
        }
        memcpy(split + dest, 
               buf->data + buf->pread, len);
        buf->pread = (buf->pread +len)%buf->size;
        
        dprintk ("function: %s pread=%08x pwrite=%08x\n", __FUNCTION__,
                buf->pread, buf->pwrite);
        wake_up(&buf->queue);
        return blen;
}

static void 
gpioirq(struct saa7146* saa, void *data)
{
        struct dvb_struct *dvb=(struct dvb_struct *) data;
        u32 rxbuf, txbuf;
        int len;
        
        //printk("GPIO0 irq\n");        

        if (dvb->debitype !=-1)
                printk("GPIO0 irq oops\n");
       
        spin_lock(&dvb->debilock);

	ARM_ClearIrq(dvb);

        saa7146_write(dvb->saa_mem, IER, 
                      saa7146_read(dvb->saa_mem, IER) & ~MASK_19 );
        saa7146_write(dvb->saa_mem, ISR, MASK_19 );

        dvb->debitype = irdebi(dvb, DEBINOSWAP, IRQ_STATE, 0, 2);
        dvb->debilen  = irdebi(dvb, DEBINOSWAP, IRQ_STATE_EXT, 0, 2);
        dvb->debibuf  = 0;
        rxbuf=irdebi(dvb, DEBINOSWAP, RX_BUFF, 0, 2);
        txbuf=irdebi(dvb, DEBINOSWAP, TX_BUFF, 0, 2);
        len=(dvb->debilen+3)&(~3);

        dprintk("GPIO0 irq %d %d\n", dvb->debitype, dvb->debilen);
        print_time("gpio");

        dprintk("GPIO0 irq %02x\n", dvb->debitype&0xff);        
        switch (dvb->debitype&0xff) {

        case DATA_TS_PLAY:
        case DATA_PES_PLAY:
                break;

        case DATA_CI_PUT:
        {
                int avail, split=0, pwrite;
                ring_buffer_t *cibuf=&dvb->ci_wbuffer;

                pwrite=cibuf->pwrite;
                avail=pwrite-cibuf->pread;
                if (avail<0) 
                        avail+=cibuf->size;
                if (avail<=2) {
                        iwdebi(dvb, DEBINOSWAP, IRQ_STATE_EXT, 0, 2);
                        iwdebi(dvb, DEBINOSWAP, TX_LEN, 0, 2);
                        iwdebi(dvb, DEBINOSWAP, TX_BUFF, 0, 2);
                        break;
                } 
                len=(cibuf->data[cibuf->pread]<<8);
                len|=cibuf->data[(cibuf->pread+1)%cibuf->size];
                if (avail<len+2) {
                        iwdebi(dvb, DEBINOSWAP, IRQ_STATE_EXT, 0, 2);
                        iwdebi(dvb, DEBINOSWAP, TX_LEN, 0, 2);
                        iwdebi(dvb, DEBINOSWAP, TX_BUFF, 0, 2);
                        break;
                } 
                cibuf->pread=(cibuf->pread+2)%cibuf->size;
                
                if (pwrite<cibuf->pread)
                        split=cibuf->size-cibuf->pread;
                if (split && split<len) {
                        int todo=len-split;
                        memcpy(dvb->debi_virt, cibuf->data+cibuf->pread, split);
                        memcpy(dvb->debi_virt+split, cibuf->data, todo);
                } else
                        memcpy(dvb->debi_virt, cibuf->data+cibuf->pread, len);
                cibuf->pread=(cibuf->pread+len)%cibuf->size;
                wake_up(&cibuf->queue);
                iwdebi(dvb, DEBINOSWAP, TX_LEN, len, 2);
                iwdebi(dvb, DEBINOSWAP, IRQ_STATE_EXT, len, 2);
                wait_for_debi_done(dvb);
                saa7146_write(dvb->saa_mem, IER, 
                              saa7146_read(dvb->saa_mem, IER) | MASK_19 );
                if (len<5) len=5; /* we want a real DEBI DMA */
                iwdebi(dvb, DEBISWAB, DPRAM_BASE+txbuf, 0, (len+3)&~3);
                spin_unlock(&dvb->debilock);
                return;
        }

        case DATA_MPEG_PLAY:
                if (!dvb->playing) {
                        iwdebi(dvb, DEBINOSWAP, IRQ_STATE_EXT, 0, 2);
                        iwdebi(dvb, DEBINOSWAP, TX_LEN, 0, 2);
                        iwdebi(dvb, DEBINOSWAP, TX_BUFF, 0, 2);
                        break;
                }
                len=0;
                if (dvb->debitype&0x100) {
                        spin_lock(&dvb->aout.lock);
                        len=pes_play(dvb->debi_virt, &dvb->aout, 2048);
                        spin_unlock(&dvb->aout.lock);
                }
                if (len<=0 && (dvb->debitype&0x200)
                        &&dvb->videostate.playState!=VIDEO_FREEZED) {
                        spin_lock(&dvb->avout.lock);
                        len=pes_play(dvb->debi_virt, &dvb->avout, 2048);
                        spin_unlock(&dvb->avout.lock);
                }
                if (len<=0) {
                        iwdebi(dvb, DEBINOSWAP, IRQ_STATE_EXT, 0, 2);
                        iwdebi(dvb, DEBINOSWAP, TX_LEN, 0, 2);
                        iwdebi(dvb, DEBINOSWAP, TX_BUFF, 0, 2);
                        break;
                } 
                dprintk("GPIO0 PES_PLAY len=%04x\n", len);        
                iwdebi(dvb, DEBINOSWAP, TX_LEN, len, 2);
                iwdebi(dvb, DEBINOSWAP, IRQ_STATE_EXT, len, 2);
                wait_for_debi_done(dvb);
                saa7146_write(dvb->saa_mem, IER, 
                              saa7146_read(dvb->saa_mem, IER) | MASK_19 );

                iwdebi(dvb, DEBISWAB, DPRAM_BASE+txbuf, 0, (len+3)&~3);
                spin_unlock(&dvb->debilock);
                return;

        case DATA_BMP_LOAD:
                len=dvb->debilen;
                if (!len) {
                        dvb->bmp_state=BMP_LOADED;
                        iwdebi(dvb, DEBINOSWAP, IRQ_STATE_EXT, 0, 2);
                        iwdebi(dvb, DEBINOSWAP, TX_LEN, 0, 2);
                        iwdebi(dvb, DEBINOSWAP, TX_BUFF, 0, 2);
                        wake_up(&dvb->bmpq);
                        break;
                }
                if (len>dvb->bmplen)
                        len=dvb->bmplen;
                if (len>2*1024)
                        len=2*1024;
                iwdebi(dvb, DEBINOSWAP, TX_LEN, len, 2);
                iwdebi(dvb, DEBINOSWAP, IRQ_STATE_EXT, len, 2);
                memcpy(dvb->debi_virt, dvb->bmpbuf+dvb->bmpp, len);
                dvb->bmpp+=len;
                dvb->bmplen-=len;
                wait_for_debi_done(dvb);
                saa7146_write(dvb->saa_mem, IER, 
                              saa7146_read(dvb->saa_mem, IER) | MASK_19 );
                if (len<5) len=5; /* we want a real DEBI DMA */
                iwdebi(dvb, DEBISWAB, DPRAM_BASE+txbuf, 0, (len+3)&~3);
                spin_unlock(&dvb->debilock);
                return;

        case DATA_CI_GET:
        case DATA_COMMON_INTERFACE:
        case DATA_FSECTION:
        case DATA_IPMPE:
        case DATA_PIPING:
                if (!len || len>4*1024) {
                        iwdebi(dvb, DEBINOSWAP, RX_BUFF, 0, 2);
                        break;
                }                  /* yes, fall through */
        case DATA_TS_RECORD:
        case DATA_PES_RECORD:
                saa7146_write(dvb->saa_mem, IER, 
                              saa7146_read(dvb->saa_mem, IER) | MASK_19);
                irdebi(dvb, DEBISWAB, DPRAM_BASE+rxbuf, 0, len);
                spin_unlock(&dvb->debilock);
                return;

        case DATA_DEBUG_MESSAGE:
                if (!len || len>0xff) {
                        iwdebi(dvb, DEBINOSWAP, RX_BUFF, 0, 2);
                        break;
                }
                saa7146_write(dvb->saa_mem, IER, 
                              saa7146_read(dvb->saa_mem, IER) | MASK_19);
                irdebi(dvb, DEBISWAB, Reserved, 0, len);
                spin_unlock(&dvb->debilock);
                return;

        case DATA_IRCOMMAND: 
                IR_handle(dvb,swahw32(irdebi(dvb, DEBINOSWAP, Reserved, 0, 4)));
                iwdebi(dvb, DEBINOSWAP, RX_BUFF, 0, 2);
                break;

        default:
                printk("gpioirq unknown type=%d len=%d\n", 
                       dvb->debitype, dvb->debilen);
                break;
        }      
        ARM_ClearMailBox(dvb);
        dvb->debitype=-1;
        spin_unlock(&dvb->debilock);
        dprintk("GPIO0 irq exit 0\n");        
}


/****************************************************************************
 * DEBI command polling 
 ****************************************************************************/


static int OutCommand(struct dvb_struct *dvb, u16* buf, int length)
{
        int i;
        u32	start;

        if (!dvb->arm_ready)
                return -1;
        start = jiffies;
        while ( rdebi(dvb, DEBINOSWAP, COMMAND, 0, 2 ) )
        {
                ddelay(1);
                if ((jiffies - start) > ARM_WAIT_FREE) {
                        printk("outcommand error 1\n");
                        //arm_error(dvb);
                        return -1;
                }
        }

#ifndef _NOHANDSHAKE
        start = jiffies;
        while ( rdebi(dvb, DEBINOSWAP, HANDSHAKE_REG, 0, 2 ) ) 
        {
                ddelay(1);
                if ((jiffies - start) > ARM_WAIT_SHAKE) {
                        printk("outcommand error 2\n");
                        //arm_error(dvb);
                        return -1;
                }
        }
#endif

        start = jiffies;
        while ( rdebi(dvb, DEBINOSWAP, MSGSTATE, 0, 2) & OSDQFull )
        {
                ddelay(1);
                if ((jiffies - start) > ARM_WAIT_OSD)
                {
                        printk("outcommand error 3\n");
                        //arm_error(dvb);
                        return -1;
                }
        }
        for (i=2; i<length; i++)
                wdebi(dvb, DEBINOSWAP, COMMAND + 2*i, (u32) buf[i], 2);

        if (length)
                wdebi(dvb, DEBINOSWAP, COMMAND + 2, (u32) buf[1], 2);
        else
                wdebi(dvb, DEBINOSWAP, COMMAND + 2, 0, 2);
        wdebi(dvb, DEBINOSWAP, COMMAND, (u32) buf[0], 2);
        return 0;
}

inline static int 
SOutCommand(struct dvb_struct *dvb, u16* buf, int length)
{
        int ret;
        
        if (!dvb->arm_ready)
                return -1;
        down_interruptible(&dvb->dcomlock);
        ret=OutCommand(dvb, buf, length);
        up(&dvb->dcomlock);
        return ret;
}


static int outcom(struct dvb_struct *dvb, int type, int com, int num, ...)
{
	va_list args;
        u16 buf[num+2];
        int i;

        buf[0]=(( type << 8 ) + com); 
        buf[1]=num;

        if (num) {
                va_start(args, num);
                for (i=0; i<num; i++)
                        buf[i+2]=va_arg(args, u32);
                va_end(args);
        }

        return SOutCommand(dvb, buf, num+2);
}

int SendCICommand(struct dvb_struct *dvb, u8 subcom, u8 *Params, u8 ParamLen)
{
        int i;
        u16 CommandBuffer[18] = { ((COMTYPE_COMMON_IF << 8) + subcom),
                                  16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
        
	for(i=0; (i<ParamLen)&&(i<32); i++)	
	{
		if(i%2 == 0)
			CommandBuffer[(i/2)+2] = (u16)(Params[i]) << 8;
		else
			CommandBuffer[(i/2)+2] |= Params[i];
	}

        return SOutCommand(dvb, CommandBuffer, 18);
}


static int CommandRequest(struct dvb_struct *dvb, u16 *Buff, int length, u16 *buf, int n)
{
        s16 i;
        u32 start;
       
        if (!dvb->arm_ready)
                return -1;
        down_interruptible(&dvb->dcomlock);
        OutCommand(dvb, Buff, length);
        
        start = jiffies;
        while ( rdebi(dvb, DEBINOSWAP, COMMAND, 0, 2) ) 
        {
#ifdef _NOHANDSHAKE
                ddelay(1);
#endif
                if ((jiffies - start) > ARM_WAIT_FREE) {
                        printk("commandrequest error\n");
                        up(&dvb->dcomlock);
                        //arm_error(dvb);
                        return -1;
                }
        }
        
#ifndef _NOHANDSHAKE
        start = jiffies;
        while ( rdebi(dvb, DEBINOSWAP, HANDSHAKE_REG, 0, 2 ) ) {
                ddelay(1);
                if ((jiffies - start) > ARM_WAIT_SHAKE) {
                        printk("commandrequest error\n");
                        up(&dvb->dcomlock);
                        //arm_error(dvb);
                        return -1;
                }
        }
#endif

        for (i=0; i<n; i++)
                buf[i] = rdebi(dvb, DEBINOSWAP, COM_BUFF + 2*i, 0, 2);
        
	up(&dvb->dcomlock);
        return 0;
}


static inline int 
RequestParameter(struct dvb_struct *dvb, u16 tag, u16* Buff, s16 length)
{
        return CommandRequest(dvb, &tag, 0, Buff, length);
}


/****************************************************************************
 * Firmware commands 
 ****************************************************************************/


inline static int 
SendDAC(struct dvb_struct *dvb, u8 addr, u8 data)
{
        return outcom(dvb, COMTYPE_AUDIODAC, AudioDAC, 2, addr, data);
}

int static 
SetVolume(struct dvb_struct *dvb, int volleft, int volright)
{
        int err;
        
        switch (dvb->adac_type) {
        case DVB_ADAC_TI:
                volleft=(volleft*256)/946;
                volright=(volright*256)/946;
                if (volleft > 0x45)
                        volleft=0x45;
                if (volright > 0x45)
                        volright=0x45;
                err=SendDAC(dvb, 3, 0x80 + volleft);
                if (err)
                        return err;
                return SendDAC(dvb, 4, volright);
                
        case DVB_ADAC_CRYSTAL:
                volleft=127-((volleft*127)/255);
                volright=127-((volright*127)/255);
                i2c_writereg(dvb, 0x20, 0x03, volleft);
                i2c_writereg(dvb, 0x20, 0x04, volright);
                return 0;
        }
        return 0;
}

#ifdef USE_OSD

inline static int ResetBlend(struct dvb_struct *dvb, u8 windownr)
{
        return outcom(dvb, COMTYPE_OSD, SetNonBlend, 1, windownr);
}

inline static int SetColorBlend(struct dvb_struct *dvb, u8 windownr)
{
        return outcom(dvb, COMTYPE_OSD, SetCBlend, 1, windownr); 
}

inline static int SetWindowBlend(struct dvb_struct *dvb, u8 windownr, u8 blending)
{
        return outcom(dvb, COMTYPE_OSD, SetWBlend, 2, windownr, blending); 
}

inline static int SetBlend_(struct dvb_struct *dvb, u8 windownr,
                     OSDPALTYPE colordepth, u16 index, u8 blending)
{
        return outcom(dvb, COMTYPE_OSD, SetBlend, 4,
                      windownr, colordepth, index, blending);
} 

inline static int SetColor_(struct dvb_struct *dvb, u8 windownr,
                     OSDPALTYPE colordepth, u16 index, u16 colorhi, u16 colorlo)
{
        return outcom(dvb, COMTYPE_OSD, SetColor, 5,
                      windownr, colordepth, index, colorhi, colorlo);
} 

inline static int BringToTop(struct dvb_struct *dvb, u8 windownr)
{
        return outcom(dvb, COMTYPE_OSD, WTop, 1, windownr);
} 

inline static int SetFont(struct dvb_struct *dvb, u8 windownr, u8 fontsize,
                   u16 colorfg, u16 colorbg)
{
        return outcom(dvb, COMTYPE_OSD, Set_Font, 4,
                      windownr, fontsize, colorfg, colorbg);
} 

static int FlushText(struct dvb_struct *dvb)
{
        u32 start;

        down_interruptible(&dvb->dcomlock);
        start = jiffies;
        while ( rdebi(dvb, DEBINOSWAP, BUFF1_BASE, 0, 2 ) ) {
                ddelay(1); 
                if ((jiffies - start) > ARM_WAIT_OSD) {
                        printk("outtext error\n");
                        up(&dvb->dcomlock);
                        //arm_error(dvb);
                        return -1;
                }
        }
        up(&dvb->dcomlock);
        return 0;
}

static int WriteText(struct dvb_struct *dvb, u8 win, u16 x, u16 y, u8* buf)
{
        int i, ret;
        u32 start;
        int length=strlen(buf)+1;
        u16 cbuf[5] = { (COMTYPE_OSD<<8) + DText, 3, win, x, y };
        
        down_interruptible(&dvb->dcomlock);
        start = jiffies;
        while ( rdebi(dvb, DEBINOSWAP, BUFF1_BASE, 0, 2 ) ) {
                ddelay(1);
                if ((jiffies - start) > ARM_WAIT_OSD) {
                        printk("outtext error\n");
                        up(&dvb->dcomlock);
                        //arm_error(dvb);
                        return -1;
                }
        }
#ifndef _NOHANDSHAKE
        start = jiffies;
        while ( rdebi(dvb, DEBINOSWAP, HANDSHAKE_REG, 0, 2 ) ) {
                ddelay(1);
                if ((jiffies - start) > ARM_WAIT_SHAKE) {
                        printk("outtext error\n");
                        up(&dvb->dcomlock);
                        //arm_error(dvb);
                        return -1;
                }
        }
#endif
        for (i=0; i<length/2; i++)
                wdebi(dvb, DEBINOSWAP, BUFF1_BASE + i*2, 
                      swab16(*(u16 *)(buf+2*i)), 2);
        if (length&1)
                wdebi(dvb, DEBINOSWAP, BUFF1_BASE + i*2, 0, 2);
        ret=OutCommand(dvb, cbuf, 5);
        up(&dvb->dcomlock);
        return ret;
}

inline static int DrawLine(struct dvb_struct *dvb, u8 windownr, 
                    u16 x, u16 y, u16 dx, u16 dy, u16 color)
{
        return outcom(dvb, COMTYPE_OSD, DLine, 6,
                      windownr, x, y, dx, dy, color);
} 

inline static int DrawBlock(struct dvb_struct *dvb, u8 windownr, 
                    u16 x, u16 y, u16 dx, u16 dy, u16 color)
{
        return outcom(dvb, COMTYPE_OSD, DBox, 6,
                      windownr, x, y, dx, dy, color);
} 

inline static int HideWindow(struct dvb_struct *dvb, u8 windownr)
{
        return outcom(dvb, COMTYPE_OSD, WHide, 1, windownr);
} 

inline static int MoveWindowRel(struct dvb_struct *dvb, u8 windownr, u16 x, u16 y)
{
        return outcom(dvb, COMTYPE_OSD, WMoveD, 3, windownr, x, y);
} 

inline static int MoveWindowAbs(struct dvb_struct *dvb, u8 windownr, u16 x, u16 y)
{
        return outcom(dvb, COMTYPE_OSD, WMoveA, 3, windownr, x, y);
} 

inline static int DestroyOSDWindow(struct dvb_struct *dvb, u8 windownr)
{
        return outcom(dvb, COMTYPE_OSD, WDestroy, 1, windownr);
} 

#if 0
static void DestroyOSDWindows(struct dvb_struct *dvb)
{
        int i;

        for (i=1; i<7; i++)
                outcom(dvb, COMTYPE_OSD, WDestroy, 1, i);
} 
#endif

static inline int 
CreateOSDWindow(struct dvb_struct *dvb, u8 windownr,
                           DISPTYPE disptype, u16 width, u16 height)
{
        return outcom(dvb, COMTYPE_OSD, WCreate, 4,
                      windownr, disptype, width, height);
} 


static OSDPALTYPE bpp2pal[8]={Pal1Bit, Pal2Bit, 0, Pal4Bit, 0, 0, 0, Pal8Bit}; 
static DISPTYPE   bpp2bit[8]={BITMAP1, BITMAP2, 0, BITMAP4, 0, 0, 0, BITMAP8}; 

static inline int 
LoadBitmap(struct dvb_struct *dvb, u16 format, u16 dx, u16 dy, int inc, u8* data)
{
        int bpp;
        int i;
        int d, delta; 
        u8 c;
        DECLARE_WAITQUEUE(wait, current);
        
        if (dvb->bmp_state==BMP_LOADING) {
                add_wait_queue(&dvb->bmpq, &wait);
                while (1) {
                        set_current_state(TASK_INTERRUPTIBLE);
                        if (dvb->bmp_state!=BMP_LOADING
                            || signal_pending(current))
                                break;
                        schedule();
                }
                current->state=TASK_RUNNING;
                remove_wait_queue(&dvb->bmpq, &wait);
        }
        if (dvb->bmp_state==BMP_LOADING)
                return -1;
        dvb->bmp_state=BMP_LOADING;
        if      (format==BITMAP8) { bpp=8; delta = 1; } 
        else if (format==BITMAP4) { bpp=4; delta = 2; }
        else if (format==BITMAP2) { bpp=2; delta = 4; }
        else if (format==BITMAP1) { bpp=1; delta = 8; }
        else {
                dvb->bmp_state=BMP_NONE;
                return -1;
        }
        dvb->bmplen= ((dx*dy*bpp+7)&~7)/8; 
        dvb->bmpp=0;
        if (dvb->bmplen>32768) {
                dvb->bmp_state=BMP_NONE;
                return -1;
        }
        for (i=0; i<dy; i++) {
                if (copy_from_user(dvb->bmpbuf+1024+i*dx, data+i*inc, dx)) { 
                        dvb->bmp_state=BMP_NONE;
                        return -1;
                }
        }
        if (format != BITMAP8) {
                for (i=0; i<dx*dy/delta; i++) {
                        c = ((u8 *)dvb->bmpbuf)[1024+i*delta+delta-1];
                        for (d=delta-2; d>=0; d--) {
                                c |= (((u8 *)dvb->bmpbuf)[1024+i*delta+d] 
                                      << ((delta-d-1)*bpp));
                                ((u8 *)dvb->bmpbuf)[1024+i] = c;
                        }
                }
        }
        dvb->bmplen+=1024;
        return outcom(dvb, COMTYPE_OSD, LoadBmp, 3, format, dx, dy);
} 

static int 
BlitBitmap(struct dvb_struct *dvb, u16 win, u16 x, u16 y, u16 trans)
{
        DECLARE_WAITQUEUE(wait, current);
        
        if (dvb->bmp_state==BMP_NONE)
                return -1;
        if (dvb->bmp_state==BMP_LOADING) {
                add_wait_queue(&dvb->bmpq, &wait);
                while (1) {
                        set_current_state(TASK_INTERRUPTIBLE);
                        if (dvb->bmp_state!=BMP_LOADING
                            || signal_pending(current))
                                break;
                        schedule();
                }
                current->state=TASK_RUNNING;
                remove_wait_queue(&dvb->bmpq, &wait);
        }
        if (dvb->bmp_state==BMP_LOADED)
                return outcom(dvb, COMTYPE_OSD, BlitBmp, 4, win, x, y, trans);
        return -1;
} 

static inline int 
ReleaseBitmap(struct dvb_struct *dvb)
{
        if (dvb->bmp_state!=BMP_LOADED)
                return -1;
        dvb->bmp_state=BMP_NONE;
        return outcom(dvb, COMTYPE_OSD, ReleaseBmp, 0);
} 

static u32 RGB2YUV(u16 R, u16 G, u16 B)
{
        u16 y, u, v;
        u16 Y, Cr, Cb;

        y = R * 77 + G * 150 + B * 29;  // Luma=0.299R+0.587G+0.114B 0..65535
        u = 2048+B * 8 -(y>>5);    // Cr 0..4095
        v = 2048+R * 8 -(y>>5);    // Cb 0..4095

        Y=y/256;
        Cb=u/16;
        Cr=v/16;

        return Cr|(Cb<<16)|(Y<<8);
}

static void
OSDSetColor(struct dvb_struct *dvb, u8 color, u8 r, u8 g, u8 b, u8 blend)
{
        u16 ch, cl;
        u32 yuv;

        yuv=blend ? RGB2YUV(r,g,b) : 0;
        cl=(yuv&0xffff);
        ch=((yuv>>16)&0xffff);
        SetColor_(dvb, dvb->osdwin, bpp2pal[dvb->osdbpp[dvb->osdwin]],
                  color, ch, cl);
        SetBlend_(dvb, dvb->osdwin, bpp2pal[dvb->osdbpp[dvb->osdwin]],
                  color, ((blend>>4)&0x0f));
}

static int
OSDSetBlock(struct dvb_struct *dvb, int x0, int y0, int x1, int y1, int inc, u8 *data)
{
        uint w, h, bpp, bpl, size, lpb, bnum, brest;
        int i;

        w=x1-x0+1; h=y1-y0+1;
        if (inc<=0)
                inc=w; 
        if (w<=0 || w>720 || h<=0 || h>576) 
                return -1;
        bpp=dvb->osdbpp[dvb->osdwin]+1; 
        bpl=((w*bpp+7)&~7)/8; 
        size=h*bpl;
        lpb=(32*1024)/bpl; 
        bnum=size/(lpb*bpl);
        brest=size-bnum*lpb*bpl;

        for (i=0; i<bnum; i++) {
                LoadBitmap(dvb, bpp2bit[dvb->osdbpp[dvb->osdwin]], w, lpb, inc, data); 
                BlitBitmap(dvb, dvb->osdwin, x0, y0+i*lpb, 0);
                data+=lpb*inc; 
        }
        if (brest) {
                LoadBitmap(dvb, bpp2bit[dvb->osdbpp[dvb->osdwin]], w, brest/bpl, inc, data); 
                BlitBitmap(dvb, dvb->osdwin, x0, y0+bnum*lpb, 0);
        }
        ReleaseBitmap(dvb);
        return 0;
}

static int 
OSD_DrawCommand(struct dvb_struct *dvb, osd_cmd_t *dc)
{
        switch (dc->cmd) {
        case OSD_Close:
                DestroyOSDWindow(dvb, dvb->osdwin);
                return 0;
        case OSD_Open:
                dvb->osdbpp[dvb->osdwin]=(dc->color-1)&7;
                CreateOSDWindow(dvb, dvb->osdwin, bpp2bit[dvb->osdbpp[dvb->osdwin]],
                                dc->x1-dc->x0+1, dc->y1-dc->y0+1);
                if (!dc->data) {
                        MoveWindowAbs(dvb, dvb->osdwin, dc->x0, dc->y0);
                        SetColorBlend(dvb, dvb->osdwin);
                }
                return 0;
        case OSD_Show:
                MoveWindowRel(dvb, dvb->osdwin, 0, 0);
                return 0;
        case OSD_Hide:
                HideWindow(dvb, dvb->osdwin);
                return 0;
        case OSD_Clear:
                DrawBlock(dvb, dvb->osdwin, 0, 0, 720, 576, 0);
                return 0;
        case OSD_Fill:
                DrawBlock(dvb, dvb->osdwin, 0, 0, 720, 576, dc->color);
                return 0;
        case OSD_SetColor:
                OSDSetColor(dvb, dc->color, dc->x0, dc->y0, dc->x1, dc->y1); 
                return 0;
        case OSD_SetPalette:
        {      
                int i, len=dc->x0-dc->color+1;
                u8 colors[len*4];

                if (copy_from_user(colors, dc->data, sizeof(colors)))
                        return -EFAULT;
                for (i=0; i<len; i++)
                        OSDSetColor(dvb, dc->color+i,
                                    colors[i*4]  , colors[i*4+1],
                                    colors[i*4+2], colors[i*4+3]);
                return 0;
        }
        case OSD_SetTrans: 
                return 0;
        case OSD_SetPixel:
                DrawLine(dvb, dvb->osdwin,
                         dc->x0, dc->y0, 0, 0,
                         dc->color);
                return 0;
        case OSD_GetPixel: 
                return 0;

        case OSD_SetRow:   
                dc->y1=dc->y0;
        case OSD_SetBlock:
                OSDSetBlock(dvb, dc->x0, dc->y0, dc->x1, dc->y1, dc->color, dc->data);
                return 0;

        case OSD_FillRow:
                DrawBlock(dvb, dvb->osdwin, dc->x0, dc->y0,
                          dc->x1-dc->x0+1, dc->y1,
                          dc->color);
                return 0;
        case OSD_FillBlock:
                DrawBlock(dvb, dvb->osdwin, dc->x0, dc->y0,
                          dc->x1-dc->x0+1, dc->y1-dc->y0+1,
                          dc->color);
                return 0;
        case OSD_Line:
                DrawLine(dvb, dvb->osdwin,
                         dc->x0, dc->y0, dc->x1-dc->x0, dc->y1-dc->y0,
                         dc->color);
                return 0;
        case OSD_Query: 
                return 0;
        case OSD_Test:
                return 0;
        case OSD_Text:
        {
                char textbuf[240];
                
                if (strncpy_from_user(textbuf, dc->data, 240)<0)
                        return -EFAULT;
                textbuf[239]=0;
                if (dc->x1>3) 
                        dc->x1=3;
                SetFont(dvb, dvb->osdwin, dc->x1,
                        (u16) (dc->color&0xffff), (u16) (dc->color>>16));
                FlushText(dvb);
                WriteText(dvb, dvb->osdwin, dc->x0, dc->y0, textbuf);
                return 0;
        }
        case OSD_SetWindow:
                if (dc->x0<1 || dc->x0>7)
                        return -EINVAL;
                dvb->osdwin=dc->x0;
                return 0;
        case OSD_MoveWindow:
                MoveWindowAbs(dvb, dvb->osdwin, dc->x0, dc->y0);
                SetColorBlend(dvb, dvb->osdwin);
                return 0;
        default:
                return -EINVAL;
        }
}
#endif /* USE_OSD */


/* get version of the firmware ROM, RTSL, video ucode and ARM application  */

static void 
firmversion(struct dvb_struct *dvb)
{
        u16 buf[20];

        u16 tag = ((COMTYPE_REQUEST << 8) + ReqVersion);
        
        RequestParameter(dvb, tag, buf, 16);
        
        dvb->arm_fw=(buf[0] << 16) + buf[1];
        dvb->arm_rtsl=(buf[2] << 16) + buf[3];
        dvb->arm_vid=(buf[4] << 16) + buf[5];
        dvb->arm_app=(buf[6] << 16) + buf[7];
        dvb->avtype=(buf[8] << 16) + buf[9];

        printk("dvb%d: AV711%d - firm %08x, rtsl %08x, vid %08x, app %08x\n", 
               dvb->num, dvb->avtype, dvb->arm_fw, 
               dvb->arm_rtsl, dvb->arm_vid, dvb->arm_app);
        return;
}

static int 
waitdebi(struct dvb_struct *dvb, int adr, int state)
{
        int k;
        
        for (k=0; k<100; k++, udelay(500)) {
                if (irdebi(dvb, DEBINOSWAP, adr, 0, 2) == state) 
                        return 0;
        }
        return -1;
}


static int 
load_dram(struct dvb_struct *dvb, u32 *data, int len)
{
        int i;
        int blocks, rest;
        u32 base, bootblock=BOOT_BLOCK;
        
        blocks=len/BOOT_MAX_SIZE;
        rest=len % BOOT_MAX_SIZE;
        base=DRAM_START_CODE;
        
        for (i=0; i<blocks; i++) {
                if (waitdebi(dvb, BOOT_STATE, BOOTSTATE_BUFFER_EMPTY) < 0)
                        return -1;
                dprintk("Writing DRAM block %d\n",i);
                iwdebi(dvb, DEBISWAB, bootblock,
                       i*(BOOT_MAX_SIZE)+(u32)data,
                       BOOT_MAX_SIZE);
                bootblock^=0x1400;
                iwdebi(dvb, DEBISWAB, BOOT_BASE, cpu_to_be32(base), 4);
                iwdebi(dvb, DEBINOSWAP, BOOT_SIZE, BOOT_MAX_SIZE, 2);
                iwdebi(dvb, DEBINOSWAP, BOOT_STATE, BOOTSTATE_BUFFER_FULL, 2);
                base+=BOOT_MAX_SIZE;
        }
        
        if (rest > 0) {
                if (waitdebi(dvb, BOOT_STATE, BOOTSTATE_BUFFER_EMPTY) < 0)
                        return -1;
                if (rest>4)
                        iwdebi(dvb, DEBISWAB, bootblock, i*(BOOT_MAX_SIZE)+(u32)data, rest);
                else
                        iwdebi(dvb, DEBISWAB, bootblock, i*(BOOT_MAX_SIZE)-4+(u32)data, rest+4);
                
                iwdebi(dvb, DEBISWAB, BOOT_BASE, cpu_to_be32(base), 4);
                iwdebi(dvb, DEBINOSWAP, BOOT_SIZE, rest, 2);
                iwdebi(dvb, DEBINOSWAP, BOOT_STATE, BOOTSTATE_BUFFER_FULL, 2);
        }
        if (waitdebi(dvb, BOOT_STATE, BOOTSTATE_BUFFER_EMPTY) < 0)
                return -1;
        iwdebi(dvb, DEBINOSWAP, BOOT_SIZE, 0, 2);
        iwdebi(dvb, DEBINOSWAP, BOOT_STATE, BOOTSTATE_BUFFER_FULL, 2);
        if (waitdebi(dvb, BOOT_STATE, BOOTSTATE_BOOT_COMPLETE) < 0)
                return -1;
        return 0;
}


#define INC_BOOT
//#define INC_FIRM
//#define FIRM_MOD

static u8 
bootcode[] = {
        0xea, 0x00, 0x00, 0x0e, 0xe1, 0xb0, 0xf0, 0x0e, /* 0x0000 */
        0xe2, 0x5e, 0xf0, 0x04, 0xe2, 0x5e, 0xf0, 0x04,
        0xe2, 0x5e, 0xf0, 0x08, 0xe2, 0x5e, 0xf0, 0x04,
        0xe2, 0x5e, 0xf0, 0x04, 0xe2, 0x5e, 0xf0, 0x04,
        0x2c, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x0c,
        0x00, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x34,
        0x00, 0x00, 0x00, 0x00, 0xa5, 0xa5, 0x5a, 0x5a,
        0x00, 0x1f, 0x15, 0x55, 0x00, 0x00, 0x00, 0x09,
        0xe5, 0x9f, 0xd0, 0x5c, 0xe5, 0x9f, 0x40, 0x54, /* 0x0040 */
        0xe3, 0xa0, 0x00, 0x00, 0xe5, 0x84, 0x00, 0x00,
        0xe5, 0x84, 0x00, 0x04, 0xe1, 0xd4, 0x10, 0xb0,
        0xe3, 0x51, 0x00, 0x00, 0x0a, 0xff, 0xff, 0xfc,
        0xe1, 0xa0, 0x10, 0x0d, 0xe5, 0x94, 0x30, 0x04,
        0xe1, 0xd4, 0x20, 0xb2, 0xe2, 0x82, 0x20, 0x3f,
        0xe1, 0xb0, 0x23, 0x22, 0x03, 0xa0, 0x00, 0x02,
        0xe1, 0xc4, 0x00, 0xb0, 0x0a, 0xff, 0xff, 0xf4,
        0xe8, 0xb1, 0x1f, 0xe0, 0xe8, 0xa3, 0x1f, 0xe0, /* 0x0080 */
        0xe8, 0xb1, 0x1f, 0xe0, 0xe8, 0xa3, 0x1f, 0xe0,
        0xe2, 0x52, 0x20, 0x01, 0x1a, 0xff, 0xff, 0xf9,
        0xe2, 0x2d, 0xdb, 0x05, 0xea, 0xff, 0xff, 0xec,
        0x2c, 0x00, 0x03, 0xf8, 0x2c, 0x00, 0x04, 0x00,
};

#ifdef INC_FIRM
#include "dvb_firm.h"
#endif

#ifdef FIRM_MOD
unsigned char *dvb_dpram_addr, *dvb_root_addr;
int dvb_dpram_len, dvb_root_len;
static int firm_mod=-1;
#endif

static int 
bootarm(struct dvb_struct *dvb)
{
        u32 *lfirm;
        int len;
        int result;
        u32 tmp;
        int i;

        setgpio(dvb, RESET_LINE, GPIO_OUTLO);

        /* Disable DEBI and GPIO irq */
        saa7146_write(dvb->saa_mem, IER, 
                      saa7146_read(dvb->saa_mem, IER) & 
                      ~(MASK_19 | MASK_03));
        saa7146_write(dvb->saa_mem, ISR, (MASK_19 | MASK_03));

        /* enable DEBI */
        saa7146_write(dvb->saa_mem, MC1, 0x08800880);
        saa7146_write(dvb->saa_mem, DD1_STREAM_B, 0x00000000);
        saa7146_write(dvb->saa_mem, MC2, (MASK_09 | MASK_25 | MASK_10 | MASK_26));
        
        /* test DEBI */
        iwdebi(dvb, DEBISWAP, DPRAM_BASE, 0x76543210, 4);
        if ((tmp=irdebi(dvb, DEBINOSWAP, DPRAM_BASE, 0, 4))!=0x10325476) {
                printk("dvb: debi test in bootarm() failed: "
                       "%08x != %08x\n", tmp, 0x10325476);;
                return -1;
        }
        for (i=0; i<8192; i+=4) 
                iwdebi(dvb, DEBISWAP, DPRAM_BASE+i, 0x00, 4);
        dprintk("bootarm: debi test OK\n");;

        /* boot */
        dprintk("bootarm: load boot code\n");
#ifndef INC_BOOT
        if (
#ifdef __DVB_PACK__
            !(len=firmread("Boot_up.axf", (char**) &lfirm)) &&
#endif
            !(len=firmread(DVB_FIRM_PATH "Boot_up.axf", (char**) &lfirm)))
                        return -1;
#endif

        setgpio(dvb, ARM_IRQ_LINE, GPIO_IRQLO);
        //setgpio(dvb, DEBI_DONE_LINE, GPIO_INPUT);
        //setgpio(dvb, 3, GPIO_INPUT);

#ifdef INC_BOOT  
        iwdebi(dvb, DEBISWAB, DPRAM_BASE, (u32) bootcode, sizeof(bootcode));
#else
        iwdebi(dvb, DEBISWAB, DPRAM_BASE, (u32) lfirm, len);
#endif
        iwdebi(dvb, DEBINOSWAP, BOOT_STATE, BOOTSTATE_BUFFER_FULL, 2);
        
#ifndef INC_BOOT  
        vfree(lfirm);
#endif
        wait_for_debi_done(dvb);
        setgpio(dvb, RESET_LINE, GPIO_OUTHI);
        current->state=TASK_INTERRUPTIBLE;
        schedule_timeout(HZ);
			dprintk("probiere mal " DVB_FIRM_PATH );
        
        dprintk("bootarm: load dram code\n");
#ifdef INC_FIRM
        if (readfirm) {
#else
#ifdef FIRM_MOD
        if (firm_mod<0 || !dvb_root_len) {
#endif
#endif
                if (
#ifdef __DVB_PACK__
                        !(len=firmread("arm_firm/Root", (char**) &lfirm)) &&
                        !(len=firmread("Root", (char**) &lfirm)) &&
#endif
                        !(len=firmread(DVB_FIRM_PATH "Root", (char**) &lfirm)))
                        return -1;
                result=load_dram(dvb, lfirm, len);
                vfree(lfirm);
                if (result<0)
                        return -1;
#ifdef INC_FIRM
        } else
                if (load_dram(dvb, (u32 *)Root, sizeof(Root))<0)
                        return -1;
#else
#ifdef FIRM_MOD
        } else
                if (load_dram(dvb, (u32 *)dvb_root_addr, dvb_root_len)<0)
                        return -1;
#endif
#endif
        setgpio(dvb, RESET_LINE, GPIO_OUTLO);
        mdelay(1);
        

        dprintk("bootarm: load dpram code\n");
#ifdef INC_FIRM
        if (readfirm) {
#else
#ifdef FIRM_MOD
        if (firm_mod<0 || !dvb_dpram_len) {
                printk("no dpram_len\n");
#endif
#endif
                if (
#ifdef __DVB_PACK__
                        !(len=firmread("arm_firm/Dpram", (char**) &lfirm)) &&
                        !(len=firmread("Dpram", (char**) &lfirm)) &&
#endif
                        !(len=firmread(DVB_FIRM_PATH "Dpram", (char**) &lfirm)))
                        return -1;
                iwdebi(dvb, DEBISWAB, DPRAM_BASE, (u32) lfirm, len);
                vfree(lfirm);
#ifdef INC_FIRM
        } else 
                iwdebi(dvb, DEBISWAB, DPRAM_BASE, (u32) Dpram, sizeof(Dpram));
#else
#ifdef FIRM_MOD
        } else
                iwdebi(dvb, DEBISWAB, DPRAM_BASE, (u32) dvb_dpram_addr, 
                       dvb_dpram_len);
#endif
#endif        
        wait_for_debi_done(dvb);

        setgpio(dvb, RESET_LINE, GPIO_OUTHI);
        mdelay(800);

        //ARM_ClearIrq(dvb); 
        ARM_ResetMailBox(dvb); 
        saa7146_write(dvb->saa_mem, ISR, (MASK_19 | MASK_03));
        saa7146_write(dvb->saa_mem, IER, 
                      saa7146_read(dvb->saa_mem, IER) | MASK_03 );

        dvb->arm_errors=0;
        dvb->arm_ready=1;
        return 0;
}

static int
SetPIDs(struct dvb_struct *dvb, u16 vpid, u16 apid, u16 ttpid, 
        u16 subpid, u16 pcrpid)
{
        return outcom(dvb, COMTYPE_PIDFILTER, MultiPID, 5, 
                      pcrpid, vpid, apid, ttpid, subpid);
}

static void
ChangePIDs(struct dvb_struct *dvb, u16 vpid, u16 apid, u16 ttpid, 
        u16 subpid, u16 pcrpid)
{
        if (!(vpid&0x8000))  dvb->pids[DMX_PES_VIDEO]=vpid;
        if (!(apid&0x8000))  dvb->pids[DMX_PES_AUDIO]=apid;
        if (!(ttpid&0x8000)) dvb->pids[DMX_PES_TELETEXT]=ttpid;
        if (!(pcrpid&0x8000)) dvb->pids[DMX_PES_PCR]=pcrpid;
        dvb->pids[DMX_PES_SUBTITLE]=0;
	if (dvb->frontend.tuning!=FE_STATE_AFC) return;
        outcom(dvb, COMTYPE_PIDFILTER, MultiPID, 5, 
               pcrpid, vpid, apid, ttpid, subpid);
}

static void
SetFEPIDs(struct dvb_struct *dvb)
{
        SetPIDs(dvb, dvb->pids[DMX_PES_VIDEO], 
                dvb->pids[DMX_PES_AUDIO], 
                dvb->pids[DMX_PES_TELETEXT], 0, 
                dvb->pids[DMX_PES_PCR]);
}

static void 
SetMode(struct dvb_struct *dvb, int mode)
{
        outcom(dvb, COMTYPE_ENCODER, LoadVidCode, 1, mode);
        
        if (!dvb->playing) {
                ChangePIDs(dvb, dvb->pids[DMX_PES_VIDEO], 
                           dvb->pids[DMX_PES_AUDIO], 
                           dvb->pids[DMX_PES_TELETEXT],  
                           0, dvb->pids[DMX_PES_PCR]);
                outcom(dvb, COMTYPE_PIDFILTER, Scan, 0);
        }
}

inline static void 
TestMode(struct dvb_struct *dvb, int mode)
{
        outcom(dvb, COMTYPE_ENCODER, SetTestMode, 1, mode);
}

inline static void 
VidMode(struct dvb_struct *dvb, int mode)
{
        outcom(dvb, COMTYPE_ENCODER, SetVidMode, 1, mode);
}
           

static int inline
vidcom(struct dvb_struct *dvb, u32 com, u32 arg)
{
        return outcom(dvb, 0x80, 0x02, 4, 
                      (com>>16), (com&0xffff), 
                      (arg>>16), (arg&0xffff));
}

static int inline
audcom(struct dvb_struct *dvb, u32 com)
{
        return outcom(dvb, 0x80, 0x03, 4, 
                      (com>>16), (com&0xffff));
}

inline static void 
Set22K(struct dvb_struct *dvb, int state)
{
        if (dvb->card_type==DVB_CARD_TT_SIEMENS)
                outcom(dvb, COMTYPE_AUDIODAC, (state ? ON22K : OFF22K), 0);
        if (dvb->card_type==DVB_CARD_TT_BUDGET)
                setgpio(dvb, 3, (state ? GPIO_OUTHI : GPIO_OUTLO));
}


/* Diseqc functions only for TT Budget card */
/* taken from the Skyvision DVB driver by 
   Ralph Metzler <rjkm@metzlerbros.de> */


inline static void 
DiseqcSendBit(struct dvb_struct *dvb, int data)
{
        setgpio(dvb, 3, GPIO_OUTHI);
        udelay(data ? 500 : 1000);
        setgpio(dvb, 3, GPIO_OUTLO);
        udelay(data ? 1000 : 500);
}

static void 
DiseqcSendByte(struct dvb_struct *dvb, int data)
{
        int i, par=1, d;

        for (i=7; i>=0; i--) 
        {
                d=(data>>i)&1;
                par^=d;
                DiseqcSendBit(dvb, d);
        }
        DiseqcSendBit(dvb, par);
}

static void 
DiseqcSendMsg(struct dvb_struct *dvb, int frame, int adr, int com, int data, int swi)
{
        setgpio(dvb, 3, GPIO_OUTLO);

        mdelay(16);

        DiseqcSendByte(dvb, frame);
        DiseqcSendByte(dvb, adr);
        DiseqcSendByte(dvb, com);
        DiseqcSendByte(dvb, data);

        mdelay(16);
        
        if (swi) 
                DiseqcSendByte(dvb, 0xff);
        else {
                setgpio(dvb, 3, GPIO_OUTHI);
                udelay(12500);
                setgpio(dvb, 3, GPIO_OUTLO);
        }

        ddelay(2);

        Set22K(dvb, dvb->sec.ttk);
}


inline static void
SetDiSEqC(struct dvb_struct *dvb, int input)
{
        dprintk("setdis: %d\n", input);
        if (input<0 || input>15)
                return;
        if (dvb->card_type==DVB_CARD_TT_SIEMENS) {
                outcom(dvb, COMTYPE_AUDIODAC, SendDiSEqC, 6, 4,
                       (input&4) ? 1 : 0,
                       0xe0, 0x10, 0x38, 0xf0+input);
                outcom(dvb, COMTYPE_AUDIODAC, SendDiSEqC, 6, 4,
                       (input&4) ? 1 : 0,
                       0xe1, 0x10, 0x38, 0xf0+input);
        } else {
                DiseqcSendMsg(dvb, 0xe0, 0x10, 0x38, 0xf0+input, (input&4));
                DiseqcSendMsg(dvb, 0xe1, 0x10, 0x38, 0xf0+input, (input&4));
        }
}

inline static int
SendDiSEqCMsg(struct dvb_struct *dvb, int len, u8 *msg, int burst)
{
        int i;

        switch (dvb->card_type) {
        case DVB_CARD_TT_SIEMENS:
        {
                u16 buf[18] = { ((COMTYPE_AUDIODAC << 8) + SendDiSEqC),
                                16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
                
                if (len>10)
                        len=10;
                buf[1] = len+2;
                buf[2] = len;
                
                if (burst!=-1)
                  buf[3]=burst ? 0x01 : 0x00;
                else
                  buf[3]=0xffff;
                
                for (i=0; i<len; i++)
                  buf[i+4]=msg[i];
                
                SOutCommand(dvb, buf, 18);
                break;
        }

        case DVB_CARD_TT_BUDGET:
                setgpio(dvb, 3, GPIO_OUTLO);
                
                mdelay(16);
                
                for (i=0; i<len; i++)
                        DiseqcSendByte(dvb, msg[i]);

                mdelay(16);
                
                if (burst!=-1) {
                        if (burst) 
                                DiseqcSendByte(dvb, 0xff);
                        else {
                                setgpio(dvb, 3, GPIO_OUTHI);
                                udelay(12500);
                                setgpio(dvb, 3, GPIO_OUTLO);
                        }
                        
                        ddelay(2);
                }
                Set22K(dvb, dvb->sec.ttk);

                break;

        default:
                return -1;
        }
        return 0;
}

/****************************************************************************
 * I2C client commands
 ****************************************************************************/

void tuning_complete_cb(void *priv)
{
        struct dvb_struct *dvb=(struct dvb_struct *) priv;
        
        SetFEPIDs(dvb);
        outcom(dvb, COMTYPE_PIDFILTER, Scan, 0);
}

static inline int
i2c_writereg(struct dvb_struct *dvb, u8 id, u8 reg, u8 val)
{
        u8 msg[2]={ reg, val }; 
        struct i2c_adapter *adap=dvb->i2cbus;
        struct i2c_msg msgs;

        msgs.flags=0;
        msgs.addr=id/2;
        msgs.len=2;
        msgs.buf=msg;
        return i2c_transfer(adap, &msgs, 1);
}

static inline int
msp_writereg(struct dvb_struct *dvb, u8 dev, u16 reg, u16 val)
{
        u8 msg[5]={ dev, reg>>8, reg&0xff, val>>8 , val&0xff }; 
        struct i2c_adapter *adap=dvb->i2cbus;
        struct i2c_msg msgs;

        msgs.flags=0;
        msgs.addr=0x40;
        msgs.len=5;
        msgs.buf=msg;
        return i2c_transfer(adap, &msgs, 1);
}

static inline u8
i2c_readreg(struct dvb_struct *dvb, u8 id, u8 reg)
{
        struct i2c_adapter *adap=dvb->i2cbus;
        u8 mm1[] = {0x00};
        u8 mm2[] = {0x00};
        struct i2c_msg msgs[2];

        msgs[0].flags=0;
        msgs[1].flags=I2C_M_RD;
        msgs[0].addr=msgs[1].addr=id/2;
        mm1[0]=reg;
        msgs[0].len=1; msgs[1].len=1;
        msgs[0].buf=mm1; msgs[1].buf=mm2;
        i2c_transfer(adap, msgs, 2);

        return mm2[0];
}


static inline void 
demod_writereg(struct dvb_struct *dvb, u8 adr, u8 val)
{
        u8 msg[2]={ adr, val }; 
  
        dvb_frontend_demod_command(&dvb->frontend, FE_WRITEREG, &msg); 
}

static inline u8
demod_readreg(struct dvb_struct *dvb, u8 adr)
{
        u8 msg[2]={ adr, 0 }; 
  
        dvb_frontend_demod_command(&dvb->frontend, FE_READREG, &msg); 
        return msg[1];
}


static void 
SetVoltage(struct dvb_struct *dvb, int power, int voltage)
{
        if (!dvb->frontend.demod)
                return;

        switch (dvb->frontend.demod_type) {
        case DVB_DEMOD_STV0299:
                demod_writereg(dvb, 0x0c, 
                               (demod_readreg(dvb, 0x0c)&0x0f)|
                               (power ? 0x40 : 0)|(voltage ? 0x10 : 0));
                break;
        case DVB_DEMOD_VES1893:
                demod_writereg(dvb, 0x1f, (power ? 0x20 : 0)|(voltage ? 0x10 : 0));
                break;
        case DVB_DEMOD_TDA8083:
                demod_writereg(dvb, 0x20, (voltage ? 0x11 : 0x01));
                break;
        default:
                break;
        }
}

static void
InitFront(struct dvb_struct *dvb)
{
        FrontendParameters para;
        dvb_front_t *fe;
        struct i2c_adapter *adap=dvb->i2cbus;
        int i;
        
        fe=&dvb->frontend;
        fe->tuner_type = -1;

        for (i=0; i<adap->client_count; i++) {
                if (!adap->clients[i])
                        continue;
                dprintk("adapter id = %02x  addr = %02x\n",
                        adap->clients[i]->driver->id,
                        adap->clients[i]->addr*2);
                
                switch (adap->clients[i]->driver->id) {
                case I2C_DRIVERID_TUNER:
                        fe->tuner = adap->clients[i];
                        break;
                case I2C_DRIVERID_VES1893:
                        fe->type = DVB_S;
                        fe->demod_type = DVB_DEMOD_VES1893;
                        fe->tuner_type = TUNER_SP5659;
                        fe->demod = adap->clients[i];
                        break;
                case I2C_DRIVERID_VES1820:
                        fe->type = DVB_C;
                        fe->demod_type = DVB_DEMOD_VES1820;
                        fe->tuner_type = TUNER_DVBC;
                        if (dvb->pdev->subsystem_vendor==0x13c2)
                                fe->tuner_type = TUNER_SP5659_C;
                        fe->demod = adap->clients[i];
                        break;
                case I2C_DRIVERID_L64781:
			fe->type = DVB_T;
                        fe->demod_type = DVB_DEMOD_L64781;
                        fe->tuner_type = TUNER_TSA5060;
                        fe->demod = adap->clients[i];
                        break;
                case I2C_DRIVERID_SP8870:
			fe->type = DVB_T;
			fe->demod_type = DVB_DEMOD_SP8870;
			fe->tuner_type = TUNER_SP5668;
			fe->demod = adap->clients[i];
			break;
		case I2C_DRIVERID_SAA7113:
                        printk("dvb%d: SAA7113 detected\n", dvb->num);
                        break;
                case I2C_DRIVERID_STV0299:
                        fe->type = DVB_S;
                        fe->demod_type = DVB_DEMOD_STV0299;
                        fe->tuner_type = TUNER_TSA5059;
                        fe->demod = adap->clients[i];
                        break;
                case I2C_DRIVERID_TDA8083:
                        fe->type = DVB_S;
                        fe->demod_type = DVB_DEMOD_TDA8083;
                        fe->tuner_type = TUNER_TSA5522;
                        fe->demod = adap->clients[i];
                        break;
                }
        }

        fe->priv=(void *)dvb;
        fe->complete_cb=tuning_complete_cb;
        fe->start_cb=0;
        fe->i2cbus=dvb->i2cbus;
        dvb_frontend_init(fe);

        dvb->powerstate=FE_POWER_ON;
        dvb->sec.power=1; 
        dvb->pids[DMX_PES_PCR]=0;

        if (!fe->tuner || !fe->demod) 
                return;

        switch (fe->type) {
        case DVB_S:
                saa7146_write(dvb->saa_mem, MC2, 
                              (MASK_09 | MASK_25 | MASK_10 | MASK_26));
                if (!init_chan)
                        break;
                
                if (init_chan==2) {
                        // Astra n-tv default
                        ChangePIDs(dvb, 0x00a2, 0x0060, 0x0037, 0, 0x00a2);
                        secSetTone(dvb, SEC_TONE_ON);
                        secSetVoltage(dvb, SEC_VOLTAGE_13);
                        para.Frequency=12669500-10600000;
                        para.u.qpsk.SymbolRate=22000000;
                        para.u.qpsk.FEC_inner=0;
                } else {
                        // pro sieben
                        ChangePIDs(dvb, 0x00ff, 0x0100, 0x0020, 0, 0x00ff);
                        secSetTone(dvb, SEC_TONE_ON);
                        secSetVoltage(dvb, SEC_VOLTAGE_13);
                        para.Frequency=12480000-10600000;
                        para.u.qpsk.SymbolRate=27500000;
                        para.u.qpsk.FEC_inner=0;
                }
		break;
        case DVB_C:
                demod_writereg(dvb, 0x0f, 0x28); //use 0x68 for analog 
                if (!init_chan)
                        break;
                ChangePIDs(dvb, 0x0262, 0x026c, 0, 0, 0x0262);
                para.Frequency=394000000;
                para.u.qam.SymbolRate=6900000;
                para.u.qam.FEC_inner=0;
                para.u.qam.QAM=QAM_64;
                break;
        case DVB_T:
//              if (!init_chan)
//                      break;
                ChangePIDs(dvb, 110, 120, 0, 0, 0);
                para.Frequency=730000000;
                para.u.ofdm.bandWidth=BANDWIDTH_8_MHZ;
                para.u.ofdm.HP_CodeRate=FEC_2_3;
                para.u.ofdm.LP_CodeRate=FEC_1_2;
                para.u.ofdm.Constellation=QAM_16;
                para.u.ofdm.TransmissionMode=TRANSMISSION_MODE_2K;
                para.u.ofdm.guardInterval=GUARD_INTERVAL_1_8;
                para.u.ofdm.HierarchyInformation=HIERARCHY_NONE;
                break;
        }
	tune(dvb, &para);
	SetVolume(dvb, 0xff, 0xff);

        if (dvb->card_type==DVB_CARD_TT_SIEMENS) {
                VidMode(dvb, vidmode);
                //iwdebi(dvb, DEBINOSWAP, RX_BUFF, 0, 2);
                //iwdebi(dvb, DEBINOSWAP, TX_BUFF, 0, 2);
                //ARM_ResetMailBox(dvb);
        }
}




/****************************************************************************
 * I/O buffer management and control
 ****************************************************************************/

static int sw2mode[16] = {
        VIDEO_MODE_PAL, VIDEO_MODE_NTSC, VIDEO_MODE_NTSC, VIDEO_MODE_PAL,
        VIDEO_MODE_NTSC, VIDEO_MODE_NTSC, VIDEO_MODE_PAL, VIDEO_MODE_NTSC,
        VIDEO_MODE_PAL, VIDEO_MODE_PAL, VIDEO_MODE_PAL, VIDEO_MODE_PAL,
        VIDEO_MODE_PAL, VIDEO_MODE_PAL, VIDEO_MODE_PAL, VIDEO_MODE_PAL,
};

static void
get_video_format(struct dvb_struct *dvb, u8 *buf, int count)
{
        int i;
	int hsize,vsize;
        int sw;
        u8 *p;

        if (dvb->sinfo)
                return;
        for (i=7; i<count-10; i++) {
                p=buf+i;
                if (p[0] || p[1] || p[2]!=0x01 || p[3]!=0xb3)
                        continue;
                p+=4;
                hsize = ((p[1] &0xF0) >> 4) | (p[0] << 4);
                vsize = ((p[1] &0x0F) << 8) | (p[2]);
                sw = (p[3]&0x0F);
                SetMode(dvb, sw2mode[sw]);
                dprintk("dvb: playback %dx%d fr=%d\n", hsize, vsize, sw);
                dvb->sinfo=1;
                break;
        }
}

static void
play_video_cb(u8 *buf, int count, void *priv)
{
        struct dvb_struct *dvb=(struct dvb_struct *) priv;

        if ((buf[3]&0xe0)==0xe0) {
                get_video_format(dvb, buf, count);
                ring_buffer_write(&dvb->avout, buf, count, 0, 0);
        } else
                ring_buffer_write(&dvb->aout, buf, count, 0, 0);
}

static void
play_audio_cb(u8 *buf, int count, void *priv)
{
        struct dvb_struct *dvb=(struct dvb_struct *) priv;
        
        ring_buffer_write(&dvb->aout, buf, count, 0, 0);
}

#define FREE_COND (ring_buffer_free(&dvb->avout)>=20*1024 && ring_buffer_free(&dvb->aout)>=20*1024)

static ssize_t 
dvb_play(struct dvb_struct *dvb, const u8 *buf,
         unsigned long count, int nonblock, int type, int umem)
{
        unsigned long todo = count, n;

        if (!dvb->kbuf[type])
                return -ENOBUFS;
        if (nonblock && !FREE_COND)
                return -EWOULDBLOCK;
                
        while (todo>0) {
                if (!FREE_COND) {
                        if (nonblock)
                                return count-todo;
                        if (wait_event_interruptible(dvb->avout.queue,
                                                     FREE_COND))
                        	return count-todo;
                } 
		n=todo;
                if (n>IPACKS*2)
                        n=IPACKS*2;
                if (umem) {
                        if (copy_from_user(dvb->kbuf[type], buf, n)) 
                                return -EFAULT;
                        instant_repack(dvb->kbuf[type], n, &dvb->ipack[type]);
                } else
                        instant_repack((u8 *)buf, n, &dvb->ipack[type]);
                todo -= n;
                buf += n;
        }
	return count-todo;
}

static ssize_t 
dvb_aplay(struct dvb_struct *dvb, const u8 *buf,
         unsigned long count, int nonblock, int type)
{
        unsigned long todo = count, n;

        if (!dvb->kbuf[type])
                return -ENOBUFS;
        if (nonblock && ring_buffer_free(&dvb->aout)<20*1024)
                return -EWOULDBLOCK;
                
        while (todo>0) {
                if (ring_buffer_free(&dvb->aout)<20*1024) {
                        if (nonblock)
                                return count-todo;
                        if (wait_event_interruptible(dvb->aout.queue,
                                                     (ring_buffer_free(&dvb->aout)>=
                                                      20*1024)))
                        	return count-todo;
                } 
		n=todo;
                if (n>IPACKS*2)
                        n=IPACKS*2;
                if (copy_from_user(dvb->kbuf[type], buf, n)) 
                        return -EFAULT;
                instant_repack(dvb->kbuf[type], n, &dvb->ipack[type]);
                todo -= n;
                buf += n;
        }
	return count-todo;
}

void init_p2t(p2t_t *p, dvb_demux_feed_t *feed)
{
	memset(p->pes,0,TS_SIZE);
	p->counter = 0;
	p->pos = 0;
	p->frags = 0;
	if (feed) p->feed = feed;
}

void clear_p2t(p2t_t *p)
{
	memset(p->pes,0,TS_SIZE);
	p->counter = 0;
	p->pos = 0;
	p->frags = 0;
}


long int find_pes_header(u8 const *buf, long int length, int *frags)
{
	int c = 0;
	int found = 0;

	*frags = 0;

	while (c < length-3 && !found) {
		if (buf[c] == 0x00 && buf[c+1] == 0x00 && 
		    buf[c+2] == 0x01) {
			switch ( buf[c+3] ) {
				
			case PROG_STREAM_MAP:
			case PRIVATE_STREAM2:
			case PROG_STREAM_DIR:
			case ECM_STREAM     :
			case EMM_STREAM     :
			case PADDING_STREAM :
			case DSM_CC_STREAM  :
			case ISO13522_STREAM:
			case PRIVATE_STREAM1:
			case AUDIO_STREAM_S ... AUDIO_STREAM_E:
			case VIDEO_STREAM_S ... VIDEO_STREAM_E:
				found = 1;
				break;
				
			default:
				c++;
				break;
			}	
		} else c++;
	}
	if (c == length-3 && !found){
		if (buf[length-1] == 0x00) *frags = 1;
		if (buf[length-2] == 0x00 &&
		    buf[length-1] == 0x00) *frags = 2;
		if (buf[length-3] == 0x00 &&
		    buf[length-2] == 0x00 &&
		    buf[length-1] == 0x01) *frags = 3;
		return -1;
	}

	return c;
}

void pes_to_ts( u8 const *buf, long int length, u16 pid, p2t_t *p)
{
	int c,c2,l,add;
	int check,rest;

	c = 0;
	c2 = 0;
	if (p->frags){
		check = 0;
		switch(p->frags){
		case 1:
			if ( buf[c] == 0x00 && buf[c+1] == 0x01 ){
				check = 1;
				c += 2;
			}
			break;
		case 2:
			if ( buf[c] == 0x01 ){
				check = 1;
				c++;
			}
			break;
		case 3:
			check = 1;
		}
		if(check){
			switch ( buf[c] ) {
				
			case PROG_STREAM_MAP:
			case PRIVATE_STREAM2:
			case PROG_STREAM_DIR:
			case ECM_STREAM     :
			case EMM_STREAM     :
			case PADDING_STREAM :
			case DSM_CC_STREAM  :
			case ISO13522_STREAM:
			case PRIVATE_STREAM1:
			case AUDIO_STREAM_S ... AUDIO_STREAM_E:
			case VIDEO_STREAM_S ... VIDEO_STREAM_E:
				p->pes[0] = 0x00;
				p->pes[1] = 0x00;
				p->pes[2] = 0x01;
				p->pes[3] = buf[c];
				p->pos=4;
				memcpy(p->pes+p->pos,buf+c,(TS_SIZE-4)-p->pos);
				c += (TS_SIZE-4)-p->pos;
				p_to_t(p->pes,(TS_SIZE-4),pid,&p->counter,
				       p->feed);
				clear_p2t(p);
				break;
				
			default:
				c=0;
				break;
			}
		}
		p->frags = 0;
	}
		
	if (p->pos){
		c2 = find_pes_header(buf+c,length-c,&p->frags);
		if (c2 >= 0 && c2 < (TS_SIZE-4)-p->pos){
			l = c2+c;
		} else l = (TS_SIZE-4)-p->pos;
		memcpy(p->pes+p->pos,buf,l);
		c += l;
		p->pos += l;
		p_to_t(p->pes,p->pos,pid,&p->counter, p->feed);
		clear_p2t(p);
	}
			
	add = 0;
	while (c < length){
		c2 = find_pes_header(buf+c+add,length-c-add,&p->frags);
		if (c2 >= 0) {
			c2 += c+add;
			if (c2 > c){
				p_to_t(buf+c,c2-c,pid,&p->counter,
				       p->feed);
				c = c2;
				clear_p2t(p);
				add = 0;
			} else add = 1;
		} else {
			l = length-c;
			rest = l % (TS_SIZE-4);
			l -= rest;
			p_to_t(buf+c,l,pid,&p->counter,
			       p->feed);
			memcpy(p->pes,buf+c+l,rest);
			p->pos = rest;
			c = length;
		}
	}
}


int write_ts_header2(u16 pid, u8 *counter, int pes_start, u8 *buf, u8 length)
{
	int i;
	int c = 0;
	int fill;
	u8 tshead[4] = { 0x47, 0x00, 0x00, 0x10}; 
        

	fill = (TS_SIZE-4)-length;
        if (pes_start) tshead[1] = 0x40;
	if (fill) tshead[3] = 0x30;
        tshead[1] |= (u8)((pid & 0x1F00) >> 8);
        tshead[2] |= (u8)(pid & 0x00FF);
        tshead[3] |= ((*counter)++ & 0x0F) ;
        memcpy(buf,tshead,4);
	c+=4;


	if (fill){
		buf[4] = fill-1;
		c++;
		if (fill >1){
			buf[5] = 0x00;
			c++;
		}
		for ( i = 6; i < fill+4; i++){
			buf[i] = 0xFF;
			c++;
		}
	}

        return c;
}


void p_to_t(u8 const *buf, long int length, u16 pid, u8 *counter, 
            dvb_demux_feed_t *feed)
{
  
	int l, pes_start;
	u8 obuf[TS_SIZE];
	long int c = 0;

	pes_start = 0;
	if ( length > 3 && 
             buf[0] == 0x00 && buf[1] == 0x00 && buf[2] == 0x01 )
		switch (buf[3]){
			case PROG_STREAM_MAP:
			case PRIVATE_STREAM2:
			case PROG_STREAM_DIR:
			case ECM_STREAM     :
			case EMM_STREAM     :
			case PADDING_STREAM :
			case DSM_CC_STREAM  :
			case ISO13522_STREAM:
			case PRIVATE_STREAM1:
			case AUDIO_STREAM_S ... AUDIO_STREAM_E:
			case VIDEO_STREAM_S ... VIDEO_STREAM_E:
				pes_start = 1;
				break;
				
			default:
				break;
		}			

	while ( c < length ){
		memset(obuf,0,TS_SIZE);
		if (length - c >= (TS_SIZE-4)){
			l = write_ts_header2(pid, counter, pes_start
					     , obuf, (TS_SIZE-4));
			memcpy(obuf+l, buf+c, TS_SIZE-l);
			c += TS_SIZE-l;
		} else { 
			l = write_ts_header2(pid, counter, pes_start
					     , obuf, length-c);
			memcpy(obuf+l, buf+c, TS_SIZE-l);
			c = length;
		}
                feed->cb.ts(obuf, 188, 0, 0, &feed->feed.ts, DMX_OK); 
		pes_start = 0;
	}
}


/****************************************************************************
 * V4L SECTION
 ****************************************************************************/

static struct video_channel dvb_channels[1] = {
        { 0, "DVB", 1, 3, 1, 1},
};

static int dvb_ioctl(struct video_device *dev, unsigned int cmd, void *arg)
{
	struct dvb_struct* dvb = (struct dvb_struct*)dev;
        
	switch (cmd)
	{
		/* Get capabilities */
        case VIDIOCGCAP:
        {
                struct video_capability b;
                
                dprintk(KERN_ERR "dvb: VIDIOCGCAP called\n");

                strcpy(b.name, &dev->name[0]);
			
                b.type 		= dvb->video.type;
						
                b.channels	= 1;	
                b.audios	= 2;
                b.maxwidth	= 768;
                b.maxheight	= 576;
                b.minwidth	= 32;
                b.minheight	= 32;
                
                if(copy_to_user(arg,&b,sizeof(b)))
                        return -EFAULT;
                return 0;
        }
		/* Get channel info (sources) */
        case VIDIOCGCHAN:
        {
                struct video_channel v;
                
                dprintk(KERN_ERR "dvb: VIDIOCGCHAN called\n");
                
                if(copy_from_user(&v, arg,sizeof(v)))
                        return -EFAULT;
                
                /* copy the desired structure to v */
                memcpy(&v, &dvb_channels[v.channel], sizeof(v));
                
                if(copy_to_user(arg,&v,sizeof(v)))
                        return -EFAULT;
                
                return 0;
                
        }
		/* Set channel 	*/
        case VIDIOCSCHAN:
        {
                struct video_channel v;
		if(copy_from_user(&v, arg,sizeof(v)))
			return -EFAULT;
		
                
		if (v.channel>0)
			return -EINVAL;
		if (v.norm > 1)
			return -EOPNOTSUPP;

                dvb->vidmode=v.norm;
                SetMode(dvb, v.norm);
                dvb->saa->mode=v.norm;
		return 0;
        }
                
		/* Get tuner abilities */
        case VIDIOCGTUNER:
        {
                struct video_tuner v;
                
                dprintk(KERN_ERR "dvb: VIDIOCGTUNER called\n");
                
                if(copy_from_user(&v,arg,sizeof(v)))
                        return -EFAULT;
                
                /* only channel 0 has a tuner */	
                if(v.tuner != 0 )
                        return -EINVAL;
                
                /* fill the structure */
                strcpy(v.name, "DVB");
                v.rangelow  = 0x00000000;
                v.rangehigh = 0xffffffff;
                
                v.flags= VIDEO_TUNER_PAL | VIDEO_TUNER_NTSC;
                v.mode = dvb->vidmode;
		
                /* fixme: fill in signal strength here */
                v.signal = 0xffff;
                
                if(copy_to_user(arg,&v,sizeof(v)))
                        return -EFAULT;
                
                return 0;
        }
		/* Tune the tuner for the current channel */
        case VIDIOCSTUNER:
        {
                struct video_tuner v;
                
                dprintk(KERN_ERR "dvb: VIDIOCSTUNER called\n");
                
                if(copy_from_user(&v, arg, sizeof(v)))
                        return -EFAULT;
                
                /* only channel 0 has a tuner */
                if( v.tuner != 0)
                        return -EINVAL;
                
                /* check if format is supported */
                if(    v.mode != VIDEO_MODE_PAL 
                       && v.mode != VIDEO_MODE_NTSC
                    /* && v.mode != VIDEO_MODE_SECAM */ )
                       return -EOPNOTSUPP;
                
                dvb->vidmode=v.mode;
                SetMode(dvb, v.mode);
                
                return 0;
        }
		/* Get picture properties */
        case VIDIOCGPICT:
        {
                struct video_picture p;
                
                saacomm(SAA7146_V4L_GPICT, &p);
                
                if(copy_to_user(arg, &p, sizeof(struct video_picture)))
                        return -EFAULT;
                
                dprintk(KERN_ERR "dvb: VIDIOCGPICT called: b:%d c:%d s:%d d:%d p:%d\n",p.brightness, p.contrast, p.colour, p.depth, p.palette);
                
                return 0;
        }
		/* Set picture properties */
        case VIDIOCSPICT:
        {
                struct video_picture p;
                
                if(copy_from_user(&p, arg,sizeof(struct video_picture)))
                        return -EFAULT;
                
                dprintk(KERN_ERR "dvb: VIDIOCGPICT called: b:%d c:%d s:%d d:%d p:%d\n",p.brightness, p.contrast, p.colour, p.depth, p.palette);
                
		switch( p.palette ) {

		case VIDEO_PALETTE_RGB555:
		case VIDEO_PALETTE_RGB565:
		case VIDEO_PALETTE_RGB24:
		case VIDEO_PALETTE_RGB32:
		case VIDEO_PALETTE_UYVY:
		case VIDEO_PALETTE_YUV422P:
		case VIDEO_PALETTE_YUV420P:
		case VIDEO_PALETTE_YUV411P:
			break;
		default:
			return -EINVAL;
		}

                saacomm(SAA7146_V4L_SPICT, &p);
                
                return 0;
                
        }
		/* Set the video overlay window */
        case VIDIOCSWIN:
        {
                struct video_window w;
                
                if(copy_from_user(&w,arg,sizeof(struct video_window)))
                        return -EFAULT;
                
                dprintk(KERN_ERR "dvb: VIDIOCSWIN called: clips: %d, x:%d, y:%d, h:%d, w:%d\n", w.clipcount,w.x, w.y, w.height,w.width);
                
                saacomm(SAA7146_V4L_SWIN, &w);
                
                return 0;
        }
		/* Set the video overlay window - passes clip list for hardware smarts , chromakey etc */
        case VIDIOCGWIN:
        {
                dprintk(KERN_ERR "dvb: VIDIOCGWIN called\n");
                return 0;
        }
		/* Start, end capture */
        case VIDIOCCAPTURE:
        {
                int v;
                
                if(copy_from_user(&v, arg,sizeof(v)))
                        return -EFAULT;
                
                dprintk(KERN_ERR "dvb: VIDIOCCAPTURE called, mode:%d (0=disable)\n",v);
                
                saacomm(SAA7146_V4L_CCAPTURE, &v);
                
                return 0;
        }
		/* Get frame buffer */
        case VIDIOCGFBUF:
        {
                struct video_buffer b;
                
                dprintk(KERN_ERR "dvb: VIDIOCGFBUF called\n");
                
                saacomm(SAA7146_V4L_GFBUF, &b);
                
                if(copy_to_user(arg, &b,sizeof(struct video_buffer)))
                        return -EFAULT;
                
                return 0;
                
        }
		/* Set frame buffer - root only */
        case VIDIOCSFBUF:
        {
                struct video_buffer b;
                u32 vid = (vidmem << 16) | vidlow;
                
                if(copy_from_user(&b, arg,sizeof(struct video_buffer)))
                        return -EFAULT;
                
                /* see if vidmem-override is requested */
                if( 0 != vidmem ) {
                        printk(KERN_ERR "dvb: video-memory-override. (0x%08x)\n",vid);
                        b.base = (void*)vid;
                }
                
                saacomm(SAA7146_V4L_SFBUF, &b);
                
                dprintk(KERN_ERR "dvb: VIDIOCSFBUF called\n");
                
                return 0;		
        }
		/* Video key event - to dev 255 is to all - cuts capture on all DMA windows with this key (0xFFFFFFFF == all) */
        case VIDIOCKEY:
        {
                dprintk(KERN_ERR "dvb: VIDIOCKEY called\n");
                return 0;
        }
		/* Get audio info */	
        case VIDIOCGAUDIO:
        {
                struct video_audio v;
                
                v.flags = VIDEO_AUDIO_MUTABLE;
                /* let's auto-detect */
                return 0;
        }
		/* Audio source, mute etc */
        case VIDIOCSAUDIO:
        {
          //struct video_audio v;
                
                
                return 0;
        }
                /* Sync with mmap grabbing */
        case VIDIOCSYNC:
        {
                int i = 0, frame = 0;
                
                if(copy_from_user((void *)&frame,arg,sizeof(int)))
                        return -EFAULT;
                
                dprintk(KERN_ERR "dvb: VIDIOCSYNC called - frame: %d\n",frame);
                
                /* simply pass the requested frame-number to the corresponding
                   saa7146-function ... */
                i = saacomm(SAA7146_V4L_CSYNC, &frame);
                
                return i;
        }
		/* Grab frames */
        case VIDIOCMCAPTURE:
        {
                struct video_mmap vm;
                int i = 0;
                
                if(copy_from_user((void *)&vm, (void *) arg, sizeof(vm)))
                        return -EFAULT;
                
                dprintk(KERN_ERR "dvb: VIDIOCMCAPTURE called: fr:%d,"
                        "fmt:%d, w:%d, h:%d\n",
                        vm.frame, vm.format,vm.width, vm.height);
                switch( vm.format ) {
		case VIDEO_PALETTE_YUV422P:
		case VIDEO_PALETTE_YUV420P:
		case VIDEO_PALETTE_YUV411P:
			return -EINVAL;
		}

                /* simply pass the structure for the requested frame-number
                   to the corresponding saa7146-function ... */
                i = saacomm(SAA7146_V4L_CMCAPTURE, &vm);
                
                return i;
        }
                
		/* Memory map buffer info */
        case VIDIOCGMBUF:
        {
                struct video_mbuf mbuf;
                
                dprintk(KERN_ERR "dvb: VIDIOCGMBUF called\n");
                saacomm(SAA7146_V4L_GMBUF, &mbuf);
                
                if (copy_to_user((void *)arg, &mbuf, sizeof(struct video_mbuf)))
                        return -EFAULT;
                
                return 0;
                
        }
		/* Get attached units */
        case VIDIOCGUNIT:
        {
                struct video_unit vu;
                dprintk(KERN_ERR "dvb: VIDIOCGUNIT called\n");
                
                if(copy_to_user((void *)arg, (void *)&vu, sizeof(vu)))
                        return -EFAULT;
                return 0;
        }
		
        case VIDIOCGCAPTURE:
        {
                dprintk(KERN_ERR "dvb: VIDIOCGCAPTURE called\n");
                return 0;
        }
        case VIDIOCSCAPTURE:
        {
                dprintk(KERN_ERR "dvb: VIDIOCSCAPTURE called\n");
                return 0;
        }

        default:
                return -ENOIOCTLCMD;
	}
        return 0;
}

static int dvb_init_done(struct video_device *dev)
{
	fprintk(KERN_ERR "dvb: dvb_init_done called\n");
	return 0;
}

static int dvb_mmap(struct video_device *dev, const char *adr, unsigned long size)
{
	struct dvb_struct* dvb = (struct dvb_struct*)dev;
	struct saa7146_mmap_struct m;
	
	fprintk(KERN_ERR "dvb: dvb_mmap called, adr:%p, size:0x%08x\n",adr,(u32)size);
        
	m.adr = adr;
	m.size = size;
        
	if ( 0 != saacomm(SAA7146_DO_MMAP, &m) ) {
		printk(KERN_ERR "dvb: dvb_mmap failed!\n");
		return -1;
	}
        
	return 0;
}

static int dvb_open(struct video_device *dev, int flags)
{
	MOD_INC_USE_COUNT;
	return 0;   
}

static void dvb_close(struct video_device *dev)
{
	fprintk(KERN_ERR "dvb: dvb_close called\n");
        MOD_DEC_USE_COUNT;  
}

static unsigned int dvb_poll(struct dvb_struct *dvb, struct file *file,
	poll_table *wait)
{
        unsigned int mask=0;

        if (dvb->playing) {
                if (FREE_COND)
                        return (POLLOUT | POLLWRNORM);

                poll_wait(file, &dvb->avout.queue, wait);
          
                if (FREE_COND)
                        mask |= (POLLOUT | POLLWRNORM);
        } else /* if not playing: may play if asked for */
                mask = (POLLOUT | POLLWRNORM);
                
        return mask;
}

static unsigned int dvb_apoll(struct dvb_struct *dvb, struct file *file,
	poll_table *wait)
{
        unsigned int mask=0;

        if (dvb->playing) {
                if (ring_buffer_free(&dvb->aout)>20*1024)
                        return (POLLOUT | POLLWRNORM);

                poll_wait(file, &dvb->aout.queue, wait);
          
                if (ring_buffer_free(&dvb->aout)>20*1024)
                        mask |= (POLLOUT | POLLWRNORM);
        } else /* if not playing: may play if asked for */
                mask = (POLLOUT | POLLWRNORM);
                
        return mask;
}

/* template for video_device-structure */
static struct video_device dvb_template = {
name:	        "DVB Board",
type:	        VID_TYPE_TUNER		|
                VID_TYPE_CAPTURE	|
                VID_TYPE_OVERLAY	|
                VID_TYPE_CLIPPING	|
                VID_TYPE_FRAMERAM	|
                VID_TYPE_SCALES,
hardware:       VID_HARDWARE_SAA7146,
open:           dvb_open,
close:          dvb_close,
read:           NULL,
write:	        NULL,		
poll:	        NULL,
ioctl: 	        dvb_ioctl,
mmap:	        dvb_mmap,
initialize:	dvb_init_done,
priv:   	NULL,
busy:           0,
minor:          0
};

static int vid_register(struct dvb_struct *dvb)
{
        /* Copy structure */
        memcpy( &dvb->video, &dvb_template, sizeof(struct video_device));

#if LINUX_VERSION_CODE < 0x020405
        if( 0 > (video_register_device(&dvb->video, VFL_TYPE_GRABBER))) {
#else
        if( 0 > (video_register_device(&dvb->video, VFL_TYPE_GRABBER, -1))) {
#endif
                printk(KERN_ERR "dvb: can't register videodevice\n");
                return -1;
        }
        return 0;
}

static inline int vid_unregister(struct dvb_struct *dvb)
{
        if(dvb->video.minor!=-1)
                video_unregister_device(&dvb->video);
        return 0;
}

/****************************************************************************
 * END OF V4L SECTION
 ****************************************************************************/


/****************************************************************************
 * DVB API SECTION
 ****************************************************************************/


/******************************************************************************
 * hardware filter functions
 ******************************************************************************/

static int 
StartHWFilter(dvb_demux_filter_t *dvbdmxfilter)
{
        dvb_demux_feed_t *dvbdmxfeed=dvbdmxfilter->feed;
        struct dvb_struct *dvb=
                (struct dvb_struct *) dvbdmxfeed->demux->priv;
        u16 buf[20];
        int ret, i;
        u16 handle;
        u16 mode=0x0320;
        
        if (dvbdmxfilter->type==DMX_TYPE_SEC) {
                buf[4]=(dvbdmxfilter->filter.filter_value[0]<<8)|
                        dvbdmxfilter->filter.filter_mask[0];
                for (i=3; i<18; i++)
                        buf[i+4-2]=(dvbdmxfilter->filter.filter_value[i]<<8)|
                                    dvbdmxfilter->filter.filter_mask[i];
                mode=4;
        } else
        if ((dvbdmxfeed->ts_type & TS_PACKET) &&
            !(dvbdmxfeed->ts_type & TS_PAYLOAD_ONLY)) 
                init_p2t(&dvb->p2t_filter[dvbdmxfilter->index], dvbdmxfeed);

        buf[0] = (COMTYPE_PID_FILTER << 8) + AddPIDFilter; 
        buf[1] = 16;	
        buf[2] = dvbdmxfeed->pid;
        buf[3] = mode;

        ret=CommandRequest(dvb, buf, 20, &handle, 1);          
        if (ret<0)
                return ret;

        dvb->handle2filter[handle]=dvbdmxfilter;
        dvbdmxfilter->hw_handle=handle;

        return ret;
}

static int 
StopHWFilter(dvb_demux_filter_t *dvbdmxfilter)
{
        struct dvb_struct *dvb=
          (struct dvb_struct *) dvbdmxfilter->feed->demux->priv;
        u16 buf[3];
        u16 answ[2];
        int ret;
        u16 handle;
                
        handle=dvbdmxfilter->hw_handle;
        if (handle>32) {
                dprintk("dvb: StopHWFilter tried to stop invalid filter %d.\n",
                       handle);
                dprintk("dvb: filter type = %d\n",  dvbdmxfilter->type);
                return 0;
        }

        dvb->handle2filter[handle]=NULL;

        buf[0] = (COMTYPE_PID_FILTER << 8) + DelPIDFilter; 
        buf[1] = 1;	
        buf[2] = handle;
        ret=CommandRequest(dvb, buf, 3, answ, 2);          

        if (answ[1] != handle) {
                dprintk("dvb: filter %d shutdown error :%d\n", handle, answ[1]);
                ret=-1;
        }
        return ret;
}


static int 
dvb_write_to_decoder(dvb_demux_feed_t *dvbdmxfeed, u8 *buf, size_t count)
{
        dvb_demux_t *dvbdmx=dvbdmxfeed->demux;
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdmx->priv;
        ipack *ipack=&dvb->ipack[dvbdmxfeed->pes_type];

        switch (dvbdmxfeed->pes_type) {
        case 0:
                if (dvb->audiostate.streamSource==AUDIO_SOURCE_MEMORY) {
                        return -EINVAL;
                }
                break;
        case 1:
                if (dvb->videostate.streamSource==VIDEO_SOURCE_MEMORY) {
                        return -EINVAL;
                }
                break;
        default:
                return -1;
        }

        if (!(buf[3]&0x10)) { // no payload?
                return -1;
        }
        if (buf[1]&0x40)
                send_ipack_rest(ipack);

        if (buf[3]&0x20) {  // adaptation field?
                count-=buf[4]+1;
                buf+=buf[4]+1;
                if (!count) {
                        return 0;
                }
        }
        
        instant_repack(buf+4, count-4, &dvb->ipack[dvbdmxfeed->pes_type]);
        return 0;
}

static void
dvb_feed_start_pid(dvb_demux_feed_t *dvbdmxfeed)
{
        dvb_demux_t *dvbdmx=dvbdmxfeed->demux;
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdmx->priv;
        u16 *pid=dvbdmx->pids, npids[5];
        int i;
        
        npids[0]=npids[1]=npids[2]=npids[3]=0xffff;
        npids[4]=0xffff;
        i=dvbdmxfeed->pes_type;
        npids[i]=(pid[i]&0x8000) ? 0 : pid[i];
        if ((i==2) && npids[i] && (dvbdmxfeed->ts_type & TS_PACKET)) {
                npids[i]=0;
                ChangePIDs(dvb, npids[1], npids[0], npids[2], npids[3], npids[4]);
                StartHWFilter(dvbdmxfeed->filter); 
                return;
        }
        if (dvbdmxfeed->pes_type<=2)
                ChangePIDs(dvb, npids[1], npids[0], npids[2], npids[3], npids[4]);
        if (dvbdmxfeed->pes_type<2 && npids[0])
                if (dvb->frontend.tuning==FE_STATE_AFC)
                        outcom(dvb, COMTYPE_PIDFILTER, Scan, 0);
        if ((dvbdmxfeed->ts_type & TS_PACKET)) { 
                if (dvbdmxfeed->pes_type == 0 && 
                    !(dvbdmx->pids[0]&0x8000))
                        AV_StartRecord(dvb, RP_AUDIO, 
                                       dvbdmxfeed);
                if (dvbdmxfeed->pes_type == 1 && 
                    !(dvbdmx->pids[1]&0x8000))
                        AV_StartRecord(dvb, RP_VIDEO, 
                                       dvbdmxfeed);
        }
}

static void
dvb_feed_stop_pid(dvb_demux_feed_t *dvbdmxfeed)
{
        dvb_demux_t *dvbdmx=dvbdmxfeed->demux;
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdmx->priv;
        u16 *pid=dvbdmx->pids, npids[5];
        int i;
        
        if (dvbdmxfeed->pes_type<=1) {
                AV_Stop(dvb, dvbdmxfeed->pes_type ? 
                        RP_VIDEO : RP_AUDIO);
                if (!dvb->rec_mode)
                        dvbdmx->recording=0;
                if (!dvb->playing)
                        dvbdmx->playing=0;
        }
        npids[0]=npids[1]=npids[2]=npids[3]=0xffff;
        npids[4]=0xffff;
        i=dvbdmxfeed->pes_type;
        switch (i) {
        case 2: //teletext
                if (dvbdmxfeed->ts_type & TS_PACKET)
                        StopHWFilter(dvbdmxfeed->filter); 
                npids[2]=0;
                break;
        case 0:
        case 1:
        case 4:
                if (!pids_off) 
                        return;
                npids[i]=(pid[i]&0x8000) ? 0 : pid[i];
                break;
        }
        ChangePIDs(dvb, npids[1], npids[0], npids[2], npids[3], npids[4]);
}

static int 
dvb_start_feed(dvb_demux_feed_t *dvbdmxfeed)
{
        dvb_demux_t *dvbdmx=dvbdmxfeed->demux;
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdmx->priv;

        if (!dvbdmx->dmx.frontend)
                return -EINVAL;

        if (dvb->card_type==DVB_CARD_TT_BUDGET) 
                return TTBStart(dvb); 

        if (dvbdmxfeed->pid>0x1fff) 
                return -1;

        if (dvbdmxfeed->type == DMX_TYPE_TS) {
	        if ((dvbdmxfeed->ts_type & TS_DECODER) 
		    && (dvbdmxfeed->pes_type<DMX_TS_PES_OTHER)) {

		        switch (dvbdmx->dmx.frontend->source) {
			case DMX_MEMORY_FE: 
			        if (dvbdmxfeed->ts_type & TS_DECODER)
				       if (dvbdmxfeed->pes_type<2 && 
                                           !(dvbdmx->pids[0]&0x8000) &&
					   !(dvbdmx->pids[1]&0x8000)) {
                                               ring_buffer_flush(&dvb->avout);
                                               ring_buffer_flush(&dvb->aout);
                                               AV_StartPlay(dvb,RP_AV);
                                               dvbdmx->playing=1;
					}
				break;
			default:
                                dvb_feed_start_pid(dvbdmxfeed);
				break;
			}
		} else
		        if ((dvbdmxfeed->ts_type & TS_PACKET) &&
                            (dvbdmx->dmx.frontend->source!=DMX_MEMORY_FE))
                                StartHWFilter(dvbdmxfeed->filter); 
        }
        
        if (dvbdmxfeed->type == DMX_TYPE_SEC) {
                int i;

	        for (i=0; i<dvbdmx->filternum; i++) {
		        if (dvbdmx->filter[i].state!=DMX_STATE_READY)
			        continue;
			if (dvbdmx->filter[i].type!=DMX_TYPE_SEC)
			        continue;
			if (dvbdmx->filter[i].filter.parent!=&dvbdmxfeed->feed.sec)
			        continue;
			dvbdmx->filter[i].state=DMX_STATE_GO;
                        if (dvbdmx->dmx.frontend->source!=DMX_MEMORY_FE)
                                StartHWFilter(&dvbdmx->filter[i]); 
                }
	}
        return 0;
}

static void 
restart_feeds(struct dvb_struct *dvb)
{
        dvb_demux_t *dvbdmx=&dvb->demux;
        dvb_demux_feed_t *feed;
        int mode;
        int i;

        mode=dvb->playing;
        dvb->playing=0;
        dvb->rec_mode=0;

        for (i=0; i<dvbdmx->filternum; i++) {
                feed=&dvbdmx->feed[i];
                if (feed->state==DMX_STATE_GO)
                        dvb_start_feed(feed);
        }

        if (mode)
                AV_StartPlay(dvb, mode);
        
}

static int 
dvb_stop_feed(dvb_demux_feed_t *dvbdmxfeed)
{
        dvb_demux_t *dvbdmx=dvbdmxfeed->demux;
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdmx->priv;

        if (dvb->card_type==DVB_CARD_TT_BUDGET) 
                return TTBStop(dvb); 

        if (dvbdmxfeed->type == DMX_TYPE_TS) {
                if (dvbdmxfeed->ts_type & TS_DECODER) {
                        if (dvbdmxfeed->pes_type>=DMX_TS_PES_OTHER ||
                            !dvbdmx->pesfilter[dvbdmxfeed->pes_type]) 
                                return -EINVAL;
                        dvbdmx->pids[dvbdmxfeed->pes_type]|=0x8000;
                        dvbdmx->pesfilter[dvbdmxfeed->pes_type]=0;
                }
                if (dvbdmxfeed->ts_type & TS_DECODER &&
		    (dvbdmxfeed->pes_type<DMX_TS_PES_OTHER)) {
                        dvb_feed_stop_pid(dvbdmxfeed);
                } else 
                        if ((dvbdmxfeed->ts_type & TS_PACKET) &&
                            (dvbdmx->dmx.frontend->source!=DMX_MEMORY_FE))
                                StopHWFilter(dvbdmxfeed->filter); 
        }
        
        if (dvbdmxfeed->type == DMX_TYPE_SEC) {
                int i;

	        for (i=0; i<dvbdmx->filternum; i++)
		        if (dvbdmx->filter[i].state==DMX_STATE_GO && 
			    dvbdmx->filter[i].filter.parent==&dvbdmxfeed->feed.sec) {
			        dvbdmx->filter[i].state=DMX_STATE_READY;
                                if (dvbdmx->dmx.frontend->source!=DMX_MEMORY_FE)
                                        StopHWFilter(&dvbdmx->filter[i]); 
                }
	}
        return 0;
}


/******************************************************************************
 * SEC device file operations
 ******************************************************************************/

static int 
secSetTone(struct dvb_struct *dvb, secToneMode mode)
{
        int val;

        switch (mode) {
        case SEC_TONE_ON:
                val=1;
                break;
        case SEC_TONE_OFF:
                val=0;
                break;
        default:
          return -EINVAL;
        }
        dvb->sec.ttk=val;
        Set22K(dvb, val);
        return 0;
}

static int 
secSetVoltage(struct dvb_struct *dvb, secVoltage voltage)
{
        int power=1, volt=0;
                       
        switch (voltage) {
        case SEC_VOLTAGE_LT: //FIXME: Test which one is true
        case SEC_VOLTAGE_OFF:
                power=0;
                break;
        case SEC_VOLTAGE_13:
                volt=0;
                break;
        case SEC_VOLTAGE_18:
                volt=1;
                break;
        case SEC_VOLTAGE_13_5:
        case SEC_VOLTAGE_18_5:
        default:
                return -EINVAL;
        }
        dvb->sec.volt=volt;
        // this guarantees that running SEC commands are done atomically
        Set22K(dvb, dvb->sec.ttk);
        SetVoltage(dvb, power, volt);
        return 0;
}

static int 
secSendSequence(struct dvb_struct *dvb, struct secCmdSequence *seq)
{
        int i, ret, burst, len;
        struct secCommand scommands;
        u8 msg[16];

        switch (seq->miniCommand) {
        case SEC_MINI_NONE:
                burst=-1;
                break;
        case SEC_MINI_A:
                burst=0;
                break;
        case SEC_MINI_B:
                burst=1;
                break;
        default:
                return -EINVAL;
        }

        for (i=0; i<seq->numCommands; i++) {
                if (copy_from_user(&scommands, &seq->commands[i], 
                                   sizeof(struct secCommand)))
                        continue;
                switch (scommands.type) {
                case SEC_CMDTYPE_DISEQC:
                        len=scommands.u.diseqc.numParams;
                        if (len>SEC_MAX_DISEQC_PARAMS)
                                return -EINVAL;

                        msg[0]=0xe0;
                        msg[1]=scommands.u.diseqc.addr;
                        msg[2]=scommands.u.diseqc.cmd;
                        memcpy(msg+3, &scommands.u.diseqc.params, len);
                        SendDiSEqCMsg(dvb, len+3, msg, -1);
                        /* repeat for the case of cascaded (and/or hearing 
                           impared) switches */
                        msg[0]=0xe1;
                        SendDiSEqCMsg(dvb, len+3, msg, -1);
                        break;
                case SEC_CMDTYPE_PAUSE:
                        //FIXME: fix firmware to allow programmed pause length
                        SendDiSEqCMsg(dvb, 0, msg, -1);
                        break;
                case SEC_CMDTYPE_VSEC:
                default:
                        return -EINVAL;
                }
        }
        
        if (burst!=-1)
                SendDiSEqCMsg(dvb, 0, msg, burst);
        
        ret=secSetVoltage(dvb, seq->voltage);
        if (ret<0)
                return ret;
        return secSetTone(dvb, seq->continuousTone);
}


/******************************************************************************
 * CI link layer file ops (FIXME: move this to separate module later)
 ******************************************************************************/

int ci_ll_init(ring_buffer_t *cirbuf, ring_buffer_t *ciwbuf, int size)
{
        ring_buffer_init(cirbuf, vmalloc(size), size);
        ring_buffer_init(ciwbuf, vmalloc(size), size);
        return 0;
}

void ci_ll_release(ring_buffer_t *cirbuf, ring_buffer_t *ciwbuf)
{
        vfree(cirbuf->data);
        cirbuf->data=0;
        vfree(ciwbuf->data);
        ciwbuf->data=0;
}


int ci_ll_reset(ring_buffer_t *cibuf, struct file *file, int slots, ca_slot_info_t *slot)
{
	int i;

        if (ring_buffer_free(cibuf)<8)
                return -EBUSY;
	for (i=0; i<2; i++) 
		if (slots&(1<<i)) {
                        cibuf->data[cibuf->pwrite]=0x00;   
                        cibuf->data[(cibuf->pwrite+1)%cibuf->size]=0x06;
                        cibuf->data[(cibuf->pwrite+2)%cibuf->size]=i;
                        cibuf->data[(cibuf->pwrite+3)%cibuf->size]=0x00;
                        cibuf->data[(cibuf->pwrite+4)%cibuf->size]=0xff;
                        cibuf->data[(cibuf->pwrite+5)%cibuf->size]=0x02;
                        cibuf->data[(cibuf->pwrite+6)%cibuf->size]=0x00;
                        cibuf->data[(cibuf->pwrite+7)%cibuf->size]=0x00;
                        cibuf->pwrite=(cibuf->pwrite+8)%cibuf->size;
                        slot[i].flags=0;
                }
        return 0;
}

static ssize_t 
ci_ll_write(ring_buffer_t *cibuf, struct file *file, const char *buf, size_t count, loff_t *ppos)
{
	int free, split;
        int32_t pread;
        int non_blocking=file->f_flags&O_NONBLOCK;
	
        if (count>2048)
                return -EINVAL;
	pread=cibuf->pread; 
	free=pread-cibuf->pwrite;
	if (free<=0) 
                free+=cibuf->size;
	if (count+2>=free) { 
                if (non_blocking)
                        return -EWOULDBLOCK;
                if (wait_event_interruptible(cibuf->queue,
                                             (ring_buffer_free(cibuf)>count+2)))
                        return 0;
        }
        cibuf->data[cibuf->pwrite]=(count>>8);   
	cibuf->data[(cibuf->pwrite+1)%cibuf->size]=(count&0xff); 
        cibuf->pwrite=(cibuf->pwrite+2)%cibuf->size;

	if (pread>cibuf->pwrite)
		split=0;
	else
		split=cibuf->size-cibuf->pwrite;
        if (split && split<count) {
                if (copy_from_user(cibuf->data + cibuf->pwrite, buf, split))
                        return -EFAULT;
		if (copy_from_user(cibuf->data, buf+split, count-split))
                        return -EFAULT;
        } else 
		if (copy_from_user(cibuf->data + cibuf->pwrite, buf, count))
                        return -EFAULT;
        cibuf->pwrite=(cibuf->pwrite+count)%cibuf->size;
	return count;
}

static ssize_t 
ci_ll_read(ring_buffer_t *cibuf, struct file *file, char *buf, size_t count, loff_t *ppos)
{
        int split=0, avail, pwrite;
        int non_blocking=file->f_flags&O_NONBLOCK;
	
	if (!cibuf->data || !count)
	        return 0;
	if (non_blocking && (ring_buffer_empty(cibuf)))
	        return -EWOULDBLOCK;
	if (wait_event_interruptible(cibuf->queue,
				     !ring_buffer_empty(cibuf)))
		return 0;
        pwrite=cibuf->pwrite;
	avail=pwrite - cibuf->pread;
	if (avail<0) 
                avail+=cibuf->size;
	if (avail<4) 
                return 0;
	count=(cibuf->data[cibuf->pread]<<8);  
	count|=cibuf->data[(cibuf->pread+1)%cibuf->size];
	if (avail<count+2) 
                return -EINVAL;
        cibuf->pread=(cibuf->pread+2)%cibuf->size;	

        if (pwrite<cibuf->pread)
                split=cibuf->size-cibuf->pread;
	if (split && split<count) {
		if (copy_to_user(buf, cibuf->data+cibuf->pread, split))
			return -EFAULT;
		if (copy_to_user(buf+split, cibuf->data, count-split))
			return -EFAULT;
	} else 
		if (copy_to_user(buf, cibuf->data+cibuf->pread, count))
			return -EFAULT;
	cibuf->pread=(cibuf->pread + count)%cibuf->size;
	return count;
}

static unsigned int 
ci_ll_poll(ring_buffer_t *rbuf, ring_buffer_t *wbuf, 
           struct file *file, poll_table *wait)
{
        unsigned int mask=0;

        if (!ring_buffer_empty(rbuf))
                mask|=POLLIN;
        if (ring_buffer_avail(wbuf)>1024)
                mask|=POLLOUT;
        if (mask)
                return mask;
        
        poll_wait(file, &rbuf->queue, wait);
        
        if (!ring_buffer_empty(rbuf))
                mask|=POLLIN;
        if (ring_buffer_avail(wbuf)>1024)
                mask|=POLLOUT;
        
        return mask;
}

static int 
ca_ioctl(struct dvb_device *dvbdev, struct file *file, 
          unsigned int cmd, unsigned long arg)
{
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdev->priv;
        void *parg=(void *)arg;

        switch (cmd) {
        case CA_RESET:
                
#ifdef NEW_CI
                
                return ci_ll_reset(&dvb->ci_wbuffer, file, arg, &dvb->ci_slot[0]);
#endif                        
                break;
                
        case CA_GET_CAP:
        {
                ca_cap_t cap;
                
                cap.slot_num=2;
#ifdef NEW_CI
                cap.slot_type=CA_CI_LINK;
#else
                cap.slot_type=CA_CI;
#endif
                cap.descr_num=16;
                cap.descr_type=CA_ECD;
                if(copy_to_user(parg, &cap, sizeof(cap)))
                        return -EFAULT;
        }
        break;
        case CA_GET_SLOT_INFO:
        {
                ca_slot_info_t info;
                
                if (copy_from_user(&info, parg, sizeof(info)))
                        return -EFAULT;
                if (info.num>1)
                        return -EINVAL;
                if (copy_to_user(parg, &dvb->ci_slot[info.num],
                                 sizeof(info)))
                        return -EFAULT;
        }
        break;
        case CA_GET_MSG:
                
                break;
        case CA_SEND_MSG:
                
                break;
                
        case CA_SET_DESCR:
        {
                ca_descr_t descr;
                
                if(copy_from_user(&descr, parg, sizeof(descr)))
                        return -EFAULT;
                if(descr.index>=16)
                        return -EINVAL;
                if(descr.parity>1)
                        return -EINVAL;
                outcom(dvb, COMTYPE_PIDFILTER, SetDescr, 5, 
                       (descr.index<<8)|descr.parity,
                       (descr.cw[0]<<8)|descr.cw[1],
                       (descr.cw[2]<<8)|descr.cw[3],
                       (descr.cw[4]<<8)|descr.cw[5],
                       (descr.cw[6]<<8)|descr.cw[7]);
        }
        break;
        default:
                return -EINVAL;
        }
        return 0;
}


/******************************************************************************
 * DVB device file operations
 ******************************************************************************/

#define INFU 32768

static dvb_devs_t dvbs_devs = {
        9,
        { 
                DVB_DEVICE_VIDEO_0, DVB_DEVICE_AUDIO_0, 
                DVB_DEVICE_SEC_0,   DVB_DEVICE_FRONTEND_0, 
                DVB_DEVICE_DEMUX_0, DVB_DEVICE_DVR_0, 
                DVB_DEVICE_CA_0,    DVB_DEVICE_NET_0,
                DVB_DEVICE_OSD_0
        },
        { INFU, INFU, INFU, INFU, INFU, 1, 1, 1, INFU },
        { 1, 1, 1, 1, INFU, 1, 1, 1, INFU }
};

static dvb_devs_t dvbc_devs = {
        9,
        { 
                DVB_DEVICE_VIDEO_0, DVB_DEVICE_AUDIO_0, 
                -1,                 DVB_DEVICE_FRONTEND_0, 
                DVB_DEVICE_DEMUX_0, DVB_DEVICE_DVR_0, 
                DVB_DEVICE_CA_0,    DVB_DEVICE_NET_0,
                DVB_DEVICE_OSD_0
        },
        { INFU, INFU, 0, INFU, INFU, 1, 1, 1, INFU },
        { 1, 1, 0, 1, INFU, 1, 1, 1, INFU }
};

static dvb_devs_t ttbs_devs = {
        9,
        { 
                -1, -1, 
                DVB_DEVICE_SEC_0,   DVB_DEVICE_FRONTEND_0, 
                DVB_DEVICE_DEMUX_0, DVB_DEVICE_DVR_0, 
                -1,                 DVB_DEVICE_NET_0,
		-1
        },
        { 0, 0, INFU, INFU, INFU, 1, 0, 1, INFU },
        { 0, 0, 1, 1, INFU, 1, 0, 1, INFU }
};

static dvb_devs_t ttbc_devs = {
        9,
        {
                -1, -1,
                -1 ,                DVB_DEVICE_FRONTEND_0,
                DVB_DEVICE_DEMUX_0, DVB_DEVICE_DVR_0,
                -1,                 DVB_DEVICE_NET_0,
		-1
        },
        { 0, 0, 0, INFU, INFU, 1, 0, 1, INFU },
        { 0, 0, 0, 1, INFU, 1, 0, 1, INFU }
};


static inline int
num2type(struct dvb_struct *dvb, int num)
{
        if (!dvb->dvb_devs)
                return -2;
        if (num>=dvb->dvb_devs->num)
                return -3;
        return dvb->dvb_devs->tab[num];
}


static int 
dvbdev_open(struct dvb_device *dvbdev, int num, 
            struct inode *inode, struct file *file)
{
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdev->priv;
        int type=num2type(dvb, num);
        int ret=0;

        if (type<0)
                return -EINVAL;

        if (dvb->users[num] >= dvb->dvb_devs->max_users[num])
                return -EBUSY;

	if ((file->f_flags&O_ACCMODE)!=O_RDONLY) 
                if (dvb->writers[num] >= dvb->dvb_devs->max_writers[num])
		        return -EBUSY;

	switch (type) {
	case DVB_DEVICE_VIDEO_0:
                ring_buffer_flush(&dvb->aout);
                ring_buffer_flush(&dvb->avout);
                dvb->video_blank=true;
                dvb->audiostate.AVSyncState=true;
                dvb->videostate.streamSource=VIDEO_SOURCE_DEMUX;
                break;
	case DVB_DEVICE_AUDIO_0:
                ring_buffer_flush(&dvb->aout);
                dvb->audiostate.streamSource=AUDIO_SOURCE_DEMUX;
                break;
	case DVB_DEVICE_SEC_0:
                if (file->f_flags&O_NONBLOCK) 
                        ret=-EWOULDBLOCK;
                break;
	case DVB_DEVICE_FRONTEND_0:
                dprintk ("try to open DEVICE_FRONTEND_0\n");
                break;
	case DVB_DEVICE_DEMUX_0:
                if ((file->f_flags&O_ACCMODE)!=O_RDWR)
                        return -EINVAL;
                ret=DmxDevFilterAlloc(&dvb->dmxdev, file);
                break;
	case DVB_DEVICE_DVR_0:
                ret=DmxDevDVROpen(&dvb->dmxdev, file);
                break;
	case DVB_DEVICE_CA_0:
                break;
	case DVB_DEVICE_OSD_0:
                break;
	case DVB_DEVICE_NET_0:
                break;
        default:
                return -EINVAL;
	}
	if (ret<0) 
	        return ret;
	if ((file->f_flags&O_ACCMODE)!=O_RDONLY)
		dvb->writers[num]++;
        dvb->users[num]++;
	MOD_INC_USE_COUNT;
        return ret;
}

static int 
dvbdev_close(struct dvb_device *dvbdev, int num, 
             struct inode *inode, struct file *file)
{
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdev->priv;
        int type=num2type(dvb, num);
        int ret=0;

        if (type<0)
                return -EINVAL;

	switch (type) {
	case DVB_DEVICE_VIDEO_0:
	        AV_Stop(dvb, RP_VIDEO);
		break;
	case DVB_DEVICE_AUDIO_0:
	        AV_Stop(dvb, RP_AUDIO);
                break;
	case DVB_DEVICE_SEC_0:
                break;
	case DVB_DEVICE_FRONTEND_0:
                break;
	case DVB_DEVICE_DEMUX_0:
                ret=DmxDevFilterFree(&dvb->dmxdev, file);
                break;
	case DVB_DEVICE_DVR_0:
                ret=DmxDevDVRClose(&dvb->dmxdev, file);
                break;
	case DVB_DEVICE_CA_0:
                break;
	case DVB_DEVICE_OSD_0:
                break;
	case DVB_DEVICE_NET_0:
                break;
        default:
                return -EINVAL;
	}
        /*
	if (ret<0) 
	        return ret;
        */
	if ((file->f_flags&O_ACCMODE)!=O_RDONLY)
		dvb->writers[num]--;
        dvb->users[num]--;
	MOD_DEC_USE_COUNT;
        return ret;
}

static ssize_t 
dvbdev_read(struct dvb_device *dvbdev, int num, 
            struct file *file, char *buf, size_t count, loff_t *ppos)
{
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdev->priv;
        int type=num2type(dvb, num);

	switch (type) {
	case DVB_DEVICE_VIDEO_0:
                return -EINVAL;
	case DVB_DEVICE_AUDIO_0:
                return -EINVAL;
	case DVB_DEVICE_DEMUX_0:
                return DmxDevRead(&dvb->dmxdev, file, buf, count, ppos);
	case DVB_DEVICE_DVR_0:
                return DmxDevDVRRead(&dvb->dmxdev, file, buf, count, ppos);
	case DVB_DEVICE_CA_0:
                return ci_ll_read(&dvb->ci_rbuffer, file, buf, count, ppos);
	default:
	        return -EOPNOTSUPP;
	}
        return 0;
}

static ssize_t 
dvbdev_write(struct dvb_device *dvbdev, int num,
             struct file *file, 
             const char *buf, size_t count, loff_t *ppos)
{
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdev->priv;
        int type=num2type(dvb, num);
        int ret=0;

	switch (type) {
	case DVB_DEVICE_VIDEO_0:
                if (dvb->videostate.streamSource!=VIDEO_SOURCE_MEMORY) {
                        ret=-EPERM;
                        break;
                }
                ret=dvb_play(dvb, buf, count, file->f_flags&O_NONBLOCK, 1, 1);
                break;

	case DVB_DEVICE_AUDIO_0:
                if (dvb->audiostate.streamSource!=AUDIO_SOURCE_MEMORY) {
                        ret=-EPERM;
                        break;
                }
                ret=dvb_aplay(dvb, buf, count, file->f_flags&O_NONBLOCK, 0);
                break;

	case DVB_DEVICE_DVR_0:
                ret=DmxDevDVRWrite(&dvb->dmxdev, file, buf, count, ppos);
                break;

	case DVB_DEVICE_CA_0:
                ret=ci_ll_write(&dvb->ci_wbuffer, file, buf, count, ppos);
                break;

	default:
	        ret=-EOPNOTSUPP;
                break;
	}
        return ret;
}


static int 
tune(struct dvb_struct *dvb, FrontendParameters *para)
{
        if (dvb->frontend.type==DVB_S)
                para->Frequency*=1000;
        SetPIDs(dvb, 0, 0, 0, 0, 0);
        return dvb_frontend_tune(&dvb->frontend, para);
}
       

u8 iframe_header[] = { 0x00, 0x00, 0x01, 0xe0, 0x00, 0x00, 0x80, 0x00, 0x00 };

#define MIN_IFRAME 400000

static void 
play_iframe(struct dvb_struct *dvb, u8 *buf, unsigned int len, int nonblock)
{
        int i, n=1;
       
        if (!(dvb->playing&RP_VIDEO)) {
                AV_StartPlay(dvb, RP_VIDEO);
                n=MIN_IFRAME/len+1;
        }
        dvb_play(dvb, iframe_header, sizeof(iframe_header), 0, 1, 0);
        for (i=0; i<n; i++) 
                dvb_play(dvb, buf, len, 0, 1, 1);
        send_ipack_rest(&dvb->ipack[1]);
}


static int 
video_ioctl(struct dvb_device *dvbdev, struct file *file, 
            unsigned int cmd, unsigned long arg)
{
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdev->priv;
        void *parg=(void *)arg;
        int ret=0;
        
        if (((file->f_flags&O_ACCMODE)==O_RDONLY) &&
            (cmd!=VIDEO_GET_STATUS))
                return -EPERM;
        
        switch (cmd) {
        case VIDEO_STOP:
                dvb->videostate.playState=VIDEO_STOPPED;
                if (dvb->videostate.streamSource==VIDEO_SOURCE_MEMORY)
                        AV_Stop(dvb, RP_VIDEO);
                else
                        vidcom(dvb, 0x000e, 
                               dvb->videostate.videoBlank ? 0 : 1);
                dvb->trickmode=TRICK_NONE;
                break; 
                
        case VIDEO_PLAY:
                dvb->trickmode=TRICK_NONE;
                if (dvb->videostate.playState==VIDEO_FREEZED) {
                        dvb->videostate.playState=VIDEO_PLAYING;
                        vidcom(dvb, 0x000d, 0);
                } 
                
                if (dvb->videostate.streamSource==VIDEO_SOURCE_MEMORY) {
                        if (dvb->playing==RP_AV) {
                                outcom(dvb, COMTYPE_REC_PLAY, __Stop, 0);
                                dvb->playing&=~RP_VIDEO;
                        }
                        AV_StartPlay(dvb,RP_VIDEO);
                        vidcom(dvb, 0x000d, 0);
                } else {
                        //AV_Stop(dvb, RP_VIDEO);
                        vidcom(dvb, 0x000d, 0);
                }
                dvb->videostate.playState=VIDEO_PLAYING;
                break;
                
        case VIDEO_FREEZE:
                dvb->videostate.playState=VIDEO_FREEZED;
                if (dvb->playing&RP_VIDEO) 
                        outcom(dvb, COMTYPE_REC_PLAY, __Pause, 0);
                else
                        vidcom(dvb, 0x0102, 1);
                dvb->trickmode=TRICK_FREEZE;
                break;
                
        case VIDEO_CONTINUE:
                if (dvb->playing&RP_VIDEO) 
                        outcom(dvb, COMTYPE_REC_PLAY, __Continue, 0);
                vidcom(dvb, 0x000d, 0);
                dvb->videostate.playState=VIDEO_PLAYING;
                dvb->trickmode=TRICK_NONE;
                break;
                
        case VIDEO_SELECT_SOURCE:
                dvb->videostate.streamSource=(videoStreamSource_t) arg;
                break;
                
        case VIDEO_SET_BLANK:
                dvb->videostate.videoBlank=(boolean) arg;
                break;
                
        case VIDEO_GET_STATUS:
                if (copy_to_user(parg, &dvb->videostate, 
                                 sizeof(struct videoStatus)))
                        ret=-EFAULT;
                break;
                
        case VIDEO_GET_EVENT:
                //FIXME: write firmware support for this
                ret=-EOPNOTSUPP;
                
        case VIDEO_SET_DISPLAY_FORMAT:
        {
                videoDisplayFormat_t format=(videoDisplayFormat_t) arg;
                u16 val=0;
                
                switch(format) {
                case VIDEO_PAN_SCAN:
                        val=VID_PAN_SCAN_PREF;
                        break;
                        
                case VIDEO_LETTER_BOX:
                        val=VID_VC_AND_PS_PREF;
                        break;
                        
                case VIDEO_CENTER_CUT_OUT:
                        val=VID_CENTRE_CUT_PREF;
                        break;
                        
                default:
                        ret=-EINVAL;
                        break;
                }
                if (ret<0)
                        break;
                dvb->videostate.videoFormat=format;
                ret=outcom(dvb, COMTYPE_ENCODER, SetPanScanType, 
                           1, (u16) val);
                break;
        }
        
        case VIDEO_SET_FORMAT:
                if (arg>1) {
                        ret=-EINVAL;
                        break;
                }
                dvb->display_ar=arg;
                ret=outcom(dvb, COMTYPE_ENCODER, SetMonitorType, 
                           1, (u16) arg);
                break;
                
        case VIDEO_STILLPICTURE:
        { 
                struct videoDisplayStillPicture pic;
                
                if(copy_from_user(&pic, parg, 
                                  sizeof(struct videoDisplayStillPicture))) {
                        ret=-EFAULT;
                        break;
                }
                ring_buffer_flush(&dvb->avout);
                play_iframe(dvb, pic.iFrame, pic.size, 
                            file->f_flags&O_NONBLOCK);
                break;
        }
        
        case VIDEO_FAST_FORWARD:
                //note: arg is ignored by firmware
                if (dvb->playing&RP_VIDEO) 
                        outcom(dvb, COMTYPE_REC_PLAY, 
                               __Scan_I, 2, AV_PES, 0);
                else 
                        vidcom(dvb, 0x16, arg); 
                dvb->trickmode=TRICK_FAST;
                dvb->videostate.playState=VIDEO_PLAYING;
                break;
                
        case VIDEO_SLOWMOTION:
                if (dvb->playing&RP_VIDEO) {
                        outcom(dvb, COMTYPE_REC_PLAY, __Slow, 2, 0, 0);
                        vidcom(dvb, 0x22, arg);
                } else {
                        vidcom(dvb, 0x0d, 0);
                        vidcom(dvb, 0x0e, 0);
                        vidcom(dvb, 0x22, arg);
                }
                dvb->trickmode=TRICK_SLOW;
                dvb->videostate.playState=VIDEO_PLAYING;
                break;
                
        case VIDEO_GET_CAPABILITIES:
        {
                int cap=VIDEO_CAP_MPEG1|
                        VIDEO_CAP_MPEG2|
                        VIDEO_CAP_SYS|
                        VIDEO_CAP_PROG;
                
                if (copy_to_user(parg, &cap, sizeof(cap))) 
                        ret=-EFAULT;
                break;
        }
        
        case VIDEO_CLEAR_BUFFER:
                ring_buffer_flush(&dvb->avout);
                reset_ipack(&dvb->ipack[1]);
                
                if (dvb->playing==RP_AV) {
                        outcom(dvb, COMTYPE_REC_PLAY, 
                               __Play, 2, AV_PES, 0);
                        if (dvb->trickmode==TRICK_FAST)
                                outcom(dvb, COMTYPE_REC_PLAY, 
                                       __Scan_I, 2, AV_PES, 0);
                        if (dvb->trickmode==TRICK_SLOW) {
                                outcom(dvb, COMTYPE_REC_PLAY, __Slow, 2, 0, 0);
                                vidcom(dvb, 0x22, arg);
                        }
                        if (dvb->trickmode==TRICK_FREEZE)
                                vidcom(dvb, 0x000e, 1);
                }
                break;
                
        case VIDEO_SET_STREAMTYPE:
                
                break;
                
        default:
                ret=-ENOIOCTLCMD;
                break;
        }
        return ret;
}

static int 
audio_ioctl(struct dvb_device *dvbdev, struct file *file, 
            unsigned int cmd, unsigned long arg)
{
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdev->priv;
        void *parg=(void *)arg;
        int ret=0;

        if (((file->f_flags&O_ACCMODE)==O_RDONLY) &&
            (cmd!=AUDIO_GET_STATUS))
                return -EPERM;
        
        switch (cmd) {
        case AUDIO_STOP:
                if (dvb->audiostate.streamSource==AUDIO_SOURCE_MEMORY)
                        AV_Stop(dvb, RP_AUDIO);
                else
                        audcom(dvb, 1);
                dvb->audiostate.playState=AUDIO_STOPPED;
                break;
                
        case AUDIO_PLAY:
                if (dvb->audiostate.streamSource==AUDIO_SOURCE_MEMORY)
                        AV_StartPlay(dvb, RP_AUDIO);
                audcom(dvb, 2);
                dvb->audiostate.playState=AUDIO_PLAYING;
                break;
                
        case AUDIO_PAUSE:
                audcom(dvb, 1);
                dvb->audiostate.playState=AUDIO_PAUSED;
                break;
                
        case AUDIO_CONTINUE:
                if (dvb->audiostate.playState==AUDIO_PAUSED) {
                        dvb->audiostate.playState=AUDIO_PLAYING;
                        audcom(dvb, 0x12);
                } 
                break;
                
        case AUDIO_SELECT_SOURCE:
                dvb->audiostate.streamSource=(audioStreamSource_t) arg;
                break;
                
        case AUDIO_SET_MUTE:
        {
                audcom(dvb, arg ? 1 : 2);
                dvb->audiostate.muteState=(boolean) arg;
                break;
        }
        
        case AUDIO_SET_AV_SYNC:
                dvb->audiostate.AVSyncState=(boolean) arg;
                audcom(dvb, arg ? 0x0f : 0x0e);
                break;
                
        case AUDIO_SET_BYPASS_MODE:
                ret=-EINVAL;
                break;
                
        case AUDIO_CHANNEL_SELECT:
                dvb->audiostate.channelSelect=(audioChannelSelect_t) arg;
                
                switch(dvb->audiostate.channelSelect) {
                case AUDIO_STEREO:
                        audcom(dvb, 0x80);
                        break;
                        
                case AUDIO_MONO_LEFT:
                        audcom(dvb, 0x100);
                        break;
                        
                case AUDIO_MONO_RIGHT:
                        audcom(dvb, 0x200);
                        break;
                        
                default:
                        ret=-EINVAL;
                        break;
                }
                break;
                
        case AUDIO_GET_STATUS:
                if(copy_to_user(parg, &dvb->audiostate, 
                                sizeof(struct audioStatus)))
                        ret=-EFAULT;
                break;
                
        case AUDIO_GET_CAPABILITIES:
        {
                int cap=AUDIO_CAP_LPCM|
                        AUDIO_CAP_MP1|
                        AUDIO_CAP_MP2;
                
                if (copy_to_user(parg, &cap, 
                                 sizeof(cap)))
                        ret=-EFAULT;
                break;
        }
        case AUDIO_CLEAR_BUFFER:
                ring_buffer_flush(&dvb->aout);
                reset_ipack(&dvb->ipack[0]);
                if (dvb->playing==RP_AV)
                        outcom(dvb, COMTYPE_REC_PLAY, 
                               __Play, 2, AV_PES, 0);
                break;
        case AUDIO_SET_ID:
                
                break;
        case AUDIO_SET_MIXER:
        {
                struct audioMixer amix;
                
                
                if(copy_from_user(&amix, parg, 
                                  sizeof(struct audioMixer))) {
                        ret=-EFAULT;
                        break;
                }
                SetVolume(dvb, amix.volume_left, amix.volume_right);
                break;
        }
        case AUDIO_SET_STREAMTYPE:
                break;
        default:
                ret=-ENOIOCTLCMD;
                break;
        }
        return ret;
}

static int 
frontend_ioctl(struct dvb_device *dvbdev, struct file *file, 
               unsigned int cmd, unsigned long arg)
{
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdev->priv;
        void *parg=(void *)arg;

        switch (cmd) {
                
        case FE_SELFTEST:
                break;
                
        case FE_SET_POWER_STATE:
                switch (arg) {
                case FE_POWER_ON:
                        dvb->powerstate=arg;
                        SetVoltage(dvb,1,0);
                        return 0;
                case FE_POWER_SUSPEND:
                case FE_POWER_STANDBY:
                case FE_POWER_OFF:
                        dvb->powerstate=FE_POWER_OFF;
                        SetVoltage(dvb,0,0);
                        return 0;
                default:
                        return -EINVAL;
                }
                dvb_frontend_demod_command(&dvb->frontend, FE_SET_POWER_STATE, &arg);
                break;
                
        case FE_GET_POWER_STATE:
                if(copy_to_user(parg, &dvb->powerstate, sizeof(u32)))
                        return -EFAULT;
                break;
                
        case FE_READ_STATUS:
        {
                FrontendStatus stat=0;
                
                dvb_frontend_demod_command(&dvb->frontend, FE_READ_STATUS, &stat);
                if (dvb->sec.power)
                        stat|=FE_HAS_POWER;
                if(copy_to_user(parg, &stat, sizeof(stat)))
                        return -EFAULT;
                break;
        }
        
        case FE_READ_BER:
        {
                uint32_t ber;
                
                if (!dvb->sec.power)
                        return -ENOSIGNAL;
                dvb_frontend_demod_command(&dvb->frontend, FE_READ_BER, &ber);
                if(copy_to_user(parg, &ber, sizeof(ber)))
                        return -EFAULT;
                break;
        }
        
        case FE_READ_SIGNAL_STRENGTH:
        {
                int32_t signal;
                
                if (!dvb->sec.power)
                        return -ENOSIGNAL;
                dvb_frontend_demod_command(&dvb->frontend, FE_READ_SIGNAL_STRENGTH, &signal);
                if(copy_to_user(parg, &signal, sizeof(signal)))
                        return -EFAULT;
                break;
        }
        
        case FE_READ_SNR:
        {
                int32_t snr;
                
                if (!dvb->sec.power)
                        return -ENOSIGNAL;
                dvb_frontend_demod_command(&dvb->frontend, FE_READ_SNR, &snr);
                if(copy_to_user(parg, &snr, sizeof(snr)))
                        return -EFAULT;
                break;
        }
        
        case FE_READ_UNCORRECTED_BLOCKS: 
        {
                u32 ublocks;
                
                if (!dvb->sec.power)
                        return -ENOSIGNAL;
                if (dvb_frontend_demod_command(&dvb->frontend, FE_READ_UNCORRECTED_BLOCKS, 
                                               &ublocks)<0)
                        return -ENOSYS; 
                if(copy_to_user(parg, &ublocks, sizeof(ublocks)))
                        return -EFAULT;
                        break;
        }                
        
        case FE_GET_NEXT_FREQUENCY:
        {
                uint32_t freq;
                
                if (copy_from_user(&freq, parg, sizeof(freq)))
                        return -EFAULT;
                
                if (dvb->frontend.type==DVB_S)
                        // FIXME: how does one calculate this?
                        freq+=1000; //FIXME: KHz like in QPSK_TUNE??
                else
                        freq+=1000000;
                
                if (copy_to_user(parg, &freq, sizeof(freq)))
                        return -EFAULT;
                break;
        }
        
        case FE_GET_NEXT_SYMBOL_RATE:
        {
                uint32_t rate;
                
                if(copy_from_user(&rate, parg, sizeof(rate)))
                        return -EFAULT;
                
                if (dvb->frontend.type==DVB_C) {
                        if (rate < 1725000)
                                rate = 1725000;
                        else if (rate < 3450000)
                                rate = 3450000;
                        else if (rate < 5175000)
                                rate = 5175000;
                        else if (rate < 5500000)
                                rate = 5500000;
                        else if (rate < 6875000)
                                rate = 6875000;
                        else if (rate < 6900000)
                                rate = 6900000;
                        else
                                return -EINVAL;
                }
                // FIXME: how does one really calculate this?
                else if (rate<5000000)
                        rate+=500000;
                else if(rate<10000000)
                        rate+=1000000;
                else if(rate<30000000)
                        rate+=2000000;
                else
                        return -EINVAL;
                
                if(copy_to_user(parg, &rate, sizeof(rate)))
                        return -EFAULT;
                break;
        }
        
        case FE_GET_FRONTEND:
        {
                if(copy_to_user(parg, &dvb->frontend.param,
                                sizeof(FrontendParameters)))
                        return -EFAULT;
                break;
        }
        
        case FE_SET_FRONTEND:
        {
                FrontendParameters para;
                
                if ((file->f_flags&O_ACCMODE)==O_RDONLY) 
                        return -EPERM;
                if(copy_from_user(&para, parg, sizeof(para)))
                        return -EFAULT;
                return tune(dvb, &para);
        }
        
        case FE_GET_EVENT:
        { 
                FrontendEvent event;
                int ret;
                
                ret=dvb_frontend_get_event(&dvb->frontend, &event, 
                                           file->f_flags&O_NONBLOCK);
                if (ret<0)
                        return ret; 
                if(copy_to_user(parg, &event, sizeof(event)))
                        return -EFAULT;
                break;
        }
        
        case FE_GET_INFO:
        {        
                FrontendInfo feinfo;
                
                switch (dvb->frontend.type) {
                case DVB_S:
                        feinfo.type=FE_QPSK;
                        feinfo.minFrequency=500;  //KHz?
                                feinfo.maxFrequency=2700000;
                                break;
                case DVB_C:
                        feinfo.type=FE_QAM;
                        feinfo.minFrequency=40000000; 
                        feinfo.maxFrequency=870000000;
                        break;
                case DVB_T:
                        feinfo.type=FE_OFDM;
                        feinfo.minFrequency=470000000;
                        feinfo.maxFrequency=860000000;
                        break;
                }
                
                feinfo.minSymbolRate=500000;
                feinfo.maxSymbolRate=30000000;
                feinfo.hwType=0;    //??
                feinfo.hwVersion=0;
                
                if(copy_to_user(parg, &feinfo, sizeof(feinfo)))
                        return -EFAULT;
                break;
        }
        
        default:
                return -ENOIOCTLCMD;
                
        }
        return 0;
}

static int 
sec_ioctl(struct dvb_device *dvbdev, struct file *file, 
          unsigned int cmd, unsigned long arg)
{
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdev->priv;
        void *parg=(void *)arg;

        if (file->f_flags&O_NONBLOCK) 
                return -EWOULDBLOCK;
        switch (cmd) {
        case SEC_GET_STATUS: 
        {        
                struct secStatus status;
                
                status.busMode=SEC_BUS_IDLE;
                
                if (dvb->secbusy) 
                        status.busMode=SEC_BUS_BUSY;
                if (!dvb->sec.power) 
                        status.busMode=SEC_BUS_OFF;
                
                status.selVolt=(dvb->sec.volt ?
                                SEC_VOLTAGE_18 :
                                SEC_VOLTAGE_13);
                
                status.contTone=(dvb->sec.ttk ?
                                 SEC_TONE_ON :
                                 SEC_TONE_OFF);
                if (copy_to_user(parg, &status, sizeof(status)))
                        return -EFAULT;
        }
        break;
        case SEC_RESET_OVERLOAD:
                if ((file->f_flags&O_ACCMODE)==O_RDONLY) 
                        return -EPERM;
                // overload control not supported by Siemens DVB card
                break;
                
        case SEC_SEND_SEQUENCE:
        {
                struct secCmdSequence seq;
                
                if(copy_from_user(&seq, parg, sizeof(seq)))
                        return -EFAULT;
                
                if ((file->f_flags&O_ACCMODE)==O_RDONLY) 
                        return -EPERM;
                
                dvb_frontend_stop(&dvb->frontend);
                SetPIDs(dvb, 0, 0, 0, 0, 0);
                return secSendSequence(dvb, &seq);
        }
        case SEC_SET_TONE:
        {
                secToneMode mode = (secToneMode) arg;
                
                if ((file->f_flags&O_ACCMODE)==O_RDONLY) 
                        return -EPERM;
                
                dvb_frontend_stop(&dvb->frontend);
                SetPIDs(dvb, 0, 0, 0, 0, 0);
                return secSetTone(dvb, mode);
        }
        case SEC_SET_VOLTAGE:
        {
                secVoltage val = (secVoltage) arg;
                
                if ((file->f_flags&O_ACCMODE)==O_RDONLY) 
                        return -EPERM;
                
                dvb_frontend_stop(&dvb->frontend);
                SetPIDs(dvb, 0, 0, 0, 0, 0);
                return secSetVoltage(dvb, val);
        }
        default:
                return -EINVAL;
        }
        return 0;
}

static int 
dvbdev_ioctl(struct dvb_device *dvbdev, int num, 
             struct file *file, unsigned int cmd, unsigned long arg)
{
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdev->priv;
        void *parg=(void *)arg;
        int type=num2type(dvb, num);

	switch (type) {
	case DVB_DEVICE_VIDEO_0:
                return video_ioctl(dvbdev, file, cmd, arg);

	case DVB_DEVICE_AUDIO_0:
                return audio_ioctl(dvbdev, file, cmd, arg);

	case DVB_DEVICE_FRONTEND_0:
                return frontend_ioctl(dvbdev, file, cmd, arg);

	case DVB_DEVICE_SEC_0:
                return sec_ioctl(dvbdev, file, cmd, arg);
                
        case DVB_DEVICE_DEMUX_0:
                return DmxDevIoctl(&dvb->dmxdev, file, cmd, arg);

	case DVB_DEVICE_DVR_0:
                return DmxDevDVRIoctl(&dvb->dmxdev, file, cmd, arg);

        case DVB_DEVICE_CA_0:
                return ca_ioctl(dvbdev, file, cmd, arg);

        case DVB_DEVICE_OSD_0: /* FIXME: get rid of */
        {
#ifdef USE_OSD
                switch (cmd) {
                case OSD_SEND_CMD:
                {
                        osd_cmd_t doc;

                        if(copy_from_user(&doc, parg, sizeof(osd_cmd_t)))
                                return -EFAULT;
                        return OSD_DrawCommand(dvb, &doc);
                }
                default:
                        return -EINVAL;
                }
#endif
                break;
        }
	case DVB_DEVICE_NET_0:  /* FIXME: move to dvb_net.c */
        {
                if (((file->f_flags&O_ACCMODE)==O_RDONLY))
                        return -EPERM;
                
                switch (cmd) {
                case NET_ADD_IF:
                {
                        struct dvb_net_if dvbnetif;
                        int result;

                        if(copy_from_user(&dvbnetif, parg, sizeof(dvbnetif)))
                                return -EFAULT;
                        result=dvb_net_add_if(&dvb->dvb_net, dvbnetif.pid);
                        if (result<0)
                                return result;
                        dvbnetif.if_num=result;
                        if(copy_to_user(parg, &dvbnetif, sizeof(dvbnetif)))
                                return -EFAULT;
                        break;
                }
                case NET_REMOVE_IF:
                        return dvb_net_remove_if(&dvb->dvb_net, (int) arg);
                default:
                        return -EINVAL;
                }
                break;
        }
        default:
                return -EOPNOTSUPP;
        }
        return 0;
}

static unsigned int 
dvbdev_poll(struct dvb_device *dvbdev, int num, 
            struct file *file, poll_table * wait)
{
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdev->priv;
        int type=num2type(dvb, num);

	switch (type) {
	case DVB_DEVICE_FRONTEND_0:
                return dvb_frontend_poll(&dvb->frontend, file, wait);

        case DVB_DEVICE_DEMUX_0:
                return DmxDevPoll(&dvb->dmxdev, file, wait);

	case DVB_DEVICE_DVR_0:
                return DmxDevDVRPoll(&dvb->dmxdev, file, wait);

        case DVB_DEVICE_VIDEO_0:
                return dvb_poll(dvb, file, wait);
                
        case DVB_DEVICE_AUDIO_0:
                return dvb_apoll(dvb, file, wait);

        case DVB_DEVICE_CA_0:
                return ci_ll_poll(&dvb->ci_rbuffer, &dvb->ci_wbuffer, file, wait);

	default:
	        return -EOPNOTSUPP;
        }

        return 0;
}


static int 
dvbdev_device_type(struct dvb_device *dvbdev, unsigned int num)
{
        struct dvb_struct *dvb=(struct dvb_struct *) dvbdev->priv;

        return num2type(dvb, num);
}

/******************************************************************************
 * driver registration 
 ******************************************************************************/

static int 
dvb_register(struct dvb_struct *dvb)
{
        int ret, i;
        struct dvb_device *dvbd=&dvb->dvb_dev;
        dmx_frontend_t *dvbfront=&dvb->hw_frontend;
        dvb_demux_t *dvbdemux=&dvb->demux;

        if (dvb->registered)
                return -1;
        dvb->registered=1;

        dvb->secbusy=0;

	dvb->audiostate.AVSyncState=0;
	dvb->audiostate.muteState=0;
	dvb->audiostate.playState=AUDIO_STOPPED;
	dvb->audiostate.streamSource=AUDIO_SOURCE_DEMUX;
	dvb->audiostate.channelSelect=AUDIO_STEREO;
	dvb->audiostate.bypassMode=0;

	dvb->videostate.videoBlank=0;
	dvb->videostate.playState=VIDEO_STOPPED;
	dvb->videostate.streamSource=VIDEO_SOURCE_DEMUX;
	dvb->videostate.videoFormat=VIDEO_FORMAT_4_3;
	dvb->videostate.displayFormat=VIDEO_CENTER_CUT_OUT;
        dvb->display_ar=VIDEO_FORMAT_4_3;

        memcpy(dvb->demux_id, "demux0_0", 9);
        dvb->demux_id[7]=dvb->num+0x30;
        dvbdemux->priv=(void *) dvb;

        if (dvb->card_type==DVB_CARD_TT_SIEMENS) {
                for (i=0; i<32; i++)
                        dvb->handle2filter[i]=NULL;

                dvbdemux->filternum=32;
                dvbdemux->feednum=32;
                dvbdemux->start_feed=dvb_start_feed;
                dvbdemux->stop_feed=dvb_stop_feed;
                dvbdemux->write_to_decoder=dvb_write_to_decoder;
                
                dvbdemux->dmx.vendor="TI";
                dvbdemux->dmx.model="AV7110";
                dvbdemux->dmx.id=dvb->demux_id;
                dvbdemux->dmx.capabilities=(DMX_TS_FILTERING|
                                             DMX_SECTION_FILTERING|
                                             DMX_MEMORY_BASED_FILTERING);

                DvbDmxInit(&dvb->demux);


                dvbfront->id="hw_frontend";
                dvbfront->vendor="VLSI";

                switch (dvb->frontend.type) {
                default:
                case DVB_S:
                        dvbfront->model="DVBS";
                        dvbfront->source=DMX_SATELLITE_FE;
                        break;
                case DVB_C:
                        dvbfront->model="1820";
                        dvbfront->source=DMX_CABLE_FE;
                        break;
                case DVB_T:
                        dvbfront->model="DVBT";
                        dvbfront->source=DMX_TERRESTRIAL_FE;
                        break;
                }
                dvb->dmxdev.filternum=32;
                dvb->dmxdev.demux=&dvbdemux->dmx;
                dvb->dmxdev.capabilities=0;

                DmxDevInit(&dvb->dmxdev);
        }

        if (dvb->card_type==DVB_CARD_TT_BUDGET) {
                dvbdemux->filternum=256;
                dvbdemux->feednum=256;
                dvbdemux->start_feed=dvb_start_feed;
                dvbdemux->stop_feed=dvb_stop_feed;
                dvbdemux->write_to_decoder=0;

                dvbdemux->dmx.vendor="CIM";
                dvbdemux->dmx.model="sw";
                dvbdemux->dmx.id=dvb->demux_id;
                dvbdemux->dmx.capabilities=(DMX_TS_FILTERING|
                                             DMX_SECTION_FILTERING|
                                             DMX_MEMORY_BASED_FILTERING);

                DvbDmxInit(&dvb->demux);

                dvbfront->id="hw_frontend";
                dvbfront->vendor="VLSI";
                switch (dvb->frontend.type) {
                default:
                case DVB_S:
                        dvbfront->model="DVBS";
                        dvbfront->source=DMX_SATELLITE_FE;
                        break;
                case DVB_C:
                        dvbfront->model="1820";
                        dvbfront->source=DMX_CABLE_FE;
                        break;
                case DVB_T:
                        dvbfront->model="DVBT";
                        dvbfront->source=DMX_TERRESTRIAL_FE;
                        break;
                }
                
                dvb->dmxdev.filternum=256;
                dvb->dmxdev.demux=&dvbdemux->dmx;
                dvb->dmxdev.capabilities=0;

                DmxDevInit(&dvb->dmxdev);
        }

        ret=dvbdemux->dmx.add_frontend(&dvbdemux->dmx, 
                                        &dvb->hw_frontend);
        if (ret<0)
                return ret;
        
        dvb->mem_frontend.id="mem_frontend";
        dvb->mem_frontend.vendor="memory";
        dvb->mem_frontend.model="sw";
        dvb->mem_frontend.source=DMX_MEMORY_FE;
        ret=dvbdemux->dmx.add_frontend(&dvbdemux->dmx, 
                                        &dvb->mem_frontend);
        if (ret<0)
                return ret;
        
        ret=dvbdemux->dmx.connect_frontend(&dvbdemux->dmx, 
                                            &dvb->hw_frontend);
        if (ret<0)
                return ret;

        // init and register dvb device structure
        dvbd->priv=(void *) dvb;
        dvbd->open=dvbdev_open;
        dvbd->close=dvbdev_close;
        dvbd->read=dvbdev_read;
        dvbd->write=dvbdev_write;
        dvbd->ioctl=dvbdev_ioctl;
        dvbd->poll=dvbdev_poll;
        dvbd->device_type=dvbdev_device_type;
        
        for (i=0; i<DVB_DEVS_MAX; i++) 
                dvb->users[i]=dvb->writers[i]=0;

        dvb->dvb_devs=0;
        if (dvb->card_type==DVB_CARD_TT_SIEMENS) {
                if (dvb->frontend.type == DVB_S)
                        dvb->dvb_devs=&dvbs_devs;
                if (dvb->frontend.type == DVB_C || dvb->frontend.type == DVB_T)
                        dvb->dvb_devs=&dvbc_devs;
        }
        if (dvb->card_type==DVB_CARD_TT_BUDGET) {
                if (dvb->frontend.type == DVB_S)
                        dvb->dvb_devs=&ttbs_devs;
                if (dvb->frontend.type == DVB_C || dvb->frontend.type == DVB_T)
                        dvb->dvb_devs=&ttbc_devs;
        }
        
        dvb->dvb_net.card_num=dvb->num;
        dvb_net_init(&dvb->dvb_net, &dvbdemux->dmx);
        return dvb_register_device(dvbd);
}


static void 
dvb_unregister(struct dvb_struct *dvb)
{
        dvb_demux_t *dvbdemux=&dvb->demux;

        if (!dvb->registered)
                return;
        dvb_net_release(&dvb->dvb_net);
        dvbdemux->dmx.close(&dvbdemux->dmx);
        dvbdemux->dmx.remove_frontend(&dvbdemux->dmx, &dvb->hw_frontend);
        dvbdemux->dmx.remove_frontend(&dvbdemux->dmx, &dvb->mem_frontend);

        DmxDevRelease(&dvb->dmxdev);
        DvbDmxRelease(&dvb->demux);
        dvb_unregister_device(&dvb->dvb_dev);
}

/****************************************************************************
 * INITIALIZATION
 ****************************************************************************/

static int __init
get_card_type(struct pci_dev *dev)
{
        int type=-1; 

        switch (dev->subsystem_vendor) {

        case 0x110a: /* Siemens cable card */
                type=DVB_CARD_TT_SIEMENS;
                break;

        case 0x13c2:
                switch (dev->subsystem_device) {

                case 0x1003: /* TT-Budget and WinTV-NOVA-S, both tuner types */
                case 0x1013: /* also TT-Budget ("SATELCO Multimedia card") */
                case 0x1004: /* WinTV-NOVA-C */
                case 0x1005: /* WinTV-NOVA-T */
                        type=DVB_CARD_TT_BUDGET;
                        break;                  

                case 0x0000: /* Siemens cable and sat card (maybe older version?) */
                case 0x0001: /* Technotrend, Hauppauge (maybe retail version ?)   */
                case 0x0002: /* dito   */
                case 0x1002: /* dito   */
                case 0x0003: /* Hauppauge/TT V2.1, new design and new tuner */
                case 0x0004: /* new type but with old tuner ? */
                case 0x0008: /* TT DVB-T */
                        type=DVB_CARD_TT_SIEMENS;
                        break;                  
                }
                break;                  

        default:
                break;
        }
        return type;
}


__init static int 
init_card(struct dvb_struct *dvb, int num, 
          struct i2c_adapter *adap, int card_type,
          struct pci_dev *pdev)
{
        /* clear everything */
        memset(dvb, 0, sizeof(struct dvb_struct));

        /* PCI device and card type */
        dvb->pdev=pdev;
        dvb->card_type=card_type;
        
        /* locks for data transfers from/to AV7110 */
        spin_lock_init (&dvb->debilock);
        sema_init(&dvb->dcomlock, 1);
        init_waitqueue_head(&dvb->debiq);
        dvb->debilock=SPIN_LOCK_UNLOCKED;
        dvb->debitype=-1;

        dvb->num=num;
        dvb->i2cbus=adap;
        dvb->saa=(struct saa7146 *) (adap->data);
        dvb->saa_mem=dvb->saa->mem;

        /* default frontend type and ADAC */
        dvb->frontend.type = DVB_S;
        dvb->adac_type = adac;

        /* default OSD window */
        dvb->osdwin=1;

        /* ARM "watchdog" */
	init_waitqueue_head(&dvb->arm_wait);
        dvb->arm_thread=0;
     
        dvb->vidmode=VIDEO_MODE_PAL;

        init_ipack(&dvb->ipack[0], IPACKS, play_audio_cb);
        dvb->ipack[0].data=(void *) dvb;
        init_ipack(&dvb->ipack[1], IPACKS, play_video_cb);
        dvb->ipack[1].data=(void *) dvb;
        

        /* allocate and init buffers */

        dvb->debi_virt=pci_alloc_consistent(dvb->pdev, 8192, &dvb->debi_bus);
        if (!dvb->debi_virt) 
                return -1;

        dvb->iobuf=vmalloc(AVOUTLEN+AOUTLEN+BMPLEN+4*IPACKS);
        if (!dvb->iobuf) 
                return -1;

        ring_buffer_init(&dvb->avout, dvb->iobuf, AVOUTLEN);
        ring_buffer_init(&dvb->aout, dvb->iobuf+AVOUTLEN, AOUTLEN);

        /* init BMP buffer */
        dvb->bmpbuf=dvb->iobuf+AVOUTLEN+AOUTLEN;
        init_waitqueue_head(&dvb->bmpq);
        
        dvb->kbuf[0]=(u8 *)(dvb->iobuf+AVOUTLEN+AOUTLEN+BMPLEN);
        dvb->kbuf[1]=dvb->kbuf[0]+2*IPACKS;

        /* CI link layer buffers */
        ci_ll_init(&dvb->ci_rbuffer, &dvb->ci_wbuffer, 8192);


        /* handle different card types */

        /* load firmware into AV7110 cards */
        if (dvb->card_type==DVB_CARD_TT_SIEMENS) {
                bootarm(dvb);
                firmversion(dvb);
                kernel_thread(arm_thread, (void *)dvb, 0);
        }

        if (dvb->card_type==DVB_CARD_TT_BUDGET) {
                saa7146_write(dvb->saa_mem, DD1_INIT, 0x02000600);
                saa7146_write(dvb->saa_mem, MC2, 
                              (MASK_09 | MASK_25 | MASK_10 | MASK_26));
                setgpio(dvb, 2, GPIO_OUTHI); /* frontend power on */
        }

        /* figure out and init frontend */
        InitFront(dvb);

        /* remaining inits according to card and frontend type */

        if (dvb->card_type==DVB_CARD_TT_SIEMENS) {
                
                /* register video4linux device */
                if(vid_register(dvb) < 0)
                        return -1;
                
                if (i2c_writereg(dvb, 0x20, 0x00, 0x00)==1) {
                        dprintk("dvb%d: Crystal audio DAC detected\n", dvb->num);
                        dvb->adac_type = DVB_ADAC_CRYSTAL;
                        i2c_writereg(dvb, 0x20, 0x01, 0xd2);
                        i2c_writereg(dvb, 0x20, 0x02, 0x49);
                        i2c_writereg(dvb, 0x20, 0x03, 0x00);
                        i2c_writereg(dvb, 0x20, 0x04, 0x00);
                }
                
                if (dvb->frontend.type == DVB_C) {
                        if (i2c_writereg(dvb, 0x80, 0x0, 0x80)==1) {
                                i2c_writereg(dvb, 0x80, 0x0, 0);
                                printk("dvb%d: DVB-C analog module detected, "
                                       "initializing MSP3400\n", dvb->num);
                                ddelay(10);
                                msp_writereg(dvb, 0x12, 0x0013, 0x0c00);
                                msp_writereg(dvb, 0x12, 0x0000, 0x7f00); // loudspeaker + headphone
                                msp_writereg(dvb, 0x12, 0x0008, 0x0220); // loudspeaker source
                                msp_writereg(dvb, 0x12, 0x0004, 0x7f00); // loudspeaker volume
                                msp_writereg(dvb, 0x12, 0x000a, 0x0220); // SCART 1 source
                                msp_writereg(dvb, 0x12, 0x0007, 0x7f00); // SCART 1 volume
                                msp_writereg(dvb, 0x12, 0x000d, 0x4800); // prescale SCART
                        }

                        // switch DVB SCART on
                        outcom(dvb, COMTYPE_AUDIODAC, MainSwitch, 1, 0);
                        outcom(dvb, COMTYPE_AUDIODAC, ADSwitch, 1, 1);
                        
                        //setgpio(dvb, 1, GPIO_OUTHI); // RGB on, SCART pin 16
                        //setgpio(dvb, 3, GPIO_OUTLO); // SCARTpin 8
                        dvb->adac_type = DVB_ADAC_NONE;
                }
        }

        dvb_register(dvb);

        return 0;
}

__init static int 
find_saas(void)
{
        struct dvb_struct *dvb;
        struct saa7146 *saa;
        int card_type;
        int i, num=0;

#ifdef FIRM_MOD
        dvb_dpram_addr=dvb_root_addr=NULL;
        dvb_dpram_len=dvb_root_len=0;
        firm_mod=request_module("dvb_firm");
        printk("adpram=%08x, aroot=%08x, ldpram=%08x, lroot=%08x\n",
               (u32)dvb_dpram_addr, (u32)dvb_root_addr,
               dvb_dpram_len, dvb_root_len);
        printk("adpram=%08x, aroot=%08x, ldpram=%08x, lroot=%08x\n",
               (u32)virt_to_phys(dvb_dpram_addr), (u32)virt_to_phys(dvb_root_addr),
               dvb_dpram_len, dvb_root_len);
        printk("dpram[0]=%08x\n", *(int *)dvb_dpram_addr);
        printk("root[0]=%08x\n", *(int *)dvb_root_addr);

#endif
        for (i=0; i<8; i++) {
                saa=saa7146_get_handle(i);
                if (!saa)
                        continue;
                card_type=get_card_type(saa->device);
                if (card_type<0)
                        continue;
                dvb=&dvbs[num];
                init_card(dvb, num, saa->i2c_adap, card_type, saa->device);
                num++;
        }
        return num;
}

static void 
dvb_irq(struct saa7146 *saa, u32 isr, void *data) 
{
        if (isr&MASK_19)
                debiirq(saa, data);
        if (isr&MASK_03)
                gpioirq(saa, data);
        if (isr&MASK_10)
                vpeirq(saa, data);
        if (isr&MASK_07)
                fidbirq(saa, data);
}

static int 
dvb_command(struct saa7146* saa, void *p, unsigned int cmd, void *arg)
{
        switch(cmd) {
        case SAA7146_SUSPEND:
                printk("dvb_suspend()\n");
                break;
        case SAA7146_RESUME:
                printk("dvb_resume()\n");
                break;
        default:
                return -ENOIOCTLCMD;
        }
        return 0;
}

static int dvb_extnum=0;

static int 
dvb_attach(struct saa7146* adap, void** p)
{
        if (dvb_extnum==MAX_NUM_DVB)
                return -1;
	*(struct dvb_struct**)p = &dvbs[dvb_extnum++];
        return 0;
}

static int 
dvb_detach(struct saa7146* adap, void** p)
{
        if (dvb_extnum==0)
                return -1;
     
        saa7146_write(adap->mem, IER, 
                      saa7146_read(adap->mem, IER) & 
                      ~(MASK_19|MASK_03));

        dvb_extnum--;
	*(struct dvb_struct**)p = 0;
        return 0;
}

static void 
dvb_inc_use(struct saa7146* adap)
{
#ifdef MODULE
	MOD_INC_USE_COUNT;
#endif
}

static void 
dvb_dec_use(struct saa7146* adap)
{
#ifdef MODULE
	MOD_DEC_USE_COUNT;
#endif
}

static struct saa7146_extension dvb_extension = {
	"dvb extension\0",
	MASK_07|MASK_10|MASK_19|MASK_03|MASK_27,
	dvb_irq,
	dvb_command,
	dvb_attach,
	dvb_detach,
	dvb_inc_use,
	dvb_dec_use
};	

static int __init 
init_dvb(void) 
{
	int result = 0;
	num_dvb = 0;

	dprintk("dvb: init_dvb\n");

	if( 0 != (result = saa7146_add_extension(&dvb_extension))) {
		printk("dvb: extension registration failed, module not inserted.\n");
		return result;
	}

        if (!(num_dvb=find_saas()))
        {
	  printk(KERN_ERR "dvb: no dvb(s) found!\n");
	  return -ENODEV;
	}

        dprintk("dvb: dvbdevice found\n");

        printk(KERN_ERR "dvb: %d dvb(s) found!\n", num_dvb);
        return result;
}

static void __exit 
exit_dvb(void)
{
	int i;

	fprintk("dvb: ==> cleanup_module\n");

	/* release the saa7146s */
	for( i = 0; i < num_dvb; i++) {
		dvbs[i].arm_rmmod=1;
		wake_up_interruptible(&dvbs[i].arm_wait);
                while (dvbs[i].arm_thread)
                        ddelay(1);
                dvb_frontend_exit(&dvbs[i].frontend);

                SetVoltage(&dvbs[i],0,0);
                saa7146_write(dvbs[i].saa_mem, IER, 
                              saa7146_read(dvbs[i].saa_mem, IER) & 
                              ~(MASK_19 | MASK_03));
                saa7146_write(dvbs[i].saa_mem, ISR,(MASK_19 | MASK_03));
                
                ci_ll_release(&dvbs[i].ci_rbuffer, &dvbs[i].ci_wbuffer);
                free_ipack(&dvbs[i].ipack[0]);
                free_ipack(&dvbs[i].ipack[1]);
                vfree(dvbs[i].iobuf);
                pci_free_consistent(dvbs[i].pdev, 8192, 
                                    dvbs[i].debi_virt,
                                    dvbs[i].debi_bus);
                
                if (dvbs[i].card_type==DVB_CARD_TT_SIEMENS) {
                        vid_unregister(&dvbs[i]);
                }
                dvb_unregister(&dvbs[i]);
        }
        if ( saa7146_del_extension(&dvb_extension)) 
                printk(KERN_ERR "dvb: extension deregistration failed.\n");

	printk(KERN_ERR "dvb: %d dvb(s) released.\n", num_dvb);

}

module_init(init_dvb);
module_exit(exit_dvb);


/*
 * Local variables:
 * c-indent-level: 8
 * c-brace-imaginary-offset: 0
 * c-brace-offset: -8
 * c-argdecl-indent: 8
 * c-label-offset: -8
 * c-continued-statement-offset: 8
 * c-continued-brace-offset: 0
 * indent-tabs-mode: nil
 * tab-width: 8
 * End:
 */
