/*******************************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.debug.internal.ui.contexts;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.ListenerList;
import org.eclipse.debug.internal.ui.contexts.provisional.IDebugContextListener;
import org.eclipse.debug.internal.ui.contexts.provisional.IDebugContextManager;
import org.eclipse.debug.internal.ui.contexts.provisional.IDebugContextProvider;
import org.eclipse.debug.internal.ui.views.ViewContextManager;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.ui.IWindowListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

/**
 * @since 3.2
 */
public class DebugContextManager implements IDebugContextManager {
	
	private static DebugContextManager fgDefault;
	private Map fServices = new HashMap();
	private ListenerList fGlobalListeners = new ListenerList();
	
	private class WindowListener implements IWindowListener {

		/* (non-Javadoc)
		 * @see org.eclipse.ui.IWindowListener#windowActivated(org.eclipse.ui.IWorkbenchWindow)
		 */
		public void windowActivated(IWorkbenchWindow window) {
		}

		/* (non-Javadoc)
		 * @see org.eclipse.ui.IWindowListener#windowDeactivated(org.eclipse.ui.IWorkbenchWindow)
		 */
		public void windowDeactivated(IWorkbenchWindow window) {			
		}

		/* (non-Javadoc)
		 * @see org.eclipse.ui.IWindowListener#windowClosed(org.eclipse.ui.IWorkbenchWindow)
		 */
		public void windowClosed(IWorkbenchWindow window) {
			DebugWindowContextService service = (DebugWindowContextService) fServices.remove(window);
			if (service != null) {
				service.dispose();
			}
		}

		/* (non-Javadoc)
		 * @see org.eclipse.ui.IWindowListener#windowOpened(org.eclipse.ui.IWorkbenchWindow)
		 */
		public void windowOpened(IWorkbenchWindow window) {
		}
		
	}
	
	private DebugContextManager() {
		PlatformUI.getWorkbench().addWindowListener(new WindowListener());
	}
	
	public static IDebugContextManager getDefault() {
		if (fgDefault == null) {
			fgDefault = new DebugContextManager();
			// create the model context bindigg manager at the same time
			DebugModelContextBindingManager.getDefault();
			// create view manager
			ViewContextManager.getDefault();			
		}
		return fgDefault;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.contexts.IDebugContextManager#addDebugContextProvider(org.eclipse.debug.ui.contexts.IDebugContextProvider)
	 */
	public void addDebugContextProvider(IDebugContextProvider provider) {
		IWorkbenchPart part = provider.getPart();
		IWorkbenchWindow window = part.getSite().getWorkbenchWindow();
		DebugWindowContextService service = createService(window);
		service.addProvider(provider);
	}
	
	protected DebugWindowContextService createService(IWorkbenchWindow window) {
		DebugWindowContextService service = (DebugWindowContextService) fServices.get(window);
		if (service == null) {
			service = new DebugWindowContextService(window);
			fServices.put(window, service);
			// TODO: register 'null' provider (global)
			
			// register global listeners
			Object[] listeners = fGlobalListeners.getListeners();
			for (int i = 0; i < listeners.length; i++) {
				IDebugContextListener listener = (IDebugContextListener) listeners[i];
				service.addDebugContextListener(listener);
			}
		}
		return service;
	}
	
	protected IDebugContextService getService(IWorkbenchWindow window) {
		return (DebugWindowContextService) fServices.get(window);
	}	

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.contexts.IDebugContextManager#removeDebugContextProvider(org.eclipse.debug.ui.contexts.IDebugContextProvider)
	 */
	public void removeDebugContextProvider(IDebugContextProvider provider) {
		IWorkbenchPart part = provider.getPart();
		IWorkbenchWindow window = part.getSite().getWorkbenchWindow();
		DebugWindowContextService service = (DebugWindowContextService) fServices.get(window);
		if (service != null) {
			service.removeProvider(provider);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.contexts.IDebugContextManager#addDebugContextListener(org.eclipse.debug.ui.contexts.IDebugContextListener, org.eclipse.ui.IWorkbenchWindow)
	 */
	public void addDebugContextListener(IDebugContextListener listener, IWorkbenchWindow window) {
		IDebugContextService service = createService(window);
		service.addDebugContextListener(listener);			
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.contexts.IDebugContextManager#removeDebugContextListener(org.eclipse.debug.ui.contexts.IDebugContextListener, org.eclipse.ui.IWorkbenchWindow)
	 */
	public void removeDebugContextListener(IDebugContextListener listener, IWorkbenchWindow window) {
		IDebugContextService service = getService(window);
		if (service != null) {
			service.removeDebugContextListener(listener);
		}	
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.contexts.IDebugContextManager#addDebugContextListener(org.eclipse.debug.ui.contexts.IDebugContextListener, org.eclipse.ui.IWorkbenchWindow, java.lang.String)
	 */
	public void addDebugContextListener(IDebugContextListener listener, IWorkbenchWindow window, String partId) {
		DebugWindowContextService service = createService(window);
		service.addDebugContextListener(listener, partId);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.contexts.IDebugContextManager#removeDebugContextListener(org.eclipse.debug.ui.contexts.IDebugContextListener, org.eclipse.ui.IWorkbenchWindow, java.lang.String)
	 */
	public void removeDebugContextListener(IDebugContextListener listener, IWorkbenchWindow window, String partId) {
		IDebugContextService service = getService(window);
		if (service != null) {
			service.removeDebugContextListener(listener, partId);
		}		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.contexts.IDebugContextManager#getActiveContext(org.eclipse.ui.IWorkbenchWindow)
	 */
	public ISelection getActiveContext(IWorkbenchWindow window) {
		IDebugContextService service = getService(window);
		if (service != null) {
			return service.getActiveContext();
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.contexts.IDebugContextManager#getActiveContext(org.eclipse.ui.IWorkbenchWindow, java.lang.String)
	 */
	public ISelection getActiveContext(IWorkbenchWindow window, String partId) {
		IDebugContextService service = getService(window);
		if (service != null) {
			return service.getActiveContext(partId);
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.internal.ui.contexts.IDebugContextManager#addDebugContextListener(org.eclipse.debug.internal.ui.contexts.IDebugContextListener)
	 */
	public void addDebugContextListener(IDebugContextListener listener) {
		fGlobalListeners.add(listener);
		DebugWindowContextService[] services = getServices();
		for (int i = 0; i < services.length; i++) {
			DebugWindowContextService service = services[i];
			service.addDebugContextListener(listener);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.internal.ui.contexts.IDebugContextManager#removeDebugContextListener(org.eclipse.debug.internal.ui.contexts.IDebugContextListener)
	 */
	public void removeDebugContextListener(IDebugContextListener listener) {
		fGlobalListeners.remove(listener);
		DebugWindowContextService[] services = getServices();
		for (int i = 0; i < services.length; i++) {
			DebugWindowContextService service = services[i];
			service.removeDebugContextListener(listener);
		}
	}
	
	/**
	 * Returns the existing context services.
	 * 
	 * @return existing context services
	 */
	private DebugWindowContextService[] getServices() {
		Collection sevices = fServices.values();
		return (DebugWindowContextService[]) sevices.toArray(new DebugWindowContextService[sevices.size()]);
	}
	
}
