/***************************************************************************
 *   Copyright (C) 2006, 2007 by Niklas Knutsson                           *
 *   nq@altern.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "recurrenceeditwidget.h"
#include "budget.h"
#include "recurrence.h"

#include <kglobal.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kdeversion.h>
#include <kstatusbar.h>
#include <kaccel.h>
#include <kconfig.h>
#include <kurl.h>
#include <kurldrag.h>
#include <qheader.h>
#include <kmessagebox.h>
#include <qlayout.h>
#include <klineedit.h>
#include <qcombobox.h>
#include "kdateedit.h"
#include <qlabel.h>
#include <kbuttonbox.h>
#include <kstdguiitem.h>
#include <kstdaccel.h>
#include <kaction.h>
#include <kstdaction.h>
#include <qobject.h>
#include <qpushbutton.h>
#include <ktempfile.h>
#include <qlabel.h>
#include <kseparator.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qvgroupbox.h>
#include <kpushbutton.h>
#include <ktextedit.h>
#include <qbuttongroup.h>
#include <qhbox.h>
#include <qwidgetstack.h>
#include <qspinbox.h>
#include <kcalendarsystem.h>

extern QDate addYears(const QDate &date, int nyears);

EditExceptionsDialog::EditExceptionsDialog(QWidget *parent)  : KDialogBase(parent, 0, true, i18n("Edit Exceptions"), Ok | Cancel, Ok, true) {
	setMainWidget(new QWidget(this));
	
	QHBoxLayout *exceptionsLayout = new QHBoxLayout(mainWidget(), 0, spacingHint());

	exceptionsList = new QListBox(mainWidget());
	exceptionsLayout->addWidget(exceptionsList);
	
	QVBoxLayout *buttonsLayout = new QVBoxLayout(exceptionsLayout);
	dateEdit = new KDateEdit(mainWidget());
	dateEdit->setDate(QDate::currentDate());
	buttonsLayout->addWidget(dateEdit);
	addButton = new QPushButton(i18n("Add"), mainWidget());
	buttonsLayout->addWidget(addButton);
	changeButton = new QPushButton(i18n("Change"), mainWidget());
	changeButton->setEnabled(false);
	buttonsLayout->addWidget(changeButton);
	deleteButton = new QPushButton(i18n("Delete"), mainWidget());
	deleteButton->setEnabled(false);
	buttonsLayout->addWidget(deleteButton);
	buttonsLayout->addStretch(1);

	connect(exceptionsList, SIGNAL(selectionChanged()), this, SLOT(onSelectionChanged()));
	connect(addButton, SIGNAL(clicked()), this, SLOT(addException()));
	connect(changeButton, SIGNAL(clicked()), this, SLOT(changeException()));
	connect(deleteButton, SIGNAL(clicked()), this, SLOT(deleteException()));

}
EditExceptionsDialog::~EditExceptionsDialog() {
}
void EditExceptionsDialog::setRecurrence(Recurrence *rec) {
	exceptionsList->clear();
	if(rec) {
		for(QValueVector<QDate>::size_type i = 0; i < rec->exceptions.size(); i++) {
			exceptionsList->insertItem(KGlobal::locale()->formatDate(rec->exceptions[i], true));
		}
		exceptionsList->sort();
	}
	saveValues();
}
void EditExceptionsDialog::modifyExceptions(Recurrence *rec) {
	for(uint i = 0; i < exceptionsList->count(); i++) {
		rec->addException(KGlobal::locale()->readDate(exceptionsList->text(i)));
	}
}
bool EditExceptionsDialog::validValues() {
	return true;
}
void EditExceptionsDialog::saveValues() {
	savedExceptions.clear();
	for(uint i = 0; i < exceptionsList->count(); i++) {
		savedExceptions.append(exceptionsList->text(i));
	}
}
void EditExceptionsDialog::restoreValues() {
	exceptionsList->clear();
	exceptionsList->insertStringList(savedExceptions);
}
void EditExceptionsDialog::slotOk() {
	if(validValues()) {
		saveValues();
		KDialogBase::slotOk();
	}
}
void EditExceptionsDialog::slotCancel() {
	KDialogBase::slotCancel();
	restoreValues();
}
void EditExceptionsDialog::onSelectionChanged() {
	if(exceptionsList->selectedItem()) {
		changeButton->setEnabled(true);
		deleteButton->setEnabled(true);
		dateEdit->setDate(KGlobal::locale()->readDate(exceptionsList->text(exceptionsList->currentItem())));
	} else {
		changeButton->setEnabled(false);
		deleteButton->setEnabled(false);
	}
}
void EditExceptionsDialog::addException() {
	QDate date = dateEdit->date();
	if(!date.isValid()) {
		KMessageBox::error(this, i18n("Invalid date."));
		return;
	}
	QString sdate = KGlobal::locale()->formatDate(date, true);
	if(!exceptionsList->findItem(sdate)) {
		exceptionsList->insertItem(sdate);
		exceptionsList->sort();
	}
}
void EditExceptionsDialog::changeException() {
	if(exceptionsList->currentItem() < 0) return;
	QDate date = dateEdit->date();
	if(!date.isValid()) {
		KMessageBox::error(this, i18n("Invalid date."));
		return;
	}
	QString sdate = KGlobal::locale()->formatDate(date, true);
	if(sdate == exceptionsList->currentText()) return;
	if(exceptionsList->findItem(sdate)) {
		exceptionsList->removeItem(exceptionsList->currentItem());
	} else {
		exceptionsList->changeItem(sdate, exceptionsList->currentItem());
		exceptionsList->sort();
	}
}
void EditExceptionsDialog::deleteException() {
	if(exceptionsList->currentItem() < 0) return;
	exceptionsList->removeItem(exceptionsList->currentItem());
}

EditRangeDialog::EditRangeDialog(const QDate &startdate, QWidget *parent) : KDialogBase(parent, 0, true, i18n("Edit Recurrence Range"), Ok | Cancel, Ok, true), date(startdate) {

	setMainWidget(new QWidget(this));
	QGridLayout *rangeLayout = new QGridLayout(mainWidget(), 4, 3, 0, spacingHint());

	startLabel = new QLabel(i18n("Begins on: %1").arg(KGlobal::locale()->formatDate(startdate)), mainWidget());
	rangeLayout->addMultiCellWidget(startLabel, 0, 0, 0, 2);

	rangeButtonGroup = new QButtonGroup();
	
	foreverButton = new QRadioButton(i18n("No ending date"), mainWidget());
	rangeButtonGroup->insert(foreverButton);
	rangeLayout->addMultiCellWidget(foreverButton, 1, 1, 0, 2);

	fixedCountButton = new QRadioButton(i18n("End after"), mainWidget());
	rangeButtonGroup->insert(fixedCountButton);
	rangeLayout->addWidget(fixedCountButton, 2, 0);
	fixedCountEdit = new QSpinBox(1, 9999, 1, mainWidget());
	rangeLayout->addWidget(fixedCountEdit, 2, 1);
	rangeLayout ->addWidget(new QLabel(i18n("occurrence(s)"), mainWidget()), 2, 2);

	endDateButton = new QRadioButton(i18n("End on"), mainWidget());
	rangeButtonGroup->insert(endDateButton);
	rangeLayout->addWidget(endDateButton, 3, 0);
	endDateEdit = new KDateEdit(mainWidget());
	rangeLayout->addMultiCellWidget(endDateEdit, 3, 3, 1, 2);

	setRecurrence(NULL);

	connect(fixedCountButton, SIGNAL(toggled(bool)), fixedCountEdit, SLOT(setEnabled(bool)));
	connect(endDateButton, SIGNAL(toggled(bool)), endDateEdit, SLOT(setEnabled(bool)));
			
}

EditRangeDialog::~EditRangeDialog() {
	delete rangeButtonGroup;
}

void EditRangeDialog::setStartDate(const QDate &startdate) {
	startLabel->setText(i18n("Begins on: %1").arg(KGlobal::locale()->formatDate(startdate)));
	date = startdate;
	if(!endDateButton->isChecked() && date > endDateEdit->date()) {
		endDateEdit->setDate(date);
	}
}
void EditRangeDialog::setRecurrence(Recurrence *rec) {
	if(rec && rec->fixedOccurrenceCount() > 0) {
		fixedCountButton->setChecked(true);
		fixedCountEdit->setValue(rec->fixedOccurrenceCount());
		endDateEdit->setEnabled(false);
		fixedCountEdit->setEnabled(true);
		endDateEdit->setDate(date);
	} else if(rec && !rec->endDate().isNull()) {
		endDateButton->setChecked(true);
		endDateEdit->setDate(rec->endDate());
		endDateEdit->setEnabled(true);
		fixedCountEdit->setEnabled(false);
		fixedCountEdit->setValue(1);
	} else {
		foreverButton->setChecked(true);
		endDateEdit->setEnabled(false);
		fixedCountEdit->setEnabled(false);
		endDateEdit->setDate(date);
		fixedCountEdit->setValue(1);
	}
	saveValues();
}
int EditRangeDialog::fixedCount() {
	if(fixedCountButton->isChecked()) {
		return fixedCountEdit->value();
	}
	return -1;
}
QDate EditRangeDialog::endDate() {
	if(endDateButton->isChecked()) {
		return endDateEdit->date();
	}
	return QDate();
}
void EditRangeDialog::slotOk() {
	if(validValues()) {
		saveValues();
		KDialogBase::slotOk();
	}
}
void EditRangeDialog::saveValues() {
	fixedCountEdit_value = fixedCountEdit->value();
	endDateEdit_value = endDateEdit->date();
	if(foreverButton->isChecked()) checkedButton = foreverButton;
	else if(endDateButton->isChecked()) checkedButton = endDateButton;
	else checkedButton = fixedCountButton;
}
void EditRangeDialog::restoreValues() {
	fixedCountEdit->setValue(fixedCountEdit_value);
	endDateEdit->setDate(endDateEdit_value);
	checkedButton->setChecked(true);
}
void EditRangeDialog::slotCancel() {
	KDialogBase::slotCancel();
	restoreValues();
}
bool EditRangeDialog::validValues() {
	if(endDateButton->isChecked()) {
		if(!endDateEdit->date().isValid()) {
			KMessageBox::error(this, i18n("Invalid date."));
			endDateEdit->setFocus();
			return false;
		}
		if(endDateEdit->date() < date) {
			KMessageBox::error(this, i18n("End date before start date."));
			endDateEdit->setFocus();
			return false;
		}
	}
	return true;
}


RecurrenceEditWidget::RecurrenceEditWidget(const QDate &startdate, Budget *budg, QWidget *parent, const char *name) : QWidget(parent, name), date(startdate), budget(budg) {

	QVBoxLayout *recurrenceLayout = new QVBoxLayout(this, 0, 6);
	
	recurrenceButton = new QCheckBox(i18n("Enable recurrance"), this);
	recurrenceLayout->addWidget(recurrenceButton);

	ruleGroup = new QVGroupBox(i18n("Recurrence Rule"), this);
	
	typeCombo = new QComboBox(ruleGroup);
	typeCombo->setEditable(false);
	typeCombo->insertItem(i18n("Daily"));
	typeCombo->insertItem(i18n("Weekly"));
	typeCombo->insertItem(i18n("Monthly"));
	typeCombo->insertItem(i18n("Yearly"));
	typeCombo->setCurrentItem(2);
	
	ruleStack = new QWidgetStack(ruleGroup);
	QWidget *dailyRuleWidget = new QWidget(ruleStack);
	ruleStack->addWidget(dailyRuleWidget, 0);
	QWidget *weeklyRuleWidget = new QWidget(ruleStack);
	ruleStack->addWidget(weeklyRuleWidget, 1);
	QWidget *monthlyRuleWidget = new QWidget(ruleStack);
	ruleStack->addWidget(monthlyRuleWidget, 2);
	QWidget *yearlyRuleWidget = new QWidget(ruleStack);
	ruleStack->addWidget(yearlyRuleWidget, 3);

	QVBoxLayout *dailyRuleLayout = new QVBoxLayout(dailyRuleWidget, 0, 6);
	QHBoxLayout *dailyFrequencyLayout = new QHBoxLayout(dailyRuleLayout);
	dailyFrequencyLayout->addWidget(new QLabel(i18n("Recur every"), dailyRuleWidget));
	dailyFrequencyEdit = new QSpinBox(1, 9999, 1, dailyRuleWidget);
	dailyFrequencyLayout->addWidget(dailyFrequencyEdit);
	dailyFrequencyLayout->addWidget(new QLabel(i18n("day(s)"), dailyRuleWidget));
	dailyFrequencyLayout->addStretch(1);
	
	QVBoxLayout *weeklyRuleLayout = new QVBoxLayout(weeklyRuleWidget, 0, 6);
	QHBoxLayout *weeklyFrequencyLayout = new QHBoxLayout(weeklyRuleLayout);
	weeklyFrequencyLayout->addWidget(new QLabel(i18n("Recur every"), weeklyRuleWidget));
	weeklyFrequencyEdit = new QSpinBox(1, 9999, 1, weeklyRuleWidget);
	weeklyFrequencyLayout->addWidget(weeklyFrequencyEdit);
	weeklyFrequencyLayout->addWidget(new QLabel(i18n("week(s) on:"), weeklyRuleWidget));
	weeklyFrequencyLayout->addStretch(1);
	QHBoxLayout *weeklyDaysLayout = new QHBoxLayout(weeklyRuleLayout);
	int weekStart=KGlobal::locale()->weekStartDay();
	const KCalendarSystem *calSys = KGlobal::locale()->calendar();
	for(int i = 0; i < 7; ++i ) {
		weeklyButtons[i] = new QCheckBox(calSys->weekDayName(i + 1, true), weeklyRuleWidget);
	}
	for(int i = weekStart - 1; i < 7; ++i ) {
		weeklyDaysLayout->addWidget(weeklyButtons[i]);
	}
	for(int i = 0; i < weekStart - 1; ++i ) {
		weeklyDaysLayout->addWidget(weeklyButtons[i]);
	}

	QVBoxLayout *monthlyRuleLayout = new QVBoxLayout(monthlyRuleWidget, 0, 6);
	QHBoxLayout *monthlyFrequencyLayout = new QHBoxLayout(monthlyRuleLayout);
	monthlyFrequencyLayout->addWidget(new QLabel(i18n("Recur every"), monthlyRuleWidget));
	monthlyFrequencyEdit = new QSpinBox(1, 9999, 1, monthlyRuleWidget);
	monthlyFrequencyLayout->addWidget(monthlyFrequencyEdit);
	monthlyFrequencyLayout->addWidget(new QLabel(i18n("month(s), after the start month"), monthlyRuleWidget));
	monthlyFrequencyLayout->addStretch(1);
	QButtonGroup *monthlyButtonGroup = new QButtonGroup(monthlyRuleWidget);
	monthlyButtonGroup->setFrameStyle(QFrame::NoFrame);
	monthlyRuleLayout->addWidget(monthlyButtonGroup, 1, AlignVCenter);
	QGridLayout *monthlyButtonLayout = new QGridLayout(monthlyButtonGroup, 3, 2, 0, 6);
	monthlyOnDayButton = new QRadioButton(i18n("Recur on the"), monthlyButtonGroup);
	monthlyOnDayButton->setChecked(true);
	monthlyButtonLayout->addWidget(monthlyOnDayButton, 0, 0);
	QBoxLayout *monthlyDayLayout = new QHBoxLayout();
	monthlyDayCombo = new QComboBox(monthlyButtonGroup);
	monthlyDayCombo->setSizeLimit(7);
	monthlyDayCombo->insertItem(i18n("1st"));
	monthlyDayCombo->insertItem(i18n("2nd"));
	monthlyDayCombo->insertItem(i18n("3rd"));
	monthlyDayCombo->insertItem(i18n("4th"));
	monthlyDayCombo->insertItem(i18n("5th"));
	monthlyDayCombo->insertItem(i18n("6th"));
	monthlyDayCombo->insertItem(i18n("7th"));
	monthlyDayCombo->insertItem(i18n("8th"));
	monthlyDayCombo->insertItem(i18n("9th"));
	monthlyDayCombo->insertItem(i18n("10th"));
	monthlyDayCombo->insertItem(i18n("11th"));
	monthlyDayCombo->insertItem(i18n("12th"));
	monthlyDayCombo->insertItem(i18n("13th"));
	monthlyDayCombo->insertItem(i18n("14th"));
	monthlyDayCombo->insertItem(i18n("15th"));
	monthlyDayCombo->insertItem(i18n("16th"));
	monthlyDayCombo->insertItem(i18n("17th"));
	monthlyDayCombo->insertItem(i18n("18th"));
	monthlyDayCombo->insertItem(i18n("19th"));
	monthlyDayCombo->insertItem(i18n("20th"));
	monthlyDayCombo->insertItem(i18n("21st"));
	monthlyDayCombo->insertItem(i18n("22nd"));
	monthlyDayCombo->insertItem(i18n("23rd"));
	monthlyDayCombo->insertItem(i18n("24th"));
	monthlyDayCombo->insertItem(i18n("25th"));
	monthlyDayCombo->insertItem(i18n("26th"));
	monthlyDayCombo->insertItem(i18n("27th"));
	monthlyDayCombo->insertItem(i18n("28th"));
	monthlyDayCombo->insertItem(i18n("29th"));
	monthlyDayCombo->insertItem(i18n("30th"));
	monthlyDayCombo->insertItem(i18n("31st"));
	monthlyDayCombo->insertItem(i18n("Last"));
	monthlyDayCombo->insertItem(i18n("2nd Last"));
	monthlyDayCombo->insertItem(i18n("3rd Last"));
	monthlyDayCombo->insertItem(i18n("4th Last"));
	monthlyDayCombo->insertItem(i18n("5th Last"));
	monthlyDayLayout->addWidget(monthlyDayCombo);
	monthlyDayLayout->addWidget(new QLabel(i18n("day"), monthlyButtonGroup));
	monthlyWeekendCombo = new QComboBox(monthlyButtonGroup);
	monthlyWeekendCombo->insertItem(i18n("possibly on weekend"));
	monthlyWeekendCombo->insertItem(i18n("but before weekend"));
	monthlyWeekendCombo->insertItem(i18n("but after weekend"));
	monthlyDayLayout->addWidget(monthlyWeekendCombo);
	monthlyDayLayout->addStretch(1);
	monthlyButtonLayout->addLayout(monthlyDayLayout, 0, 1);
	monthlyOnDayOfWeekButton = new QRadioButton(i18n("Recur on the"), monthlyButtonGroup);
	monthlyButtonLayout->addWidget(monthlyOnDayOfWeekButton, 1, 0);
	QBoxLayout *monthlyWeekLayout = new QHBoxLayout();
	monthlyWeekCombo = new QComboBox(monthlyButtonGroup);
	monthlyWeekCombo->insertItem(i18n("1st"));
	monthlyWeekCombo->insertItem(i18n("2nd"));
	monthlyWeekCombo->insertItem(i18n("3rd"));
	monthlyWeekCombo->insertItem(i18n("4th"));
	monthlyWeekCombo->insertItem(i18n("5th"));
	monthlyWeekCombo->insertItem(i18n("Last"));
	monthlyWeekCombo->insertItem(i18n("2nd Last"));
	monthlyWeekCombo->insertItem(i18n("3rd Last"));
	monthlyWeekCombo->insertItem(i18n("4th Last"));
	monthlyWeekCombo->insertItem(i18n("5th Last"));
	monthlyWeekLayout->addWidget(monthlyWeekCombo);
	monthlyDayOfWeekCombo = new QComboBox(monthlyButtonGroup);
	monthlyWeekLayout->addWidget(monthlyDayOfWeekCombo);
	monthlyWeekLayout->addStretch(1);
	monthlyButtonLayout->addLayout(monthlyWeekLayout, 1, 1);

	QVBoxLayout *yearlyRuleLayout = new QVBoxLayout(yearlyRuleWidget, 0, 6);
	QHBoxLayout *yearlyFrequencyLayout = new QHBoxLayout(yearlyRuleLayout);
	yearlyFrequencyLayout->addWidget(new QLabel(i18n("Recur every"), yearlyRuleWidget));
	yearlyFrequencyEdit = new QSpinBox(1, 9999, 1, yearlyRuleWidget);
	yearlyFrequencyLayout->addWidget(yearlyFrequencyEdit);
	yearlyFrequencyLayout->addWidget(new QLabel(i18n("year(s), after the start year"), yearlyRuleWidget));
	yearlyFrequencyLayout->addStretch(1);
	QButtonGroup *yearlyButtonGroup = new QButtonGroup(yearlyRuleWidget);
	yearlyButtonGroup->setFrameStyle(QFrame::NoFrame);
	yearlyRuleLayout->addWidget(yearlyButtonGroup, 1, AlignVCenter);
	QGridLayout *yearlyButtonLayout = new QGridLayout(yearlyButtonGroup, 3, 2, 0, 6);
	yearlyOnDayOfMonthButton = new QRadioButton(i18n("part before XXX of 'Recur on day XXX of month YYY'", "Recur on day"), yearlyButtonGroup);
	yearlyOnDayOfMonthButton->setChecked(true);
	yearlyButtonLayout->addWidget(yearlyOnDayOfMonthButton, 0, 0);
	QBoxLayout *yearlyMonthLayout = new QHBoxLayout();
	yearlyDayOfMonthEdit = new QSpinBox(1, 31, 1, yearlyButtonGroup);
	yearlyMonthLayout->addWidget(yearlyDayOfMonthEdit);
	yearlyMonthLayout->addWidget(new QLabel(i18n("part between XXX and YYY of 'Recur on day XXX of month YYY'", "of"), yearlyButtonGroup));
	yearlyMonthCombo = new QComboBox(yearlyButtonGroup);
	yearlyMonthLayout->addWidget(yearlyMonthCombo);
	yearlyWeekendCombo_month = new QComboBox(yearlyButtonGroup);
	yearlyWeekendCombo_month->insertItem(i18n("possibly on weekend"));
	yearlyWeekendCombo_month->insertItem(i18n("but before weekend"));
	yearlyWeekendCombo_month->insertItem(i18n("but after weekend"));
	yearlyMonthLayout->addWidget(yearlyWeekendCombo_month);
	yearlyMonthLayout->addStretch(1);
	yearlyButtonLayout->addLayout(yearlyMonthLayout, 0, 1);
	yearlyOnDayOfWeekButton = new QRadioButton(i18n("Part before NNN in 'Recur on the NNN. WEEKDAY of MONTH'", "On the"), yearlyButtonGroup);
	yearlyButtonLayout->addWidget(yearlyOnDayOfWeekButton, 1, 0);
	QBoxLayout *yearlyWeekLayout = new QHBoxLayout();
	yearlyWeekCombo = new QComboBox(yearlyButtonGroup);
	yearlyWeekCombo->insertItem(i18n("1st"));
	yearlyWeekCombo->insertItem(i18n("2nd"));
	yearlyWeekCombo->insertItem(i18n("3rd"));
	yearlyWeekCombo->insertItem(i18n("4th"));
	yearlyWeekCombo->insertItem(i18n("5th"));
	yearlyWeekCombo->insertItem(i18n("Last"));
	yearlyWeekCombo->insertItem(i18n("2nd Last"));
	yearlyWeekCombo->insertItem(i18n("3rd Last"));
	yearlyWeekCombo->insertItem(i18n("4th Last"));
	yearlyWeekCombo->insertItem(i18n("5th Last"));
	yearlyWeekLayout->addWidget(yearlyWeekCombo);
	yearlyDayOfWeekCombo = new QComboBox(yearlyButtonGroup);
	yearlyWeekLayout->addWidget(yearlyDayOfWeekCombo);
	yearlyWeekLayout->addWidget(new QLabel(i18n("part between WEEKDAY and MONTH in 'Recur on NNN. WEEKDAY of MONTH'", "of"), yearlyButtonGroup));
	yearlyMonthCombo_week = new QComboBox(yearlyButtonGroup);
	yearlyWeekLayout->addWidget(yearlyMonthCombo_week);
	yearlyWeekLayout->addStretch(1);
	yearlyButtonLayout->addLayout(yearlyWeekLayout, 1, 1);
	yearlyOnDayOfYearButton = new QRadioButton(i18n("Recur on day #"), yearlyButtonGroup);
	yearlyButtonLayout->addWidget(yearlyOnDayOfYearButton, 2, 0);
	QBoxLayout *yearlyDayLayout = new QHBoxLayout();
	yearlyDayOfYearEdit = new QSpinBox(1, 366, 1, yearlyButtonGroup);
	yearlyDayLayout->addWidget(yearlyDayOfYearEdit);
	yearlyDayLayout->addWidget(new QLabel(i18n("part after NNN of 'Recur on day #NNN of the year'", " of the year"), yearlyButtonGroup));
	yearlyWeekendCombo_day = new QComboBox(yearlyButtonGroup);
	yearlyWeekendCombo_day->insertItem(i18n("possibly on weekend"));
	yearlyWeekendCombo_day->insertItem(i18n("but before weekend"));
	yearlyWeekendCombo_day->insertItem(i18n("but after weekend"));
	yearlyDayLayout->addWidget(yearlyWeekendCombo_day);
	yearlyDayLayout->addStretch(1);
	yearlyButtonLayout->addLayout(yearlyDayLayout, 2, 1);

	recurrenceLayout->addWidget(ruleGroup);
	
	QHBoxLayout *buttonLayout = new QHBoxLayout(recurrenceLayout);
	rangeButton = new KPushButton(i18n("Range..."), this);
	buttonLayout->addWidget(rangeButton);
	exceptionsButton = new KPushButton(i18n("Exceptions..."), this);
	buttonLayout->addWidget(exceptionsButton);

	recurrenceLayout->addStretch(1);

	ruleStack->raiseWidget(2);
	recurrenceButton->setChecked(false);
	ruleGroup->setEnabled(false);
	rangeButton->setEnabled(false);
	exceptionsButton->setEnabled(false);

	rangeDialog = new EditRangeDialog(date, this);
	rangeDialog->hide();

	exceptionsDialog = new EditExceptionsDialog(this);
	exceptionsDialog->hide();

	int months = 0;
	QDate date = QDate::currentDate();
	for(int i = 0; i < 10; i++) {
		int months2 = calSys->monthsInYear(date);
		if(months2 > months) {
			int year = calSys->year(date);
			for(int i2 = months + 1; i2 <= months2; i2++) {
				yearlyMonthCombo_week->insertItem(calSys->monthName(i2, year));
				yearlyMonthCombo->insertItem(calSys->monthName(i2, year));
			}
			months = months2;
		}
		date = addYears(date, 1);
	}
	for(int i = 1; i <= 7; i++) {
		yearlyDayOfWeekCombo->insertItem(calSys->weekDayName(i));
		monthlyDayOfWeekCombo->insertItem(calSys->weekDayName(i));
	}

	connect(typeCombo, SIGNAL(activated(int)), ruleStack, SLOT(raiseWidget(int)));
	connect(rangeButton, SIGNAL(clicked()), this, SLOT(editRange()));
	connect(exceptionsButton, SIGNAL(clicked()), this, SLOT(editExceptions()));
	connect(recurrenceButton, SIGNAL(toggled(bool)), ruleGroup, SLOT(setEnabled(bool)));
	connect(recurrenceButton, SIGNAL(toggled(bool)), rangeButton, SLOT(setEnabled(bool)));
	connect(recurrenceButton, SIGNAL(toggled(bool)), exceptionsButton, SLOT(setEnabled(bool)));
	
}

RecurrenceEditWidget::~RecurrenceEditWidget() {
}

void RecurrenceEditWidget::editExceptions() {
	exceptionsDialog->exec();
	exceptionsDialog->hide();
}
void RecurrenceEditWidget::editRange() {
	rangeDialog->exec();
	rangeDialog->hide();
}

void RecurrenceEditWidget::setRecurrence(Recurrence *rec) {
	rangeDialog->setRecurrence(rec);
	exceptionsDialog->setRecurrence(rec);
	if(!rec) {
		recurrenceButton->setChecked(false);
		recurrenceButton->setChecked(false);
		ruleGroup->setEnabled(false);
		rangeButton->setEnabled(false);
		exceptionsButton->setEnabled(false);
		return;
	}
	switch(rec->type()) {
		case RECURRENCE_TYPE_DAILY: {
			DailyRecurrence *drec = (DailyRecurrence*) rec;
			dailyFrequencyEdit->setValue(drec->frequency());
			typeCombo->setCurrentItem(0);
			break;
		}
		case RECURRENCE_TYPE_WEEKLY: {
			WeeklyRecurrence *wrec = (WeeklyRecurrence*) rec;
			weeklyFrequencyEdit->setValue(wrec->frequency());
			for(int i = 0; i < 7; i++) {
				weeklyButtons[i]->setChecked(wrec->dayOfWeek(i + 1));
			}
			typeCombo->setCurrentItem(1);
			break;
		}
		case RECURRENCE_TYPE_MONTHLY: {
			MonthlyRecurrence *mrec = (MonthlyRecurrence*) rec;
			monthlyFrequencyEdit->setValue(mrec->frequency());
			if(mrec->dayOfWeek() > 0) {
				monthlyOnDayOfWeekButton->setChecked(true);
				monthlyDayOfWeekCombo->setCurrentItem(mrec->dayOfWeek() - 1);
				int week = mrec->week();
				if(week <= 0) week = 6 - week;
				monthlyWeekCombo->setCurrentItem(week - 1);
			} else {
				monthlyOnDayButton->setChecked(true);
				int day = mrec->day();
				if(day <= 0) day = 32 - day;
				monthlyDayCombo->setCurrentItem(day - 1);
				monthlyWeekendCombo->setCurrentItem(mrec->weekendHandling());
			}
			typeCombo->setCurrentItem(2);
			break;
		}
		case RECURRENCE_TYPE_YEARLY: {
			YearlyRecurrence *yrec = (YearlyRecurrence*) rec;
			yearlyFrequencyEdit->setValue(yrec->frequency());
			if(yrec->dayOfYear() > 0) {
				yearlyOnDayOfYearButton->setChecked(true);
				yearlyDayOfYearEdit->setValue(yrec->dayOfYear());
				yearlyWeekendCombo_day->setCurrentItem(yrec->weekendHandling());
				yearlyWeekendCombo_month->setCurrentItem(yrec->weekendHandling());
			} else if(yrec->dayOfWeek() > 0) {
				yearlyOnDayOfWeekButton->setChecked(true);
				yearlyDayOfWeekCombo->setCurrentItem(yrec->dayOfWeek() - 1);
				int week = yrec->week();
				if(week <= 0) week = 6 - week;
				yearlyWeekCombo->setCurrentItem(week - 1);
				yearlyMonthCombo_week->setCurrentItem(yrec->month() - 1);
			} else {
				yearlyOnDayOfMonthButton->setChecked(true);
				yearlyDayOfMonthEdit->setValue(yrec->dayOfMonth());
				yearlyMonthCombo->setCurrentItem(yrec->month() - 1);
				yearlyWeekendCombo_day->setCurrentItem(yrec->weekendHandling());
				yearlyWeekendCombo_month->setCurrentItem(yrec->weekendHandling());
			}
			typeCombo->setCurrentItem(3);
			break;
		}
	}
	ruleStack->raiseWidget(typeCombo->currentItem());
	recurrenceButton->setChecked(true);
	ruleGroup->setEnabled(true);
	rangeButton->setEnabled(true);
	exceptionsButton->setEnabled(true);
}
Recurrence *RecurrenceEditWidget::createRecurrence() {
	if(!recurrenceButton->isChecked() || !validValues()) return NULL;
	switch(typeCombo->currentItem()) {
		case 0: {
			DailyRecurrence *rec = new DailyRecurrence(budget);
			rec->set(date, rangeDialog->endDate(), dailyFrequencyEdit->value(), rangeDialog->fixedCount());
			exceptionsDialog->modifyExceptions(rec);
			return rec;
		}
		case 1: {
			WeeklyRecurrence *rec = new WeeklyRecurrence(budget);
			rec->set(date, rangeDialog->endDate(), weeklyButtons[0]->isChecked(), weeklyButtons[1]->isChecked(), weeklyButtons[2]->isChecked(), weeklyButtons[3]->isChecked(), weeklyButtons[4]->isChecked(), weeklyButtons[5]->isChecked(), weeklyButtons[6]->isChecked(), weeklyFrequencyEdit->value(), rangeDialog->fixedCount());
			exceptionsDialog->modifyExceptions(rec);
			return rec;
		}
		case 2: {
			MonthlyRecurrence *rec = new MonthlyRecurrence(budget);
			if(monthlyOnDayButton->isChecked()) {
				int day = monthlyDayCombo->currentItem() + 1;
				if(day > 31) day = 32 - day;
				rec->setOnDay(date, rangeDialog->endDate(), day, (WeekendHandling) monthlyWeekendCombo->currentItem(), monthlyFrequencyEdit->value(), rangeDialog->fixedCount());
			} else {
				int week = monthlyWeekCombo->currentItem() + 1;
				if(week > 5) week = 6 - week;
				rec->setOnDayOfWeek(date, rangeDialog->endDate(), monthlyDayOfWeekCombo->currentItem() + 1, week, monthlyFrequencyEdit->value(), rangeDialog->fixedCount());
			}
			exceptionsDialog->modifyExceptions(rec);
			return rec;
		}
		case 3: {
			YearlyRecurrence *rec = new YearlyRecurrence(budget);
			if(yearlyOnDayOfMonthButton->isChecked()) {
				rec->setOnDayOfMonth(date, rangeDialog->endDate(), yearlyMonthCombo->currentItem() + 1, yearlyDayOfMonthEdit->value(), (WeekendHandling) yearlyWeekendCombo_month->currentItem(), yearlyFrequencyEdit->value(), rangeDialog->fixedCount());
			} else if(yearlyOnDayOfWeekButton->isChecked()) {
				int week = yearlyWeekCombo->currentItem() + 1;
				if(week > 5) week = 6 - week;
				rec->setOnDayOfWeek(date, rangeDialog->endDate(), yearlyMonthCombo_week->currentItem() + 1, yearlyDayOfWeekCombo->currentItem() + 1, week, yearlyFrequencyEdit->value(), rangeDialog->fixedCount());
			} else {
				rec->setOnDayOfYear(date, rangeDialog->endDate(), yearlyDayOfYearEdit->value(), (WeekendHandling) yearlyWeekendCombo_day->currentItem(), yearlyFrequencyEdit->value(), rangeDialog->fixedCount());
			}
			exceptionsDialog->modifyExceptions(rec);
			return rec;
		}
	}
	return NULL;
}
bool RecurrenceEditWidget::validValues() {
	if(!recurrenceButton->isChecked()) return true;
	switch(typeCombo->currentItem()) {
		case 0: {
			break;
		}
		case 1: {
			bool b = false;
			for(int i = 0; i < 7; i++) {
				if(weeklyButtons[i]->isChecked()) {
					b = true;
					break;
				}
			}
			if(!b) {
				KMessageBox::error(this, i18n("No day of week selected for weekly recurrence."));
				weeklyButtons[0]->setFocus();
				return false;
			}
			break;
		}
		case 2: {
			if(KGlobal::locale()->calendarType() == "gregorian") {
				int i_frequency = monthlyFrequencyEdit->value();
				if(i_frequency % 12 == 0) {
					const KCalendarSystem *calSys = KGlobal::locale()->calendar();
					int i_dayofmonth = monthlyDayCombo->currentItem() + 1;
					int i_month = calSys->month(date);
					if(i_dayofmonth <= 31 && ((i_month == 2 && i_dayofmonth > 29) || (i_dayofmonth > 30 && (i_month == 4 || i_month == 6 || i_month == 9 || i_month == 11)))) {
						KMessageBox::error(this, i18n("Selected day will never occur with selected frequency and start date."));
						monthlyDayCombo->setFocus();
						return false;
					}
				}
			}
			break;
		}
		case 3: {
			const KCalendarSystem *calSys = KGlobal::locale()->calendar();
			if(yearlyOnDayOfMonthButton->isChecked()) {
				int i_frequency = yearlyFrequencyEdit->value();
				int i_dayofmonth = yearlyDayOfMonthEdit->value();
				int i_month = yearlyMonthCombo->currentItem() + 1;
				if(KGlobal::locale()->calendarType() == "gregorian") {
					if((i_month == 2 && i_dayofmonth > 29) || (i_dayofmonth > 30 && (i_month == 4 || i_month == 6 || i_month == 9 || i_month == 11))) {
						KMessageBox::error(this, i18n("Selected day does not exist in selected month."));
						yearlyDayOfMonthEdit->setFocus();
						return false;
					} else if(i_month != 2 || i_dayofmonth < 29) {
						break;
					}
				}
				QDate nextdate;
				calSys->setYMD(nextdate, calSys->year(date), i_month, 1);
				if(i_dayofmonth > calSys->daysInMonth(nextdate)) {
					int i = 10;
					do {
						if(i == 0) {
							KMessageBox::error(this, i18n("Selected day will never occur with selected frequency and start date."));
							yearlyDayOfMonthEdit->setFocus();
							return false;
						}
						nextdate = addYears(nextdate, i_frequency);
						calSys->setYMD(nextdate, calSys->year(nextdate), i_month, 1);
						i--;
					} while(i_dayofmonth > calSys->daysInMonth(nextdate));
				}
			} else if(yearlyOnDayOfYearButton->isChecked()) {
				int i_frequency = yearlyFrequencyEdit->value();
				int i_dayofyear = yearlyDayOfYearEdit->value();
				if(i_dayofyear > calSys->daysInYear(date)) {
					QDate nextdate = date;
					int i = 10;
					do {
						if(i == 0) {
							KMessageBox::error(this, i18n("Selected day will never occur with selected frequency and start date."));
							yearlyDayOfYearEdit->setFocus();
							return false;
						}
						nextdate = addYears(nextdate, i_frequency);
						i--;
					} while(i_dayofyear > calSys->daysInYear(nextdate));
				}
			}
			break;
		}
	}
	if(!rangeDialog->validValues()) return false;
	if(!exceptionsDialog->validValues()) return false;
	return true;
}
void RecurrenceEditWidget::setStartDate(const QDate &startdate) {
	if(!startdate.isValid()) return;
	date = startdate;
	rangeDialog->setStartDate(date);
}

#include "recurrenceeditwidget.moc"

