/* ethos-plugin-info.c
 *
 * Copyright (C) 2009 Christian Hergert <chris@dronelabs.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston MA 
 * 02110-1301 USA
 */

#include "ethos-error.h"
#include "ethos-plugin-info.h"
#include "ethos-plugin-info-private.h"

/**
 * SECTION:ethos-plugin-info
 * @title: EthosPluginInfo
 * @short_description: plugin description during runtime
 *
 * #EthosPluginInfo encapsulates information about a plugin during runtime.  The data for an
 * #EthosPluginInfo is read from a plugin description file such as "my-plugin.example-plugin".
 * This includes information about the plugin module to load, what #EthosPluginLoader should
 * be used, the author, description, and the icon for the plugin.
 *
 * The naming of the plugin description files is derived from the name of the application or
 * the name set using ethos_manager_set_app_name().  It is all lowercase and is named such as
 * "name.appname-plugin" where appname is lowercase.  Therfore, if the application was named
 * "Marina" and the plugin was named "foo", an apt name for the description file would be
 * "foo.marina-plugin".
 *
 * The plugin description file consists of a simple #GKeyFile format.  The header for key-file
 * is also derived from the application name.  Therefore, if the application was named "Marina",
 * the header for the key file would be "[Marina Plugin]".
 *
 * A plugin for an application named "Sample" may look like:
 *
 * |[
 * # example.sample-plugin
 * [Sample Plugin]
 * Name=Example
 * Loader=python
 * Module=example
 * IAge=1
 * Authors=Foo;Bar;Baz
 * Copyright=MIT X11
 * Website=http://example.com
 * ]|
 */

G_DEFINE_TYPE (EthosPluginInfo, ethos_plugin_info, G_TYPE_OBJECT);

enum
{
	LOADED,
	UNLOADED,
	LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0,};

static void
ethos_plugin_info_clear (EthosPluginInfo *plugin_info)
{
	EthosPluginInfoPrivate *priv;
	GList                  *iter;

	g_return_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info));

	priv = plugin_info->priv;

	g_free (priv->name);
	g_free (priv->module);
	g_free (priv->loader);
	g_free (priv->desc);
	g_free (priv->copyright);
	g_free (priv->website);
	g_free (priv->version);
	g_free (priv->deps);
	g_free (priv->icon);
	g_strfreev (priv->authors);

	for (iter = priv->errors; iter; iter = iter->next)
		g_error_free (iter->data);
	g_list_free (priv->errors);

	priv->errors    = NULL;
	priv->name      = NULL;
	priv->module    = NULL;
	priv->loader    = NULL;
	priv->authors   = NULL;
	priv->copyright = NULL;
	priv->website   = NULL;
	priv->version   = NULL;
	priv->deps      = NULL;
	priv->icon      = NULL;
	priv->iage      = 1;
}

static void
ethos_plugin_info_finalize (GObject *object)
{
	ethos_plugin_info_clear (ETHOS_PLUGIN_INFO (object));
	g_free (ETHOS_PLUGIN_INFO (object)->priv->id);
	g_free (ETHOS_PLUGIN_INFO (object)->priv->filename);
	G_OBJECT_CLASS (ethos_plugin_info_parent_class)->finalize (object);
}

enum
{
	PROP_0,
	PROP_ID,
	PROP_FILENAME,
	PROP_NAME,
	PROP_MODULE,
	PROP_DESCRIPTION,
	PROP_LOADER,
	PROP_AUTHORS,
	PROP_COPYRIGHT,
	PROP_WEBSITE,
	PROP_DEPENDENCIES,
	PROP_ICON,
	PROP_ACTIVE,
	PROP_VERSION,
};

static void
ethos_plugin_info_get_property (GObject    *object,
                                guint       property_id,
                                GValue     *value,
                                GParamSpec *pspec)
{
	switch (property_id) {
	case PROP_ID:
		g_value_set_string (value, ethos_plugin_info_get_id (ETHOS_PLUGIN_INFO (object)));
		break;
	case PROP_FILENAME:
		g_value_set_string (value, ethos_plugin_info_get_filename (ETHOS_PLUGIN_INFO (object)));
		break;
	case PROP_NAME:
		g_value_set_string (value, ethos_plugin_info_get_name (ETHOS_PLUGIN_INFO (object)));
		break;
	case PROP_MODULE:
		g_value_set_string (value, ethos_plugin_info_get_module (ETHOS_PLUGIN_INFO (object)));
		break;
	case PROP_DESCRIPTION:
		g_value_set_string (value, ethos_plugin_info_get_description (ETHOS_PLUGIN_INFO (object)));
		break;
	case PROP_LOADER:
		g_value_set_string (value, ethos_plugin_info_get_loader (ETHOS_PLUGIN_INFO (object)));
		break;
	case PROP_AUTHORS:
		g_value_set_boxed (value, ethos_plugin_info_get_authors (ETHOS_PLUGIN_INFO (object)));
		break;
	case PROP_COPYRIGHT:
		g_value_set_string (value, ethos_plugin_info_get_copyright (ETHOS_PLUGIN_INFO (object)));
		break;
	case PROP_WEBSITE:
		g_value_set_string (value, ethos_plugin_info_get_website (ETHOS_PLUGIN_INFO (object)));
		break;
	case PROP_VERSION:
		g_value_set_string (value, ethos_plugin_info_get_version (ETHOS_PLUGIN_INFO (object)));
		break;
	case PROP_DEPENDENCIES:
		g_value_set_string (value, ethos_plugin_info_get_dependencies (ETHOS_PLUGIN_INFO (object)));
		break;
	case PROP_ICON:
		g_value_set_string (value, ethos_plugin_info_get_icon_name (ETHOS_PLUGIN_INFO (object)));
		break;
	case PROP_ACTIVE:
		g_value_set_boolean (value, ethos_plugin_info_get_active (ETHOS_PLUGIN_INFO (object)));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
	}
}

static void
ethos_plugin_info_set_property (GObject      *object,
                                guint         property_id,
                                const GValue *value,
                                GParamSpec   *pspec)
{
	switch (property_id) {
	case PROP_ID:
		ethos_plugin_info_set_id (ETHOS_PLUGIN_INFO (object),
		                          g_value_get_string (value));
		break;
	case PROP_FILENAME:
		ethos_plugin_info_set_filename (ETHOS_PLUGIN_INFO (object),
		                                g_value_get_string (value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
	}
}

static void
ethos_plugin_info_class_init (EthosPluginInfoClass *klass)
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS (klass);
	object_class->finalize = ethos_plugin_info_finalize;
	object_class->get_property = ethos_plugin_info_get_property;
	object_class->set_property = ethos_plugin_info_set_property;
	g_type_class_add_private (object_class, sizeof(EthosPluginInfoPrivate));

	/**
	 * EthosPluginInfo:id:
	 *
	 * Unique identifier for the plugin.
	 */
	g_object_class_install_property (object_class,
	                                 PROP_ID,
	                                 g_param_spec_string ("id",
	                                                      "Id",
	                                                      "Plugin Identifier",
	                                                      NULL,
	                                                      G_PARAM_READWRITE));

	/**
	 * EthosPluginInfo:filename:
	 *
	 * The filename that was used to load the plugin.
	 */
	g_object_class_install_property (object_class,
	                                 PROP_FILENAME,
	                                 g_param_spec_string ("filename",
	                                                      "Filename",
	                                                      "Plugin filename",
	                                                      NULL,
	                                                      G_PARAM_READWRITE));

	/**
	 * EthosPluginInfo:name:
	 *
	 * The name of the plugin.
	 */
	g_object_class_install_property (object_class,
	                                 PROP_NAME,
	                                 g_param_spec_string ("name",
	                                                      "Name",
	                                                      "Plugin Name",
	                                                      NULL,
	                                                      G_PARAM_READABLE));

	/**
	 * EthosPluginInfo:description:
	 *
	 * A short description of the plugin.
	 */
	g_object_class_install_property (object_class,
	                                 PROP_DESCRIPTION,
	                                 g_param_spec_string ("description",
	                                                      "Description",
	                                                      "Plugin Description",
	                                                      NULL,
	                                                      G_PARAM_READABLE));

	/**
	 * EthosPluginInfo:loader:
	 *
	 * The name of the #EthosPluginLoader that should be used to load the plugin.
	 */
	g_object_class_install_property (object_class,
	                                 PROP_LOADER,
	                                 g_param_spec_string ("loader",
	                                                      "Loader",
	                                                      "Plugin Loader",
	                                                      NULL,
	                                                      G_PARAM_READABLE));

	/**
	 * EthosPluginInfo:module:
	 *
	 * The name of the module that should be used to load the plugin.
	 */
	g_object_class_install_property (object_class,
	                                 PROP_MODULE,
	                                 g_param_spec_string ("module",
	                                                      "Module",
	                                                      "External module to load",
	                                                      NULL,
	                                                      G_PARAM_READABLE));

	/**
	 * EthosPluginInfo:authors:
	 *
	 * The list of authors that have contributed to the plugin.
	 */
	g_object_class_install_property (object_class,
	                                 PROP_AUTHORS,
	                                 g_param_spec_boxed ("authors",
	                                                     "Authors",
	                                                     "Plugin Authors",
	                                                     G_TYPE_STRV,
	                                                     G_PARAM_READABLE));

	/**
	 * EthosPluginInfo:copyright:
	 *
	 * The copyright the plugin is released under such as "GPL-2".
	 */
	g_object_class_install_property (object_class,
	                                 PROP_COPYRIGHT,
	                                 g_param_spec_string ("copyright",
	                                                      "Copyright",
	                                                      "Plugin Copyright",
	                                                      NULL,
	                                                      G_PARAM_READABLE));

	/**
	 * EthosPluginInfo:website:
	 *
	 * The website to find more information on the plugin.
	 */
	g_object_class_install_property (object_class,
	                                 PROP_WEBSITE,
	                                 g_param_spec_string ("website",
	                                                      "Website",
	                                                      "Plugin Website",
	                                                      NULL,
	                                                      G_PARAM_READABLE));

	/**
	 * EthosPluginInfo:version:
	 *
	 * The version of the plugin.  See ethos_plugin_info_get_version().
	 */
	g_object_class_install_property (object_class,
	                                 PROP_VERSION,
	                                 g_param_spec_string ("version",
	                                                      "Version",
	                                                      "Plugin Version",
	                                                      NULL,
	                                                      G_PARAM_READABLE));

	/**
	 * EthosPluginInfo:dependencies:
	 *
	 * Dependencies that must be loaded for the plugin to load.  See
	 * ethos_plugin_info_get_dependencies().
	 */
	g_object_class_install_property (object_class,
	                                 PROP_DEPENDENCIES,
	                                 g_param_spec_string ("dependencies",
	                                                      "Dependencies",
	                                                      "Plugin Dependencies",
	                                                      NULL,
	                                                      G_PARAM_READABLE));

	/**
	 * EthosPluginInfo:icon-name:
	 *
	 * The name of the icon to display in the manager widgets.
	 */
	g_object_class_install_property (object_class,
	                                 PROP_ICON,
	                                 g_param_spec_string ("icon-name",
	                                                      "IconName",
	                                                      "Plugin Icon Name",
	                                                      NULL,
	                                                      G_PARAM_READABLE));

	/**
	 * EthosPluginInfo:active:
	 *
	 * Whether or not the plugin is currently loaded.
	 */
	g_object_class_install_property (object_class,
	                                 PROP_ACTIVE,
	                                 g_param_spec_boolean ("active",
	                                                       "Active",
	                                                       "If the plugin has been activated",
	                                                       FALSE,
	                                                       G_PARAM_READABLE));

	/**
	 * EthosPluginInfo::loaded:
	 * @plugin_info: An #EthosPluginInfo
	 *
	 * Signal emitted when a new plugin is loaded.
	 */
	signals [LOADED]
		= g_signal_new (g_intern_static_string ("loaded"),
		                G_TYPE_FROM_CLASS (klass),
		                G_SIGNAL_RUN_FIRST,
		                0,
		                NULL, NULL,
		                g_cclosure_marshal_VOID__VOID,
		                G_TYPE_NONE,
		                0);

	/**
	 * EthosPluginInfo::unloaded:
	 * @plugin_info: An #EthosPluginInfo
	 *
	 * Signal emitted when the plugin is unloaded.
	 */
	signals [UNLOADED]
		= g_signal_new (g_intern_static_string ("unloaded"),
		                G_TYPE_FROM_CLASS (klass),
		                G_SIGNAL_RUN_FIRST,
		                0,
		                NULL, NULL,
		                g_cclosure_marshal_VOID__VOID,
		                G_TYPE_NONE,
		                0);
}

static void
ethos_plugin_info_init (EthosPluginInfo *plugin_info)
{
	plugin_info->priv = G_TYPE_INSTANCE_GET_PRIVATE (plugin_info,
	                                                 ETHOS_TYPE_PLUGIN_INFO,
							 EthosPluginInfoPrivate);
}

/**
 * ethos_plugin_info_new:
 *
 * Creates a new #EthosPluginInfo instance.
 *
 * Return value: The newly created #EthosPluginInfo instance.
 */
EthosPluginInfo*
ethos_plugin_info_new (void)
{
	return g_object_new (ETHOS_TYPE_PLUGIN_INFO, NULL);
}

/**
 * ethos_plugin_info_load_from_key_file:
 * @plugin_info: An #EthosPluginInfo
 * @group: the name of the group to load
 * @key_file: A #GKeyFile
 * @error: A location for a #GError or %NULL
 *
 * Loads the plugin information from @key_file.  The information is loaded from the group named
 * @group within the keyfile.  Therfore, if @group where "Sample Plugin" then the keyfile should
 * have a group header such as "[Sample Plugin]" within it.
 *
 * Upon failure, @error is set and %FALSE is returned.
 *
 * Return value: %TRUE on success
 */
gboolean
ethos_plugin_info_load_from_key_file (EthosPluginInfo  *plugin_info,
                                      const gchar      *group,
                                      GKeyFile         *key_file,
				      GError          **error)
{
	EthosPluginInfoPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), FALSE);
	g_return_val_if_fail (key_file != NULL, FALSE);

	priv = plugin_info->priv;

	ethos_plugin_info_clear (plugin_info);

	if (!g_key_file_has_group (key_file, group)) {
		g_set_error (error, ETHOS_ERROR, ETHOS_ERROR_INVALID_KEY_FILE,
		             "Could not find group %s in key file", group);
		return FALSE;
	}

	priv->loader    = g_key_file_get_string  (key_file, group, "Loader", NULL);
	priv->authors   = g_key_file_get_string_list
	                                         (key_file, group, "Authors", NULL, NULL);
	priv->desc      = g_key_file_get_locale_string  (key_file, group, "Description", NULL, NULL);
	priv->copyright = g_key_file_get_string  (key_file, group, "Copyright", NULL);
	priv->website   = g_key_file_get_string  (key_file, group, "Website", NULL);
	priv->deps      = g_key_file_get_string  (key_file, group, "Dependencies", NULL);
	priv->name      = g_key_file_get_locale_string  (key_file, group, "Name", NULL, NULL);
	priv->module    = g_key_file_get_string  (key_file, group, "Module", NULL);
	priv->icon      = g_key_file_get_string  (key_file, group, "Icon", NULL);
	priv->version   = g_key_file_get_string  (key_file, group, "Version", NULL);
	priv->iage      = g_key_file_get_integer (key_file, group, "IAge", NULL);

	if (!priv->name) {
		g_set_error (error, ETHOS_ERROR, ETHOS_ERROR_INVALID_KEY_FILE,
		             "No \"Name\" in key file");
		return FALSE;
	}
	else if (!priv->module) {
		g_set_error (error, ETHOS_ERROR, ETHOS_ERROR_INVALID_KEY_FILE,
		             "No \"Module\" in key file");
		return FALSE;
	}

	return TRUE;
}

/**
 * ethos_plugin_info_load_from_file:
 * @plugin_info: An #EthosPluginInfo
 * @group: the keyfile group to extract information from
 * @filename: A path to the file containing the plugin info
 * @error: A location for a #GError or %NULL
 *
 * Loads the plugin information found within @filename.  If there is an
 * error, %FALSE is returned and @error will be set.
 *
 * Return value: %TRUE if the data was loaded successfully
 */
gboolean
ethos_plugin_info_load_from_file (EthosPluginInfo  *plugin_info,
                                  const gchar      *group,
                                  const gchar      *filename,
                                  GError          **error)
{
	GKeyFile      *key_file;
	gboolean       success = FALSE;
	GKeyFileFlags  flags;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), FALSE);

	/* make sure our plugin file exists */
	if (!g_file_test (filename, G_FILE_TEST_IS_REGULAR)) {
		g_set_error (error, ETHOS_ERROR, ETHOS_ERROR_FILE_NOT_FOUND,
		             "File not found: %s", filename);
		return FALSE;
	}

	key_file = g_key_file_new ();
	flags = G_KEY_FILE_KEEP_TRANSLATIONS | G_KEY_FILE_KEEP_COMMENTS;

	if (!g_key_file_load_from_file (key_file, filename, flags, error))
		goto cleanup;

	success = ethos_plugin_info_load_from_key_file (plugin_info,
	                                                group,
	                                                key_file, error);

cleanup:
	g_key_file_free (key_file);

	return success;
}

/**
 * ethos_plugin_info_load_from_data:
 * @plugin_info: A #EthosPluginInfo
 * @group: the group to load information from
 * @data: a buffer of data
 * @length: length of buffer to use
 * @error: A location for a #GError or %NULL
 *
 * Loads the plugin information from a buffer of information. The data
 * should be in #GKeyFile format.
 *
 * Return value: %TRUE on success. @error is set on failure.
 */
gboolean
ethos_plugin_info_load_from_data (EthosPluginInfo  *plugin_info,
                                  const gchar      *group,
                                  const gchar      *data,
				  gsize             length,
				  GError          **error)
{
	GKeyFile      *key_file;
	GKeyFileFlags  flags   = 0;
	gboolean       success = FALSE;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), FALSE);
	g_return_val_if_fail (data != NULL, FALSE);

	key_file = g_key_file_new ();

	if (!g_key_file_load_from_data (key_file, data, length, flags, error))
		goto cleanup;

	success = ethos_plugin_info_load_from_key_file (plugin_info, group,
	                                                key_file, error);

cleanup:
	g_key_file_free (key_file);

	return success;
}

/**
 * ethos_plugin_info_set_id:
 * @plugin_info: An #EthosPluginInfo
 * @id: the id of the plugin
 *
 * Sets the id of the plugin.  This is typically taking from the filename
 * of the description file.
 */
void
ethos_plugin_info_set_id (EthosPluginInfo *plugin_info,
                          const gchar     *id)
{
	EthosPluginInfoPrivate *priv;

	g_return_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info));

	priv = plugin_info->priv;

	g_free (priv->id);
	priv->id = g_strdup (id);
}

/**
 * ethos_plugin_info_get_id:
 * @plugin_info: An #EthosPluginInfo
 *
 * Retreives the unique id of the plugin.
 *
 * Return value: The unique id of the plugin which should not be modified or freed.
 */
G_CONST_RETURN gchar*
ethos_plugin_info_get_id (EthosPluginInfo *plugin_info)
{
	EthosPluginInfoPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), NULL);

	priv = plugin_info->priv;

	return priv->id;
}

/**
 * ethos_plugin_info_get_name:
 * @plugin_info: An #EthosPluginInfo
 *
 * Retrieves the name of the plugin.
 *
 * Return value: The name of the plugin. This value should not be freed or modified.
 */
G_CONST_RETURN gchar*
ethos_plugin_info_get_name (EthosPluginInfo *plugin_info)
{
	EthosPluginInfoPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), NULL);

	priv = plugin_info->priv;

	return priv->name;
}

/**
  * ethos_plugin_info_get_module:
  * @plugin_info: An #EthosPluginInfo
  *
  * Retreives the module that implements an #EthosPlugin.
  *
  * Return value: The module of the plugin. This value should not be
  *   modified or freed.
  */
G_CONST_RETURN gchar*
ethos_plugin_info_get_module (EthosPluginInfo *plugin_info)
{
	EthosPluginInfoPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), NULL);

	priv = plugin_info->priv;

	return priv->module;
}

/**
 * ethos_plugin_info_get_loader:
 * @plugin_info: An #EthosPluginInfo
 *
 * Retrieves the name of the #EthosPluginLoader that should load the plugin.
 *
 * Return value: The loader to use to load the plugin.  This value should
 *   not be modified or freed.
 */
G_CONST_RETURN gchar*
ethos_plugin_info_get_loader (EthosPluginInfo *plugin_info)
{
	EthosPluginInfoPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), NULL);

	priv = plugin_info->priv;

	return priv->loader;
}

/**
 * ethos_plugin_info_get_authors:
 * @plugin_info: An #EthosPluginInfo
 *
 * Retreives the authors of the plugin.
 *
 * Return value: A %NULL terminated array of strings that contain the authors.
 *   This value should not be modified or freed.
 */
gchar**
ethos_plugin_info_get_authors (EthosPluginInfo *plugin_info)
{
	EthosPluginInfoPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), NULL);

	priv = plugin_info->priv;

	return priv->authors;
}

/**
 * ethos_plugin_info_get_description:
 * @plugin_info: An #EthosPluginInfo
 *
 * Retrieves the description of the plugin
 *
 * Return value: A string containing the description.  The string
 *   should not be modified or freed.
 */
G_CONST_RETURN gchar*
ethos_plugin_info_get_description (EthosPluginInfo *plugin_info)
{
	EthosPluginInfoPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), NULL);

	priv = plugin_info->priv;

	return priv->desc;
}

/**
 * ethos_plugin_info_get_copyright:
 * @plugin_info: An #EthosPluginInfo
 *
 * Retrieves the copyright of the plugin
 *
 * Return value: A string containing the copyright.  The string
 *   should not be modified or freed.
 */
G_CONST_RETURN gchar*
ethos_plugin_info_get_copyright (EthosPluginInfo *plugin_info)
{
	EthosPluginInfoPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), NULL);

	priv = plugin_info->priv;

	return priv->copyright;
}

/**
 * ethos_plugin_info_get_website:
 * @plugin_info: An #EthosPluginInfo
 *
 * Retrieves the website url of the plugin
 *
 * Return value: A string containing the website.  The string
 *   should not be modified or freed.
 */
G_CONST_RETURN gchar*
ethos_plugin_info_get_website (EthosPluginInfo *plugin_info)
{
	EthosPluginInfoPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), NULL);

	priv = plugin_info->priv;

	return priv->website;
}

/**
 * ethos_plugin_info_get_iage:
 * @plugin_info: An #EthosPluginInfo
 *
 * Retreives the interface age that the plugin conforms to.
 *
 * Return value: a #guint containing the age.
 */
guint
ethos_plugin_info_get_iage (EthosPluginInfo *plugin_info)
{
	EthosPluginInfoPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), 0);

	priv = plugin_info->priv;

	return priv->iage;
}

/**
 * ethos_plugin_info_get_dependencies:
 * @plugin_info: An #EthosPluginInfo
 *
 * Retrieves the list of dependencies as a string.  The string is a set
 * of checks that must pass before the plugin can be loaded.
 *
 * Such as:
 *
 *  "network >= 0.1 xmpp"
 *
 * Would require the network plugin 0.1 or greater and any version of the
 * xmpp plugin.
 *
 * Return value: A string containing the dependencies or %NULL.
 */
G_CONST_RETURN gchar*
ethos_plugin_info_get_dependencies (EthosPluginInfo *plugin_info)
{
	EthosPluginInfoPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), 0);

	priv = plugin_info->priv;

	return priv->deps;
}

/**
 * ethos_plugin_info_get_active:
 * @plugin_info: An #EthosPluginInfo
 *
 * Retrieves whether or not the plugin is currently activated.
 *
 * Return value: %TRUE if the plugin is currently active
 */
gboolean
ethos_plugin_info_get_active (EthosPluginInfo *plugin_info)
{
	EthosPluginInfoPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), FALSE);

	priv = plugin_info->priv;

	return priv->active;
}

/**
 * ethos_plugin_info_set_active:
 * @plugin_info: An #EthosPluginInfo
 * @active: if the #EthosPluginInfo is active
 *
 * Set the active status of the #EthosPluginInfo.  This is useful so
 * that gui front-ends can keep track if the plugin is currently in use.
 */
void
ethos_plugin_info_set_active (EthosPluginInfo *plugin_info,
                              gboolean         active)
{
	EthosPluginInfoPrivate *priv;

	g_return_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info));

	priv = plugin_info->priv;

	priv->active = active;

	if (active)
		g_signal_emit (plugin_info, signals [LOADED], 0);
	else
		g_signal_emit (plugin_info, signals [UNLOADED], 0);
}

/**
 * ethos_plugin_info_get_filename:
 * @plugin_info: An #EthosPluginInfo
 *
 * Retreives the filename that was used to load the @plugin_info.
 *
 * Return value: A string or %NULL
 */
G_CONST_RETURN gchar*
ethos_plugin_info_get_filename (EthosPluginInfo *plugin_info)
{
	EthosPluginInfoPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), NULL);

	priv = plugin_info->priv;

	return priv->filename;
}

/**
 * ethos_plugin_info_set_filename:
 * @plugin_info: An #EthosPluginInfo
 * @filename: The filename
 *
 * Sets the filename used to load the plugin-info.
 */
void
ethos_plugin_info_set_filename (EthosPluginInfo *plugin_info,
                                const gchar     *filename)
{
	EthosPluginInfoPrivate *priv;

	g_return_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info));

	priv = plugin_info->priv;

	g_free (priv->filename);
	priv->filename = g_strdup (filename);
}

/**
 * ethos_plugin_info_get_version:
 * @plugin_info: An #EthosPluginInfo
 *
 * Retrieves the version of the plugin
 *
 * Return value: the version of the plugin
 */
G_CONST_RETURN gchar*
ethos_plugin_info_get_version (EthosPluginInfo *plugin_info)
{
	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), 0);
	return plugin_info->priv->version;
}

/**
 * ethos_plugin_info_set_version:
 * @plugin_info: An #EthosPluginInfo
 * @version: the version
 *
 * Sets the version of the plugin.
 */
void
ethos_plugin_info_set_version (EthosPluginInfo *plugin_info,
                               const gchar     *version)
{
	EthosPluginInfoPrivate *priv;

	g_return_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info));

	priv = plugin_info->priv;

	g_free (priv->version);
	priv->version = g_strdup (version);
}

/**
 * ethos_plugin_info_get_icon_name:
 * @plugin_info: An #EthosPluginInfo
 *
 * Retrieves the icon field for the #EthosPluginInfo
 *
 * Return value: A string containing the icon name or %NULL. This value
 *   should not be modified or freed.
 */
G_CONST_RETURN gchar*
ethos_plugin_info_get_icon_name (EthosPluginInfo *plugin_info)
{
	EthosPluginInfoPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), NULL);

	priv = plugin_info->priv;

	return priv->icon;
}

/**
 * ethos_plugin_info_has_errors:
 * @plugin_info: An #EthosPluginInfo
 *
 * Checks whether or not the plugin has had any errors.
 *
 * Return value: %TRUE if the plugin had an error loading.
 */
gboolean
ethos_plugin_info_has_errors (EthosPluginInfo *plugin_info)
{
	EthosPluginInfoPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), FALSE);

	priv = plugin_info->priv;

	return priv->errors != NULL;
}

/**
 * ethos_plugin_info_get_errors:
 * @plugin_info: An #EthosPluginInfo
 *
 * Retrieves a list of errors that occurred while trying to load the
 * plugin.  If no errors have occurred, NULL is returned.  The list
 * is a copy however the #GError<!-- -->s contained are not.  You should
 * free the list with g_list_free().
 *
 * Return value: A #GList containing #GError<!-- -->s that should be freed
 *   with g_list_free().
 */
GList*
ethos_plugin_info_get_errors (EthosPluginInfo *plugin_info)
{
	EthosPluginInfoPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), FALSE);

	priv = plugin_info->priv;

	return g_list_copy (priv->errors);
}

/**
 * ethos_plugin_info_add_error:
 * @plugin_info: An #EthosPluginInfo
 * @error: A #GError
 *
 * Adds an error to the list of errors preventing startup of the plugin.
 */
void
ethos_plugin_info_add_error (EthosPluginInfo *plugin_info,
                             const GError    *error)
{
	EthosPluginInfoPrivate *priv;

	g_return_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info));

	priv = plugin_info->priv;

	priv->errors = g_list_append (priv->errors,
	                              g_error_copy (error));
}
