/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: object.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include <fullengine.h>
#include "object.h"
#include "engine.h"
#include "handlemgr.h"
#include "discover.h"
#include "common.h"
#include "volume.h"
#include "internalAPI.h"


int isa_valid_input_object(ADDRESS object,
                           TAG     object_tag,
                           uint    Object_size,
                           ADDRESS object_handle,
                           ADDRESS parameters) {
    int rc = 0;

    storage_object_t * obj = (storage_object_t *) object;

    LOG_PROC_ENTRY();

    if ((object_tag == EVMS_OBJECT_TAG) ||
        (object_tag == REGION_TAG) ||
        (object_tag == SEGMENT_TAG) ||
        (object_tag == DISK_TAG)) {

        /* The object must not be corrupt. */
        if (!(obj->flags & SOFLAG_CORRUPT)) {

            /* The object must be a top level object. */
            if (is_top_object(obj)) {

                /* The object must not insist on being the top object. */
                if (!(obj->flags & SOFLAG_MUST_BE_TOP)) {
                    LOG_DEBUG("Object %s is a valid input object.\n", obj->name);
                } else {
                    LOG_ERROR("Object %s is not a valid input object.  It insists it must be a top level object.\n", obj->name);
                    rc = EINVAL;
                }

            } else {
                LOG_ERROR("Object %s is not a valid input object.  It is not a top level object.\n", obj->name);
                rc = EINVAL;
            }

        } else {
            LOG_ERROR("Object %s is not a valid input object.  It is corrupt.\n", obj->name);
            rc = EINVAL;
        }

    } else {
        switch (object_tag) {
            case PLUGIN_TAG:
                LOG_ERROR("Object is a plug-in.\n");
                break;
            case CONTAINER_TAG:
                LOG_ERROR("Object is a container.\n");
                break;
            case VOLUME_TAG:
                LOG_ERROR("Object is a volume.\n");
                break;
            default:
                LOG_ERROR("Object is of unknown type %d.\n", object_tag);
                break;
        }
        rc = EINVAL;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int delete_object(ADDRESS Object,
                         TAG     ObjectTag,
                         uint    ObjectSize,
                         ADDRESS ObjectHandle,
                         ADDRESS Parameters) {
    int rc = 0;

    storage_object_t * object = (storage_object_t *) Object;
    dlist_t            child_objects;

    LOG_PROC_ENTRY();

    child_objects = CreateList();

    if (child_objects != NULL) {
        /* Tell the plug_in that manages this object to delete it. */

        rc = object->plugin->functions.plugin->delete(object, child_objects);

        /*
         * This is a error condition cleanup function.  We don't care about
         * errors in this function.  Setting an error would abort the looping
         * of ForEachItem() which would terminate the cleanup.  We don't want
         * that to happen, so clear the error code.
         */
        rc = 0;

    } else {
        rc = ENOMEM;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int cleanup_child_objects(ADDRESS Object,
                                 TAG     ObjectTag,
                                 uint    ObjectSize,
                                 ADDRESS ObjectHandle,
                                 ADDRESS Parameters) {
    int rc = 0;

    storage_object_t * parent_object = (storage_object_t *) Object;

    LOG_PROC_ENTRY();

    /*
     * Mark the feature headers of the children dirty so that they will
     * no longer report themselves as the last feature, a.k.a. volume
     * complete.
     */
    ForEachItem(parent_object->child_objects,
                set_feature_header_dirty,
                NULL,
                TRUE);

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


BOOLEAN make_volume_if_necessary(ADDRESS   object,
                                 TAG       object_tag,
                                 uint      object_size,
                                 ADDRESS   object_handle,
                                 ADDRESS   parameters,
                                 BOOLEAN * free_memory,
                                 uint    * error) {
    BOOLEAN result = FALSE;
    int rc = 0;
    storage_object_t * obj = (storage_object_t *) object;
    u_int32_t new_minor = 0;

    LOG_PROC_ENTRY();

    /* Never free item memory. */
    *free_memory = FALSE;

    if (obj->flags & SOFLAG_MUST_BE_VOLUME) {

        /*
         * Only non EVMS objects can insist on being made into a compatibility
         * volume.
         */
        if (obj->object_type != EVMS_OBJECT) {
            char * pVolName = NULL;

            /*
             * Build the compatibility volume name by prepending the EVMS
             * dev node prefix to the object name.
             */
            pVolName = malloc(strlen(EVMS_DEV_NODE_PATH) + strlen(obj->name) + 1);
            if (pVolName != NULL) {
                strcpy(pVolName, EVMS_DEV_NODE_PATH);
                strcat(pVolName, obj->name);

                /*
                 * Reserve a minor number for the volume.  This may not be the
                 * volume's minor number when the kernel discovers it.  This
                 * just helps the Engine avoid collisions with minor numbers.
                 */

                rc = get_compatibility_minor_number(&new_minor);

                if (rc == 0) {
                    rc = make_volume(obj,
                                     pVolName,
                                     new_minor,
                                     VOLFLAG_NEW | VOLFLAG_DIRTY | VOLFLAG_NEEDS_DEV_NODE,
                                     0);

                    if (rc == 0) {
                        /*
                         * We successfully made a volume for the object.
                         * Remove the object from the list.
                         */
                        result = TRUE;
                    }
                }

                free(pVolName);

            } else {
                LOG_CRITICAL("Could not get memory for building a volume name for object %s.\n", obj->name);
                rc = ENOMEM;
            }

        } else {
            /*
             * This is an EVMS object that wants to be made into a compatibility
             * volume.  That is not allowed.  EVMS objects get made into EVMS
             * volumes, and that requires specifying a name, which the plug-in
             * can't supply.  The plug-in is a little touched, so we will touch
             * the SOFLAG_MUST_BE_VOLUME and turn it off.
             */
            obj->flags &= ~SOFLAG_MUST_BE_VOLUME;
        }
    }

    *error = rc;
    LOG_PROC_EXIT_BOOLEAN_INT(result, rc);
    return result;
}


int validate_create_parameters(plugin_handle_t     plugin_handle,
                               handle_array_t   *  objects,
                               plugin_record_t * * ppPlugRec,
                               dlist_t *           pObjectList) {
    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    *ppPlugRec = NULL;
    *pObjectList = NULL;

    rc = translate_handle(plugin_handle,
                          &object,
                          &type);

    if (rc == HANDLE_MANAGER_NO_ERROR) {
        if (type == PLUGIN) {
            plugin_record_t * pPlugRec = (plugin_record_t *) object;
            plugin_type_t plugin_type = GetPluginType(pPlugRec->id);

            *ppPlugRec = pPlugRec;

            if ((plugin_type == EVMS_DEVICE_MANAGER) ||
                (plugin_type == EVMS_SEGMENT_MANAGER) ||
                (plugin_type == EVMS_REGION_MANAGER) ||
                (plugin_type == EVMS_FEATURE) ||
                (plugin_type == EVMS_ASSOCIATIVE_FEATURE)) {
                dlist_t object_list = CreateList();

                if (object_list != NULL) {

                    rc = make_dlist(objects, object_list);

                    if (rc == 0) {

                        /*
                         * Make sure that the input object list contains
                         * top level objects and none of them insist on
                         * being a top object.
                         */
                        rc = ForEachItem(object_list,
                                         isa_valid_input_object,
                                         NULL,
                                         TRUE);
                    }

                    if (rc != DLIST_SUCCESS) {
                        DestroyList(&object_list, FALSE);
                    }

                    *pObjectList = object_list;

                } else {
                    LOG_CRITICAL("Error allocating memory for an output object list.\n");
                    rc = ENOMEM;
                }

            } else {
                LOG_ERROR("The plug-in %s is not a type that manages storage objects.\n", pPlugRec->short_name);
                rc = EINVAL;
            }

        } else {
            LOG_ERROR("The plugin_handle is not for a plug-in.\n");
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


int evms_create(plugin_handle_t    plugin_handle,
                handle_array_t   * objects,
                option_array_t   * options,
                handle_array_t * * output_objects) {
    int rc = 0;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {
        plugin_record_t * pPlugRec = NULL;
        dlist_t object_list = NULL;

        rc = validate_create_parameters(plugin_handle, objects,
                                        &pPlugRec,     &object_list);
        if (rc == 0) {

            /*
             * Make a copy of the input objects list in case we have to
             * clean up after an error.
             */
            dlist_t input_objects = CreateList();

            if (input_objects != NULL) {
                rc = CopyList(input_objects, object_list, AppendToList);

                if (rc == DLIST_SUCCESS) {
                    /*
                     * Make a list to receive the new objects from the
                     * create.
                     */
                    dlist_t new_objects = CreateList();

                    if (new_objects != NULL) {

                        /*
                         * We are finally setup to do the create.
                         */
                        rc = pPlugRec->functions.plugin->create(object_list, options, new_objects);

                        if (rc == 0) {

                            /* Remove any corrupt objects from the list. */
                            PruneList(new_objects,
                                      remove_corrupt_object,
                                      NULL);

                            /*
                             * If any of the objects insist on being made
                             * into volumes, do so.
                             */

                            rc = PruneList(new_objects,
                                           make_volume_if_necessary,
                                           NULL);

                            if (rc == 0) {
                                /*
                                 * Make sure the children of the new
                                 * objects are cleaned up.
                                 */
                                rc = ForEachItem(new_objects,
                                                 cleanup_child_objects,
                                                 NULL,
                                                 TRUE);

                                if (rc == 0) {

                                    /* Mark the new objects' feature
                                     * headers dirty.
                                     */
                                    ForEachItem(new_objects,
                                                set_feature_header_dirty,
                                                NULL,
                                                TRUE);

                                    /*
                                     * Make sure the compatiblity volume names
                                     * are in sync.
                                     */
                                    sync_compatibility_volume_names();

                                    /*
                                     * If the user wants handles for the
                                     * new objects, make the handles.
                                     */
                                    if (output_objects != NULL) {
                                        rc = make_user_handle_array(new_objects, output_objects);
                                    }
                                }
                            }

                            if (rc != 0) {

                                /* Something went wrong after the new
                                 * object(s) was/were created.  Tell the
                                 * plug-in to delete the new objects.
                                 */
                                ForEachItem(new_objects,
                                            delete_object,
                                            NULL,
                                            TRUE);
                            }
                        }

                        DestroyList(&new_objects, FALSE);

                    } else {
                        LOG_CRITICAL("Error getting memory for resulting object list.\n");
                        rc = ENOMEM;
                    }

                } else {
                    LOG_CRITICAL("Error code %d from CopyList when making a copy of the input object list.\n", rc);
                }

                DestroyList(&input_objects, FALSE);

            } else {
                LOG_CRITICAL("Error getting memory for copying the input object list.\n");
                rc = ENOMEM;
            }

            DestroyList(&object_list, FALSE);
        }

        if (rc == 0) {
            changes_pending = TRUE;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int check_for_free_space(ADDRESS object,
                                TAG     object_tag,
                                uint    object_size,
                                ADDRESS object_handle,
                                ADDRESS parameters) {
    int rc = 0;
    storage_object_t * obj = (storage_object_t *) object;

    LOG_PROC_ENTRY();

    if (obj->data_type != FREE_SPACE_TYPE) {
        rc = EINVAL;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * validate_transfer_parameters is a support function for evms_transfer()
 * below.  It does the parameter validation so that evms_transfer() can
 * do the real work of transferring the object.
 */
static int validate_transfer_parameters(object_handle_t         object_handle,
                                        plugin_handle_t         plugin_handle,
                                        object_handle_t         container_handle,
                                        storage_object_t    * * p_object,
                                        plugin_record_t     * * p_plugin,
                                        storage_container_t * * p_container) {
    int rc = 0;
    object_type_t         type;
    storage_object_t    * object = NULL;
    plugin_record_t     * plugin = NULL;
    storage_container_t * container = NULL;

    LOG_PROC_ENTRY();

    rc = translate_handle(object_handle, (void *) &object, &type);

    if (rc == HANDLE_MANAGER_NO_ERROR) {

        /* Containers can only contain storage objects. */
        if ((type == DISK) ||
            (type == SEGMENT) ||
            (type == REGION) ||
            (type == EVMS_OBJECT)) {

            /* The object must not be part of a volume. */
            if (object->volume == NULL) {

                /*
                 * If the object has any parents, they must all be free
                 * space.
                 */
                rc = ForEachItem(object->parent_objects,
                                 check_for_free_space,
                                 NULL,
                                 TRUE);

                if (rc != 0) {
                    LOG_ERROR("The object cannot be transferred since it has parent(s) that are not free space.\n");
                }

            } else {
                LOG_ERROR("The object cannot be transferred since it is part of a volume.\n");
                rc = EINVAL;
            }

        } else {
            LOG_ERROR("The object handle is not for a storage object.\n");
            rc = EINVAL;
        }

    } else {
        LOG_ERROR("The object handle parameter is not valid.  translate_handle() returned error %d.\n", rc);
    }

    if (rc == 0) {
        /* If a plugin_handle was given, get the plug-in record. */
        if (plugin_handle != 0) {
            rc = translate_handle(plugin_handle, (void *) &plugin, &type);

            if (rc == 0) {
                if (type == PLUGIN) {
                    /*
                     * Check to make sure the plug-in supports containers, i.e.,
                     * it has a function table for container functions.
                     */
                    if (plugin->container_functions == NULL) {
                        LOG_ERROR("The target plug-in does not support container functions.\n");
                        rc = EINVAL;
                    }

                } else {
                    LOG_ERROR("The plug-in handle parameter is not a handle for a plug-in.\n");
                    rc = EINVAL;
                }

            } else {
                LOG_ERROR("The plug-in handle parameter is not valid.  translate_handle() returned error %d.\n", rc);
            }
        }
    }

    if (rc == 0) {
        /* If a container_handle was given, get the container. */
        if (container_handle != 0) {
            rc = translate_handle(container_handle, (void *) &container, &type);

            if (rc == 0) {
                if (type != CONTAINER) {
                    LOG_ERROR("The container handle parameter is not a handle for a container.\n");
                    rc = EINVAL;
                }

            } else {
                LOG_ERROR("The container handle parameter is not valid.  translate_handle() returned error %d.\n", rc);
            }
        }
    }

    if (rc == 0) {
        /*
         * If both a plug-in and a container were given, make sure the plug-in
         * is the one that manages the container.
         */
        if ((plugin != NULL) && (container != NULL) &&
            (container->plugin != plugin)) {
            LOG_ERROR("Parameter conflict.  The specified container is not managed by the specified plug-in.\n");
            rc = EINVAL;
        }
    }

    /* If all went well, return the pointers. */
    if (rc == 0) {
        *p_object    = object;
        *p_plugin    = plugin;
        *p_container = container;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * evms_transfer will transfer an object to a particular plug-in and/or
 * container.  If container is 0, the plug-in will assign the object to its
 * default or unassigned container.
 */
int evms_transfer(object_handle_t  object_handle,
                  plugin_handle_t  plugin_handle,
                  object_handle_t  container_handle,
                  option_array_t * options) {

    int rc = 0;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {
        storage_object_t    * object = NULL;
        plugin_record_t     * plugin = NULL;
        storage_container_t * container = NULL;

        rc = validate_transfer_parameters(object_handle, plugin_handle, container_handle,
                                          &object, &plugin, &container);
        if (rc == 0) {
            /*
             * If a target plug-in was not specified, then the target plug-in is
             * the one that manages the target container, if a container was
             * specified.
             */
            if (plugin == NULL) {
                if (container != NULL) {
                    plugin = container->plugin;
                }
            }

            /* Check if the object is currently in a container. */
            if (object->consuming_container != NULL) {

                if (object->consuming_container->plugin == plugin) {

                    /* This is an intra-plugin transfer. */
                    rc = plugin->container_functions->transfer_object(object, container, options);

                    LOG_DEBUG("transfer_object() returned error code %d.\n", rc);

                } else {
                    /*
                     * This is an inter-plugin transfer.  Tell the plug-in that
                     * currently owns the container to which the object belongs
                     * to remove it from its current container.
                     */
                    rc = object->consuming_container->plugin->container_functions->remove_object(object);

                    LOG_DEBUG("remove_object() returned error code %d.\n", rc);

                    if (rc == 0) {
                        /*
                         * If a target plug-in/container was specified, add the
                         * object to the target plug-in/container.
                         */
                        if (plugin != NULL) {
                            rc = plugin->container_functions->add_object(object, container, options);

                            LOG_DEBUG("add_object() returned error code %d.\n", rc);
                        }

                    } else {
                        LOG_WARNING("The %s plug-in failed to remove the object with handle %d from its current container.  The error code was %d.\n", object->consuming_container->plugin->short_name,object_handle, rc);
                    }
                }

            } else {
                if (plugin != NULL) {
                    /*
                     * The object does not belong to a container.
                     * Just add it to the new container.
                     */
                    rc = plugin->container_functions->add_object(object, container, options);

                    LOG_DEBUG("add_object() returned error code %d.\n", rc);

                } else {
                    LOG_WARNING("The object does not belong to a container and no target container or plug-in was specified.  There is nothing for evms_transfer() to do.\n");
                    rc = EINVAL;
                }
            }
        }

        if (rc == 0) {
            changes_pending = TRUE;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * evms_get_object_list returns a pointer to a handle_array_t with handles for
 * storage objects, optionally filtering on the object type, the data type of
 * the object, the * feature that manages the object, and some flags.  If any of
 * the filtering parameters is 0, that filter is not used.
 */
int evms_get_object_list(object_type_t         object_type,
                         data_type_t           data_type,
                         plugin_handle_t       plugin_handle,
                         object_search_flags_t flags,
                         handle_array_t    * * object_handle_list) {
    int rc = 0;
    ADDRESS object = NULL;
    object_type_t type;
    plugin_record_t * plugin = NULL;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        if (plugin_handle != 0) {
            /* Translate the handle for the feature to make sure it is valid */
            /* and to get the plugin_record_t for the feature. */
            rc = translate_handle(plugin_handle,
                                  &object,
                                  &type);

            if (rc == HANDLE_MANAGER_NO_ERROR) {
                if (type == PLUGIN) {
                    plugin = (plugin_record_t *) object;
                } else {
                    rc = EINVAL;
                }
            }
        }

        if (rc == 0) {
            dlist_t object_list;

            /*
             * Call the internal version of engine_get_object_list.  "plugin"
             * will be NULL if the caller did not specify a plug-in, else it
             * will be a pointer to the plug-in's plugin_record_t.
             */
            rc = engine_get_object_list(object_type, data_type, plugin, flags, &object_list);

            if (rc == 0) {
                rc = make_user_handle_array(object_list, object_handle_list);

                /* We are finished with the list that was returned by */
                /* Engine_GetObjectList. */
                DestroyList(&object_list, FALSE);
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


