/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: aixregmgr
 * File: aixregmgr.h
 */ 

#ifndef _AIX_REGION_MANAGER_H_
#define _AIX_REGION_MANAGER_H_ 1


typedef u_int8_t boolean;

#define	AIX_NAME		"aix"

// Defines from the AIX LVM source code.
#define LVM_MAXLPS		65535	// max number of logical partitions allowed in an LV
#define LVM_NAMESIZ		64	// maximum size for the logical volume name
#define LVM_NUMCOPIES		3	// max number of copies allowed of a logical partition
#define LVM_MAXVGS		255
#define LVM_MAXPVS		32
#define LVM_MAXLVS		256
#define AIX_MIN_BLOCK_SIZE	4096
#define VGSA_BT_PV		127
#define NBPI			32
#define OFFSET_CONSTANT		144
#define SLEEP_TIME		0
#define AIX_DEFAULT_MIRRORING	1
#define AIX_FIRST_MIRROR	2
#define AIX_MAX_MIRRORS		3	// AIX defines ALL copies as mirrors - 3 mirrors MAX - 1 orig and 2 copies
#define AIX_MIN_MIRROR_POOL	10
#define AIX_MIRROR_POOL_CHANGE	10

// Sector locations of AIX metadata structures
#define PSN_IPL_REC		0
#define PSN_LVM_REC		7
#define PSN_LVE_OFFSET		1
#define PSN_PVH_OFFSET		17
#define PSN_PVH_INCREMENT	34
#define PSN_NAMELIST_OFFSET	33	// Negative offset from end of VGDA
#define AIX_SECTOR_SIZE		512
#define MAX_PPENT_SECTOR	16
#define MAX_SECTORS_LV_ENTRIES	16
#define MAXLVS_OFFSET		16
#define MAX_SECTORS_NAMELIST	32


// AIX On-Disk Metadata Structures


// AIX unique identifiers. Four 4-byte numbers
typedef struct _unique_id {
	u_int32_t	word1;
	u_int32_t	word2;
	u_int32_t	word3;
	u_int32_t	word4;
} unique_id;


// AIX timestamps.
typedef struct _timestruc_t {
	int	tv_sec;
	int	tv_nsec;
} timestruc_t;


// AIX IPL Record
// Always the first sector of an AIX physical volume
typedef struct ipl_rec_area {
	unsigned int	IPL_record_id;	// This physical volume contains a
					// valid IPL record if and only if
					// this field contains IPLRECID
#define IPL_REC_ID	0xc9c2d4c1	// Value is EBCIDIC 'IBMA'
	char		reserved1[20];
	unsigned int	formatted_cap;	// Formatted capacity. The number of
					// sectors available after formatting
					// The presence or absence of bad
					// blocks does not alter this value.
	char		last_head;	// THIS IS DISKETTE INFORMATION
					// The number of heads minus 1. Heads
					// are number from 0 to last_head.
	char		last_sector;	// THIS IS DISKETTE INFORMATION
					// The number of sectors per track.
					// Sectors are numbered from 1 to
					// last_sector.
	char		reserved2[6];
	unsigned int	boot_code_length; // Boot code length in sectors. A 0
					// value implies no boot code present
	unsigned int	boot_code_offset; // Boot code offset. Must be 0 if no
					// boot code present, else contains 
					// byte offset from start of boot
					// code to first instruction.
	unsigned int	boot_lv_start;	// Contains the PSN of the start of
					// the BLV.
	unsigned int	boot_prg_start;	// Boot code start. Must be 0 if no
					// boot code present, else contains
					// the PSN of the start of boot code.
	unsigned int	boot_lv_length;	// BLV length in sectors.
	unsigned int	boot_load_add;	// 512 byte boundary load address for
					// boot code.
	char		boot_frag;	// Boot code fragmentation flag. Must
					// be 0 if no fragmentation allowed,
					// else must be 0x01.
	char		boot_emulation;	// ROS network emulation flag
					// 0x0 => not an emul support image
					// 0x1 => ROS network emulation code
					// 0x2 => AIX code supporting ROS emul
	char		reserved3[2];
	ushort		basecn_length;	// Number of sectors for base
					// customization. Normal mode.
	ushort		basecs_length;	// Number of sectors for base
					// customization. Service mode.
	unsigned int	basecn_start;	// Starting PSN value for base
					// customization. Normal mode.
	unsigned int	basecs_start;	// Starting PSN value for base 
					// customization. Service mode.
	char		reserved4[24];
	unsigned int	ser_code_length;// Service code length in sectors.
					// A 0 value implies no service code
					// present.
	unsigned int	ser_code_offset;// Service code offset. Must be 0 if
					// no service code is present, else
					// contains byte offset from start of
					// service code to first instruction.
	unsigned int	ser_lv_start;	// Contains the PSN of the start of
					// the SLV.
	unsigned int	ser_prg_start;	// Service code start. Must be 0 if
					// service code is not present, else
					// contains the PSN of the start of
					// service code.
	unsigned int	ser_lv_length;	// SLV length in sectors.
	unsigned int	ser_load_add;	// 512 byte boundary load address for
					// service code.
	char		ser_frag;	// Service code fragmentation flag.
					// Must be 0 if no fragmentation
					// allowed, else must be 0x01.
	char		ser_emulation;	// ROS network emulation flag
					// 0x0 => not an emul support image
					// 0x1 => ROS network emulation code
					// 0x2 => AIX code supporting ROS emul
	char		reserved5[2];
	unique_id	pv_id;		// The unique identifier for this
					// physical volume.
	char		dummy[512 - 128 - sizeof(unique_id)];
} aix_ipl_rec_t;


// AIX LVM Record
// Always sector 7 (PSN_LVM_REC) of an AIX physical volume
typedef struct AIXlvm_rec_s {
	long		lvm_id;		// LVM id field which identifies whether the PV is a member of a volume group
#define AIX_LVM_LVMID	0x5F4C564D	// LVM id field of ASCII "_LVM"

	unique_id	vg_id;		// The id of the volume group to which this physical volume belongs
	long		lvmarea_len;	// The length of the LVM reserved area
	long		vgda_len;	// Length of the volume group descriptor area
	daddr_t		vgda_psn[2];	// The physical sector numbers of the beginning of the volume group descriptor area copies on this disk 
	daddr_t		reloc_psn;	// The physical sector number of the beginning of a pool of blocks
					// (located at the end of the PV) which are reserved for the relocation of bad blocks
	long		reloc_len;	// The length in number of sectors of the pool of bad block relocation blocks
	short int	pv_num;		// The physical volume number within the volume group of this physical volume
	short int	pp_size;	// The size in bytes for the partition, expressed as a power of 2 (i.e., the partition size is 2 to the power pp_size)
	long		vgsa_len;	// Length of the volume group status area
	daddr_t		vgsa_psn[2];	// The physical sector numbers of the beginning of the volume group status area copies on this disk
	short int	version;	// The version number of this volume group descriptor and status area
#define LVM_VERSION_1		1	// first version - AIX 3.0
#define LVM_STRIPE_ENHANCE	2	// version with striped lv's - AIX 4.1
#define LVM_1024_PPSIZE		3	// ppsizes of 512 and 1024
#define LVM_GT_1016		4	// version with support for > 1016 pps/pv
#define	LVM_MAX_VERSION		LVM_GT_1016 // max version #

	char res1 [450];		// reserved area
} aix_lvm_rec_t;



// AIX Volume Group Status Area (VGSA)
typedef struct _vgsa_area {
	timestruc_t	b_tmstamp;	// Beginning timestamp
	unsigned int	pv_missing[(LVM_MAXPVS + (NBPI -1)) / NBPI];	// Bit per PV
	unsigned char	stalepp[LVM_MAXPVS][VGSA_BT_PV];
	short		factor;
	char		resv[10];	// Padding
	timestruc_t	e_tmstamp;	// Ending timestamp
} vgsa_area;


// AIX Volume Group Header
// Beginning of the Volume Group Descriptor Area (VGDA)
typedef struct _vg_header {
	timestruc_t	vg_timestamp;	// Time of last update
	unique_id	vg_id;		// Unique id for volume group
	short		numlvs;		// Number of lvs in vg
	short		maxlvs;		// Max number of lvs allowed in vg
	short		pp_size;	// Size of pps in the vg
	short		numpvs;		// Number of pvs in the vg
	short		total_vgdas;	// Number of copies of vg
					// descriptor area on disk
	short		vgda_size;	// Size of volume group descriptor
	short		bigvg;
	short		quorum;
	short		auto_varyon;
	int		checksum;
	int		bigda_size;
} vg_header;
 

// AIX Logical Volume Entry
// Array of LV Entries immediately follows the VG header.
typedef struct _lv_entries {
	short		lvname;		// Name of LV
	short		res1;		// Reserved area
	int		maxsize;	// Maximum number of partitions allowed
	char		lv_state;	// State of logical volume
	char		mirror;		// None, single, or double
	short		mirror_policy;	// Type of writing used to write
	int		num_lps;	// Number of logical partitions on the lv - Base 1
	char		permissions;	// Read-write or read-only
	char		bb_relocation;	// Specifies if bad block
					// relocation is desired
	char		write_verify;	// Verify all writes to the LV
	char		mirwrt_consist;	// Mirror write consistency flag
	unsigned short	stripe_exp;	// Stripe size in exponent value
	unsigned short	striping_width;	// Stripe width
	unsigned short	lv_avoid;
	unsigned short	child_minor_num;
	char		res4[4];	// Reserved area on disk
} lv_entries;

 
// AIX Physical Volume Header
// First PV Header starts immediately after the array of LV entries.
// Each PV header is immeidately followed by the PP map for that PV.
typedef struct _pv_header {
	unique_id	pv_id;		// Unique identifier of PV
	unsigned short	pp_count;	// Number of physical partitions
					// on PV
	char		pv_state;	// State of physical volume
	char		res1;		// Reserved area on disk
	daddr_t		psn_part1;	// Physical sector number of 1st pp
	short		pvnum_vgdas;	// Number of vg descriptor areas
					// on the physical volume
	short		pv_num;		// PV number
	long		res2;		// Reserved area on disk
} pv_header;
 

// AIX Physical Partition Entry.
// PP entries immediately follow the PV header of the PV they belong to.
typedef struct _pp_entries {
	short	lv_index;	// Index to lv pp is on
	short	res_1;		// Reserved area on disk
	long	lp_num;		// Logical part. number
	char	copy;		// The copy of the logical partition
				// that this pp is allocated for
	char	pp_state;	// Current state of pp
	char	fst_alt_vol;	// PV where partition allocation for
				// first mirror begins
	char	snd_alt_vol;	// PV where partition allocation for
				// second mirror begins
	short	fst_alt_part;	// Partition to begin first mirror
	short	snd_alt_part;	// Partition to begin second mirror 
	double	res_3;		// Reserved area on disk
	double	res_4;		// Reserved area on disk
} pp_entries;


// List of names of AIX Logical Volumes
// 32 sectors immediately before the VG trailer.
typedef struct _namelist {
	char	name[LVM_MAXLVS][LVM_NAMESIZ];
} namelist;
 

// AIX Volume Group Trailer
// Last sector of the VGDA.
typedef struct _vg_trailer {
	timestruc_t	timestamp;	// Time of last update
	short		concurrency;	// MS Nibble = concurrent capable
					// LS Nibble = concurrent auto-varyon
	short		res_2;		// Reserved area on disk
	int		res_3;		// Reserved area on disk
	double		res_4;		// Reserved area on disk
	double		res_5;		// reserved area on disk
} vg_trailer;


// End of AIX metadata structures.


// AIX Region Manager internal data structures.


// Parent structure for all VGDA metadata structures.
typedef struct aix_vgda_s {
	vg_header	* vg_head;
	lv_entries	* lv_array;
	pv_header	* pv_headers[LVM_MAXPVS];
	pp_entries	* pp_array[LVM_MAXPVS];
	namelist	* lv_names;
	vg_trailer	* vg_tail;
} aix_vgda_t;


// Structure describing one logical-to-physical-partition mapping.
typedef struct aix_lp_map_entry_s {
	storage_object_t	* object;
	unsigned long		pp_number;
	unsigned long		flags;
} aix_lp_map_entry_t;

#define AIX_LP_MISSING	(1 << 0)


// Private data describing an AIX LV.
typedef struct aix_region_data_s {
	lv_entries		* lv;
	aix_lp_map_entry_t	* lp_map[AIX_MAX_MIRRORS];
	unsigned int		flags;
} aix_region_data_t;

#define AIX_REGION_EXPORTED	(1 << 0)
#define AIX_REGION_INCOMPLETE	(1 << 1)


// Private data describing an AIX PV.
typedef struct aix_pv_data_s {
	aix_ipl_rec_t		* ipl;
	aix_lvm_rec_t		* lvm;
	vg_header		* vg_head[2];
	vg_trailer		* vg_tail[2];
	pv_header		* pv_head;
	pp_entries		* pp_map;
	unsigned long		pv_state;	// PV_STATE_*
	unsigned long		flags;
} aix_pv_data_t;

#define AIX_PV_STATE_VALID		0	// Both VGDAs are valid and match.
#define AIX_PV_STATE_FIRST_VGDA		1	// Only the first VGDA is valid.
#define AIX_PV_STATE_SECOND_VGDA	2	// Only the second VGDA is valid.
#define AIX_PV_STATE_EITHER_VGDA	-1	// Both VGDAs are valid, but do not match each other.


// Private data describing an AIX VG.
typedef struct aix_container_data_s {
	aix_vgda_t		* vgda;
	vgsa_area		* vgsa;
	storage_object_t	* freespace;
	storage_object_t	* regions[LVM_MAXLVS];
	unsigned long		pp_count;
	unsigned long		flags;		// AIX_CONTAINER_*
} aix_container_data_t;

#define AIX_CONTAINER_INVALID_VG_HEADER	(1 << 0)


// Global symbols exported from aixregmgr.c
extern plugin_record_t * evms_plugin_records[];


// Global data used by aixregmgr
extern engine_functions_t		* aix_engine;
extern plugin_record_t			AIX_Plugin;
extern plugin_record_t			* aix_plugin;
extern dlist_t				aix_container_list;


// Log message macros
#define MESSAGE(msg, args...)		aix_engine->user_message(aix_plugin, NULL, NULL, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_CRITICAL(msg, args...)	aix_engine->write_log_entry(CRITICAL,	aix_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_SERIOUS(msg, args...)	aix_engine->write_log_entry(SERIOUS,	aix_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_ERROR(msg, args...)		aix_engine->write_log_entry(ERROR,	aix_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_WARNING(msg, args...)	aix_engine->write_log_entry(WARNING,	aix_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG(msg, args...)		aix_engine->write_log_entry(DEFAULT,	aix_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_DETAILS(msg, args...)	aix_engine->write_log_entry(DETAILS,	aix_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_DEBUG(msg, args...)		aix_engine->write_log_entry(DEBUG,	aix_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_EXTRA(msg, args...)		aix_engine->write_log_entry(EXTRA,	aix_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_PROC(msg, args...)		aix_engine->write_log_entry(ENTRY_EXIT,	aix_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_EVERYTHING(msg, args...)	aix_engine->write_log_entry(EVERYTHING,	aix_plugin, "%s: " msg, __FUNCTION__ , ## args)

#define LOG_ENTRY			LOG_PROC("Entering\n")
#define LOG_EXIT(x)			LOG_PROC("Exiting: rc = %d\n", x)
#define RETURN(x)			do { LOG_EXIT(x); return(x); } while (0)

#define BUG()				do {	LOG_CRITICAL("BUG at line %d of %s\n", __LINE__, __FILE__); \
						fprintf(stderr, "BUG at line %d of %s. Function %s\n", __LINE__, __FILE__, __FUNCTION__); \
						exit(-EFAULT); } while (0)


#define COMPARE_UNIQUE_IDS(id1, id2)	( (id1).word1 == (id2).word1 && \
					  (id1).word2 == (id2).word2 && \
					  (id1).word3 == (id2).word3 && \
					  (id1).word4 == (id2).word4 )
#define COMPARE_TIMESTAMPS(t1, t2)	( (t1).tv_sec  == (t2).tv_sec && \
					  (t1).tv_nsec == (t2).tv_nsec )

#define PP_SIZE_TO_VSECTORS(pp_size)	(1 << ((pp_size) - EVMS_VSECTOR_SIZE_SHIFT))

// FOR_EACH: Requires an "int rc" variable in the calling function.
//           The "list" variable must be a dlist_t type.
//           The "item" variable can be any pointer type.
//           No items may be removed from the list during this loop.
#define FOR_EACH(item, list)	for ( rc = GoToStartOfList(list); \
					!rc && (item = aix_get_list_item(list)); \
					rc = NextItem(list) )


#include "aix_containers.h"
#include "aix_dlist.h"
#include "aix_io.h"
#include "aix_pv.h"
#include "aix_regions.h"

#endif

